/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RegularJUnitProvider.java,v 1.6 2007/10/30 13:45:37 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.event.IJUnitTestSuiteFactoryEvent;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.event.IJUnitTestSuiteFactoryListener;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.event.JUnitTestSuiteCreatedEvent;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.event.JUnitTestSuiteDetachedEvent;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.event.JUnitTestSuiteFactoryEventManager;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNodeListener;
import org.eclipse.hyades.test.ui.navigator.ITypeProvider;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderContext;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderProxyNode;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jdt.core.ElementChangedEvent;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IElementChangedListener;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaElementDelta;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.dom.CompilationUnit;

/**
 * A provider showing Java projects content, with non-JUnit classes filtered out.
 * @author jcanches
 * @author Paul E. Slauenwhite
 * @version October 20, 2007
 * @since 4.3
 */
public class RegularJUnitProvider implements ITypeProvider, IElementChangedListener, IDisposable, IJUnitTestSuiteFactoryListener {

    /** The file proxy manager used to cache file proxies */
    protected IFileProxyManager fileProxyManager;
	protected IProxyNodeListener refresher;

	private Map projectToProxyMap = new HashMap();
	
	public RegularJUnitProvider() {
        
	}
	
    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.ITypeProvider#get(org.eclipse.core.resources.IProject, java.lang.String)
     */
    public ITypeProviderProxyNode get(IProject project, String type) {
    	IJavaProject jproject = JavaCore.create(project);
        if(!jproject.exists()) return null;
        RegularJUnitProviderProxyNode proxy = (RegularJUnitProviderProxyNode) projectToProxyMap.get(jproject);
        if(proxy == null) {
        	proxy = new RegularJUnitProviderProxyNode(this, jproject, type, project);
    		projectToProxyMap.put(jproject, proxy);
        }
        if (proxy != null && proxy.getChildren().length == 0) return null;
        return proxy;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.ITypeProvider#init(org.eclipse.hyades.test.ui.navigator.ITypeProviderContext)
     */
    public void init(ITypeProviderContext context) {
        fileProxyManager = context.getFileProxyManager();
        if (!context.isStaticView()) {
        	refresher = context.getProxyNodeListener();
        } else {
        	refresher = null;
        }
		if (refresher != null) {
			JavaCore.addElementChangedListener(this, ElementChangedEvent.POST_CHANGE);
	        JUnitTestSuiteFactoryEventManager.getInstance().addListener(this);
		}
    }

    public synchronized void dispose() {
    	// disposed must be set to true before cancelling the current job, otherwise another
    	// job may be triggered while we cancel this one.
    	if (refresher != null) {
	    	JUnitTestSuiteFactoryEventManager.getInstance().removeListener(this);
	        JavaCore.removeElementChangedListener(this);
    	}
        Iterator it = projectToProxyMap.values().iterator();
        while (it.hasNext()) {
			RegularJUnitProviderProxyNode node = (RegularJUnitProviderProxyNode) it.next();
			node.dispose();
		}
        projectToProxyMap.clear();
    }

	public void elementChanged(ElementChangedEvent event) {
		IJavaElementDelta[] children = event.getDelta().getAffectedChildren();
		for(int i = 0; i < children.length; i++) {
			IJavaProject jproject = (IJavaProject) children[i].getElement();
			RegularJUnitProviderProxyNode node = (RegularJUnitProviderProxyNode) projectToProxyMap.get(jproject);
			if (children[i].getKind() == IJavaElementDelta.REMOVED) {
				// Project deleted: forget the reference we hold to its proxy
				if (node != null) {
					projectToProxyMap.remove(jproject);
				}
			}
			if (node != null) {
				int previousChildrenCount = node.getChildren().length;
				Object lowestChange = node.elementChanged(children[i]);
				if (lowestChange == node) {
					int newChildrenCount = node.getChildren().length;
					if ((previousChildrenCount == 0 || newChildrenCount == 0) && previousChildrenCount != newChildrenCount) {
						lowestChange = jproject.getProject();
					}
				}
				if (lowestChange != null) {
					refresher.nodeChanged(lowestChange);
				}
			}
		}
	}
	
	/**
	 * Convenience method to get the proxy node associated to an arbitrary java element.
	 * @param element An opened java element.
	 * @return A proxy node, or null if no proxy node is associated to the java element.
	 */
	protected JavaElementProxyNode getProxyNode(IJavaElement element) {
		if (element instanceof IJavaProject) {
			return (RegularJUnitProviderProxyNode) projectToProxyMap.get(element);
		}
		IJavaElement parent = element.getParent();
		if (parent != null) {
			JavaElementProxyNode parentNode = getProxyNode(parent);
			if (parentNode instanceof JavaParentElementProxyNode) {
				return ((JavaParentElementProxyNode)parentNode).getChildProxy(element);
			}
		}
		return null;
	}
	
	/**
	 * Basic JavaElementDelta that describes a change to an element, with a set of children
	 * deltas.
	 * 
     * @author jcanches
     * @author Paul E. Slauenwhite
     * @version October 20, 2007
     * @since 4.3
	 */
	static class BasicJavaElementChangeDelta implements IJavaElementDelta {
		// These constants should not collide with JDT's ones (hopefully, unless JDT defines
		// about a dozen of new event flags, or unless they redistribute their constants).
		public static final int F_TEST_SUITE_ATTACHED = 0x80000000;
		public static final int F_TEST_SUITE_DETACHED = 0x40000000;
		private static IJavaElementDelta[] NO_CHIDLREN = new IJavaElementDelta[0];
		private IJavaElement element;
		private IJavaElementDelta[] children;
		private int flags;
		public BasicJavaElementChangeDelta(IJavaElement element, IJavaElementDelta[] children, int flags) {
			this.element = element;
			if (children == null) this.children = NO_CHIDLREN;
			else this.children = children;
			this.flags = flags;
		}
		public IJavaElementDelta[] getAddedChildren() {
			return NO_CHIDLREN;
		}
		public IJavaElementDelta[] getAffectedChildren() {
			return children;
		}
		public IJavaElementDelta[] getChangedChildren() {
			return children;
		}
		public CompilationUnit getCompilationUnitAST() {
			return null;
		}
		public IJavaElement getElement() {
			return element;
		}
		public int getFlags() {
			return flags;
		}
		public int getKind() {
			return IJavaElementDelta.CHANGED;
		}
		public IJavaElement getMovedFromElement() {
			return null;
		}
		public IJavaElement getMovedToElement() {
			return null;
		}
		public IJavaElementDelta[] getRemovedChildren() {
			return NO_CHIDLREN;
		}
		public IResourceDelta[] getResourceDeltas() {
			return null;
		}
		public IJavaElementDelta[] getAnnotationDeltas(){
			return null;
		}
	}
	
	private IJavaElementDelta makeDeltaTree(IJUnitTestSuiteFactoryEvent event) {
		IJavaElementDelta delta = null;
		ICompilationUnit cu = null;
		if (event instanceof JUnitTestSuiteCreatedEvent) {
			JUnitTestSuiteCreatedEvent e = (JUnitTestSuiteCreatedEvent)event;
			cu = e.getCompilationUnit();
			delta = new BasicJavaElementChangeDelta(cu, null, BasicJavaElementChangeDelta.F_TEST_SUITE_ATTACHED);
		} else if (event instanceof JUnitTestSuiteDetachedEvent) {
			JUnitTestSuiteDetachedEvent e = (JUnitTestSuiteDetachedEvent)event;
			cu = e.getCompilationUnit();
			delta = new BasicJavaElementChangeDelta(cu, null, BasicJavaElementChangeDelta.F_TEST_SUITE_DETACHED);
		}
		if (delta != null && cu != null) {
			return makeDeltaTree(cu.getParent(), delta);
		}
		return null;
	}
	
	private IJavaElementDelta makeDeltaTree(IJavaElement element, IJavaElementDelta child) {
		IJavaElementDelta delta = new BasicJavaElementChangeDelta(element, new IJavaElementDelta[] {child}, IJavaElementDelta.F_CHILDREN);
		if (element instanceof IJavaModel) return delta;
		else return makeDeltaTree(element.getParent(), delta);
	}

	public void onEvent(IJUnitTestSuiteFactoryEvent event) {
		// Convert this internal event to a java element delta, so the event can be processed
		// using the regular event processing.
		IJavaElementDelta delta = makeDeltaTree(event);
		if (delta != null) {
			elementChanged(new ElementChangedEvent(delta, IJavaElementDelta.CHANGED));
		}
	}

}
