/********************************************************************** 
 * Copyright (c) 2005, 2007 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: SourceFolderProxyNode.java,v 1.8 2007/06/01 14:51:54 jgout Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator;

import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IExtendedProxyNode;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IResourceChangeListenerProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestFolderContentValidator;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.ISharedImages;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.swt.graphics.Image;

/** Proxy Node for Source folder of a java project. 
 * @author jgout
 * @since 4.0
 */
public class SourceFolderProxyNode implements IResourceChangeListenerProxyNode, IExtendedProxyNode {

    private IPackageFragmentRoot root;
    private IProxyNode[] children;
    private Object parent;
    private static final IProxyNode[] NO_CHILDREN = new IProxyNode[0];
    private ITestFolderContentValidator validator;
    private IFileProxyManager fileProxyManager;
    
    public static SourceFolderProxyNode create(IPackageFragmentRoot root, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, Object parent) {
        SourceFolderProxyNode proxy = new SourceFolderProxyNode(root, validator, fileProxyManager, parent);
        if (proxy.getChildren().length > 0) {
            return proxy;
        }
        return null;
    }

    protected SourceFolderProxyNode(IPackageFragmentRoot root, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, Object parent) {
        init(root, validator, fileProxyManager, parent, this);
    }
    
    protected SourceFolderProxyNode(IPackageFragmentRoot root, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, Object parent, Object claimedParent) {
    	init(root, validator, fileProxyManager, parent, claimedParent);
    }

	private void init(IPackageFragmentRoot root,
			ITestFolderContentValidator validator,
			IFileProxyManager fileProxyManager, Object parent, Object claimedParent) {
		this.parent = parent;
        this.root = root;
        this.validator = validator;
        this.fileProxyManager = fileProxyManager;
        List _children = new LinkedList();
        try {
            //- first handle non java files and folders.
            Object[] nonJava = root.getNonJavaResources();
            for (int i = 0; i < nonJava.length; i++) {
                Object res = nonJava[i];
                if(res instanceof IFile && validator.isFileOk((IFile)res)) {
                    IProxyNode proxy = fileProxyManager.getProxy((IFile)res, claimedParent);
                    if(proxy != null && validator.isProxyOk(proxy)) {
                        _children.add(proxy);
                        fileProxyManager.cacheProxy((IFile)res, proxy);
                    }
                } else if(res instanceof IContainer) {
                    //- this is basic folder under a package (there should be an exclude rule on the project)
                    IProxyNode proxy = JUnitFolderProxyNode.create((IContainer)res, validator, fileProxyManager, false, claimedParent);
                    if(proxy != null && validator.isProxyOk(proxy)) {
                        _children.add(proxy);
                    }
                }
            }
            //- then handle the packages
            IJavaElement[] fragments = root.getChildren();
            for (int i = 0; i < fragments.length; i++) {
                IPackageFragment frag = (IPackageFragment)fragments[i];
                IProxyNode proxy = PackageProxyNode.create(frag, validator, fileProxyManager, claimedParent);
                if(proxy != null) {
                    _children.add(proxy);
                }
            }
            children = (IProxyNode[]) _children.toArray(new IProxyNode[_children.size()]);
        } catch (JavaModelException e) {
            ToolsUiPlugin.logError(e);
            children = NO_CHILDREN;
        }
	}
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getText()
     */
    public String getText() {
        return root.getElementName();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getImage()
     */
    public Image getImage() {
        return JavaUI.getSharedImages().getImage(ISharedImages.IMG_OBJS_PACKFRAG_ROOT);
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getParent()
     */
    public Object getParent() {
        return parent;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getChildren()
     */
    public IProxyNode[] getChildren() {
        return children;
    }

    /* (non-Javadoc)
     * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
     */
    public Object getAdapter(Class adapter) {
        if(adapter == IContainer.class) {
            try {
                return root.getUnderlyingResource();
            } catch (JavaModelException e) {
                ToolsUiPlugin.logError(e);
                return null;
            }
        }
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxy#getUnderlyingResource()
     */
    public IResource getUnderlyingResource() {
        try {
            return root.getUnderlyingResource();
        } catch (JavaModelException e) {
            ToolsUiPlugin.logError(e);
            return null;
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxy#getIdentifier()
     */
    public String getIdentifier() {
        return root.getElementName();
    }
    
    /**
     * Returns the child proxy associated to the given resource (folder or file)
     * @param res a resource (folder or file)
     * @return the proxy that represents the given resource
     */
    private IProxyNode getChildProxy(IResource res) {
        for (int i = 0; i < children.length; i++) {
            if (res.equals(children[i].getUnderlyingResource())) {
                return children[i];
            }
        }
        return null;
    }

    private void replaceChildProxy(IProxyNode oldProxy, IProxyNode newProxy) {
        for (int i = 0; i < children.length; i++) {
            if(children[i] == oldProxy) {
                children[i] = newProxy;
                break;
            }
        }
    }
 
    private boolean addChildProxy(IProxyNode proxy) {
        if(proxy == null) return false;
        IProxyNode[] newChildren = new IProxyNode[children.length + 1];
        for (int i = 0; i < children.length; i++) {
            //- if the proxy to add is already a child do not add it
            if(children[i].equals(proxy)) {
                return false;
            }
            newChildren[i] = children[i];
        }
        newChildren[children.length] = proxy;
        //- change the children of the current proxy node
        children = newChildren;
        return true;
    }

    private void removeChildProxy(IProxyNode proxy) {
        IProxyNode[] newChildren = new IProxyNode[children.length - 1];
        int j = 0;
        for (int i = 0; i < children.length; i++) {
            if(children[i] != proxy) {
                newChildren[j++] = children[i];
            }
        }
        //- change the children of the current proxy node
        children = newChildren;
    }

    public IProxyNode resourceChanged(IResourceDelta rootDelta) {
        IProxyNode lowestChanged = null;
        IResourceDelta[] res = rootDelta.getAffectedChildren();
        for (int i = 0; i < res.length; i++) {
            int kind = res[i].getKind();
            IProxyNode lowestChild = null;
            if (kind == IResourceDelta.ADDED) {
                lowestChild = resourceChangedADDED(res[i]);
            } else if (kind == IResourceDelta.REMOVED) {
                lowestChild = resourceChangedREMOVED(res[i]);
            } else if (kind == IResourceDelta.CHANGED) {
                lowestChild = resourceChangedCHANGED(res[i]);
            }
            if(lowestChild != null) {
                //- the current child has reported a change in its content
                if(lowestChanged == null) {
                    //- not a child modified yet, so the lowest is this child
                    lowestChanged = lowestChild;
                } else {
                    //- this child is not the first child to change, so the lowest is the parent
                    lowestChanged = this;
                }
            }
        }
        return lowestChanged;
    }

    private IProxyNode resourceChangedCHANGED(IResourceDelta delta) {
        IProxyNode lowestChanged = null;
        IResource resource = delta.getResource();
        int resType = resource.getType();
        if (resType == IResource.FOLDER) {
            lowestChanged = resourceChangedCHANGEDFolder(delta);
        } else if(resType == IResource.FILE) {
            lowestChanged = resourceChangedCHANGEDFile(delta);
        }
        return lowestChanged;
    }

    /**
     * Treats the case of the resource changed event on a file.
     * @param delta the delta containing the change.
     * @return the lowest proxy node that contains all changes of the given delta change or
     * <code>null</code> if nothing should been done.
     */
    private IProxyNode resourceChangedCHANGEDFile(IResourceDelta delta) {
        IFile file = (IFile)delta.getResource();
        IProxyNode lowestChanged = null;
        if ((delta.getFlags() & IResourceDelta.CONTENT) != 0) {
            //- the change comes from the content of a file
            IProxyNode proxy = getChildProxy(file);
            if(proxy != null) {
                if(validator.isFileOk(file)) {
                    fileProxyManager.uncacheProxy(file);
                    //- the updated file is associated to a child of this source folder, we need to update the proxy.
                    IProxyNode newProxy = fileProxyManager.updateProxy(file, this);
                    if(validator.isProxyOk(newProxy)) {
                        replaceChildProxy(proxy, newProxy);
                        fileProxyManager.cacheProxy(file, newProxy);
                    } else {
                        //- updated proxy for the given file is no longer valid, new to remove the child of 'this'
                        removeChildProxy(proxy);
                    }
                    lowestChanged = this;
                }
            } else {
                //- even if the proxy is not yet created, we need to check that content change do not add now this proxy
                if(validator.isFileOk(file)) {
                    IProxyNode newProxy = fileProxyManager.getProxy(file, this);
                    if(validator.isProxyOk(newProxy)) {
                        boolean wasAdded = addChildProxy(newProxy);
                        if(wasAdded) {
                            fileProxyManager.cacheProxy(file, newProxy);
                            lowestChanged = this;
                        }
                    }
                }
            }
        }
        return lowestChanged;
    }

    /** 
     * Treats the case of the resource changed event on a folder.
     * @param delta the delta containing the change.
     * @return the lowest proxy node that contains all changes of the given delta change or
     * <code>null</code> if nothing should been done.
     */
    private IProxyNode resourceChangedCHANGEDFolder(IResourceDelta delta) {
        IFolder folder = (IFolder)delta.getResource();
        IProxyNode lowestChanged = null;
        IProxyNode lowestChild = null;
        IProxyNode proxy = getChildProxy(folder);
        if(proxy != null) {
            if(proxy instanceof IResourceChangeListenerProxyNode) {
                //- this means that proxy (child of 'this') is impacted by the delta change
                lowestChanged = ((IResourceChangeListenerProxyNode)proxy).resourceChanged(delta);
                //- after an update of an existing container child (folder or package) proxy node 
                //- we need to check that there are still children in that proxy in order to display it
                if(proxy.getChildren().length == 0) {
                    removeChildProxy(proxy);
                    lowestChanged = this;
                }
            }
        } else {
            //- no child folder/package found
            //- maybe the current change is contained by a new folder/package not yet created
            lowestChanged = PackageProxyNode.create(folder, delta, validator, fileProxyManager, this);
            if(lowestChanged != null) {
                //- this means that the delta has produced a not empty package proxy node, need to add it as child to 'this'
                addChildProxy(lowestChanged);
                lowestChanged = this;
            }
        }
        //- we need to propagate change to all child packages
        IResourceDelta[] affectedChildren = delta.getAffectedChildren();
        for (int i = 0; i < affectedChildren.length; i++) {
            if(affectedChildren[i].getResource().getType() == IResource.FOLDER) {
                lowestChild = resourceChangedCHANGED(affectedChildren[i]);
                if(lowestChanged != null) {
                    //- this means that this is not the first change found under the source folder (this).
                    lowestChanged = this;
                } else {
                    //- this is the first change found in that hierarchy
                    lowestChanged = lowestChild;
                }
            }
        }
        return lowestChanged;
    }

    private IProxyNode resourceChangedADDED(IResourceDelta delta) {
        IResource resource = delta.getResource();
        int resType = resource.getType();
        //- a resource (folder or file) has been added to the file system
        if (resType == IResource.FILE) {
            //- new file check if this file should kept
            if (validator.isFileOk((IFile) resource)) {
                IProxyNode newProxy = fileProxyManager.getProxy((IFile)resource, this);
                //- check if new proxy should be added as a child of current
                if (validator.isProxyOk(newProxy)) {
                    boolean wasAdded = addChildProxy(newProxy);
                    if(wasAdded) {
                        fileProxyManager.cacheProxy((IFile)resource, newProxy);
                        return this;
                    }
                }
            }
        } else if (resType == IResource.FOLDER) {
            IFolder folder = (IFolder)resource;
            IPackageFragment frag = (IPackageFragment) JavaCore.create(folder);
            if(frag != null) {
                IProxyNode proxy = PackageProxyNode.create(frag, validator, fileProxyManager, this);
                if(proxy != null) {
                    addChildProxy(proxy);
                    return this;
                }
            }
        }
        return null;
    }

    private IProxyNode resourceChangedREMOVED(IResourceDelta delta) {
        IResource resource = delta.getResource();
        int resType = resource.getType();
        IProxyNode proxy2Remove = getChildProxy(resource);
        //- a resource (folder or file) has been removed from the file system
        if (resType == IResource.FILE) {
            if(proxy2Remove != null) {
                //- the proxy to remove is one of the current node children, we need to remove it form the cache
                fileProxyManager.uncacheProxy((IFile)resource);
                //- and remove it from the children list of the current folder
                removeChildProxy(proxy2Remove);
                return this;
            } else {
                //- proxy not found among children so this file removal has no impacts in this current folder
                return null;
            }
        } else if (resType == IResource.FOLDER) {
            if(proxy2Remove != null) {
                //- recursive call to remove children
                if (proxy2Remove instanceof IResourceChangeListenerProxyNode) {
                    ((IResourceChangeListenerProxyNode)proxy2Remove).resourceChanged(delta);
                    removeChildProxy(proxy2Remove);
                    return this;
                } else {
                    //- this should never arrive, because the proxy of a folder
                    //- should be a IResourceChangeListenerProxyNode
                    UiPlugin.logError("The proxy node derived from a folder should be a IResourceChangeListenerProxyNode"); // $NON-NLS-1$ //$NON-NLS-1$
                }
            }
        }
        return null;
    }

	public IResource getCorrespondingResource() {
		return getUnderlyingResource();
	}
        
}
