/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitTestSuiteNewWizard.java,v 1.24 2010/05/05 12:32:22 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.wizard;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.tools.core.common.TestCommon;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.AutomaticDependencyUpdater;
import org.eclipse.hyades.test.tools.core.internal.java.codegen.JUnitGenerator;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitModelUpdater;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.tools.ui.internal.resources.ToolsUiPluginResourceBundle;
import org.eclipse.hyades.test.tools.ui.java.TestJavaImages;
import org.eclipse.hyades.test.ui.wizard.JavaTestLocationPage;
import org.eclipse.hyades.test.ui.wizard.TestSuiteNewWizard;
import org.eclipse.hyades.test.ui.wizard.TestSuiteNewWizard2;
import org.eclipse.hyades.ui.internal.wizard.AttributeWizardPage;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.tptp.platform.common.ui.wizard.LocationPage;

/**
 * <p>Extension of the {@link TestSuiteNewWizard} wizard for creating JUnit test suites.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Julien Canches
 * @author  Jerome Gout
 * @author  Paul E. Slauenwhite
 * @version May 4, 2010
 * @since   February 1, 2005
 */
public class JUnitTestSuiteNewWizard extends TestSuiteNewWizard2 {
	
	private SourcePage sourcePage;
	
    /**
     * Constructor for TestSuiteNewWizard
     */
    public JUnitTestSuiteNewWizard() {
        super();
        setWindowTitle(ToolsUiPluginResourceBundle.WIZ_NEW_JUNIT_TST_SUITE_PG_TTL); 
        setDefaultPageImageDescriptor(TestJavaImages.INSTANCE.getImageDescriptor(TestJavaImages.IMG_WIZBAN_NEW_JUNIT_TESTSUITE));

        IDialogSettings dialogSettings = ToolsUiPlugin.getDefault().getDialogSettings();
        IDialogSettings settings = dialogSettings.getSection("JUnitTestSuiteNewWizard"); //$NON-NLS-1$
        if (settings == null) settings = dialogSettings.addNewSection("JUnitTestSuiteNewWizard"); //$NON-NLS-1$
        setDialogSettings(settings);
    }

    /**
     * Returns the type for the test suite created by this wizard.
     * @return String
     */
    protected String getType() {
        return TestCommon.JUNIT_TEST_SUITE_TYPE;
    }

    /**
     * @see org.eclipse.jface.wizard.IWizard#addPages()
     */
    public void addPages() {
    	sourcePage.setTitle(ToolsUiPluginResourceBundle.WIZ_NEW_JUNIT_TST_SUITE_PG_TTL); 
    	addPage(sourcePage);
    	
    	super.addPages();
    	
        getLocationPage().setTitle(ToolsUiPluginResourceBundle.WIZ_NEW_JUNIT_TST_SUITE_PG_TTL); 
        getLocationPage().setDescription(ToolsUiPluginResourceBundle.WIZ_TST_SUITE_GEN_PG_LOC_DSC); 

        getAttributeWizardPage().setTitle(ToolsUiPluginResourceBundle.WIZ_NEW_JUNIT_TST_SUITE_PG_TTL); 
        getAttributeWizardPage().setDescription(ToolsUiPluginResourceBundle.WIZ_NEW_JUNIT_TST_SUITE_PG_ATT_DSC); 
    }

    /**
     * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
     */
    protected void initPages() {
    	IStructuredSelection structuredSelection = adjustSeletion();
    	
    	initLocationPage(structuredSelection);
    	initSourcePage(structuredSelection, getLocationPage());
    	
    	setAttributeWizardPage(new AttributeWizardPage("nameDescription")); //$NON-NLS-1$
    }

	/**
	 * @param structuredSelection
	 * @return
	 */
	protected void initLocationPage(IStructuredSelection structuredSelection) {
		
		JavaTestLocationPage locationPage = new JavaTestLocationPage("location", structuredSelection); //$NON-NLS-1$

		setLocationPage(locationPage);
    	
    	String fileExtension = getFileExtension();
    	if (fileExtension != null) {
    		locationPage.setFileExtension(fileExtension);
    		locationPage.setFileName(""); //$NON-NLS-1$
    	}
	}

	/**
	 * @param structuredSelection
	 * @param locationPage
	 */
	protected void initSourcePage(IStructuredSelection structuredSelection, LocationPage locationPage) {
		SourcePage sourcePage_ = new SourcePage();
    	setSourcePage(sourcePage_);
    	sourcePage_.setWizard(this);
    	sourcePage_.init(structuredSelection, locationPage);
	}
    
	protected void adjustTestSuite(ITestSuite testSuite) {
	    // Set the name of the implementor class
		String qualifiedName = this.sourcePage.getQualifiedTypeName();
		testSuite.getImplementor().setResource(qualifiedName);
		// Set the source folder
		IResource resource;
		try {
			resource = this.sourcePage.getPackageFragmentRoot().getCorrespondingResource();
		} catch (JavaModelException e) {
			resource = null;
			ToolsUiPlugin.logError(e);
		}
		if (resource != null) {
			testSuite.getImplementor().setLocation(resource.getFullPath().toString());
		}
		if (!this.sourcePage.isModelBehavior()) {
			JUnitModelUpdater.setImplementorBlock(testSuite.getImplementor(), null);
		}
		try {
			// Generate the source code
			generateCode(testSuite, new NullProgressMonitor());
		} catch (Exception e) {
			ToolsUiPlugin.logError(e);
		}
		// Don't invoke super.adjustTestSuite().
	}
	
	protected SourcePage getSourcePage() {
		return this.sourcePage;
	}
	
	protected JUnitGenerator createGenerator(ITestSuite testSuite) {
		return new JUnitGenerator(testSuite, new AutomaticDependencyUpdater(), getSourcePage().getSuperClass());
	}
	
	private void generateCode(ITestSuite testSuite, IProgressMonitor monitor) throws Exception {
		monitor.beginTask("", 4); //$NON-NLS-1$
		try {
			JUnitGenerator generator = createGenerator(testSuite);
			RefactoringStatus status = generator.checkAllConditions(new SubProgressMonitor(monitor, 1));
			if (status.hasFatalError()) {
				throw new IllegalStateException("Refactor conditions not met"); // TODO Treat this appropriately //$NON-NLS-1$
			}
			Change change = generator.createChange(new SubProgressMonitor(monitor, 1));
			change.perform(new SubProgressMonitor(monitor, 2));
		} finally {
			monitor.done();
		}
	}
	
	protected void setSourcePage(SourcePage page) {
		this.sourcePage = page;
	}
}