/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitWorkspaceContentProvider.java,v 1.10 2010/05/06 12:28:35 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.wizard;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitModelUpdater;
import org.eclipse.hyades.test.tools.core.java.IJUnitTestSuiteFactory;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.tools.ui.internal.resources.ToolsUiPluginResourceBundle;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.StandardJavaElementContentProvider;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.osgi.util.NLS;

/**
 * <p>JUnitWorkspaceContentProvider.java</cp>
 * 
 * 
 * @author  Jerome Gout
 * @author  Julien Canches
 * @author  Paul Slauenwhite
 * @version May 6, 2010
 * @since   March 25, 2005
 */
public class JUnitWorkspaceContentProvider extends StandardJavaElementContentProvider {
	
	private Map children = new HashMap();
	private IRunnableContext context;
	private IJUnitTestSuiteFactory factory;
	
	public void setJUnitTestSuiteFactory(IJUnitTestSuiteFactory factory) {
		if (factory != this.factory) {
			this.factory = factory;
			children.clear();
		}
	}
	
	private Object[] initChildren(Object element) {
		//- java project
		if (element instanceof IJavaProject) {
			IJavaProject jProject = (IJavaProject) element;
			class ComputeProjectChildrenOperation implements IRunnableWithProgress {
				private IJavaProject project;
				private List projectChildren = new ArrayList();
				public ComputeProjectChildrenOperation(IJavaProject project) {
					this.project = project;
				}
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
		            try {
						IPackageFragment [] packages = project.getPackageFragments();
						monitor.beginTask(NLS.bind(ToolsUiPluginResourceBundle.JUnitWorkspaceContentProvider_SEARCH_TASK, project.getElementName()), packages.length); 
						try {
							for(int p = 0; p < packages.length; p++) {
								// This is the time-consuming operation
								Object [] directChildren = initChildren(packages[p]);
								if(directChildren.length > 0) {
									//- this package contains children, we need to keep it
									projectChildren.add(packages[p]);
									//- store the association between this package and its children
									children.put(packages[p], directChildren);
								}
								monitor.worked(1);
								if (monitor.isCanceled()) return;
							}
						} finally {
							monitor.done();
						}
					} catch (JavaModelException e) {
						ToolsUiPlugin.logError(e);
					}
				}
				public Object[] getProjectChildren() {
					return projectChildren.toArray();
				}
			}
			ComputeProjectChildrenOperation operation = new ComputeProjectChildrenOperation(jProject);
			try {
				if (this.context != null) {
					this.context.run(/*fork*/false, /*cancelable*/true, operation);
				} else {
					operation.run(new NullProgressMonitor());
				}
			} catch (InvocationTargetException e) {
				ToolsUiPlugin.logError(e);
			} catch (InterruptedException e) {
				ToolsUiPlugin.logError(e);
			}
			return operation.getProjectChildren();
			
		//- package
		} else if (element instanceof IPackageFragment) {
			List ret = new ArrayList();
			IPackageFragment pack = (IPackageFragment)element;
			try {
				IJavaElement [] compilationUnits = pack.getChildren();
				for(int c = 0; c < compilationUnits.length; c++) {
					if(isDisplayed(compilationUnits[c])) {
						//- Actually, the current compilation unit is a JUnit script, we need to keep it
						ret.add(compilationUnits[c]);
					}
				}
				return ret.toArray();
			} catch (JavaModelException e) {
				ToolsUiPlugin.logError(e);
				return new Object[0];				
			} 
		}
		return new Object[0];
	}

	/** Checks whether the given java element is a JUnit test script
	 * @param element 
	 * @return
	 */
	private boolean isDisplayed(IJavaElement element) {
		if (element instanceof ICompilationUnit) {
			ICompilationUnit cu = (ICompilationUnit)element;
			if (factory.isSupported(cu)) {
				try {
					return JUnitModelUpdater.findTestSuiteFile((IFile)cu.getUnderlyingResource()) == null;
				} catch (JavaModelException e) {
					return false;
				} catch (CoreException e) {
					return false;
				}
			}
		}
		return false;
	}

	public Object[] getChildren(Object element) {
		if (element instanceof IJavaModel) {
			Object[] projects = super.getChildren(element);
			return filterProjects(projects);
		}
		if (!children.containsKey(element)) {
			Object[] ch = initChildren(element);
			children.put(element, ch);
			return ch;
		} else {
			return (Object[])children.get(element);
		}
	}
	
	private Object[] filterProjects(Object[] projects) {
		if (factory == null) {
			return NO_CHILDREN;
		}
		ArrayList result = new ArrayList(projects.length);
		for (int i = 0; i < projects.length; i++) {
			if (factory.isSupported((IJavaProject)projects[i])) {
				result.add(projects[i]);
			}
		}
		return result.toArray();
	}
	
	public boolean hasChildren(Object element) {
		if (element instanceof IJavaProject) {
			// Always assume true. Digging into the project would be too
			// time-consuming.
			return true;
		} else if (element instanceof IPackageFragment) {
			// This information is already in the cache.
			return getChildren(element).length != 0;
		} else if (element instanceof ICompilationUnit) {
			return false;
		}
		return super.hasChildren(element);
	}
	
	public void setRunnableContext(IRunnableContext context) {
		this.context = context;
	}
	
}
