/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SourcePage.java,v 1.26 2010/05/06 12:28:35 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.wizard;

import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.AutomaticDependencyUpdater;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.Helper;
import org.eclipse.hyades.test.tools.core.internal.java.codegen.JUnitGenerator.JUnitProjectDependencyUpdater;
import org.eclipse.hyades.test.tools.core.internal.util.PluginProjectUtil;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.tools.ui.internal.resources.ToolsUiPluginResourceBundle;
import org.eclipse.hyades.test.tools.ui.java.internal.util.ContextIds;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeHierarchy;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.wizards.NewTypeWizardPage;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.tptp.platform.common.ui.wizard.LocationPage;

/**
 * <p>Source page.</p>
 * 
 * 
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version May 6, 2010
 * @since   March 23, 2005
 */
public class SourcePage extends NewTypeWizardPage implements SelectionListener {
	
	private Button modelBehaviorRadio = null;
	private Button codeBehaviorRadio = null;
	private boolean isModelBehavior = false;
	private boolean dontAskMissingLibsAgain = false;
	private LocationPage nextPage = null;
	
	/**
	 * Field name as used in handleFieldChanged
	 */
	private static final String BEHAVIOR = "behavior"; //$NON-NLS-1$

	/**
	 * Constructor for ProjectAndSourcePage
	 */
	public SourcePage()
	{
		super(/*isClass=*/true, "ProjectAndSourcePage"); //$NON-NLS-1$
		setTitle(ToolsUiPluginResourceBundle.JAVA_GEN_WTITLE); 
		setDescription(ToolsUiPluginResourceBundle.JAVA_GEN_LDESC); 
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
	 */
	public void dispose()
	{
		super.dispose();
	}
	
	/**
	 * Should be called from the wizard with the initial selection. The wizard
	 * must have been set using {@link #setWizard(org.eclipse.jface.wizard.IWizard)} prior to invoking this
	 * method.
	 */
	public void init(IStructuredSelection selection, LocationPage locationPage) {
		this.nextPage = locationPage;
		IJavaElement element = getInitialJavaElement(selection);

		// Order matters here (to avoid the same message being displayed twice)
		setModelBehavior(true);
		initContainerPage(element);
		initTypePage(element);
		setDefautSuperClass();
		doStatusUpdate();
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		
		int nColumns = 4;
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(nColumns, false));
		composite.setLayoutData(GridDataUtil.createFill());
			
		createContainerControls(composite, nColumns);		
		createPackageControls(composite, nColumns);
		createSeparator(composite, nColumns);
		createTypeNameControls(composite, nColumns);
		createSeparator(composite, nColumns);
		createBehaviorControls(composite, nColumns);
		createSeparator(composite, nColumns);
		createSuperClassControls(composite, nColumns);
		
		setControl(composite);

		Dialog.applyDialogFont(composite);
		ToolsUiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(parent, ToolsUiPlugin.getID() + ContextIds.INV_TEST_DLG);
	}
	
	protected void createBehaviorControls(Composite composite, int numColumns) {
		Label label = new Label(composite, SWT.NONE);
		label.setText(ToolsUiPluginResourceBundle.SourcePage_behaviorModePrompt); 
		GridData gridData = new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false, numColumns, 1);
		label.setLayoutData(gridData);

		label = new Label(composite, SWT.NONE); // To fill the free cell

		Composite radioGroup = new Composite(composite, SWT.LINE_SOLID);
		GridLayout radioLayout = new GridLayout();
		radioGroup.setLayout(radioLayout);
		gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false, numColumns - 2, 1);
		radioGroup.setLayoutData(gridData);
		
		modelBehaviorRadio = new Button(radioGroup, SWT.RADIO);
		modelBehaviorRadio.setText(ToolsUiPluginResourceBundle.SourcePage_modelBehaviorChoice); 
		modelBehaviorRadio.addSelectionListener(this);
		
		codeBehaviorRadio = new Button(radioGroup, SWT.RADIO);
		codeBehaviorRadio.setText(ToolsUiPluginResourceBundle.SourcePage_codeBehaviorChoice); 
		
		if (isModelBehavior()) {
			modelBehaviorRadio.setSelection(true);
		} else {
			codeBehaviorRadio.setSelection(true);
		}

		label = new Label(composite, SWT.NONE); // To fill the free cell
	}
	
	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
		// NOP
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e)
	{
		if (e.widget == modelBehaviorRadio) {
			setModelBehavior(modelBehaviorRadio.getSelection());
			handleFieldChanged(BEHAVIOR);
		}
	}
	
	public void setModelBehavior(boolean state) {
		this.isModelBehavior = state;
	}
	
	public boolean isModelBehavior() {
		return this.isModelBehavior;
	}
	
	protected void setDefautSuperClass() {
		if (isModelBehavior()) {
			setSuperClass(Helper.HYADES_TEST_CASE_CLASS_NAME, false);
		} else {
			setSuperClass(Helper.JUNIT_TEST_CASE_CLASS_NAME, true);
		}
	}
	
	protected void behaviorChanged() {
		setDefautSuperClass();
		fContainerStatus = containerChanged();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.ui.wizards.NewContainerWizardPage#handleFieldChanged(java.lang.String)
	 */
	protected void handleFieldChanged(String fieldName) {
		super.handleFieldChanged(fieldName);
		if (BEHAVIOR.equals(fieldName)) {
			behaviorChanged();
		}
		doStatusUpdate();
	}
	
	private void doStatusUpdate() {
		
		//Note: This test suite name MUST be set BEFORE the status is updated so the wizard can correctly determine if its finished.
		//Set the test suite name based on the test class name:
		if (nextPage != null) {
			nextPage.setFileName(getTypeName());
		}
		
		//Update the status:
		updateStatus(new IStatus[] {
				fContainerStatus,
				fPackageStatus,
				fTypeNameStatus,
				fSuperClassStatus});		
	}
	
	/**
	 * <p>Resolves the qualified type name of the test, in the 
	 * format:</p>
	 * 
	 * <p><code>[&lt;package name&gt;.]&lt;class name&gt;</code></p>
	 * 
	 * @return The qualified type name of the test.
	 */
	public String getQualifiedTypeName() {
		
		String packageName = getPackageText();
		
		if ((packageName != null) && (packageName.trim().length() > 0)){
			return (packageName + '.' + getTypeName());
		} 
			
		return (getTypeName());
	}

	public static boolean isTestImplementor(IType type) throws JavaModelException {
		ITypeHierarchy typeHier= type.newSupertypeHierarchy(null);
		IType[] superInterfaces= typeHier.getAllInterfaces();
		for (int i= 0; i < superInterfaces.length; i++) {
			if (superInterfaces[i].getFullyQualifiedName().equals(Helper.JUNIT_TEST_CLASS_NAME))
				return true;
		}
		return false;
	}

	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.ui.wizards.NewTypeWizardPage#superClassChanged()
	 */
	protected IStatus superClassChanged() {
		IStatus status = super.superClassChanged();
		if (status.getSeverity() != IStatus.OK) return status;
		IPackageFragmentRoot root = getPackageFragmentRoot();
		if (root != null && root.exists()) {
			try {
				IType type = root.getJavaProject().findType(getSuperClass());
				if (type != null) {
					if (!isTestImplementor(type)) {
						return new Status(IStatus.ERROR, ToolsUiPlugin.getID(), 1, ToolsUiPluginResourceBundle.SourcePage_invalidSuperClassError, null); 
					}
				} else {
					return new Status(IStatus.WARNING, ToolsUiPlugin.getID(), 1, ToolsUiPluginResourceBundle.SourcePage_superClassNotFound, null); 
				}
			} catch (JavaModelException e) {
				ToolsUiPlugin.logError(e);
			}
		}
		return status;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.ui.wizards.NewTypeWizardPage#containerChanged()
	 */
	protected IStatus containerChanged() {
		IStatus containerStatus= super.containerChanged();
		dontAskMissingLibsAgain = false;
		if (!containerStatus.matches(IStatus.ERROR)) {
			IStatus projectStatus= validateProjectClasspath();
			if (!projectStatus.isOK()) {
				return projectStatus;
			}
		}
		return containerStatus;
	}
	
	/**
	 * The method is called when the container has changed to validate if the project
	 * has the required jars in its classpath for declaring a JUnit/Hyades Test.
	 * @return the status of the validation
	 */
	protected IStatus validateProjectClasspath() {
		Status status= new Status(IStatus.OK, ToolsUiPlugin.getID(), 1, "", null); //$NON-NLS-1$
		IPackageFragmentRoot root= getPackageFragmentRoot();
		if (root == null)
			return status;
		
		IJavaProject jp= root.getJavaProject();
		
		JUnitProjectDependencyUpdater updater = new JUnitProjectDependencyUpdater(
				new AutomaticDependencyUpdater(), isModelBehavior());
		Collection missingLibraries = updater.previewAdjustProject(jp.getProject());
		if (missingLibraries.isEmpty()) return status;
		
		if (!dontAskMissingLibsAgain) {
			String message = computeMessage(jp.getProject(), missingLibraries);
			if (MessageDialog.openQuestion(getShell(), ToolsUiPluginResourceBundle.SourcePage_not_on_buildpath_title, message)) { 
				try {
					updater.adjustProject(jp.getProject(), new NullProgressMonitor());
				} catch (CoreException e) {
					ToolsUiPlugin.logError(e);
				}
				return status;	
			} else {
				dontAskMissingLibsAgain = true;
			}
		}
		if (missingLibraries.size() == 1) {
			return new Status(IStatus.WARNING, ToolsUiPlugin.getID(), 1, NLS.bind(ToolsUiPluginResourceBundle.SourcePage_error_jarNotOnbuildpath, missingLibraries.toString()), null); 
		} else {
			return new Status(IStatus.WARNING, ToolsUiPlugin.getID(), 1, NLS.bind(ToolsUiPluginResourceBundle.SourcePage_error_jarsNotOnbuildpath, missingLibraries.toString()), null); 
		}
	}
    
    /**
     * Compute the message to be displayed. For more details, see bugzilla #175824: Confusing informational dialog.
     * @param project the project
     * @param missingLibraries the missign libraries
     * @return the message that informs the user of the missing libraries/JARs
     * 
     * @author pnedelec
     * @since 4.4
     */
    private String computeMessage(IProject project, Collection missingLibraries) {
        String librariesAsText = ""; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        Iterator it = missingLibraries.iterator();
        boolean moreThanOneElement = missingLibraries.size() > 1;
        while (it.hasNext()) {
            if (moreThanOneElement) {
                librariesAsText += "\n\t- "; //$NON-NLS-1$
            }
            librariesAsText += (String) it.next();
        }
        String message;
        
        if (PluginProjectUtil.isStrictPluginProject(project)) {
            //- The delegate is a PluginProjectDependencyUpdater 
            if (missingLibraries.size() == 1) {
                message = NLS.bind(ToolsUiPluginResourceBundle.SourcePage_lib_not_on_buildpath_message, (new String[] { librariesAsText, project.getName()}));
            } else {
                message = NLS.bind(ToolsUiPluginResourceBundle.SourcePage_libs_not_on_buildpath_message, (new String[] { project.getName(), librariesAsText}));
            }
        } 
        else {
            //- The delegate is a JavaProjectDependencyUpdater
            if (missingLibraries.size() == 1) {
                message = NLS.bind(ToolsUiPluginResourceBundle.SourcePage_jar_not_on_buildpath_message, (new String[] { librariesAsText, project.getName()}));
            } else {
                message = NLS.bind(ToolsUiPluginResourceBundle.SourcePage_jars_not_on_buildpath_message, (new String[] { project.getName(), librariesAsText}));
            }
        }
      
        return message;
    }

}
