/* ***********************************************************
 * Copyright (c) 2005,2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExportToXML.java,v 1.2 2006/05/11 18:17:33 dnsmith Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/
package org.eclipse.tptp.exportviewergenerator.sample.xml;


import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.extension.IExportViewer;
import org.eclipse.hyades.ui.internal.wizard.LocationPage;
import org.eclipse.hyades.ui.report.ReportGeneratorWizard;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

/**
 *
 */
public class ExportToXML extends ReportGeneratorWizard {

	private final static String newLine = System.getProperties().getProperty("line.separator");
	
    private final static String FILE_EXTENSION = "xml"; //$NON-NLS-1$
	
    private LocationPage locationPage;
	
	public ExportToXML() {
		super();
        setWindowTitle("Export to XML sample"); //$NON-NLS-1$
	}

	public boolean isAvailable(ISelection selection) {
		
		if(selection == null || selection.isEmpty())
			return false;
		
		Object sel = ((StructuredSelection)selection).getFirstElement();
		if(!(sel instanceof IExportViewer))
			return false;
		
		Object[] controls = ((IExportViewer)sel).getViewerControls();
		if(controls == null)
			return false;
		
		for(int idx=0; idx<controls.length; idx++)
		{
			if(!(controls[idx] instanceof Tree))
				return false;
		}
				
		return true;
	}
	
    /**
     * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
     */
    protected void initPages() {
        locationPage = new LocationPage("location", adjustLocation()); //$NON-NLS-1$
        locationPage.setTitle(TraceMessages._14); //$NON-NLS-1$
        locationPage.setDescription(TraceMessages._15); //$NON-NLS-1$
        locationPage.setFileExtension(FILE_EXTENSION);
    }

    protected IStructuredSelection adjustLocation() {
        IStructuredSelection structuredSelection = getSelection();
        
        Object selection = structuredSelection.getFirstElement();
        if (selection instanceof IResource) {
            IResource res = (IResource) selection;
            IContainer parent = res.getParent();
            if ((parent != null) && parent.exists()) {
                structuredSelection = new StructuredSelection(parent);
            }
        }
        return structuredSelection;
    }

	
    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#generate(org.eclipse.core.runtime.IProgressMonitor)
     */
    public IFile generate(IProgressMonitor monitor) throws Exception {
    	
    	Object sel = getSelection().getFirstElement();
    	if(sel == null || !(sel instanceof IExportViewer))
    		return null;
      
        monitor.beginTask("", 4); //$NON-NLS-1$             
    	
        IFile reportFile = getReportFile();
        if (reportFile.exists()) {
            if (!MessageDialog.openQuestion(getShell(), "W_QUESTION", //$NON-NLS-1$
                    "FILE_EXISTS")) //$NON-NLS-1$
                return null;
        }
        
        monitor.worked(1);
        export((IExportViewer)sel, reportFile.getLocation().toOSString());
        monitor.worked(1);

        reportFile.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        monitor.done();
           
        return reportFile;
    }
	
	private IFile getReportFile() {
        IFile file = null;
        IPath path = locationPage.getContainerFullPath();
        IResource res = createContainer(path);
        path = new Path(locationPage.getFileName()); //$NON-NLS-1$
        file = ((IContainer) res).getFile(path);
        return file;
    }
    
    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#addReportPages()
     */
    protected void addReportPages() throws Exception {
        addPage(locationPage);
    }
  
    /**
     * @see org.eclipse.jface.wizard.IWizard#canFinish()
     */
    public boolean canFinish() {
    	
        return locationPage.getItemName().length() > 0 && locationPage.getContainerFullPath() != null;
    }
    
    private IContainer createContainer(IPath path) {
        IContainer container = null;
        IWorkspace workbench = ResourcesPlugin.getWorkspace();
        int segCount = path.segmentCount();

        for (int idx = 0; idx < segCount; idx++) {
            if (idx == 0) {
                //project
                IProject project = workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());
                if (project == null || !project.exists()) {
                    //create the project
                    try {
                        project.create(null);
                    } catch (Exception e) {
                    	UIPlugin.getDefault().log(e);
                        e.printStackTrace();
                    }
                }
                try {
                    project.open(null);
                } catch (Exception e) {
                }
                container = project;
            } else // (idx > 1)
            { //folder
                IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));
                if (folder == null || !folder.exists()) {
                    //create the folder
                    try {
                        folder.create(false, true, null);
                    } catch (Exception e) {
                    	UIPlugin.getDefault().log(e);
                        e.printStackTrace();
                    }
                }
                container = folder;
            }
        }
        try {
            container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        } catch (CoreException exc) {
            UIPlugin.getDefault().log(exc);
            exc.printStackTrace();
        }
        return container;
    }
    
    /**
     * 
     * @param viewer
     * @return
     */
	private boolean export(IExportViewer viewer, String fileName) {
		
		if(fileName == null)
		{
			MessageDialog.openError(getShell(), TraceMessages.TRC_MSGT, TraceMessages._6);
		    return false;
		}
		
		Writer writer = null;
		try {
			File file = new File(fileName);
			if(file.isDirectory())
			{
				MessageDialog.openError(getShell(), TraceMessages.TRC_MSGT, TraceMessages._7);
				return false;				
			}
			OutputStream outputStream = new FileOutputStream(file.toString(), false);
			writer = new OutputStreamWriter(outputStream, "UTF8");
		} catch (IOException ioException) {
			ioException.printStackTrace();			
			return false;
		}
		
		StringBuffer buffer = new StringBuffer();        
		buffer.append(printHeader());
		buffer.append(printTitle(viewer));
		
		for(int idx=0; idx<viewer.getViewerControls().length; idx++)
		{
			buffer.append(printColumns((Tree)viewer.getViewerControls()[idx]));
		}
		
		buffer.append(printFooter());
		
		try {
			writer.write(buffer.toString());
			writer.flush();
			writer.close();
		} catch (Exception exception) {
			exception.printStackTrace();
		}
		
		return true;
	}
    
	protected String printHeader()
	{
		final String newLine = System.getProperties().getProperty("line.separator");
		StringBuffer buffer = new StringBuffer();
		buffer.append("<html"+getDirection()+">" + newLine);
		buffer.append("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">" + newLine
		              + "<head>" + newLine
		              + "<STYLE TYPE=\"text/css\">"  + newLine
		              + "p, table, td, th {  font-family: arial, helvetica, geneva; font-size: 10pt}" + newLine
		              + "body {  font-family: arial, helvetica, geneva; font-size: 10pt; clip:   rect(   ); margin-top: 5mm; margin-left: 3mm}" + newLine
		              + "</STYLE>" + newLine
		              + "</head>" + newLine);
		buffer.append("<body text=\"#000000\" bgcolor=\"#ffffff\">" + newLine);
		
		return buffer.toString();
	}
	
	protected String printTitle(IExportViewer viewer)
	{
		final String newLine = System.getProperties().getProperty("line.separator");
		StringBuffer buffer = new StringBuffer();
		
		String caption = viewer.getViewerTitle();	
		
		buffer.append("<caption align=top><B>" + newLine);
		buffer.append(caption + newLine);
		buffer.append("</B></caption>" + newLine);
        return buffer.toString();		
	}
	
	protected String printFooter()
	{
		final String newLine = System.getProperties().getProperty("line.separator");
		
		StringBuffer buffer = new StringBuffer();		
		buffer.append("</body>" + newLine);
		buffer.append("</html>" + newLine);
		return buffer.toString();
	}
	
	protected String printColumns(Tree tree)
	{		
		StringBuffer buffer = new StringBuffer();
		buffer.append("<data>" + newLine); //$NON-NLS-1$

		String[] columnHeaders = new String[tree.getColumnCount()];
		for (int idx = 0; idx < tree.getColumnCount(); idx++) {
			String columnText = tree.getColumn(idx).getText();

			if (columnText.indexOf("<") == 0) //$NON-NLS-1$
				columnText =  columnText.substring(1);
			else if (columnText.indexOf(">") == 0) //$NON-NLS-1$
				columnText =  columnText.substring(1);
			
			columnHeaders[idx] = columnText.replace(' ', '_').replace('(', '_').replace(')', '_').replaceAll("%", Messages.getString("ExportToXML.6"));
		}
		
		printTreeItems(tree.getItems(), 0, buffer, columnHeaders);

		buffer.append("</data>" + newLine); //$NON-NLS-1$
		
		return buffer.toString();
	}
	
	protected void printTreeItems(TreeItem[] treeItems, int level, StringBuffer buffer, String[] columnHeaders)
	{
		for (int kdx = 0; kdx < treeItems.length; kdx++)
		{
			buffer.append("<element>"); //$NON-NLS-1$
	
			for (int ndx = 0; ndx < columnHeaders.length; ndx++)
			{
				if (columnHeaders[ndx].length() > 0)
				{
					String headerSuffix;
					if (ndx == 0 && level > 0)
						headerSuffix = "_" + Messages.getString("ExportToXML.7");
					else
						headerSuffix = "";
					
					buffer.append("<"+columnHeaders[ndx]+headerSuffix+">"); //$NON-NLS-1$ //$NON-NLS-2$
					buffer.append(fixStringForXML(treeItems[kdx].getText(ndx)));
					buffer.append("</"+columnHeaders[ndx]+headerSuffix+">"); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
			
			buffer.append("</element>"); //$NON-NLS-1$
			buffer.append(newLine);

			if (treeItems[kdx].getExpanded())
				printTreeItems(treeItems[kdx].getItems(), level+1, buffer, columnHeaders);
		}
	}
	
	private String fixStringForXML(String s)
	{
		return s.replace('<', '(').replace('>', ')');
	}
	
	private String getDirection()
    {
    	if (Window.getDefaultOrientation() == SWT.RIGHT_TO_LEFT)
    		return " dir=\"rtl\"";
    	else
    		return "";
    }    
	
}
