/*******************************************************************************
 * Copyright (c) 2005,2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.tptp.filtertype.extension.sample.filter.definitions;

import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.trace.TracePackage;
import org.eclipse.hyades.ui.filters.IFilterAttribute;
import org.eclipse.tptp.filtertype.extension.sample.FilterTypeSamplePlugin;

/**
 * This defines a sample filter attribute.  It is kept simple and also
 * to display that attribute can be rather dynamic.  The user
 * can construct any WhereExpression from the operator/value pair they wish 
 * for the filter.
 * 
 * The operators in this sample are "Package starts with..." and "Package ends with...". 
 * This filter attribute basically would return packages that start with or end with the given value.
 * The user cannot enter a value with a '*' character, this is shown in the validate method.
 *
 */
public class SampleFilterAttribute implements IFilterAttribute{
	private String id;
	
	private static final String STARTS_WITH_OPERATOR = FilterTypeSamplePlugin.getResourceString("STR_FILTER_ATTRIBUTE_OP_STARTSWITH");
	private static final String ENDS_WITH_OPERATOR = FilterTypeSamplePlugin.getResourceString("STR_FILTER_ATTRIBUTE_OP_ENDSWITH");
	private static final String[] operators =
	{
		STARTS_WITH_OPERATOR,
		ENDS_WITH_OPERATOR
	};
	
	/**
	 * The constructor sets the id, this id must be unique from the others
	 * in the advanced tab.
	 * @param _id
	 */
	public SampleFilterAttribute(String _id)
	{
		id = _id;
	}
	
	/**
	 * @see IFilterAttribute#id()
	 */
	public String id() {
		return id;
	}

	/**
	 * @see IFilterAttribute#displayText()
	 */
	public String displayText() {
		return FilterTypeSamplePlugin.getResourceString("STR_FILTER_ATTRIBUTE_TEXT");
	}

	/**
	 * @see IFilterAttribute#validate(java.lang.String, java.lang.String)
	 */	
	public String validate(String operator, String value) {
		if (value.indexOf('*') >=0)
			return FilterTypeSamplePlugin.getResourceString("STR_FILTER_ATTRIBUTE_VALIDATE_ERROR");
		
		return null;
	}

	/**
	 * @see IFilterAttribute#operators()
	 */
	public String[] operators() {
		return operators;
	}

	/**
	 * @see IFilterAttribute#getWhereExpression(java.lang.String, java.lang.String)
	 */
	public WhereExpression getWhereExpression(String operator, String value) {
		if (STARTS_WITH_OPERATOR.equals(operator))
			value = value + "*";
		else
			value = "*" + value;

		//We construct a BinaryExpression, that is a WhereExpression.  It is "package_name like value* or *value", depending
		//  whether starts with or ends with was selected.  LogicalExpressions could also be constructed with more complexity
		//  as an implementor sees necessary.
		return ExpressionUtil.createBinaryExpression(TracePackage.eINSTANCE.getTRCPackage_Name(), RelationalOperators.get(RelationalOperators.LIKE), value, true);
	}

	/**
	 * @see IFilterAttribute#getValue(org.eclipse.hyades.models.hierarchy.extensions.WhereExpression)
	 */	
	public String getValue(WhereExpression whereExpression) {
		if (whereExpression instanceof BinaryExpression)
		{
		 	BinaryExpression be = (BinaryExpression)whereExpression;
		 	return getDisplayValue(((SimpleOperand)be.getRightOperands().get(0)).getRawValue()); 	
		}
		
		return null;
	}
	
	/**
	 * Returns the display value, visible in the UI, from the given value
	 * taken from the WhereExpression
	 * @param value
	 * @return
	 */	
	protected String getDisplayValue(String value)
	{
		if (value.startsWith("*"))
			return value.substring(1);
		else if (value.endsWith("*"))
			return value.substring(0, value.length()-1);
		
		return value;
	}		

	/**
	 * @see IFilterAttribute#getOperator(org.eclipse.hyades.models.hierarchy.extensions.WhereExpression)
	 */	
	public String getOperator(WhereExpression whereExpression) {
		if (whereExpression instanceof BinaryExpression)
		{
		 	BinaryExpression be = (BinaryExpression)whereExpression;
		 	String value = ((SimpleOperand)be.getRightOperands().get(0)).getRawValue();
		 	
		 	if (value.startsWith("*"))
		 		return ENDS_WITH_OPERATOR;
		 	else if (value.endsWith("*"))
		 		return STARTS_WITH_OPERATOR;
		}
		
		return null;
	}

	/**
	 * @see IFilterAttribute#defaultValue()
	 */	
	public String defaultValue() {
		return "";
	}

}
