/*******************************************************************************
 * Copyright (c) 2005,2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.tptp.filtertype.extension.sample.views;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.ResultEntry;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.util.internal.HierarchyXMIQueryEngine;
import org.eclipse.hyades.models.hierarchy.util.internal.QueryUtils;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TracePackage;
import org.eclipse.hyades.trace.ui.HyadesConstants;

/**
 * This class has utility methods for filtering source objects.  It contains setup methods
 * to setup the source, and setup the filter before calling the engine.  It also contains code
 * to call the filtering engine and use its results.
 *
 */
public class FilterTypeSampleFilteringUtil {

	/**
	 * Returns the list of AbstractTRCProcess objects, from the selected object
	 * @param object
	 * @return
	 */
	public static List getProcessList(EObject object) {
		List processList = new ArrayList();
		
		if(object instanceof TRCMonitor)
		{
			for (int i = 0; i < ((TRCMonitor)object).getNodes().size(); i++)
				processList.addAll(getProcessList((TRCNode)((TRCMonitor)object).getNodes().get(i)));
		}
		else if(object instanceof TRCNode)
		{
			for (int i = 0; i < ((TRCNode)object).getProcessProxies().size(); i++)
				processList.addAll(getProcessList((TRCProcessProxy)((TRCNode)object).getProcessProxies().get(i)));
		}
		else if(object instanceof TRCProcessProxy)
		{
			for (int i = 0; i < ((TRCProcessProxy)object).getAgentProxies().size(); i++)
				processList.addAll(getProcessList((TRCAgentProxy)((TRCProcessProxy)object).getAgentProxies().get(i)));
		}			
		else if(object instanceof TRCAgentProxy)
		{
			if (HyadesConstants.PROFILE_AGENT_TYPE.equals(((TRCAgentProxy)object).getType()))
				processList.add(((TRCAgentProxy)object).getAgent().getProcess());
		}
		
		return processList;
	}		
		
	/**
	 * This method calls the filtering engine, which filters depending on the given filter.
	 * @param filter
	 * @param source
	 * @param outputEClass
	 * @return
	 */
	private static List getFilteredElements(SimpleSearchQuery filter, EObject[] source, EClass outputEClass) {
		if (source == null)
			return new ArrayList();

		if (source.length == 0)
			return new ArrayList();
		
		if (filter == null)
			return new ArrayList();

		//below we setup the source the filter and search will work with
		filter.getSources().clear();
		Resource resource = null;
		for (int i = 0; i < source.length; i++)
		{
			if (source[i] instanceof EObject)
			{
				filter.getSources().add(EcoreUtil.getURI((EObject)source[i]).toString());
				
				if (resource == null)
					resource = ((EObject)source[i]).eResource(); 
			}
		}
		
		if (resource == null)
			return new ArrayList();
		
		//below we set the output element, in this class we can just have the TRCPackage type, as shown
		//   in the call to this method.
		filter.getOutputElements().clear();
		SimpleOperand outputElement = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		outputElement.setType(outputEClass);
		filter.getOutputElements().add(outputElement);
		
		//Here we create an engine instance, and execute the query the results contain the filters
		HierarchyXMIQueryEngine engine = new HierarchyXMIQueryEngine();
		QueryResult result = engine.executeQuery(filter, resource.getResourceSet(),Collections.EMPTY_LIST);
		if(result==null)
			return new ArrayList();
		return (List)((ResultEntry)result.getResultEntries().get(0)).getValue();
	}	

	/**
	 * Returns the filtered packages with the given filter and source list
	 * @param filter
	 * @param source
	 * @return
	 */
	public static List getFilteredPackages(SimpleSearchQuery filter, EObject[] source) {
		if (filter == null)
			filter = QueryUtils.getEmptyQuery();
		else
			filter = (SimpleSearchQuery)EcoreUtil.copy(filter);
		
		//We copy the filter above, as we are altering it below before sending it to the filtering engine
		//addInConditions, adds conditions to the filter before sending it to the engine, these conditions
		//  are needed so that the engine could relate features
		addInConditions((LogicalExpression)filter.getWhereExpression());
		
		//Gets the filtered packages from the (modified) filter, the source, and given the output type, we
		//  want package objects.  This method makes the call to the filtering engine.
		List allPackages = getFilteredElements(filter, source, TracePackage.eINSTANCE.getTRCPackage());
		
		//below we only wish to return packages that have some classes called
		List list = new ArrayList();
		for(int i=0; i < allPackages.size(); i++)
		{
			TRCPackage pack = (TRCPackage)allPackages.get(i);
			if(pack.getClasses().size() > 0)
			   list.add(pack);
		}
		return list;  
	}	
	
	/**
	 * Adds the possible in conditions for this sample
	 * @param logicalExpression
	 */
	private static void addInConditions(LogicalExpression logicalExpression) {
		Map cp = QueryUtils.getClassPredicatesIndex(logicalExpression.getSearchQuery());
		
		//If the filter has some feature attribute based on the method, say method_name, since the output are
		//  the packages we need to add two in conditions, method in class_methods and class in packages_classes,
		//  as shown below.
		//If filter has some feature attribute based on the class, say class_name, since the output are
		//  the packages we need to add the in condition,  class in packages_classes, as shown below.
		//These in conditions are required for the engine, so that it knows how to relate possible output packages
		//  with given method or class filtering.  A similar principle within the model would need to be applied
		//  for other levels, TRCObject, etc.
	    //The output element is important, filter attributes need to be related to it.  So if the output element
		//  were a TRCClass and we just had some method_name filter, we would just need to add method in class_methods.
		
		if(cp.containsKey(TracePackage.eINSTANCE.getTRCMethod()))
		{
			addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCClass(), TracePackage.eINSTANCE.getTRCPackage_Classes());
			addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCMethod(), TracePackage.eINSTANCE.getTRCClass_Methods());
		}
		else if(cp.containsKey(TracePackage.eINSTANCE.getTRCClass()))
			addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCClass(), TracePackage.eINSTANCE.getTRCPackage_Classes());
	}
	
	/**
	 * This constructs a binary expression, IN conditions with the given left and right operands.
	 * So that a left IN right binary expression is created, and added to the given logical expression.
	 * @param logicalExpression
	 * @param left
	 * @param right
	 */
	private static void addInCondition(LogicalExpression logicalExpression, EClass left, EReference right)
	{
		BinaryExpression binaryExpression = ExtensionsFactory.eINSTANCE.createBinaryExpression();

		SimpleOperand lhs = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		lhs.setType(left);
		
		binaryExpression.setOperator(RelationalOperators.IN_LITERAL);

		SimpleOperand rhs = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		rhs.setFeature(right);
		
		binaryExpression.setLeftOperand(lhs);
		binaryExpression.getRightOperands().add(rhs);
		
		logicalExpression.getArguments().add(binaryExpression);
	}	
}
