/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BaseConnectUtil.java,v 1.16 2010/12/10 20:17:46 jptoomey Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.security.internal.util;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import javax.net.ssl.TrustManager;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.execution.security.IKeystoreManager;
import org.eclipse.hyades.execution.security.ISecureClientParameters;
import org.eclipse.hyades.execution.security.KeystoreHelper;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.local.control.Application;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.internal.execution.local.control.NodeImpl;
import org.eclipse.hyades.internal.execution.local.control.SecureConnectionImpl;
import org.eclipse.hyades.internal.execution.security.DuplicateUserException;
import org.eclipse.hyades.internal.execution.security.SecureConnectionRequiredException;
import org.eclipse.hyades.internal.execution.security.UntrustedAgentControllerException;
import org.eclipse.hyades.internal.execution.security.User;
import org.eclipse.hyades.internal.execution.security.UserFactory;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;

public class BaseConnectUtil {
	private static final String TPTPECLIPSE_SECURE_STORAGE_SECURITY_PROVIDER = "TPTPEclipseSecureStorageSecurityProvider";
	public static final int CONNECTION_SUCCESS = 0;
	public static final int CONNECTION_CONTROLLER_ERROR = 1;
	public static final int CONNECTION_HOST_ERROR = 2;
	public static final int CONNECTION_PORT_ERROR = 3;
	public static final int CONNECTION_SECURITY_NOT_SUPPORTED = 4;
	public static final int CONNECTION_SECURITY_REQUIRED = 5;
	public static final int CONNECTION_AUTHENTICATION_SUCCESS = 6;
	public static final int CONNECTION_AUTHENTICATION_FAILED = 7;
	public static final int CONNECTION_UNTRUSTED = 8;

	/* Bug 145071 begins */
	public static final int AUTHENTICATION_SUCCESS = 0;
	public static final int AUTHENTICATION_FAILED = 1;
	public static final int AUTHENTICATION_ABORTED = 2;
	/* Bug 145071 ends */

	protected static int CERTIFICATE_UNKNOWN = 0;
	protected static int CERTIFICATE_SAVED = 1;
	protected static int CERTIFICATE_BYPASSED = 2;
	protected static int CERTIFICATE_CANCELLED = 3;

	public static final String PLUGIN_ID = "org.eclipse.hyades.ui"; //$NON-NLS-1$
	public static final String COMMON_UI_PLUGIN_ID = "org.eclipse.tptp.platform.common.ui"; //$NON-NLS-1$
	protected Application _app;
	protected String _port; // insecured port - do not overwrite with secure port
	protected String _hostName;
	protected InetAddress _hostAddr;
	protected String _userId;
	protected String _userPw;
	protected Node _node;
	protected ConnectErrorMsg _connectErrorMessage;
	protected String _keyStoreDir;
	protected String _keyStorePassword;
	protected SecureConnectionRequiredException _secureConnectionRequiredException = null;

	// private Throwable _pendingException = null;
	protected IConnectUtilUI connectionUI;

	protected class ConnectErrorMsg {
		protected String _errorMessage;

		protected Status _status;

		public ConnectErrorMsg(String errorMessage, Status status) {
			_errorMessage = errorMessage;
			_status = status;
		}

		public String getErrorMessage() {
			return _errorMessage;
		}

		public Status getErrorStatus() {
			return _status;
		}
	}

	public BaseConnectUtil(String hostName, String port, Application app) {
		this(hostName, port, null, app);
	}

	public BaseConnectUtil(String hostName, String port, String userId, Application app) {
		this(hostName, port, userId, app, null, null);
	}

	public BaseConnectUtil(String hostName, String port, String userId, Application app, String keyStoreDir, String keyStorePassword) {
		_hostName = hostName;
		_port = port;
		_userId = userId;
		_app = app;
		this.connectionUI = createConnectUI();

		_connectErrorMessage = null;
		_keyStoreDir = keyStoreDir;
		_keyStorePassword = keyStorePassword;
	}

	public BaseConnectUtil(InetAddress hostAddr, String port, Application app) {
		this(hostAddr, port, null, app);
	}

	public BaseConnectUtil(InetAddress hostAddr, String port, String userId, Application app) {
		this.connectionUI = createConnectUI();
		_hostAddr = hostAddr;
		_hostName = _hostAddr.getHostName();
		_port = port;
		_userId = userId;
		_app = app;
		if (_keyStoreDir == null) {
			_keyStoreDir = CommonPlugin.getKeyStoreLocation();
		}
	}

	protected IConnectUtilUI createConnectUI() {
		return new NullConnectUtilUI();
	}

	protected ISecureClientParameters getClientParms() 
	{
		IExtensionRegistry extensionRegistry = Platform.getExtensionRegistry();		
		ISecureClientParameters parameters = getClientParms (extensionRegistry.getConfigurationElementsFor(PLUGIN_ID, "JSSESecurityProviders"));		
		parameters = parameters == null ? getClientParms (extensionRegistry.getConfigurationElementsFor(COMMON_UI_PLUGIN_ID, "JSSESecurityProviders")) : parameters;
		return parameters;
	}

	
	protected ISecureClientParameters getClientParms(IConfigurationElement[] config)
	{		
		for (int idx = 0; idx < config.length; idx++) {
			IConfigurationElement elem = config[idx];

			// String typeAttr = elem.getAttribute("name");
			String classAttr = elem.getAttribute("class");

			/* For each of our security providers, instantiate the instances */
			if (classAttr != null) {
				try {
					Object realization = elem.createExecutableExtension("class");
					if (realization instanceof ISecureClientParameters) {
						// ISecureClientParameters parms =
						// (ISecureClientParameters) realization;
						/* Add the provider from the realization */
						// Security.addProvider(parms.getSecurityProvider());
						// parms.getKeystoreManager().setProvider(parms.getSecurityProvider());
						return (ISecureClientParameters) realization;
					}
				} catch (CoreException e) {
				}
			}
		}
		return null;
	}
	
	
	protected void showErrorDialogForConnect(ConnectErrorMsg message) {
		if (message != null) {
			openErrorDialog(SecurityMessages._41, message.getErrorMessage(), message.getErrorStatus());
		}
	}

	public void showErrorDialogForConnect() {
		showErrorDialogForConnect(_connectErrorMessage);
	}

	public final int connect() {
		return connect(true);
	}

	/**
	 * Connect, if password specified, use it if necessary instead of popping up
	 * UI, password specified also requires a user id to be specified.
	 * 
	 * @param password
	 *            password for the specified user
	 * @param showErrorMsgs
	 *            determines if error messages should be displayed
	 * @return the connect return code
	 */
	public final int connect(String password, boolean showErrorMsgs) {
		int rc_connect = CONNECTION_HOST_ERROR;
		int rc_authenticate = AUTHENTICATION_FAILED;

		/*
		 * Make sure _hostAddr is valid
		 */
		try {
			if (_hostAddr == null) {
				_hostAddr = InetAddress.getByName(_hostName);
			}
		} catch (UnknownHostException exc) {
			if (showErrorMsgs) {
				Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._43, _hostName), null);
				showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
			}
			return CONNECTION_HOST_ERROR;
		}

		/*
		 * Create the node
		 */
		if (_node == null) {
			try {
				_node = NodeFactory.createNode(_hostAddr, _app);
			} catch (UnknownHostException exc) {
				if (showErrorMsgs) {
					Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._43, _hostName), exc);
					showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
				}
				return CONNECTION_HOST_ERROR;
			}
		}

		/*
		 * Primary connection checking
		 */
		//239310  Compare port number too.
		if(!_node.isConnected()|| _node.getConnection().getPort()!= Integer.parseInt(_port)) {
			/*
			 * Establish a connection first
			 */
			rc_connect = insecureConnect(showErrorMsgs);
			if (rc_connect == CONNECTION_SECURITY_REQUIRED) {
				if (_secureConnectionRequiredException != null) {
					try {
						rc_connect = secureConnect(showErrorMsgs);
						if (rc_connect != CONNECTION_SUCCESS) {
							return rc_connect; //return failure, no need to proceed any further for security dialog
						}
					} catch (UntrustedCertificateAgentControllerException e) {
						if (saveCertificateUI(e.getKeyStore(), e.getCertificate())) {
							try {
								rc_connect = secureConnect(showErrorMsgs);
								if (rc_connect != CONNECTION_SUCCESS) {
									return rc_connect; //return failure, no need to proceed any further for security dialog
								}
							} catch (UntrustedCertificateAgentControllerException e1) {
								e1.printStackTrace();
							}
						} else {
							/* user has cancelled the certificate dialog */
							resetConnection();
							return CONNECTION_HOST_ERROR;
						}
					}
				}
			}
			else if(rc_connect == CONNECTION_SUCCESS) {
				/** return connect success without security requirement **/
				return CONNECTION_SUCCESS;
			}
		}
		else { /* Primary connection already exists */
			rc_connect = CONNECTION_SUCCESS;

			/* return success if no further security requirement */
			if (!(_node.getConnection() instanceof SecureConnectionImpl)) {
				return CONNECTION_SUCCESS;
			}
		}

		/* Return failing code if there is no successful primary connection before any security connection requirement */
		if (rc_connect != CONNECTION_SUCCESS) {
			return rc_connect;
		}

		/**************************************************/
		/** primary connection success beyond this point **/
		/**************************************************/

		/*
		 * Authenticate the user ID and password here
		 */
		if (_node.getConnection() instanceof SecureConnectionImpl) { /* secure connection is required */
			/*
			 * Only authenticate if not yet done so
			 */
			if(!((NodeImpl)_node).isUserAuthenticated()) {
				/* Will use stored password first since it is most likely the correct one */
				rc_authenticate = authenticateUser(_userId, _userPw);
				if(rc_authenticate != AUTHENTICATION_SUCCESS) {
					_userPw = password; // Retry with supplied password
					rc_authenticate = authenticateUser(_userId, _userPw);
					// Next see if the username and password for this host
					// are provided by a mapping extension
					if(rc_authenticate != AUTHENTICATION_SUCCESS) {
						StringBuffer preferenceUserPw = new StringBuffer();
						StringBuffer preferenceUserId = new StringBuffer();

						boolean preferenceCred = getPreferenceCredentials(_hostName, 
								preferenceUserId, preferenceUserPw);
						
						if (preferenceCred) {
							rc_authenticate = authenticateUser(preferenceUserId
									.toString(), preferenceUserPw.toString());
						}
					}
					// None of those attempts worked.  Prompt the user for the
					// required credentials
					if(rc_authenticate != AUTHENTICATION_SUCCESS) {
						int retry = 0;
						do {
							IConnectUtilUser user = connectionUI.promptAuthentication(getHostName(), _userId);
							if(user != null) {
								_userId = user.getName();
								_userPw = user.getPassword();
								rc_authenticate = authenticateUser(_userId, _userPw);
								retry++;
							}
							else { // Cancel is pressed
								rc_authenticate = AUTHENTICATION_ABORTED;
								break;
							}
						} while((rc_authenticate != AUTHENTICATION_SUCCESS) && (retry < 3));
					}
				}
			}
			else {
				rc_authenticate = AUTHENTICATION_SUCCESS;
			}

			/* Check the authentication status and determine if connection is successful */
			if(rc_authenticate == AUTHENTICATION_FAILED) {
				if (showErrorMsgs) {
					Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._52, _hostName), null);
					showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
				}
				
				// Bug 323137 
				return CONNECTION_AUTHENTICATION_FAILED; 
			}
		}

		// Bug 323137 - Disambiguate an auth failure from connection failure
		if( rc_connect == CONNECTION_SUCCESS ) {
			if( rc_authenticate == AUTHENTICATION_SUCCESS )
				return CONNECTION_SUCCESS;
			else 
				return CONNECTION_AUTHENTICATION_FAILED;
		} else {
			return CONNECTION_HOST_ERROR;
		}
	}

	private boolean getPreferenceCredentials(String hostname, StringBuffer preferenceUserId, StringBuffer preferenceUserPw) {
		boolean found = false;
		IExtensionRegistry extensionRegistry = Platform.getExtensionRegistry();		
		IConfigurationElement[] elements = extensionRegistry.getConfigurationElementsFor(COMMON_UI_PLUGIN_ID, TPTPECLIPSE_SECURE_STORAGE_SECURITY_PROVIDER);
		for (int i = 0; i < elements.length && !found; i++) {
			IConfigurationElement configurationElement = elements[i];
			if (configurationElement.getAttribute("class") != null) {
				try {
					ITPTPSecureStorageProvider provider = (ITPTPSecureStorageProvider) 
						configurationElement.createExecutableExtension("class");
					found = provider.getCredentialsForHost(hostname, 
							preferenceUserId, preferenceUserPw);
				} catch (CoreException e) {
					// The use of a secure storage provider is always optional, 
					// so this method shouldn't propogate any failures upward.
					CommonPlugin.getLogger().logError(e);
					return false;
				}
			}
		}
		
		return found;
	}

	public final int reconnect(String password, boolean showErrorMsgs) {
		return connect(password, showErrorMsgs);
	}

	public final int connectConnection(String password, boolean showErrorMsgs) {
		return connect(password, showErrorMsgs);
	}

	protected int insecureConnect(boolean showErrorMsgs) {

		try {
			_node.setUser(null);
			_node.setSecurityParameters(null);
			_node.connect(Integer.parseInt(_port));
		} catch (SecureConnectionRequiredException exc) {
			_secureConnectionRequiredException = exc;
			return CONNECTION_SECURITY_REQUIRED;
		} catch (UntrustedAgentControllerException exc) {
			/* Should not get this under insecured mode */
			exc.printStackTrace();
			return CONNECTION_SECURITY_REQUIRED;
		} catch (AgentControllerUnavailableException exc) {
			resetConnection();
			if (showErrorMsgs) {
				showConnectionError(exc);
			}
			return CONNECTION_CONTROLLER_ERROR;
		} catch (NumberFormatException exc) {
			resetConnection();
			if (showErrorMsgs) {
				Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._44, String.valueOf(_port)), exc);
				showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
			}
			return CONNECTION_PORT_ERROR;
		}

		return CONNECTION_SUCCESS;
	}

	protected boolean loadKeyStore(boolean showErrorMsgs) {
		if(_node != null) {
			ISecureClientParameters param = _node.getSecurityParameters();
			if(param != null) {
				IKeystoreManager km = param.getKeystoreManager();
				if(km != null) {
					try {
						km.loadKeystore(_keyStoreDir, _keyStorePassword);
						return true;
					} catch (IOException e) {
						resetConnection();
						if (showErrorMsgs) {
							Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._69, null), e);
							showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
						}
					} catch (CertificateException e) {
						resetConnection();
						if (showErrorMsgs) {
							Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._46, _keyStoreDir), e);
							showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
						}
					} catch (KeyManagementException e) {
						resetConnection();
						if (showErrorMsgs) {
							Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._48, _keyStoreDir), e);
							showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
						}
					} catch (KeyStoreException e) {
						resetConnection();
						if (showErrorMsgs) {
							Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._49, _keyStoreDir), e);
							showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
						}
					} catch (NoSuchAlgorithmException e) {
						resetConnection();
						if (showErrorMsgs) {
							Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._47, _keyStoreDir), e);
							showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
						}
					} catch (UnrecoverableKeyException e) {
						resetConnection();
						if (showErrorMsgs) {
							Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._50, _keyStoreDir), e);
							showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
						}
					}
				}
			}
		}
		return false;
	}

	protected void showConnectionError(Exception exc) {
		try {
			if (_hostName.equals("localhost") || InetAddress.getByName(_hostName).equals(InetAddress.getLocalHost())) {// localhost
				Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._67, "localhost"), exc);
				showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
				return;
				
			} else {
				Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._67, _hostName), exc);
				showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
			}
		} catch (Exception exec) {
			exec.printStackTrace();
		}
	
	}

	protected int secureConnect(boolean showErrorMsgs) throws UntrustedCertificateAgentControllerException {
		String keyStoreDir = null;
		String keyStorePassword = null;
		int securedPort;

		securedPort = (int)_secureConnectionRequiredException.getSecurePort();

		try {
			if (_node.getSecurityParameters() == null) {
				ISecureClientParameters parms = getClientParms();
				if (parms == null) {
					resetConnection();
					if (showErrorMsgs) {
						Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._71, _hostName), null);
						showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
					}
					return CONNECTION_SECURITY_NOT_SUPPORTED;
				} else {
					/* Load the keystore */
					_node.setSecurityParameters(parms);

					try {
						keyStoreDir = _node.getSecurityParameters().getKeystoreFileName();
						keyStorePassword = _node.getSecurityParameters().getKeystoreFilepassword();

						/* Update the keystore dir and password if no exception */
						_keyStoreDir = keyStoreDir;
						_keyStorePassword = keyStorePassword;
					} catch (RuntimeException e) {
						//e.printStackTrace();
						// if exception occurs try the keyStoreDir and
						// keyStorePassword supplied in the context
						
					}

					/*
					 * Load the keystore
					 */
					if(!loadKeyStore(showErrorMsgs)) {
						return CONNECTION_HOST_ERROR;
					}
				}
			}

			_node.setUser(UserFactory.getUser(_app, _userId));
			_node.connect(securedPort);

		} catch (SecureConnectionRequiredException e) {
			// Should not be getting this
			e.printStackTrace();
			return CONNECTION_SECURITY_NOT_SUPPORTED;
		} catch (UntrustedAgentControllerException e) {
			throw updateCertificateException(e);
		} catch (NumberFormatException e) {
			resetConnection();
			if (showErrorMsgs) {
				Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._44, String.valueOf(securedPort)), e);
				showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
			}
			return CONNECTION_HOST_ERROR;
		} catch (AgentControllerUnavailableException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return CONNECTION_HOST_ERROR;
//		} catch (NoSuchProviderException e) {
//			resetConnection();
//			if (showErrorMsgs) {
//				Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, NLS.bind(SecurityMessages._58, null), e);
//				showErrorDialogForConnect(new ConnectErrorMsg(NLS.bind(SecurityMessages._42, _hostName), err));
//			}
//			return CONNECTION_HOST_ERROR;
		}
		return CONNECTION_SUCCESS;
	}

	public final int connect(boolean showErrorMsgs) {
		return connect(null, showErrorMsgs);
	}

	public String getUserId() {
		return _userId;
	}

	public Node getNode() {
		return _node;
	}

	public void resetConnection() {
		if (_userId != null) {
			UserFactory.removeUser(_app, _userId);
		}
		if (_node != null) {
			_node.setSecurityParameters(null);
		}
	}

	public int authenticateUser(String userId, String userPw) {
		User user = null;

		if (userId != null && !userId.equals("")) {
			user = UserFactory.getUser(_app, userId);
			if (UserFactory.getUser(_app, userId) != null) {// remove
				UserFactory.removeUser(_app, userId);
			}
			try {
				user = UserFactory.createUser(_app, userId, userPw);
			} catch (DuplicateUserException e) {
				e.printStackTrace();
			}
		}

		if ((user == null) || (userPw == null)) {
			return AUTHENTICATION_ABORTED;
		} else {
			user.setPassword(userPw);
		}

		/* Bug 145071 begins */
		if (user == null) {
			return AUTHENTICATION_ABORTED;
		} else {
			_node.setUser(user);

			if (_node instanceof NodeImpl) {
				boolean status = ((NodeImpl) _node).authenticateUser();
				if (status) {
					return AUTHENTICATION_SUCCESS;
				} else {
					return AUTHENTICATION_FAILED;
				}
			} else {
				return AUTHENTICATION_FAILED;
			}
		}
		/* Bug 145071 ends */
	}

	/**
	 * Add certificate information and keystore information so that the user
	 * interface can present this information to the ui.
	 */
	protected UntrustedCertificateAgentControllerException updateCertificateException(UntrustedAgentControllerException exc) {
		TrustManager[] tManagers = _node.getSecurityParameters().getKeystoreManager().getTrustManagers();
		X509Certificate cert = null;
		try {
			for (int i = 0; i < tManagers.length; i++) {
				Method getReceivedCerts = tManagers[i].getClass().getMethod("getReceivedCerts", null);
				Object certs = getReceivedCerts.invoke(tManagers[i], null);
				if (certs != null) {
					cert = (X509Certificate) ((Object[]) certs)[0];
				}
			}
		} catch (Exception e) {
			System.out.println(e.toString());
		}
		KeyStore keyStore = creatKeyStore();
		UntrustedCertificateAgentControllerException newExc = new UntrustedCertificateAgentControllerException(exc);
		newExc.setCertificate(cert);
		newExc.setKeyStore(keyStore);
		return newExc;
	}

	public KeyStore creatKeyStore() {

		try {
			return KeystoreHelper.createKeyStore(_keyStoreDir, _keyStorePassword);
		} catch (KeyStoreException exc) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, NLS.bind(SecurityMessages._57, _keyStoreDir), exc);
			connectionUI.openErrorDialog(SecurityMessages._41, SecurityMessages._59, err);
		} catch (NoSuchAlgorithmException e) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, NLS.bind(SecurityMessages._47, _keyStoreDir), e);
			connectionUI.openErrorDialog(SecurityMessages._41, SecurityMessages._59, err);
		} catch (CertificateException exc) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, NLS.bind(SecurityMessages._46, _keyStoreDir), exc);
			connectionUI.openErrorDialog(SecurityMessages._41, SecurityMessages._59, err);
		} catch (NoSuchProviderException e) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, SecurityMessages._58, e);
			connectionUI.openErrorDialog(SecurityMessages._41, NLS.bind(SecurityMessages._57, _keyStoreDir), err);
		} catch (IOException e) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, NLS.bind(SecurityMessages._61, _keyStoreDir), e);
			connectionUI.openErrorDialog(SecurityMessages._41, SecurityMessages._59, err);
		}
		return null;
	}

	public IStatus saveCertificate(KeyStore ks, String alias, X509Certificate cert) {
		try {
			ks.setCertificateEntry(alias, cert);
			KeystoreHelper.persistKeyStore(ks, _keyStoreDir, _keyStorePassword);
			loadKeyStore(true); // Reload after saving the certificate
			return Status.OK_STATUS;
		} catch (IOException e) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, NLS.bind(SecurityMessages._54, _keyStoreDir), e);
			connectionUI.openErrorDialog(SecurityMessages._41, SecurityMessages._53, err);
			return err;
		} catch (CertificateException exc) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, NLS.bind(SecurityMessages._55, _keyStoreDir), exc);
			connectionUI.openErrorDialog(SecurityMessages._41, SecurityMessages._53, err);
			return err;
		} catch (KeyStoreException exc) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, NLS.bind(SecurityMessages._56, _keyStoreDir), exc);
			connectionUI.openErrorDialog(SecurityMessages._41, SecurityMessages._53, err);
			return err;
		} catch (NoSuchAlgorithmException exc2) {
			Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, NLS.bind(SecurityMessages._47, _keyStoreDir), exc2);
			connectionUI.openErrorDialog(SecurityMessages._41, SecurityMessages._53, err);
			return err;
		}
	}

	protected boolean saveCertificateUI(KeyStore keyStore, X509Certificate cert) {
		/*
		 * RKD: First we open a dialog stating that the server is not trusted
		 * and ask the user to continue
		 */
		int rc = 0;
		int status = CERTIFICATE_UNKNOWN;
		boolean ok = false;
		do {
			String[] btnLabels = new String[] { SecurityMessages._32, SecurityMessages._33, SecurityMessages._34 };
			rc = openQuestionDialog(SecurityMessages._41, NLS.bind(SecurityMessages._70, _hostName), btnLabels);
			// import certificate
			if (rc == 0) { // Hit OK to save the certificate
				ok = openSaveCertificateDialog(keyStore);
				if (ok) { // If certificate saved successfully
					saveCertificate(keyStore, connectionUI.getAlias(), cert);
					status = CERTIFICATE_SAVED;
				}
				else { // If decided not to save at this point, ask if wanted to bypass
					ok = openQuestionDialog(SecurityMessages._17, SecurityMessages._76, null) == 0;
					if(ok) {
						ISecureClientParameters params = _node.getSecurityParameters();
						if (params != null) {
							params.disableServerAuthentication();
						}
						status = CERTIFICATE_BYPASSED;
					}
					else {
						status = CERTIFICATE_CANCELLED;
					}
				}
			}
			else if (rc == 1) { // Hit Cancel to bypass the certificate
				ok = openQuestionDialog(SecurityMessages._17, SecurityMessages._76, null) == 0;
				if(ok) {
					ISecureClientParameters params = _node.getSecurityParameters();
					if (params != null) {
						params.disableServerAuthentication();
					}
					status = CERTIFICATE_BYPASSED;
				}
				else {
					status = CERTIFICATE_CANCELLED;
				}
			}
			else { // Hit View to view the certificate
				openCertificateDetailDialog(cert);
			}
		} while (status == CERTIFICATE_UNKNOWN);

		if((status == CERTIFICATE_CANCELLED)) {
			return false;
		}
		else {
			return true;
		}
	}

	public static boolean isLocalHost(String hostname) {
		String name = "localhost";
		if (name.equalsIgnoreCase(hostname))
			return true;

		try {
			InetAddress a1 = InetAddress.getByName(hostname);
			InetAddress a2 = InetAddress.getLocalHost();
			
			if(a1.getHostAddress().equals(a2.getHostAddress())){
				return true;
			}
		} catch (UnknownHostException exc) {
			return false;
		}

		return false;
	}

	protected void openCertificateDetailDialog(X509Certificate certificate) {
		connectionUI.openCertificateDetailDialog(certificate);
	}

	protected int openQuestionDialog(String title, String msg, String[] btnLabels) {
		return connectionUI.openQuestionDialog(title, msg, btnLabels);
	}

	protected boolean openSaveCertificateDialog(KeyStore ks) {
		return connectionUI.openSaveCertificateDialog(ks);
	}

	protected void openErrorDialog(String title, String message, IStatus err) {
		connectionUI.openErrorDialog(title, message, err);
	}

	public IConnectUtilUI getConnectionUI() {
		return connectionUI;
	}

	public void setConnectionUI(IConnectUtilUI connectionUI) {
		this.connectionUI = connectionUI;
	}

	public String getHostName() {
		return _hostName;
	}

	public String getPort() {
		return _port; // Always return the insecure port
	}

	public void setUserId(String userId) {
		_userId = userId;
	}

	public void setUserPassword(String userPw) {
		_userPw = userPw;
	}
}
