/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileUtil.java,v 1.3 2010/06/25 20:32:58 jwest Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

/**
 * Contains utility methods to manipulate {@link java.io.File} instances and streams.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class FileUtil
{
	/**
	 * Returns a path that uses '/' as the path separator and that
	 * doesn't end with a path separator.
	 * 
	 * @param path
	 * @return String
	 */
	public static String normalizePath(String path)
	{
		if(path == null)
			return null;
			
		path = path.trim().replace('\\', '/');
		if(path.endsWith("/") && (!path.endsWith("//")))
			path = path.substring(0, (path.length()-"/".length()));
			
		return path;
	}

	/**
	 * Returns a normalized path that is the result of appending
	 * path2 to path1.
	 * @param path1
	 * @param path2
	 * @return String
	 */
	public static String appendPath(String path1, String path2)
	{
		path1 = normalizePath(path1);
		path2 = normalizePath(path2);
		
		if((path1 == null) || (path1.trim().length() == 0))
			return path2;
			
		if((path2 == null) || (path2.trim().length() == 0))
			return path1;
					
		if(path1.endsWith("//"))
		{
			if(path2.startsWith("/") && (path2.length() > 1))
				path2 = path2.substring(1);
		}
		else if(!path2.startsWith("/"))
			path2 = "/" + path2;
			
		return path1 + path2;
	}

	/**
	 * Copies one stream to another synchronizing the operations (thread save).
	 * @param inputStream The source of the copy
	 * @param outputStream The target of the copy
	 * @return int The number of copied bytes.
	 * @throws IOException
	 */
	public static int streamCopy(InputStream inputStream, OutputStream outputStream)
	throws IOException
	{
		synchronized(inputStream)
		{
			synchronized(outputStream)
			{
				int totalBytes = 0;
				int length = 0;
				byte[] buffer = new byte[256];
				while((length = inputStream.read(buffer)) != -1)
				{
					totalBytes += length;
					outputStream.write(buffer, 0, length);
				}
				
				return totalBytes;
			}
		}
	}
	
	/**
	 * Returns a normalized path that doesn't contain the last
	 * segment from the specified path.  The last segment is any
	 * text at the right of the last file separator. 
	 * @param path
	 * @return String
	 */
	public static String removeLastSegment(String path)
	{
		path = normalizePath(path);

		int index = path.lastIndexOf("/");
		if(index < 0)
			return "";

		return path.substring(0, index);
	}

	/**
	 * Returns the last segment of a path.  The last segment is any
	 * text of the right of the last path separator (\ or /).
	 * 
	 * The returned string never starts or ends with a path separator.
	 * 
	 * @param path
	 * @return String
	 */
	public static String getLastSegment(String path)
	{
		path = normalizePath(path);

		int index = path.lastIndexOf("/");
		if(index >= 0)
			path = path.substring(index + 1);

		return path;
	}

	/**
	 * Returns a file handle that is an absolute path to an existing
	 * file or directory.
	 * 
	 * @param file
	 * @return File
	 */
	public static File getValidFileOrDirectory(File file)
	{
		if((file == null) || (!file.exists()))
			return null;

		try
		{
			return file.getAbsoluteFile().getCanonicalFile();
		}
		catch(IOException ioE)
		{
		}

		return null;
	}

	/**
	 * Writes a String to a file.  It is possible to specify 
	 * the encoding to be used and if the String should be appended to file.
	 * 
	 * <p>If the String is not appended the file will be overwritten.
	 * 
	 * <p>Returns true if no errors have happened. 
	 * 
	 * @param encoding
	 * @param file
	 * @param append
	 * @param text
	 * @return boolean
	 */
	public static boolean writeToFile(String encoding, File file, boolean append, String text)
	{
		if(file == null)
			return false;

		if(!append)
		{
			File validFile = getValidFileOrDirectory(file);
			if((validFile != null) && file.isFile() && file.exists())
			{
				validFile.delete();
				file = validFile;
			}
		}
				
		if((text == null) || (text.length() == 0))
			return true;

		try
		{
			return writeToOutputStream(encoding, new FileOutputStream(file.toString(), append), text);
		}
		catch(FileNotFoundException e)
		{
		}
		
		return false;
	}
	
	/**
	 * Writes a string to a output stream.  It is possible to specify 
	 * the encoding.
	 * 
	 * <p>Returns true if no errors have happened. 
	 * 
	 * @param encoding
	 * @param outputStream
	 * @param text
	 * @return boolean
	 */
	public static boolean writeToOutputStream(String encoding, OutputStream outputStream, String text)
	{
		try
		{
			Writer out = null;
			try
			{
				if(encoding != null)	
					out = new OutputStreamWriter(outputStream, encoding);
				else
					out = new OutputStreamWriter(outputStream);
					
				out.write(text);
				out.flush();
			}
			finally
			{
				if(outputStream != null)
					outputStream.close();
			}
		}
		catch(IOException ioE)
		{
			return false;
		}

		return true;
	}
	
	/**
	 * Returns a String with the content of a file.  It is possible to specify the
	 * encoding and if the file should be closed or not.
	 * 
	 * @param encoding
	 * @param file
	 * @param keepOpen
	 * @return String
	 * @throws IOException
	 */
	public static String readFromFile(String encoding, File file, boolean keepOpen)
	throws IOException
	{
		if(file == null)
			return null;

		if(!file.exists())
			return null;
				
		return readFromInputStream(encoding, new FileInputStream(file), keepOpen);
	}
	
	/**
	 * Returns the data read from a InputStream as a String.  It is possible to specify
	 * the encoding and if the file should be closed or not.
	 * 
	 * @param encoding
	 * @param inputStream
	 * @param keepOpen
	 * @return String
	 * @throws IOException
	 */
	public static String readFromInputStream(String encoding, InputStream inputStream, boolean keepOpen)
	throws IOException
	{
		StringBuffer buffer = new StringBuffer();
    	try
    	{
			InputStreamReader isr = null;
			if(encoding != null)
				isr = new InputStreamReader(inputStream, encoding);
			else
				isr = new InputStreamReader(inputStream);

			Reader in = new BufferedReader(isr);
			int ch;
			while((ch = in.read()) > -1)
				buffer.append((char)ch);
			
			return buffer.toString();
    	}
		finally
		{
			if((!keepOpen) && (inputStream != null))
				inputStream.close();
		}			
	}
	
	/**
	 * Returns a Map with each entry data (byte[]) by the entry
	 * name.
	 * 
	 * <p>The entry name is the entry full path inside the zip file.
	 * 
	 * @param zipFileName
	 * @return Map
	 */
    public static Map getZipEntryDataByZipEntryName(String zipFileName)
	{
		Map entrySizeByEntryName = new Hashtable();
		Map entryDataByEntryName = new Hashtable();
		
		try
	    {
	    	ZipFile zipFile = null;
	    	try
	    	{
	    		zipFile = new ZipFile(zipFileName);
		    	for(Enumeration e=zipFile.entries(); e.hasMoreElements();)
		    	{
					ZipEntry entry=(ZipEntry)e.nextElement();
					entrySizeByEntryName.put(entry.getName(),new Integer((int)entry.getSize()));
				}
	    	}
	    	finally
	    	{
	    		if(zipFile != null)
	    			zipFile.close();
	    	}

	    	ZipInputStream zipInputStream = null;
	    	try
	    	{
	    		zipInputStream = new ZipInputStream(new BufferedInputStream(new FileInputStream(zipFileName)));
	    		for(ZipEntry entry = zipInputStream.getNextEntry(); entry != null; entry = zipInputStream.getNextEntry())
				{
					if (entry.isDirectory())
		    			continue;
		
					int size=(int)entry.getSize();
					if(size==-1)
		    			size=((Integer)entrySizeByEntryName.get(entry.getName())).intValue();

					byte[] data=new byte[size];
					int offset=0;
					int ret=0;
					while((size - offset) > 0)
		    		{
		    			ret = zipInputStream.read(data,offset,size - offset);
		   				 if(ret==-1)
							break;
		    			
		    			offset+=ret;
		    		}

					entryDataByEntryName.put(entry.getName(), data);
				}
	    	}
	    	finally
	    	{
	    		if(zipInputStream != null)
	    			zipInputStream.close();
	    	}
	    }
		catch(Exception e)
		{
	    }
	    
	    return entryDataByEntryName;
	}
	
	/**
	 * Returns the extension of a file.  The extension is any
	 * text at the right side of the last '.'.
	 * 
	 * <p>If the file has no extension or is null or is a directory
	 * this method returns an empty string. 
	 * 
	 * @param file
	 * @return String
	 */
	public static String getFileExtension(File file)
	{
		if((file == null) || file.isDirectory())
			return "";
			
		return getFileExtension(file.getName());
  	}
  	
	/**
	 * Returns the extension of a given file full path or name.  The 
	 * extension is any text at the right side of the last '.'.
	 * 
	 * <p>If the file has no extension or is null returns an 
	 * empty string. 
	 * 
	 * @param file
	 * @return String
	 */
	public static String getFileExtension(String file)
	{
		if(file == null)
			return "";
			
		int index = file.lastIndexOf(".");
		if(index < 0)
			return "";
		
		if(index == (file.length()-1))
			return "";
				
		return file.substring(index+1);
	}

	/**
	 * Returns true if the file to analyze is in the parent candidate's 
	 * hierarchy.
	 * @param parentCandidate
	 * @param fileToAnalyse
	 * @return boolean
	 */
	public static boolean isParent(File parentCandidate, File fileToAnalyse)
	{
		if((fileToAnalyse == null) || (parentCandidate == null))
			return false;
			
		parentCandidate = parentCandidate.getAbsoluteFile();
		fileToAnalyse = fileToAnalyse.getAbsoluteFile();
		
		while(fileToAnalyse != null)
		{
			if(parentCandidate.equals(fileToAnalyse))
				return true;
				
			fileToAnalyse = fileToAnalyse.getParentFile();
		}
		
		return false;
	}

  	/**
  	 * Returns the system temp directory.
  	 * @return File
  	 */
    public static File getTempDir()
    {
	    return new File(System.getProperty("java.io.tmpdir"));
	}
	
	/**
	 * Deletes the specified file.  If the file is a directory this
	 * methods deletes all its contents.
	 * 
	 * <p>Returns the number of deleted files.
	 * 
	 * @param file
	 * @return int
	 */
	public static int delete(File file)
	{
		if(file.isFile())
			return (file.delete()?1:0);
		
		if(file.isDirectory())
		{	
			int counter = 0;
			
			File[] files = file.listFiles();
			for(int i = 0; i < files.length; i++)
				counter += delete(files[i]);
				
			counter += (file.delete()?1:0);
			return counter;
		}
			
		return 0;	
	}
	
	/**
	 * Copies the specified files to a directory.  In order to be copied an element
	 * from the files array must exist and be a file.
	 * 
	 * <p>The <code>toDirectory</code> is created if necessary.
	 * 
	 * @param files
	 * @param toDirectory
	 * @param overwrite
	 * @return int The number of copied files.
	 * @throws IOException
	 */
	public static int copyFiles(File[] files, File toDirectory, boolean overwrite)
	throws IOException
	{
		if((files == null) || (files.length == 0) || (toDirectory == null))
			return 0;
			
		if(toDirectory.exists())
		{
			if(!toDirectory.isDirectory())
				return 0;
		}
		else
			toDirectory.mkdirs();
		
		int copyCount = 0;
		for(int i=0, max=files.length; i<max; i++)
		{
			if((files[i] != null) && files[i].exists() && files[i].isFile())
			{
				File newFile = new File(toDirectory, files[i].getName());
				if((!newFile.exists()) || (!newFile.isFile()))
				{
					newFile.createNewFile();
				}
				else if(!overwrite)
					continue;
				
				FileInputStream inputStream = new FileInputStream(files[i]);
				FileOutputStream outputStream = new FileOutputStream(newFile);
				try
				{
					streamCopy(inputStream, outputStream);
					copyCount++;
				}
				finally
				{
					inputStream.close();
					outputStream.close();
				}
			}
		}
		
		return copyCount;
	}
}