/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLUtil.java,v 1.4 2008/01/24 02:29:24 apnan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.hyades.ui.util.ILogger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXParseException;

/**
 * Contains utility methods to manipulate the XML files and fragments.  This classes 
 * uses the Xerces parser.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class XMLUtil
{
	protected static ILogger logger = PrintStreamLogger.SYSTEM;

	/**
	 * Gets the logger used by this class.
	 * @return ILogger
	 */
	public static ILogger getLogger()
	{
		return logger;
	}

	/**
	 * Sets the logger used by this class.
	 * @param logger
	 */
	public static void setLogger(ILogger logger)
	{
		XMLUtil.logger = logger;
	}	

	/**
	 * Replaces some of the characters in the <code>string</code> by they
	 * XML symbol.
	 * @param string
	 * @return The changed String.
	 */
	public static String useXMLSymbols(String string)
	{
		if(string != null)
		{
			string = StringUtil.replace(string, "&",  "&amp;");
			string = StringUtil.replace(string, "<",  "&lt;");
			string = StringUtil.replace(string, ">",  "&gt;");
			string = StringUtil.replace(string, "\"", "&quot;");
			string = StringUtil.replace(string, "'",  "&apos;");
			string = StringUtil.replace(string, "\t",  "&#x9;");
			string = StringUtil.replace(string, "\n",  "&#xA;");
			string = StringUtil.replace(string, "\r",  "&#xD;");
		}
		
		return string;
	}
	
	
	/**
	 * Replaces XML symbols in the <code>xml</code> by they
	 * equivalent character.
	 * @param xml
	 * @return The changed xml.
	 */
	public static String removeXMLSymbols(String xml)
	{
		if(xml != null)
		{
			xml = StringUtil.replace(xml, "&amp;",  "&");
			xml = StringUtil.replace(xml, "&lt;",   "<");
			xml = StringUtil.replace(xml, "&gt;",   ">");
			xml = StringUtil.replace(xml, "&quot;", "\"");
			xml = StringUtil.replace(xml, "&apos;", "'");
			xml = StringUtil.replace(xml, "&#x9;",	"\t");
			xml = StringUtil.replace(xml, "&#xA;",	"\n");
			xml = StringUtil.replace(xml, "&#xD;",  "\r");
		}
		
		return xml;
	}
	
	/**
	 * Returns the XML string for the specified attribute and value.  The
	 * value is passed to the {@link #useXMLSymbols(String)} method.
	 * 
	 * <p>If the <code>value</code> is <code>null</code> then the return is
	 * an empty StringBuffer.
	 * 
	 * <p>If <code>asChild</code> is <code>true</code> than this method returns 
	 * <PRE><attribute>value</attribute></PRE>.  If it <code>false</code> the
	 * return is " <i>attribute</i>="<i>value</i>""
	 * 
	 * @param attribute
	 * @param value
	 * @param asChild
	 * @return StringBuffer. 
	 */
	public static StringBuffer createXMLAttribute(String attribute, String value, boolean asChild)
	{
		if(value == null)
			return new StringBuffer(0);
		value = useXMLSymbols(value);
		
		if(asChild)
			return new StringBuffer("<").append(attribute).append(">").append(value).append("</").append(attribute).append(">");
		return new StringBuffer(" ").append(attribute).append("=\"").append(value).append("\""); 
	}
	
	
	/*
	 * Xerces utils
	 */


	/**
	 * Returns an Element with a given name that was loaded 
	 * from the specified input stream.
	 * @param inputStream
	 * @param name
	 * @return Element 
	 */
	public static Element loadDom(InputStream inputStream, String name)
	{
		if((inputStream == null) || (name == null))
			return null;
		
		Document doc = null;
		
		try
		{
			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();

			docBuilderFactory.setIgnoringComments(true);
			docBuilderFactory.setValidating(true);
	
			DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
			docBuilder.setErrorHandler(createErrorHandler(false));
			doc = docBuilder.parse(inputStream);
		}
		catch(Throwable t)
		{
			logger.logError(t);
		}
		
		if(doc == null)
			return null;
					
		if(name != null)
		{
			NodeList list = doc.getElementsByTagName(name);
			if((list.getLength() > 0) && (list.item(0) instanceof Element))
				return (Element)list.item(0);
			else
				return null;
		}
			
		return (Element)doc.getFirstChild();
	}
	
	/**
	 * Returns an Element with a given name that was loaded 
	 * from the specified string.
	 * @param xml
	 * @param name
	 * @return Element 
	 */
	public static Element loadDom(String xml, String name)
	{
		if((xml == null) || (name == null))
			return null;
		
		byte[] bytes = null;
		try {
			bytes = xml.getBytes("UTF-8");
		}
		catch (UnsupportedEncodingException e) {
			// fall back to default encoding
			bytes = xml.getBytes();
		}

		return loadDom(new ByteArrayInputStream(bytes), name);
	}
	
	/**
	 * Returns a value identified by a name in a given <code>element</code>.  This 
	 * value can correspond to a regular XML attribute value or to a child node
	 * value.  In the last case there should be only one element with the given name.
	 * 
	 * <p>This method returns null if the value is a zero-length string.
	 * 
	 * @param element
	 * @param name
	 * @return String
	 */	
	public static String getValue(Element element, String name)
	{
		if((element == null) || (name == null))
			return null;
			
		String value = element.getAttribute(name);
		if((value == null) || (value.length() == 0))
		{
			NodeList list = element.getElementsByTagName(name);
			for(int i=0, max=list.getLength(); i<max; i++)
			{
				Node textNode = list.item(i);
				Node firstChild = list.item(0).getFirstChild();
				if(firstChild != null)
				{
					String auxValue = firstChild.getNodeValue();
					if((auxValue != null) && textNode.getParentNode().equals(element))
					{
						value = auxValue;
						break;
					}
				}
			}
		}
		if(value.length()== 0)
			return null;
			
		return value;
	}
	
	/**
	 * Returns a {@link NodeList} with all the children identified by a name 
	 * in a given <code>element</code>.   
	 * @param element
	 * @param name
	 * @return NodeList
	 */	
	public static NodeList getChildrenByName(Element element, String name)
	{
		if((element == null) || (name == null))
			return null;
			
		List list = new ArrayList();
		
		NodeList children = element.getChildNodes();
		for(int i=0, max=children.getLength(); i<max; i++)
		{
			if((children.item(i) instanceof Element) && name.equals(((Element)children.item(i)).getTagName()))
				list.add(children.item(i));
		}
		
		return createNodeList(list);
	}
	
	/**
	 * Returns a {@link NodeList} with all nodes of a given list.  The 
	 * returned node list will thrown a <code>ClassCastException</code> if there 
	 * is a element in the list that is not a {@link Node}
	 * @param element
	 * @param name
	 * @return NodeList
	 */	
	public static NodeList createNodeList(final List nodes)
	{
		if(nodes == null)
			return null;
			
		NodeList nodeList = new NodeList()
		{
			public int getLength()
			{
				return nodes.size();
			}

			public Node item(int index)
			{
				return (Node)nodes.get(index);
			}
		};
		
		return nodeList;
	}
	
	/**
	 * Returns a document based on the content of the specified file.
	 * @param filePath
	 * @return Document
	 */
	public static Document getXmlDom(String filePath)
	{
		try
		{
	
			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
			/*
			docBuilderFactory.setExpandEntityReferences(true);
			docBuilderFactory.setCoalescing(true);
			docBuilderFactory.setIgnoringElementContentWhitespace(true);
			docBuilderFactory.setNamespaceAware(true);
			*/
	
			docBuilderFactory.setIgnoringComments(true);
			docBuilderFactory.setValidating(true);
	
			DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
	
			final File xmlFile = new File(filePath);
	
	
			docBuilder.setErrorHandler(createErrorHandler(false));
			return (docBuilder.parse(xmlFile));
		}
		catch (Exception e)
		{
		}
	
		return null;
	}
	
	/**
	 * Creates the error handler used by this class.
	 * @param logMessages
	 * @return ErrorHandler
	 */
	protected static ErrorHandler createErrorHandler(final boolean logMessages)
	{
		ErrorHandler errorHandler = new ErrorHandler()
		{
			public void error(SAXParseException exception)
			{
				if(logMessages)
					getLogger().logError(exception);
			}

			public void fatalError(SAXParseException exception)
			{
				if(logMessages)
					getLogger().logError(exception);
			}

			public void warning(SAXParseException exception)
			{
				if(logMessages)
					getLogger().logError(exception);
			}
		};
		
		return errorHandler;
	}
	/**
	 * Get the value of the attribute tagName in element as an integer 
	 * @param element
	 * @param tagName
	 * @return int
	 */
	public static int getIntValue(Element element, String tagName)
	{
		try
		{
			return Integer.parseInt(getValue(element, tagName));
		}
		catch(NumberFormatException nfe)
		{
		}
		
		return 0;
	}
	
	/**
	 * @param node This method takes in an XML Element
	 * @return the XML node in String form
	 */
	public static StringBuffer convertElementToString(Element node)
	{
		StringBuffer temp = new StringBuffer();
		//no attributes and no children
		if ((!node.hasAttributes()) && (!node.hasChildNodes()))
		{
			temp.append("<" + node.getTagName() + "/>");
			return temp;
		}
		//atributes and no child nodes
		else if (node.hasAttributes() && (!node.hasChildNodes()))
		{
			temp.append("<" + node.getTagName());
			
			NamedNodeMap map = node.getAttributes();
			
			for (int attributes = 0; attributes < map.getLength(); attributes ++)
			{
				Node attr = map.item(attributes);
				temp.append(" " + attr.getNodeName() + "='" + node.getAttribute(attr.getNodeName()) + "'");
			}
			temp.append("/>");
			return temp;
		}
		//no attributes, but children
		else if (!(node.hasAttributes()) && node.hasChildNodes())
		{
			temp.append("<" + node.getTagName() + ">");
			
			NodeList children = node.getChildNodes();
			
			for (int iChild = 0; iChild < children.getLength(); iChild ++)
			{
				Element eChild = (Element)children.item(iChild);
				temp.append(convertElementToString(eChild));
			}
			temp.append("</" + node.getTagName() + ">");
			return temp;
		}
		//attributes and children
		else if (node.hasAttributes() && node.hasChildNodes())
		{
			temp.append("<" + node.getTagName());
			
			NamedNodeMap map = node.getAttributes();
			
			for (int attributes = 0; attributes < map.getLength(); attributes ++)
			{
				Node attr = map.item(attributes);
				temp.append(" " + attr.getNodeName() + "='" + node.getAttribute(attr.getNodeName()) + "'");
			}
			temp.append(">");
			
			NodeList children = node.getChildNodes();
			
			for (int iChild = 0; iChild < children.getLength(); iChild ++)
			{
				Element eChild = (Element)children.item(iChild);
				temp.append(convertElementToString(eChild));
			}
			
			temp.append("</" + node.getTagName() + ">");
			return temp;
			
		}
		return temp;
	}
	
}
