/********************************************************************** 
 * Copyright (c) 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: HeadlessConnectorService.java,v 1.1 2008/02/29 01:57:33 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.tptp.platform.common.internal.services;

import java.util.Properties;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.automation.server.AbstractService;
import org.eclipse.hyades.execution.core.task.ProgressiveTask;
import org.eclipse.hyades.execution.core.util.MutableObject;
import org.eclipse.hyades.internal.execution.local.control.Application;
import org.eclipse.hyades.security.internal.util.BaseConnectUtil;

/**
 * The headless connector service will establish a connection with the Agent Controller
 * in the appropriate security mode (secure or unsecure) and return back the
 * node object that was connected through (by updating the mutable node in out
 * parameter passed in). If credentials are required for a secure connection
 * then the credentials (user and password) must be passed in as service-related 
 * properties in the service properties state.
 * <p/>
 * It is important than a mutable node is passed in, this is where the node
 * return value is set and read once the service has completed and returned back
 * to the client.
 * <p/>
 * Support properties (default values) include:
 * <ul>
 * <li>host (<code>localhost</code>)</li>
 * <li>port (<code>10002</code>)</li>
 * <li>user* (<code>null</code>)</li>
 * <li>password* (<code>null</code>)</li>
 * <li>mutableNode (<code>null</code>)</li>
 * </ul>
 * <p/>
 * *User and password are optional if connecting in a unsecure mode.
 * <p/>
 * <b>Implementation Note:</b> It is desirable for the dependency on test and profile to 
 * go away and move this service into the execution framework (as soon as the 
 * functionality supplied by the profile connect utility is migrated into the 
 * execution framework.
 * <p/>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Scott E. Schneider
 * @version February 28, 2008
 * @since   February 28, 2008
 */
public class HeadlessConnectorService extends AbstractService implements Application {

	/**
	 * Used for serialization purposes
	 */
	private static final long serialVersionUID = -5537228646033407333L;
	
	/**
	 * The connector service will retrieve the various service-related
	 * properties from the service properties state and then use the profile
	 * connect utilities to make a connection to the Agent Controller (using
	 * secure or unsecure mode).
	 * <p/>
	 * Support properties (default values) include:
	 * <ul>
	 * <li>host (<code>localhost</code>)</li>
	 * <li>port (<code>10002</code>)</li>
	 * <li>user* (<code>null</code>)</li>
	 * <li>password* (<code>null</code>)</li>
	 * <li>mutableNode (<code>null</code>)</li>
	 * </ul>
     * *User and password are optional if connecting in a unsecure mode.
	 * <p/>
	 * 
	 * @return The status (IStatus) of the service.
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public Object execute() {
		
		Properties properties = this.getProperties();
		String host = properties.getProperty("host");
		String port = properties.getProperty("port");
		String user = properties.getProperty("user");
		String password = properties.getProperty("password");
		MutableObject mutableNode = ((MutableObject) (properties.get("mutableNode")));
		
		return (this.connect((host == null ? "localhost" : host), (port == null ? "10002" : port), user, password, mutableNode));
	}

	/**
	 * Resolves the name of the service.
	 * <p/>
	 * 
	 * @return Name of the service
	 * @see java.security.Principal#getName()
	 */
	public String getName() {
		return "Agent Controller headless connector service";
	}

	/**
	 * Underlying connect method delegated on to connect utility in the 
	 * security package, used for unsecure and secure connections.
	 * 
	 * @param host
	 *            host to connect on
	 * @param port
	 *            port to use for the connection
	 * @param user
	 *            user to be used for a secure connection
	 * @param password
	 *            password to be used for a secure connection
	 * @param mutableNode
	 *            node for storing the return value
	 * @return The status (IStatus) of the service.
	 */
	private IStatus connect(final String host, final String port, final String user, final String password, final MutableObject mutableNode) {

		// A mutable object to hold the found agent
		final MutableObject mutableStatus = new MutableObject();

		// Progressive task to connect to agent controller
		ProgressiveTask connector = new ProgressiveTask("Agent Controller Connect", new Runnable() {
			
			public void run() {
				
				try {

					// Use the headless version of the internal connection utility:
					BaseConnectUtil connectUtil = new BaseConnectUtil(host, port, user, HeadlessConnectorService.this);

					int result = connectUtil.connect(password, false);

					// Handle results, success and exceptions
					switch (result) {
					
					case BaseConnectUtil.CONNECTION_SUCCESS: {
						mutableStatus.set(new Status(IStatus.OK, HeadlessConnectorService.this.getName(), result, "", null));
						mutableNode.set(connectUtil.getNode());
						break;
					}
					case BaseConnectUtil.CONNECTION_CONTROLLER_ERROR:
					case BaseConnectUtil.CONNECTION_HOST_ERROR:
					case BaseConnectUtil.CONNECTION_PORT_ERROR:
					case BaseConnectUtil.CONNECTION_SECURITY_NOT_SUPPORTED:
					default: {
						mutableStatus.set(new Status(IStatus.ERROR, HeadlessConnectorService.this.getName(), result, "", null));
					}
					}

				} catch (Throwable t) {
					//Ignore.
				}
			}
		}, new NullProgressMonitor(), 10000);

		// Blocks until runnable completes normally or is canceled
		connector.execute(ProgressiveTask.Synchronicity.SYNCHRONOUS);

		// Return status (and node) to service consumer
		return (IStatus) mutableStatus.getAndClear();
	}
}
