/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DSashLayout.java,v 1.2 2008/05/23 14:12:16 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 10 nov. 2004
 */
package org.eclipse.tptp.platform.report.ui.swt.widgets.internal;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.swt.widgets.Sash;
import org.eclipse.tptp.platform.report.drivers.ui.internal.SWTViewer;


/**
 * @author kdarcour
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 *
 * A DSashLayout controls the position and size
 * of the children of a composite widget.
 * This class will allow to create a layout with x SWTViewer
 * separated by sash which have the same style (VERTICAL or 
 * HORIZONTAL). 
 */
public class DSashLayout extends Layout {

	private double default_rate;
	private int style = SWT.HORIZONTAL;
	private boolean first_layout = true;
	private int sash_width = 4;
	
	/**
	 * Constructor
	 */
	public DSashLayout() {
		super();
	}

	/**
	 * Computes and returns the size of the specified
	 * composite's client area according to this layout.
	 * @param composite a composite widget using this layout
	 * @param wHint width (<code>SWT.DEFAULT</code> for minimum)
	 * @param hHint height (<code>SWT.DEFAULT</code> for minimum)
	 * @param flushCache <code>true</code> means flush cached layout values
	 * @return a point containing the computed size (width, height)
	 */
	protected Point computeSize(Composite composite, int wHint, int hHint, boolean flushCache) {
		return new Point(wHint, hHint);

//		Point extent;
//		if (style == SWT.HORIZONTAL) {
//			extent = layoutHorizontal (composite, false, wHint, flushCache);
//		} else {
//			extent = layoutVertical (composite, false, hHint, flushCache);
//		}
//		if (wHint != SWT.DEFAULT) extent.x = wHint;
//		if (hHint != SWT.DEFAULT) extent.y = hHint;
//		return extent;
	}
	
	private Point computeSize (Control control, boolean flushCache) {
		return control.computeSize (SWT.DEFAULT, SWT.DEFAULT, flushCache);
	}

	/**
	 * Lays out the children of the specified composite
	 * according to this layout.
	 * <p>
	 * The layout algorithm takes into account the children number
	 * and the sash in case they are used as separator.
	 *
	 * @param composite a composite widget using this layout
	 * @param flushCache <code>true</code> means flush cached layout values
	 */
	protected void layout(Composite composite, boolean flushCache) {
		Rectangle clientArea = composite.getClientArea();
		Control children[] = composite.getChildren();
		
		// Verifications
		verifyCompositeContents(composite);	// throws exception if composite child requirements are not respected
		
		int count = children.length;
		setDefaultRate(count);
		style = ((Sash)children[2]).getStyle();

		if (style == SWT.HORIZONTAL) {
			layoutHorizontal (composite, true, clientArea, flushCache);
		} 
		else {   // style == SWT.VERTICAL
			layoutVertical (composite, true, clientArea, flushCache);
		}
	}

	void layoutHorizontal (Composite composite, boolean move, Rectangle clientArea, boolean flushCache) {
		int height = clientArea.height;
		int width = clientArea.width;
		int posx = clientArea.x;
		int posy = clientArea.y;
		Control [] children = composite.getChildren ();
		int count = children.length;
		int sash_count = (count - 1)/2;
		
		Sash sash;
		Rectangle sash_rect;
		
		// compute with default_rate only for the first-first time
		if ( first_layout ) {
			int mid_width = 0;
			for (int i = 2; i < count; i = i+ 2) {
				sash = (Sash) (children[i]);
				sash_rect = sash.getBounds();
				sash_rect.x = mid_width + width / ((count + 1)/2);
				mid_width = sash_rect.x + sash_width;
				
				sash.setBounds(sash_rect.x, sash_rect.y, sash_width, height);	
			}
		}
		SWTViewer viewer;
		for (int j = 2; j < count; j = j+2) {
			viewer = (SWTViewer)(children[j-1]);
			sash = (Sash)(children[j]);
			
			int w_viewer = sash.getBounds().x - posx;  //size at the left of sash
		    
			viewer.setBounds (posx, posy, w_viewer, height);

		    posx = sash.getBounds().x + sash_width;  // Update posx
			width = posx + width - (sash.getBounds().x + sash_width +1); // Update width
			
            // compute bounds of the last viewer
			if ( j == count -1 ) {
			  SWTViewer viewer2 = (SWTViewer)(children[j+1]);
			  viewer2.setBounds(posx, posy, width, height);
			}
		}
		first_layout = false;
	}

	void layoutVertical (Composite composite, boolean move, Rectangle clientArea, boolean flushCache) {
		int height = clientArea.height;
		int width = clientArea.width;
		int posx = clientArea.x;
		int posy = clientArea.y;
		Control [] children = composite.getChildren ();
		int count = children.length;
		int sash_count = (count - 1)/2;
		
		Sash sash;
		Rectangle sash_rect;
		
		// compute with default_rate only for the first-first time
		if ( first_layout ) {
			int mid_height = 0;
			for (int i = 2; i < count; i = i+ 2) {
				sash = (Sash) (children[i]);
				sash_rect = sash.getBounds();
				sash_rect.y = mid_height + height / ((count + 1)/2);
				mid_height = sash_rect.y + sash_width + 1;
				
				sash.setBounds(sash_rect.x, sash_rect.y, width, sash_width);	
			}
		}
		for (int j = 2; j < count; j = j+2) {
			SWTViewer viewer = (SWTViewer)(children[j-1]);
			sash = (Sash)(children[j]);
			
			int h_viewer = sash.getBounds().y - posy;  //size at the top of sash
		    
			viewer.setBounds (posx, posy, width, h_viewer);

		    posy = sash.getBounds().y + sash_width;  // Update posx
			height = posy + height - (sash.getBounds().y + sash_width + 1); // Update width
			
			// compute bounds of the last viewer
			if ( j == count -1 ) {
			  SWTViewer viewer2 = (SWTViewer)(children[j+1]);
			  viewer2.setBounds(posx, posy, width, height);
			}
		}
		first_layout = false;
	}


	/**
	 * This method verifies that the children order and type are
	 * compatible with the definition of this layout. 
	 * If not, it throws IllegalArgumentException.
	 * Here is the accepted set:
	 * 		Control , (Sash , Control)+
	 * @param composite is the parent Composite using this layout
	 * @see IllegalArgumentException
	 */
	private void verifyCompositeContents( Composite composite )
	{
		Control children[] = composite.getChildren();

		// Verify there is children
		if ( children == null ) 
			throw new IllegalArgumentException("This layout should be composed of at least 3 children: a Viewer, a Sash and another Viewer");
		
		int count = children.length;
		
		// Verify there is at least 3 children
		if ( count < 3 )  
			throw new IllegalArgumentException("This layout should be composed of at least 2 Controls and 1 Sash");
		
		// Verify there is an impait number of children (3 - 5 - 7 ...)
		if ( count%2 != 0)
			throw new IllegalArgumentException("This layout should be composed of a sequence [Control , (Sash , Control)+]");
		
		// Verify the child sequence is like that: [Control , (Sash , Control)+]
		Control child = children[0];
		if (! (child instanceof SWTViewer)) {
			throw new IllegalArgumentException("First child has to be an instance of SWTViewer");
		}
		for ( int i = 1; i < count - 1 ; i = i + 2 )
		{
			child = children[i];
			if (! (child instanceof Sash)) {
				throw new IllegalArgumentException("This layout should be composed of a sequence [Control , (Sash , Control)+]");
			}
			child = children[i+1];
			if (! (child instanceof SWTViewer)) {
				throw new IllegalArgumentException("This layout should be composed of a sequence [Control , (Sash , Control)+]");
			}
		}
		
		// Verify if there is more than 3 children that all Sashs are in the same position (horizontal or vertical)
		if ( count > 3 ) {
			int style = ((Sash)children[2]).getStyle();
			for (int j = 4; j < count ; j = j+2 ) {
				if ( ! ((((Sash)children[2]).getStyle()) == style )) {
					throw new IllegalArgumentException("This layout should be composed of sashs with the same style");
				}
			}
		}
	}

	/**
	 * Returns the default rate between different SWTViewers
	 * @return a double betwwen 0 and 1
	 */
	public double getDefault_rate() {
		return default_rate;
	}
	
	/**
	 * set a default_rate between different SWTViewers. 
	 * The rate will be the same for each frame, the last frame is the rest. 
	 * @param default_rate
	 */
	public void setDefault_rate(double default_rate) {
		this.default_rate = default_rate;
	}
	
	/**
	 * Computes a default_rate depending on the child number
	 * @param count represent the child number of the parent composite
	 */
	private void setDefaultRate(int count) {
		if ( count <  3 ) this.default_rate = 1.0;
		else {
			this.default_rate = 1 / ((count + 1)/2);
		}
	}
	public int getSashWidth() {
		return sash_width;
	}
	public void setSashWidth(int sash_width) {
		this.sash_width = sash_width;
	}
}

