/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SWTPainter.java,v 1.5 2008/05/23 14:12:21 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/



package org.eclipse.tptp.platform.report.drivers.ui.internal;

import java.util.Iterator;

import org.eclipse.tptp.platform.report.core.internal.*;
import org.eclipse.tptp.platform.report.drawutil.internal.DSymbolRegistry;
import org.eclipse.tptp.platform.report.drawutil.internal.DrawUtilIGC;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.drawutil.internal.ISymbol;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.*;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.igc.brushes.internal.FullGradientBrush;
import org.eclipse.tptp.platform.report.igc.brushes.internal.GradientBrush;
import org.eclipse.tptp.platform.report.igc.internal.IBrush;
import org.eclipse.tptp.platform.report.igc.internal.IFont;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IImage;
import org.eclipse.tptp.platform.report.igc.internal.IPen;
import org.eclipse.tptp.platform.report.igc.internal.IShape;
import org.eclipse.tptp.platform.report.igc.internal.ISize;
import org.eclipse.tptp.platform.report.igc.util.internal.Font;
import org.eclipse.tptp.platform.report.igc.util.internal.GradientGenerator;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;
import org.eclipse.tptp.platform.report.render.internal.DRenderRegistry;
import org.eclipse.tptp.platform.report.render.internal.IRender;
import org.eclipse.tptp.platform.report.render.internal.NullRenderMonitor;
import org.eclipse.tptp.platform.report.tools.internal.IDIImageProvider;




/**
  * This class provides a default Paint event handler used by the SWTViewer.
  * This handler manages the SWT rendering for each item and associated cell  
  * 
  * @see SWTViewer#getDefaultMouseEventHandler()
  * 
  * This class inherits of DExtensible class. You can extend the default handler if
  * you want to render a derived JScrib Item.
  * 
  * @see DExtensible
  * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
  *   
  */ 
public class SWTPainter extends DExtensible {
	  
	public static class PainterEvent
	{
        /** Use this gc to paint */
		public IGC        gc;
        
		/** The zoom factor used with the display */
		public float     zoom; 
		
		/** The IImageProvider object used to get IImage in the document */
		public IDIImageProvider iimgProvider;
		
        /** current cell to paint */
        public TAbstractCell    cell;
        
        /** location of cell in contents coordinates */
        public int abs_cell_x, abs_cell_y; 
        
        /** paint area in viewport coordintates (cell bounds in viewport), valid after draw Cell) */
		public Rect paintArea ;

        /** clipping rectangle, intersection of paintArea and clipArea, in viewport coordinates.
         * This is the part of cell, requested to be painted. (valid after drawCell)
         */
        public Rect cellClip ;

        /** clipping rectangle in contents coordinates (this rect. is inner of viewArea)
         * this is the top-level clipping requested by original(system) paint event.
         * (not the intersection between top level clipping rect and cell area, see clipCell ).
         */
	    public Rect clipArea;

        /** style for cell, one create it is responsible to dispose it, see drawCell */		
		public IGCDStyle style; 
		
        /** viewport area in contents coordinates */        
		public Rect viewArea;
		
        /** true if draw selection is required */
        public boolean   drawSelection;
        
        /** cell focus mark */
        public TCellFocusMark    focus;
        /** true if the focus is required */
        public boolean   drawFocus;
        /** start index of selection */
        public int       start;
        /** end index of selecton */
        public int       end;   
    
        public PainterEvent( IGC _gc, TAbstractCell _cell, float _zoom,
        		           IDIImageProvider iimgp,
		                   TCellFocusMark _focus, boolean _drawFocus,
                           boolean _drawSelection, int _start, int _end,
                           int _abs_cell_x, int _abs_cell_y,
                           int vax, int vay, int vaw, int vah, //viewArea in contents coordinates
                           int clx, int cly, int clw, int clh //clip area in contents coordinates 
					   )  
        {
          gc=_gc; cell=_cell; 
		  zoom = _zoom;
		  iimgProvider = iimgp; 
          drawSelection=_drawSelection;
          focus = _focus;
          drawFocus=_drawFocus;  
          start=_start; end=_end;
          abs_cell_x=_abs_cell_x; abs_cell_y=_abs_cell_y;
          viewArea = new Rect( vax,vay,vaw,vah );
          clipArea = new Rect( clx,cly,clw,clh );
          paintArea = new Rect( 0,0,0,0 ); //completed in drawCell.
          cellClip  = new Rect( 0,0,0,0 ); //completed in drawCell.
        }
        
        //Copy a paint event...
        public PainterEvent( PainterEvent e )
        {
          gc=e.gc; cell=e.cell;
          zoom = e.zoom;
          drawSelection=e.drawSelection;
          focus = e.focus;
          drawFocus=e.drawFocus;  
          start=e.start; end=e.end;
          abs_cell_x=e.abs_cell_x; abs_cell_y=e.abs_cell_y;
          viewArea = new Rect( e.viewArea.getX(),e.viewArea.getY(),e.viewArea.getW(),e.viewArea.getH() );
          clipArea = new Rect( e.clipArea.getX(),e.clipArea.getY(),e.clipArea.getW(),e.clipArea.getH() );
          paintArea = new Rect( e.paintArea.getX(),e.paintArea.getY(),e.paintArea.getW(),e.paintArea.getH() );
          cellClip  = new Rect( e.cellClip.getX(),e.cellClip.getY(),e.cellClip.getW(),e.cellClip.getH() );
          iimgProvider = e.iimgProvider;
        }

	}

	private Rect crect = new Rect(0,0,0,0);
	private Rect rtmp = new Rect(); 
	public void doMethod(DDocument doc, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
		//e.gc.drawRectangle(e.paintArea);
		drawWallpaper( e, e.paintArea.getX(),e.paintArea.getY(), e.paintArea.getW(), e.paintArea.getH() );
        drawChildrenCell( e );
	}
	
	public void doMethod(DSection section, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
        drawChildrenCell( e );
	}
	
	public void doMethod(DHeader h, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
        drawChildrenCell( e );
	}
		
	public void doMethod(DFooter f, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
        drawChildrenCell( e );
	}
		
	public void doMethod(DPageCounter section, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
		if (!(e.cell instanceof TCellText)) return;
		drawTextCell((TCellText)e.cell, e, IDAlignment.LEFT);   
	}
	
	public void doMethod(IDLink link, DExtensible ext, Object event)
    {
    	// do nothing
    }
		
	public void doMethod(DTitle title, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
		if (!(e.cell instanceof TCellText)) return;
		drawTextCell((TCellText)e.cell, e, IDAlignment.LEFT);
	}
	
	public void doMethod(DList list, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
		TCellLinePrefix cell = (TCellLinePrefix)e.cell;
		IGCDStyle s = new IGCDStyle(cell.getEntryItem(),e.zoom);
	
		IFont oldfont = e.gc.setFont(s.getFont());	
		
		IBrush oldbrush = null;
		switch (list.getFormat())
		{
			case DStyle.GLYPH:
				 ISymbol sy = cell.getSymbol();
			     
				 rtmp.setRect(e.paintArea.getX(),e.paintArea.getY()+e.cell.getHeight()/3, 
								  e.cell.getWidth(),e.cell.getHeight()/3);
				 int c=s.getFore();
				 IPen oldpen = e.gc.setPen( new LineStylePen( c));
				 c = s.getBack();
				 oldbrush = e.gc.setBrush( new SolidBrush( c));
				 sy.draw( e.gc, rtmp);
				 e.gc.setPen(oldpen);
				 break;
			default:
			     if (cell.getNumber()!=null)
			     {
			        c=s.getFore();
			        oldbrush = e.gc.setBrush( new SolidBrush( c));
			        
					cell.setItem(cell.getEntryItem());
					drawTextCell(cell, cell.getNumber(), 0, cell.getNumber().length(), e, IDAlignment.LEFT);
					cell.setItem(list);
				 }
				 break;
		}
		
		e.gc.setBrush(oldbrush);
		e.gc.setFont(oldfont);
	}
	
	public void doMethod(DParagraph p, DExtensible ext, Object event)
	{
		drawChildrenCell((PainterEvent)event);
	}
    
	public void doMethod(DTag t, DExtensible ext, Object event)
	{
	}
	
	public void doMethod(DText text, DExtensible ext, Object event)
	{
	    PainterEvent e = (PainterEvent)event;
	    
	    if (!(e.cell instanceof TCellText)) return;
	    
		drawTextCell( ((TCellText)e.cell), (PainterEvent)event, IDAlignment.LEFT );
		drawFocus( (PainterEvent)event);
	}
		
	public void doMethod(DImage image, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
		
		if (!(e.cell instanceof TCellImage)) return;
		
		IImage pix = ((TCellImage)e.cell).getImage();
		if (pix != null)
		{
		   e.gc.drawImage(pix,0,0, pix.getWidth(), pix.getHeight(),
						  e.paintArea.getX(), e.paintArea.getY(), (int)(e.cell.getWidth()), (int)(e.cell.getHeight()));
		}
		
		drawFocus( (PainterEvent)event);
	}
	
	public void doMethod(DCellText celltext, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
		
		//if (!(e.cell instanceof TCellText)) return;
		
		if (e.cell instanceof TCellText)
		    drawTextCell( ((TCellText)e.cell), e, IDAlignment.LEFT );
		else
		{
		    drawCellBackground( e );
			drawCellBorder( e );
		}
		
		/*if (celltext.getText()==null) //((TCellText)e.cell).getText()== null)
		{
			drawCellBackground( e );
			drawCellBorder( e );
		}
		else
			drawTextCell( ((TCellText)e.cell), e, IDAlignment.LEFT );*/
			
        drawChildrenCell( e );
	}	
	
	public void doMethod(DCell cell, DExtensible ext, Object event)
	{    
		// Must be after DCellText because DCellText extends DCell...
		PainterEvent e = (PainterEvent)event;
		drawCellBackground( e );
		drawCellBorder( e );
		
        drawChildrenCell( e );
	}
	
	public void doMethod(DTable table, DExtensible ext, Object event)
	{    
		drawChildrenCell((PainterEvent)event);
	}
	
	public void doMethod(DPopup popup, DExtensible ext, Object event)
	{   
		PainterEvent e = (PainterEvent)event;	
        drawChildrenCell( e );
	}
	
	public void doMethod(IDRenderable object, DExtensible ext, Object event)
	{
	    if (object.getFirstChild() instanceof DGraphic)
	        object = (IDRenderable)object.getFirstChild();
	    
		PainterEvent e = (PainterEvent)event;
	    TAbstractCell cell = e.cell;
	    if (cell instanceof TCellFocusMark) return;
	    
        TCellData celldata = cell instanceof TCellData ? (TCellData)cell : null ;
		IRender render_ = DRenderRegistry.GetRender( object );		

		//wallpaper is applyed by this.
        IBrush ob = e.gc.getBrush();
        IPen   op = e.gc.getPen();
        IShape oc = e.gc.getClipping();
		render_.draw( e.gc, e.paintArea, object, e.zoom, e.iimgProvider, new NullRenderMonitor(), celldata );
		drawFocus( (PainterEvent)event );
        e.gc.setBrush( ob );
        e.gc.setPen( op );
        e.gc.setClipping( oc );
	}
	
    protected void drawGradientRect( IGC gc, int x, int y, int w, int h, int rgba1, int rgba2, boolean vertical )
    {
      GradientBrush nbrush = new GradientBrush(rgba1, rgba2, vertical);
      nbrush.setBounds(x, y, w, h);
      IBrush obrush = gc.setBrush(nbrush);
     
      gc.fillRect( x, y, w, h );
      gc.setBrush(obrush);
    }
    
	public void doMethod(DLine line, DExtensible ext, Object event)
	{
		PainterEvent e = (PainterEvent)event;
		
		int lw = line.getSize();
       
		IBrush oldbrush = e.gc.getBrush();
		
		crect.setRect( e.cell.getX(), e.cell.getY(), e.cell.getWidth(), e.cell.getHeight()) ;
        int ly = e.paintArea.getY()+(crect.getH()-lw)/2;       

		if( line.isType( DLine.HGRADIENT ) )
		{
		  drawGradientRect( e.gc, e.paintArea.getX(), ly, crect.getW(), lw, e.style.getFore(), e.style.getBack(), false);
		} 
		else if( line.isType( DLine.HGRADIENTR ) )
        {
          drawGradientRect( e.gc, e.paintArea.getX(), ly, crect.getW(), lw, e.style.getBack(), e.style.getFore(), false);
        } 
        else if( line.isType( DLine.HGRADIENT2 ) )
        {
          int xl = e.paintArea.getX();
          int xm = e.paintArea.getX()+crect.getW()/2;
          int xr = e.paintArea.getX()+crect.getW();
          drawGradientRect( e.gc, xl, ly, xm-xl, lw, e.style.getBack(), e.style.getFore(), false);
          drawGradientRect( e.gc, xm, ly, xr-xm, lw, e.style.getFore(), e.style.getBack(), false);
        } 
        else if( line.isType( DLine.HGRADIENT2R ) )
        {
          int xl = e.paintArea.getX();
          int xm = e.paintArea.getX()+crect.getW()/2;
          int xr = e.paintArea.getX()+crect.getW();
          drawGradientRect( e.gc, xl, ly, xm-xl, lw, e.style.getFore(), e.style.getBack(), false);
          drawGradientRect( e.gc, xm, ly, xr-xm, lw, e.style.getBack(), e.style.getFore(), false);
        } 
        else if( line.isType( DLine.VGRADIENT ) )
        {
          drawGradientRect( e.gc, e.paintArea.getX(), ly, crect.getW(), lw, e.style.getFore(), e.style.getBack(), true);
        } 
        else if( line.isType( DLine.VGRADIENTR ) )
        {
          drawGradientRect( e.gc, e.paintArea.getX(), ly, crect.getW(), lw, e.style.getBack(), e.style.getFore(), true);
        } 
        else if( line.isType( DLine.VGRADIENT2 ) )
        {
          int ym = ly+lw/2;
          int yb = ly+lw;
          drawGradientRect( e.gc, e.paintArea.getX(), ly, crect.getW(), ym-ly, e.style.getBack(), e.style.getFore(), true);
          drawGradientRect( e.gc, e.paintArea.getX(), ym, crect.getW(), yb-ym, e.style.getFore(), e.style.getBack(), true);
        } 
        else if( line.isType( DLine.VGRADIENT2R ) )
        {
          int ym = ly+lw/2;
          int yb = ly+lw;
          drawGradientRect( e.gc, e.paintArea.getX(), ly, crect.getW(), ym-ly, e.style.getFore(), e.style.getBack(), true);
          drawGradientRect( e.gc, e.paintArea.getX(), ym, crect.getW(), yb-ym, e.style.getBack(), e.style.getFore(), true);
        } 
        else
		{   //SOLID is de default   
            e.gc.setBrush(new SolidBrush(e.style.getFore()));
			e.gc.fillRect(e.paintArea.getX(), ly, crect.getW(), lw);
		}
		
		e.gc.setBrush(oldbrush);
	}
	
	public void drawFocus( PainterEvent evt )
	{
		if (!evt.drawFocus) return;
	//	if (evt.focus == null) return;
		
		IBrush oldbrush = evt.gc.setBrush(new SolidBrush(RGBA.WHITE));
		IPen open = evt.gc.setPen(new LineStylePen(RGBA.BLACK));
	    evt.gc.drawFocus(evt.paintArea.getX()-1, evt.paintArea.getY()-1, evt.paintArea.getW()+1, evt.paintArea.getH()+1);
	    evt.gc.setPen(open);
	    evt.gc.setBrush(oldbrush);
	}
	
	public void drawTextCell( TCellText cell, PainterEvent evt, int alignment )
    {
	    drawTextCell(cell, cell.getText(), cell.getStart(), cell.getLen(), evt, alignment);
    }

    public void drawTextCell( TAbstractCell cell, String text, int startPos, int len, PainterEvent evt, int alignment )
    {
        
       IGC gc = evt.gc; 
         
       Rect r_area = evt.paintArea;
       boolean hasSelection = evt.drawSelection ;
       int start = evt.start;
       int end   = evt.end;
      
	   if(text != null)
	      text = text.substring(startPos,startPos+len);
	  
	   rtmp.setRect( r_area.getX(), r_area.getY(), r_area.getW(), r_area.getH() );
	  
	   Font font = IGCDStyle.GetFont(cell.getItem(),evt.zoom);
	   
	  // int font_style = Font.NORMAL;
	   
	   IBrush oldbrush = gc.setBrush(new SolidBrush(evt.style.getBack()));
	   if (evt.style.getBack()!=0)
	      evt.gc.fillRect(evt.cellClip.getX()+1, evt.cellClip.getY()+1, evt.cellClip.getW()-1, evt.cellClip.getH()-1);    

	   gc.setBrush(new SolidBrush(evt.style.getFore()));
	   
	   IFont oldfont = gc.setFont(font);
	   DrawUtilIGC.drawText(gc,  rtmp, alignment, text);

	   if(hasSelection)
	   {
	   	  if (start>end)
	   	  {
	   	  	int tmp = start;
	   	  	start = end;
	   	  	end = tmp;
	   	  }
	   	
		  int cpos = cell.getPosition();
		  int clen = len;
		  //int clen = cell.getLen();
		  if((end>=cpos)&&(start<cpos+clen))
		  {
			String beforeS=null;
			String selectedS=null;
			if(start > cpos)
			{
				beforeS=text.substring(0,start-cpos);
				if(end<(cpos+clen-1))
					selectedS=text.substring(start-cpos,end-cpos); //+1);
				else
					selectedS=text.substring(start-cpos);
			}
			else
			{
				if(end<(cpos+clen-1))
					selectedS=text.substring(0,end-cpos); //+1);
				else
					selectedS=text;
			}
			if(beforeS != null)
			{
				ISize point = gc.textExtent(beforeS);
				rtmp.setLeft(rtmp.left()+point.getW());
			}
			if(end<(cpos+clen-1))
			{
				ISize point = gc.textExtent(selectedS);
				rtmp.setWidth(point.getW());
			}
			
			SolidBrush sb = new SolidBrush(gc.getSystemColor(IGC.C_LIST_SELECTION_BACKGROUND));
			gc.setBrush(sb);
			gc.fillRect(rtmp);
			sb.setRGBA(gc.getSystemColor(IGC.C_LIST_SELECTION_TEXT));
            gc.setBrush(sb);
            DrawUtilIGC.drawText(gc, rtmp, alignment, selectedS);
		  }
	   }
	   
	   gc.setFont(oldfont);
	   gc.setBrush(oldbrush);
     }

     protected void drawCellBorder( PainterEvent evt )
     {
       DCell dcell = (DCell)evt.cell.getItem();
       
       DRow row = (DRow)dcell.getParent();
	   DTable table = (DTable)row.getParent();
	   int border = (int)(table.getBorder()*evt.zoom);
	   if (border>0) {
         LineStylePen pen = new LineStylePen(evt.style.getFore());
	     pen.setLineWidth(border);     
       
         IPen open = evt.gc.setPen(pen);
       
	     int mid = -((border/2)+(border%2));	   
	     evt.gc.drawRect( evt.paintArea.getX()+mid,
	                    evt.paintArea.getY()+mid,
	                    evt.paintArea.getW()-(mid+mid)-1,
	                    evt.paintArea.getH()-(mid+mid)-1 );
	     evt.gc.setPen(open);
	   }
     }
     
     protected void drawCellBackground( PainterEvent evt)
	 {  
        IBrush obrush = evt.gc.setBrush(new SolidBrush(evt.style.getBack()));
        evt.gc.fillRect(evt.paintArea.getX(), evt.paintArea.getY(), evt.paintArea.getW(), evt.paintArea.getH());
    	drawWallpaper( evt, evt.paintArea.getX(),evt.paintArea.getY(), evt.paintArea.getW(), evt.paintArea.getH() );    	
     	evt.gc.setBrush(obrush);
     }

     public void drawCell( PainterEvent pe )   
     {
		IDItem item = pe.cell.getItem();
		if(item == null) return;	
		
		Rect r = Rect.Intersection( pe.abs_cell_x, pe.abs_cell_y, pe.cell.getWidth(), pe.cell.getHeight(),
		                            pe.clipArea.getX(), pe.clipArea.getY(), pe.clipArea.getW(),  pe.clipArea.getH() );  
//System.out.println("Paint/drawCell/ cell intersection ? clip="+r+" pe.clipA="+pe.clipArea);
		if (r==null) return;	

        //IGCDStyle sty=null;
        IGCDStyle saved_style = pe.style;
		try
		{	            
			pe.style = /*sty =*/ new IGCDStyle( item, pe.zoom );
			
            pe.cellClip = new Rect( r.getX(), r.getY(), r.getW(), r.getH() );
            pe.cellClip.setX(pe.cellClip.getX() - pe.viewArea.getX()); 
            pe.cellClip.setY(pe.cellClip.getY() - pe.viewArea.getY());
            
            pe.gc.setClipping(new Rect(pe.clipArea.getX()-pe.viewArea.getX(), pe.clipArea.getY()-pe.viewArea.getY(), pe.clipArea.getW(), pe.clipArea.getH()));
            
			pe.drawFocus = (pe.focus!=null)?pe.focus.hasFocus(pe.cell):false;
            
			pe.paintArea.setX( pe.abs_cell_x - pe.viewArea.getX());
			pe.paintArea.setY( pe.abs_cell_y - pe.viewArea.getY());
			pe.paintArea.setWidth( pe.cell.getWidth());
			pe.paintArea.setHeight( pe.cell.getHeight());
			
	 
	        pe.gc.setFont(pe.style.getFont());  
	        
			invokeDoMethod(item, this, pe);
  		}
		finally
		{
            pe.style = saved_style;
		}	              
     }
     
     public void drawChildrenCell( PainterEvent pe )
	 {
     	
        TAbstractCell save_cell = pe.cell;

	 	if (save_cell==null || save_cell.getChild()==null) return; 
	 
		int clip_left   = pe.clipArea.getX();
		int clip_right  = clip_left + pe.clipArea.getW();
		int clip_top    = pe.clipArea.getY();
		int clip_bottom = clip_top + pe.clipArea.getH();
    
        int save_x = pe.abs_cell_x;
        int save_y = pe.abs_cell_y;
        
		for(TAbstractCell I=save_cell.getChild();(I!=null);I=I.getNext())
		{
            //cell I rect in contents coordinates.
            int abs_I_left   = save_x    + I.getX();
            int abs_I_top    = save_y    + I.getY();
			int abs_I_right  = abs_I_left+ I.getWidth();
            int abs_I_bottom = abs_I_top + I.getHeight();
		 
		    // part of cell is visible ?
			if( ( clip_top    > abs_I_bottom )
			 || ( clip_bottom < abs_I_top   )
			 || ( clip_left   > abs_I_right  )
			 || ( clip_right  < abs_I_left   ) )
            {
              continue;
            }
    
            pe.abs_cell_x = abs_I_left;
            pe.abs_cell_y = abs_I_top;
            pe.cell = I;            
            
			//Draw current cell
            drawCell( pe );
		}
    
        pe.abs_cell_x = save_x;
        pe.abs_cell_y = save_y;
        pe.cell = save_cell;
	 }
   
     /**
      * Draw wallpaper get from cell in event in the given rectangle.
      */
//TODO: need clip area ??
     public void drawWallpaper( PainterEvent evt, int x, int y, int w, int h )
     {
       if( evt==null || evt.cell==null ) return ;
       IDItem item = evt.cell.getItem();
       if( item ==null ) return ;
       IDWallpaper bi = item.getStyle()!=null ? item.getStyle().getWallpaper() : null;
       
       if( bi==null) return ;
       
       IImage image = evt.iimgProvider.getImage( bi.getName() );
       if( image!=null )
           DrawUtilIGC.drawImage( evt.gc, image, x,y,w,h, evt.zoom, bi.getEffect());
     }
     
     /** paint a DFolder */
     public void doMethod( DFolder _folder, DExtensible _ext, Object _arg )
     {
       if( !(_arg instanceof SWTPainter.PainterEvent ) ) return ;
       
       SWTPainter.PainterEvent e = (SWTPainter.PainterEvent)_arg;
       
       //int px = e.paintArea.getX();
       //int py = e.paintArea.getY();
       
       TCellFolder cell = (TCellFolder)e.cell;
       boolean l2r = !cell.isReversed();
       
       ISymbol symbol =null;
       DFolderModel model = _folder.getFolderModel();
       if( model==null ) model = DFolderModel.GetDefault();
       IDColor fore = null, back=null;
       if( _folder.isOpen() )
       {
         symbol = model.getOpenedSymbol( l2r ); 
         fore = model.getOpenedForeColor();
         back = model.getOpenedBackColor();
         if( symbol == null ) symbol = DSymbolRegistry.GetDown();
       } else {
         symbol =  model.getClosedSymbol( l2r );
         fore = model.getClosedForeColor();
         back = model.getClosedBackColor();
         if( symbol == null ) symbol = l2r ? DSymbolRegistry.GetRight() : DSymbolRegistry.GetLeft();
       }
       
       IPen old_pen;
       if( fore!=null )
         old_pen = e.gc.setPen( new LineStylePen( RGBA.Get( fore.getRed(), fore.getGreen(), fore.getBlue() )));
       else
         old_pen = e.gc.setPen( new LineStylePen( RGBA.BLACK ));
       
       IBrush old_brush=null;
       if( back!=null ) 
         old_brush = e.gc.setBrush( new SolidBrush( RGBA.Get( back.getRed(),back.getGreen(), back.getBlue() )));
       else
         old_brush = e.gc.setBrush( new SolidBrush( IGCDStyle.GetBackColor( _folder ) ));         
       
       Rect r = new Rect( cell.getSymbolCell() );
       r.moveBy( e.paintArea.x(), e.paintArea.y() );     
       symbol.draw( e.gc, r );
       
       e.gc.setPen( old_pen );
       if( back!=null ) e.gc.setBrush( old_brush );
       
       SWTPainter painter = (SWTPainter)_ext;
       SWTPainter.PainterEvent ne = new SWTPainter.PainterEvent( e );
       TAbstractCell c;
       //header cells
       Iterator I = cell.getHeaderCellIterator(); 
       if( I!=null )
       {
         for( ; I.hasNext(); )
         {
           c = (TAbstractCell)I.next();
           ne.abs_cell_x = e.abs_cell_x + c.getX();
           ne.abs_cell_y = e.abs_cell_y + c.getY();
           ne.cell = c  ;       
           
           painter.drawCell( ne );
         }
       }
       //body cells
       I = cell.getBodyCellIterator(); 
       if( I!=null )
       {
         for( ; I.hasNext(); )
         {
           c = (TAbstractCell)I.next();
           ne.abs_cell_x = e.abs_cell_x + c.getX();
           ne.abs_cell_y = e.abs_cell_y + c.getY();
           ne.cell = c  ;       
           
           painter.drawCell( ne );
         }
       }
     }
     
   /** paint a DBorder item */
   public void doMethod( DBorder _border, DExtensible _ext, Object _arg )
   {
     if( !(_arg instanceof SWTPainter.PainterEvent ) ) return ;
     
     SWTPainter.PainterEvent e = (SWTPainter.PainterEvent)_arg;
     
     //int px = e.paintArea.getX();
     //int py = e.paintArea.getY();

     int c = IGCDStyle.GetBackColor( _border );
     SolidBrush brush = new SolidBrush();
    
     brush.setRGBA( c ); 
     
     IBrush oldb = e.gc.setBrush( brush );
     
     c = IGCDStyle.GetForeColor( _border );
     LineStylePen pen = new LineStylePen();
     pen.setRGBA( c );
          
     IPen oldp = e.gc.setPen( pen );
     
     TCellBorder cell = (TCellBorder)e.cell;
     boolean reversed = cell.isReversed();
     int ss = _border.getShadowDepth();
     if( _border.isShadowType( DBorder.S_NONE) ) ss=0;
     
     int cx = e.paintArea.x() + (reversed ? ss-1 : 0 );
     int cy = e.paintArea.y();
     int gw = cell.getW()-ss;
     int gh = cell.getH()-ss;
     
     if( _border.isShadowType( DBorder.S_HALO ))
     {
       cx += ss - (reversed ? ss-1: 0);
       cy += ss;
       gh -= ss;
       gw -= ss;
     }
     
     e.gc.fillRect( cx, cy, gw, gh );
     int line_type = _border.getLineType();
     
     for( int i=0,m=1; i<8; i++,m<<=1 )
     {
       if( (m&line_type)!=0) 
         e.gc.drawRect( cx+i, cy+i, gw-2*i, gh-2*i );
     }
     
     //no shadow ?
     if( ss== 0 || _border.isShadowType( DBorder.S_NONE ) )
     {
       e.gc.setBrush( oldb );
       e.gc.setPen( oldp );
       ((SWTPainter)_ext).drawChildrenCell( e );
       return ;
     }
     
     //shadow color:
     int sc = DColor.RGBA( _border.getShadowColor(), RGBA.DARK_GRAY );
     
     switch( _border.getShadowType() )
     {
     case DBorder.S_NONE: break;
     
     case DBorder.S_SOLID : {
       brush.setRGBA(  sc );
       e.gc.setBrush( brush );
       int h = cell.getH()-ss-ss+1;
       if( reversed )
       {
         if(  h > 0 ) e.gc.fillRect( cx-ss+1, cy+ss, ss-1, h );
         if( gw > 0 ) e.gc.fillRect( cx-ss+1, cy+cell.getH()-ss+1, gw, ss-1 );
       } else  {
         if(  h > 0 ) e.gc.fillRect( cx+gw+1, cy+ss, ss-1, h );
         if( gw > 0 ) e.gc.fillRect( cx+ss, cy+cell.getH()-ss+1, gw, ss-1 );
       }       
       break;
     }
     
     case DBorder.S_GRADIENT:{
       int bk = IGCDStyle.GetBackColor( _border.getParent() );
       if( bk==sc ) break ;
       GradientGenerator g = new GradientGenerator( sc, bk );
       int yb = cy+gh+1;
       if( reversed )
       {
         //vertical left rect.
         FullGradientBrush b = new FullGradientBrush( cx, cx-ss, g, false );
         e.gc.setBrush( b );
         e.gc.fillRect( cx-ss, cy+ss, ss-1, gh-ss+1 );
         //horizontal bottom rect:
         b.setPoints(0, yb, 0, cy+cell.getH() );
         e.gc.setBrush( b );
         e.gc.fillRect( cx, yb, gw-ss, ss-1 );          
         //left corner
         b.setPoints( cx, yb,  cx, cy+cell.getH() );
         b.setType( FullGradientBrush.T_CIRCULAR );
         e.gc.fillRect( cx-ss, yb,  ss-1, ss-1 );
         //top-left corner
         b.setPoints( cx, cy+ss, cx-ss, cy+ss );
         e.gc.fillRect( cx-ss, cy,ss-1, ss-1 );
         //bottom-right corner
         int x = cx+gw-ss+1;
         b.setPoints( x, yb, x, cy+cell.getH() );
         e.gc.fillRect( x, yb, ss-1, ss-1 );
       } else {
         int xr = cx+gw+1;
         //vertical right rect.
         FullGradientBrush b = new FullGradientBrush( cx+gw+1, cx+gw+ss, g, false );
         e.gc.setBrush( b );
         e.gc.fillRect( xr, cy+ss, ss-1, gh-ss+1 );
         //horizontal bottom rect:
         b.setPoints(0, yb, 0, cy+cell.getH() );
         e.gc.setBrush( b );
         e.gc.fillRect( cx+ss, yb, gw-ss, ss-1 );          
         //bottom-right corner
         b.setPoints( xr, yb,  cx+gw+1,cy+cell.getH() );
         b.setType( FullGradientBrush.T_CIRCULAR );
         e.gc.fillRect( xr, yb,  ss-1, ss-1 );
         //top-right corner
         b.setPoints( xr, cy+ss,  cx+gw+ss,cy+ss );
         e.gc.setBrush( b );
         e.gc.fillRect( xr, cy, ss-1, ss-1 );
         //left-bottom corner
         b.setPoints( cx+ss, yb, cx+ss, cy+cell.getH());
         e.gc.setBrush( b );
         e.gc.fillRect( cx, yb, ss, ss-1 );
       }
       break;
     }
     
     case DBorder.S_HALO:{
       int bk = IGCDStyle.GetBackColor( _border.getParent() );
       if( bk==sc ) break ;
       GradientGenerator g = new GradientGenerator( sc, bk );
       FullGradientBrush b = new FullGradientBrush(g, FullGradientBrush.T_LINEAR );
       int yt = cy-ss, yb=cy+gh+1, ybb=yb+ss-1;
       int xl = cx-ss, xr = cx+gw+1, xrr=xr+ss-1;
       //top rect
       b.setPoints( cx,cy,cx,yt);
       e.gc.setBrush( b );
       e.gc.fillRect( cx,yt,gw,ss-1);
       //right rect
       b.setPoints( xr,cy,xrr,cy);
       e.gc.setBrush( b );
       e.gc.fillRect( xr,cy,ss-1,gh);
       //bottom rect
       b.setPoints( cx,yb,cx,ybb);
       e.gc.setBrush( b );
       e.gc.fillRect( cx,yb,gw,ss-1);
       //left rect
       b.setPoints( cx,cy,xl,cy);
       e.gc.setBrush( b );
       e.gc.fillRect( xl,cy,ss-1,gh);   
       
       b.setType( FullGradientBrush.T_CIRCULAR );
       //top-left corner
       b.setPoints( cx,cy, xl,cy);
       e.gc.setBrush( b );
       e.gc.fillRect( xl,yt, ss-1,ss-1);
       //top-right corner
       b.setPoints( xr,cy, xrr,cy);
       e.gc.setBrush( b );
       e.gc.fillRect( xr,yt, ss-1,ss-1);
       //bottom-right corner
       b.setPoints( xr,yb, xrr,yb);
       e.gc.setBrush( b );
       e.gc.fillRect( xr,yb, ss-1,ss-1);
       //left-bottom corner
       b.setPoints( cx,yb, xl,yb);
       e.gc.setBrush( b );
       e.gc.fillRect( xl,yb, ss-1,ss-1);
       
       break;
     }
     }
     
     e.gc.setBrush( oldb );
     e.gc.setPen( oldp );
     
     //children:
     SWTPainter painter = (SWTPainter)_ext;
     painter.drawChildrenCell( e );
   }
}
