/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SWTViewer.java,v 1.11 2008/12/12 22:22:03 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/



/*
 * Created on 6 oct. 2003
 *
 */
package org.eclipse.tptp.platform.report.drivers.ui.internal;

import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.ACC;
import org.eclipse.swt.accessibility.AccessibleControlAdapter;
import org.eclipse.swt.accessibility.AccessibleControlEvent;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.report.chart.internal.DefaultChartRenderData;
import org.eclipse.tptp.platform.report.core.internal.DAbstractParagraph;
import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DCell;
import org.eclipse.tptp.platform.report.core.internal.DCellText;
import org.eclipse.tptp.platform.report.core.internal.DCurve;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DFolder;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DI18N;
import org.eclipse.tptp.platform.report.core.internal.DLine;
import org.eclipse.tptp.platform.report.core.internal.DLinkUtil;
import org.eclipse.tptp.platform.report.core.internal.DList;
import org.eclipse.tptp.platform.report.core.internal.DMarkerLine;
import org.eclipse.tptp.platform.report.core.internal.DParagraph;
import org.eclipse.tptp.platform.report.core.internal.DPoint;
import org.eclipse.tptp.platform.report.core.internal.DPopup;
import org.eclipse.tptp.platform.report.core.internal.DSection;
import org.eclipse.tptp.platform.report.core.internal.DTag;
import org.eclipse.tptp.platform.report.core.internal.DText;
import org.eclipse.tptp.platform.report.core.internal.DTitle;
import org.eclipse.tptp.platform.report.core.internal.IDBasicItem;
import org.eclipse.tptp.platform.report.core.internal.IDCoord;
import org.eclipse.tptp.platform.report.core.internal.IDDocument;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDLink;
import org.eclipse.tptp.platform.report.core.internal.IDRenderable;
import org.eclipse.tptp.platform.report.core.internal.IDTaggedItem;
import org.eclipse.tptp.platform.report.drawutil.internal.CursorUtil;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.ILayout;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.ILineLayout;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.IMarginLayout;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TAbstractCell;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TAbstractCellContainer;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCell;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCellContainer;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCellData;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCellFocusMark;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCellFolder;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCellIterator;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCellParagraph;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCellText;
import org.eclipse.tptp.platform.report.drivers.ui.layout.internal.TCellUtil;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.igc.internal.IBrush;
import org.eclipse.tptp.platform.report.igc.internal.IFont;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.ISize;
import org.eclipse.tptp.platform.report.igc.swt.internal.SWTGC;
import org.eclipse.tptp.platform.report.igc.util.internal.Font;
import org.eclipse.tptp.platform.report.igc.util.internal.Point;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;
import org.eclipse.tptp.platform.report.render.internal.DRenderRegistry;
import org.eclipse.tptp.platform.report.render.internal.IRender;
import org.eclipse.tptp.platform.report.render.internal.IRenderLocation;
import org.eclipse.tptp.platform.report.signals.internal.Signal;
import org.eclipse.tptp.platform.report.tools.internal.DLinkResolver;
import org.eclipse.tptp.platform.report.tools.internal.DListLevel;
import org.eclipse.tptp.platform.report.tools.internal.DTitleLevel;
import org.eclipse.tptp.platform.report.tools.internal.IDIImageProvider;
import org.eclipse.tptp.platform.report.tools.internal.IDProgressMonitor;
import org.eclipse.tptp.platform.report.tools.internal.SWTIImageProvider;
import org.eclipse.tptp.platform.report.tools.internal.SimpleProgressMonitor;
import org.eclipse.tptp.platform.report.ui.swt.widgets.internal.ScrollView;
import org.eclipse.tptp.platform.report.ui.swt.widgets.internal.ToolTipRedrawer;

import com.ibm.icu.text.DecimalFormat;
import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.text.UFormat;
import com.ibm.icu.util.ULocale;


/**
 * This class is a widget that extends a Canvas to display a JScrib IDItem.
 * This widget can render the entire or a part of a JScrib document.
 * The SWTViewer do not render directly IDItem but a hierarchy of TAbstractCell. This
 * hierarchy is a rectangle hierarchy which are arranged by a @link ILayout mechanism.
 * This class calls the SWTPainter's services to render the TAbstractCell<br>
 * 
 
 * The following example shows a DDocument in a SWTViewer.<P> 
 *
 * <PRE>
 * IDItem doc = new Document();
 * doc.addChild(new Text("Hello World"));
 * SWTViewer ws = new SWTViewer(c, parent);
 * ws.setContent(doc);
 * ws.show();
 * </PRE>
 *
 * @see TAbstractCellContainer
 * @see IDItem
 * @see ILayout
 * @see SWTPainter
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class SWTViewer extends ScrollView
{
	public static class ViewerEvent 
	{
		public TAbstractCell cell;
		public SWTViewer viewer;
	}
	
	/**
	 * This class provides the datas to manage a mouse event on a ITCell in 
	 * the SWTViewer. This event is passed in the third parameter of 
	 * MouseEventHandler's doMethod
	 * 
	 * @see TAbstractCell
	 * @see SWTViewer.MouseEventHandler
	 */
	public static class ViewerMouseEvent extends ViewerEvent 
	{	
		/** Original SWT mouse event */
		public MouseEvent event;
		/** Event kind. The values are:
		 * <UL>
		 * <LI> <b>SWT.MouseEnter</b>: the cursor enters in the cell's item.
		 * <LI> <b>SWT.MouseExit</b>: the cursor leaves in the cell's item.
		 * <LI> <b>SWT.MouseMove</b>: the cursor mouves on the cell's item.
		 * <LI> <b>SWT.MouseDown</b>: a mouse button clicks down on the cell's item.
		 * <LI> <b>SWT.MouseUp</b>: a mouse button clicks up on the cell's item.
		 * </UL> */
		public int        type;
		/** Event processor must set handled to true to break the event handlers chain (this is a return value)*/
		public boolean    handled;
		/** New cursor value requested by this event processor (this is a return value) */
		public Cursor     cursor;
		/** New tooltip text value requested by this event processor (this is a return value) */
		public String     tooltip;
		/** item's doMethod request an invalidation of it's cell, a layout must be done */
		public boolean    invalidate;
	}
	
	/**
	 * This class provides the datas to manage a mouse event on a ITCell in 
	 * the SWTViewer. This event is passed in the third parameter of 
	 * MouseEventHandler's doMethod
	 * 
	 * @see TAbstractCell
	 * @see SWTViewer.MouseEventHandler
	 */
	public static class ViewerProgressMonitor extends SimpleProgressMonitor
	{	
	    private SWTViewer viewer;

	    
	    public ViewerProgressMonitor(SWTViewer v)
	    {
	       viewer = v;  
	    }

        /**
         * @see org.eclipse.tptp.platform.report.tools.internal.IDProgressMonitor#beginTask(java.lang.String, int)
         */
        public void beginTask(String name, int total_works) {
            super.beginTask(name, total_works);
        }

        /**
         * @see org.eclipse.tptp.platform.report.tools.internal.IDProgressMonitor#done()
         */
        public void done() {
           super.done();
           oldnb=0;
        }

        /**
         * @see org.eclipse.tptp.platform.report.tools.internal.IDProgressMonitor#setCanceled(boolean)
         */
        public void setCanceled(boolean value) {
            super.setCanceled(value);
        }

        private  int oldnb = 0;
        
        /**
         * @see org.eclipse.tptp.platform.report.tools.internal.IDProgressMonitor#worked(int)
         */
        public void worked(int work) {
            super.worked(work);
            int prg = (int)(((double)work/(double)getTotalWorks())*100);
            if (prg>(oldnb+5)) {
               viewer.redraw();
               viewer.update();
               oldnb = prg;
            }
            
        }
	}
	
	/**
	 * This class provides a default Mouse event handler for used by the SWTViewer.
	 * This handler manages the activation link (popup, tooltip, scrolls,...).
	 * <P>
	 * Each doMethod for JScrib Item receives in the parameter event a ViewerMouseEvent object.
	 * <P>
	 * This class inherits of DExtensible class. You can extend the default handler by adding
	 * doMethod method if you want to add mouse event action on derived JScrib Item.<P>
	 * For example:
	 * 
	 * <PRE>
	 * public class MyMouseEventExtender implements IDExtension {
	 * 
	 *   public void updateExtensible(DExtensible e)
	 *   {
	 *     e.installDoMethods(this);
	 *   }
	 *  
	 *   public void doMethod(MyJScribItem i,  DExtensible ext, Object event)
	 *   {
	 *    ViewerMouseEvent e = (ViewerMouseEvent)event;
	 *    switch (e.type) {
	 *      case SWT.MouseEnter:
	 *           ...
	 *           break;
	 *      case SWT.MouseExit:
	 *           ....
	 *   }
	 * }
	 * ...
	 * // creates an instance of the extension
	 * MyMouseEventExtender meh = new MyMouseEventExtender();
	 * ...
	 * // the SWTViewer receives automatically the extension for MyJScribItem
	 * SWTViewer viewer = new SWTViewer(composite);
	 * ...
	 * </PRE>
	 * 
	 * @see org.eclipse.tptp.platform.report.extensionL.internal.DExtensible
	 * @see org.eclipse.tptp.platform.report.extensionL.internal.IDExtension
	 * @see #getDefaultMouseEventHandler
	 */ 
	public static class MouseEventHandler extends DExtensible
	{
		/** Manages link activation */
		public void doMethod(IDLink link, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			boolean bbutton1 = (((e.event.stateMask & SWT.BUTTON1) != 0)||(e.event.button == 1));

			switch (e.type)
			{
			
			case SWT.MouseEnter:
				e.viewer.highlightLink(link, null);	    
			break;
			
			case SWT.MouseExit:
				e.viewer.unhighlightLink(link);	         	    
			break;
			
			case SWT.MouseMove:
				if (DLinkUtil.isPopupLink(link) 
						&& (DLinkUtil.TOOLTIP.equals( DLinkUtil.getMethod(link))))
				{
					e.cursor = null;
				}
				else
					e.cursor = e.viewer.handCursor();
			
			if (!DLinkUtil.isInternalLink(link)
					&& (e.viewer.isShowExternalLinkTooltip()))
			{
				e.tooltip = link.getTarget();
			}
			break;
			
			
			case SWT.MouseDown:
				if (bbutton1)
				{
					e.viewer.activateLink(link, null);
					e.cursor = null;
					e.handled = true;		
				}
				
			break;
			}
		}
		
		/** Manages mouse event for the selection */
		public void doMethod(DDocument doc, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			
			_handleSelection(e, e.type);
			if (e.handled) return;
			
			e.cursor = null;
			e.handled = true;
		}
		
		/** Manages mouse event for the selection */
		public void doMethod(DSection section, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			_handleSelection(e, e.type);
			if (e.handled) return;
			
			e.cursor = null;
			e.handled = true;
		}
		
		/** Manages mouse event for the selection */
		public void doMethod(DTitle title, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			_handleSelection(e, e.type);
		}
		
		/** Manages mouse event for the selection */
		public void doMethod(DList list, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			_handleSelection(e, e.type);
		}
				
		/** Manages mouse event for a DFolder */
		public void doMethod( DFolder folder, DExtensible ext, Object arg )
		{     
		  if( !(arg instanceof SWTViewer.ViewerMouseEvent) ) return ;
		  
		  SWTViewer.ViewerMouseEvent e = (SWTViewer.ViewerMouseEvent)arg;
		  
		  TCellFolder cell = (TCellFolder)e.cell;
		  
		  Point p = new Point();
		  TCellUtil.GetGlobalCellXY(cell, p);
		  int evt_x = e.event.x - p.getX();
		  int evt_y = e.event.y - p.getY();
		  
		  IRect r = cell.getSymbolCell();
		  
		  if( Rect.Contains( r, evt_x, evt_y ))
		  {
		    e.cursor = new Cursor( Display.getDefault(), SWT.CURSOR_HAND );
		    if( e.type == SWT.MouseDown )
		    {
		      folder.setOpen( !folder.isOpen() );
		      //reset selection as it can be over folder and next rebuild doc select bad text area.
		      e.viewer.resetDefaultSelection();
		      e.handled = true;
		      e.invalidate = true;
		    }
		    return ;
		  }
		  
		  if(true)return;
		  
		  /*the following is not mandatory due to c=MapPointToCell() and doMethod(c) policy of SWTViewer     
		   TAbstractCell c = cell.getHeaderCell();
		   p.setX(0); p.setY(0);
		   TCellUtil.GetGlobalCellXY(c, p);
		   System.out.println(" POINT... p"+p); 
		   if( c.contains( e.event.x-p.getX(), e.event.y-p.getY()))
		   {
		   e.cell = c;
		   ext.invokeDoMethod( c.getItem(), ext, arg );
		   e.cell=cell;
		   System.out.println(" EVT... hld"+e.handled);       
		   if( !e.handled && e.type == SWT.MouseDown )
		   {
		   folder.setOpened( !folder.isOpened() );
		   //reset selection as it can be over folder and next rebuild doc select bad text area.
		    e.viewer.resetDefaultSelection();
		    e.handled = true;
		    e.invalidate = true;
		    }
		    return ;
		    }
		    
		    
		    for( Iterator I=cell.getBodyCellIterator(); I.hasNext(); )
		    {
		    c = (TAbstractCell)I.next();
		    TCellUtil.GetGlobalCellXY(c, p);
		    p.setPoint( evt_x-p.getX(), evt_y-p.getY() );
		    c = TCellUtil.MapPointToCell( c, p );
		    e.cell = c;
		    if( c==null ) continue;
		    System.out.println("BODY evt="+evt_x+","+evt_y+" cell="+c+" item="+c.getItem());       
		    ext.invokeDoMethod( c.getItem(), ext, arg );
		    if( e.handled ) break;
		    }
		    
		    e.cell = cell; */   
		}
		/** Manages mouse event for the selection */
		public void doMethod(DParagraph p, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			_handleSelection(e, e.type);		
		}
		
		/** Manages mouse event for the selection */
		public void doMethod(DTag t, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			_handleSelection(e, e.type);
		}
		
		/** Manages mouse event for the selection and link activation */
		public void doMethod(DText text, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			
			IDLink link = null;
			switch (e.type)
			{
			case SWT.MouseEnter:
			case SWT.MouseExit:
				_handleSelection(e, e.type);
			if (e.handled) return;
			
			link = e.viewer.getLinkFromCell( e );
			if (link!=null)
				e.viewer.defaultMouseEventHandler.invokeDoMethod(link, this, e);
			break;
			
			case SWT.MouseDown:
				link = e.viewer.getLinkFromCell( e );
			if (link!=null)
				e.viewer.defaultMouseEventHandler.invokeDoMethod(link, this, e);
			
			if (!e.handled)
				_handleSelection(e, e.type); 
			break;		
			
			case SWT.MouseMove:
				_handleSelection(e, e.type);
			if (e.handled) return;
			
			link = e.viewer.getLinkFromCell( e );
			if (link!=null)
				e.viewer.defaultMouseEventHandler.invokeDoMethod(link, this, e);
			break;
			
			case SWT.MouseUp:
				_handleSelection(e, e.type);
			break;
			
			}
			
		}
		
		/** Manages mouse event for the selection and link activation */
		public void doMethod(IDBasicItem basic_item, DExtensible ext, Object event)
		{ 
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			
			switch (e.type)
			{
			case SWT.MouseEnter:
			case SWT.MouseExit:
			case SWT.MouseMove:
			case SWT.MouseDown:
				IDLink link = e.viewer.getLinkFromCell( e );
			if (link!=null)
				e.viewer.defaultMouseEventHandler.invokeDoMethod(link, this, e);
			break;
			}	
		}
		
		/** Manages mouse event for the selection */
		public void doMethod(DCellText celltext, DExtensible ext, Object event)
		{
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			
			_handleSelection(e, e.type);
			if (e.handled) return;
			
			e.cursor = null;
			e.handled = true;
			
		}
		
		/** Manages mouse event for the selection */
		public void doMethod(DCell cell, DExtensible ext, Object event)
		{    
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			_handleSelection(e, e.type);
			if (e.handled) return;
			
			e.cursor  = null;
			e.handled = true;
		}
		
		/** Do nothing */
		public void doMethod(DPopup popup, DExtensible ext, Object event)
		{}
		
		private IDLink prevGraphLink;
		/** Manages mouse event on the part of a graphic:
		 * <UL>
		 * <LI>Moves 3D graphic
		 * <LI>Detect link in the graphic and call doMethod(IDLink,...
		 * <LI>Show grahic tooltip
		 * </UL> */
		public void doMethod(DGraphic graph, DExtensible ext, Object event)
		{
		    if (graph.getFirstChild() instanceof DGraphic)
		        graph = (DGraphic)graph.getFirstChild();
		    
			ViewerMouseEvent e = (ViewerMouseEvent)event;
			e.handled=true;
			IDLink link;
			switch (e.type)
			{
			case SWT.MouseMove:
				if( DGraphic.CanMoves( graph ) )
				{
					e.cursor = e.viewer.graphicMovingCursor();
				}
				
			if((e.event.stateMask & SWT.BUTTON1) != 0)
			{
				if( e.viewer.defaultProperties.activeMoveGrachic==graph 
						&& graph.canMoves() ) 
				{
					//rotate the graphic
					e.viewer.move3DView( e,  graph);
					e.handled = true;
					return ;
				}
			}
			
			//extract a link and/or build a tooltip text..
			{
				//get cell location in contents coordinates, I know there are cell.absoluteXY()
				//but I don't want to allocate a point each time ...
				int cx = 0, cy = 0;
				for( TAbstractCell c=e.cell; c!=null; c=c.getParent() ) { cx+=c.getX(); cy+=c.getY(); }
				int px =  e.event.x-cx;
				int py =  e.event.y-cy;
				IRenderLocation loc = e.viewer.locatePointInRenderable( px,py, graph, e.cell );
				link = e.viewer.extractRenderableLink( loc );
				//if have link-tooltip, no auto tooltip text is build.
				boolean do_ttip = loc!=null ;
				if( link != null )
				{
					String method = DLinkUtil.getMethod( link.getTarget() );
					do_ttip = ( !DLinkUtil.TOOLTIP.equals( method ) );
				}
				IDItem item ;
				if( do_ttip && (item=loc.getItem())!=null )
				{
					//ok, now try to build something interesting ...
					if( item instanceof DCurve )
					{
						/*TODO: locate mouse over truncated text "..." (except for Vertical text)
						 if( e.tooltip !=null && e.tooltip.indexOf( "...")<0 )
						 {
						 e.tooltip=null;
						 }*/
					}
					else if ( item instanceof DPoint )
					{
						DPoint pt = (DPoint)item;
						IDItem p = pt.getParent();
						DCurve c = p instanceof DCurve ? (DCurve)p:null ;
						p = p.getParent();
						//DGraphic g = p instanceof DGraphic ? (DGraphic)p : null;
						
                        String title = DefaultChartRenderData.getResourceString(c.getName(), 
                                (DI18N)graph.getChildOfClass(DI18N.class));
                        
						String t= (c!=null && title!=null) ? title+": " : "";
						for( IDItem ic=item.getFirstChild(); ic!=null; ic=ic.getNext() )
						{
							if( !(ic instanceof IDCoord)) continue;
							
							IDCoord coord=(IDCoord)ic;
							Object v = coord.getValue(null);
							if( v==null ) continue;
							DAxis  a = coord.getAxis();
							UFormat f=null;
							if( t.length()>0 ) t += "  "; //between coords
							if( a!=null ) 
							{
								if( a.getTitle()!=null )
								{
									t+= a.getTitle()+"="; //getName() pas mieux.
								}
								try{
									f = (UFormat)a.getProperties().get( DAxis.P_UNIT_FORMAT );
								}
								catch( ClassCastException ex ) {}
							}
							//guess a format
                            ULocale locale= graph.getLocale();
							if( f ==null )
							{
								if ( v instanceof Number )
								{
									f = DecimalFormat.getInstance(locale);
								}
								else if ( v instanceof Date )
								{                             
									f = SimpleDateFormat.getDateInstance(SimpleDateFormat.SHORT, locale);
								}
							}
							if( f!=null )
							{
								try{
									t += f.format( v ) ;
								}
								catch( Exception ex ){}
							}
							else
							{
								//TODO do this or do nothing, because toString() can contains the best like the worst ...                
								t += v.toString();
							}
							if( a!=null && a.getUnit()!=null )
							{
								t+=" "+a.getUnit();
							}
							
						}//for
						
						e.tooltip = t;
					}
					else if ( item instanceof DMarkerLine )
					{
					    DMarkerLine ml = (DMarkerLine)item;
					    if (ml.getLabel()!=null)
					        e.tooltip = ml.getLabel();
					    else
					        e.tooltip = "unamed marker line";
					    e.tooltip += " : " + ml.getValue();
					}
				}           
			}
			
			if (prevGraphLink!=link)
			{
				if (prevGraphLink!=null)
				{
					e.type = SWT.MouseExit;
					e.viewer.defaultMouseEventHandler.invokeDoMethod(prevGraphLink, this, e);
				}
				
				if (link!=null)
				{
					e.type = SWT.MouseEnter;
					e.viewer.defaultMouseEventHandler.invokeDoMethod(link, this, e);
				}
				prevGraphLink = link;
				return;
			}
			
			prevGraphLink = link;
			if (link!=null)
				e.viewer.defaultMouseEventHandler.invokeDoMethod(link, this, e);
			
			break;
			case SWT.MouseDown:
				if( DGraphic.CanMoves( graph ) )
				{
					//rotate the graphic
					e.viewer.defaultProperties.activeMoveGrachic=graph;
					e.viewer.defaultProperties.moveGraphicx_=e.event.x;
					e.viewer.defaultProperties.moveGraphicy_=e.event.y;
				}
			link = e.viewer.getLinkFromCell( e );
			if (link!=null)
				e.viewer.defaultMouseEventHandler.invokeDoMethod(link, this, e);
			
			e.handled = true;
			break;
			case SWT.MouseUp:
				e.viewer.defaultProperties.activeMoveGrachic=null;
			break;
			
			}
			
		}
		
		/** do nothing */	
		public void doMethod(DLine line, DExtensible ext, Object event)
		{}
		
		private void _handleSelection(ViewerMouseEvent e, int event_type)
		{ 
			//TAbstractCell cell = e.cell;
			Selection selection = e.viewer.getDefaultSelection();
			e.handled = false;
			
			if (selection == null) return;
			
			boolean bbutton1 = (((e.event.stateMask & SWT.BUTTON1) != 0)||(e.event.button == 1));
			switch (event_type) 
			{
			case SWT.MouseDown: 
				if(bbutton1)
				{
					if (selection.isValid()) e.viewer.resetDefaultSelection();
					e.viewer.closeChildViewer();
					
					int p = _detectSelectedChar(e);
					if(p >= 0)
					    selection.setRange(p,p);
					selection.state = Selection.SELECT_STARTED;
					e.handled = true;
				}	
			break;
			
			case SWT.MouseUp: 
				if(bbutton1)
				{
					if(selection.state == Selection.SELECT_STARTED)
					{
						selection.state = Selection.SELECT_TERMINATED;
						Object[] params = { e.viewer };
						e.viewer.selectionChanged.emit(params);
						e.handled = true;
					}
				}
			break;
			
			case SWT.MouseEnter:
			case SWT.MouseExit:
			case SWT.MouseMove: 
				if(bbutton1 && (selection.getState() == Selection.SELECT_STARTED))
				{
					int p = _detectSelectedChar(e);
					if(p >= 0)
					{
						selection.setEndPosition(p);
						Object[] params = { e.viewer };
						e.viewer.selectionChanged.emit(params);
						e.viewer.redraw();
					}
					e.handled = true;
				}
			break;
			
			default: 
				break;
			}			 
		}
		
		private int _detectSelectedChar(ViewerMouseEvent e)
		{
		    if (!(e.cell instanceof TCellText)) return -1;
			TCellText cell = (TCellText)e.cell;
			
			int posx = e.event.x;
			
			if(cell == null)	return -1;
			IDItem item = cell.getItem();
			if(item == null)	return -1;
			
			String ctext = cell.getText();
			if ((ctext != null)&&(ctext.length()>0))
			{
				int cpos = cell.getPosition();
				ctext = ctext.substring(cell.getStart(), cell.getStart()+cell.getLen());
				int clen = ctext.length();
				
				GC gc = new GC(e.viewer);
				SWTGC swtgc = new SWTGC(e.viewer.getDisplay(), gc);
				
				IFont newfont = IGCDStyle.GetFont(item, e.viewer.getZoom());
				swtgc.setFont(newfont);
				
				Point abs = new Point(0,0);
				TCellUtil.GetGlobalCellXY(cell,abs);
				
				int clenx = posx - abs.getX();
				   
				int i=0;
				int result=0;
				int diff=0;
				while (i<clen)
				{
				   String tmptext = ctext.substring(0,i+1);
				   ISize s = swtgc.textExtent(tmptext);
				   
				   if (s.getW()>=clenx) {
				   	 	diff = s.getW() - clenx;
				   	    s = swtgc.textExtent(ctext.substring(i,i+1));
				   	    if (diff < (s.getW()/2)) i++;
				   	 	break;
				   }
				   i++;
				}
				   
				result = i;
				   
				//newfont.dispose();
				gc.dispose();
				swtgc.dispose();
				return cpos+result;
			}
			else
			{
				int dir;
				if (e.viewer.getDefaultSelection().endPos>=e.viewer.getDefaultSelection().startPos)
				{
					dir = TCellUtil.LEFT_DIRECTION;
				}
				else
				{       	       
					dir = TCellUtil.RIGHT_DIRECTION;
				}
				TCellText i = TCellUtil.MapPointToClosestTextCell(e.cell, new Point(e.event.x, e.event.y), dir);
				if (i!=null)
				{
					int p = (dir==TCellUtil.LEFT_DIRECTION)? i.getPosition()+i.getLen()-1:
						i.getPosition();
					return p;  
				}
				return -1;
			} 	 
		}
		
	}
	
	/**
	 * This class provides the datas to manage a keyboard event on a ITCell in 
	 * the SWTViewer. This event is passed in the third parameter of 
	 * KeyEventHandler's doMethod
	 * 
	 * @see TAbstractCell
	 * @see SWTViewer.KeyEventHandler
	 */
	public static class ViewerKeyEvent extends ViewerEvent
	{	
		/** original SWT mouse event */
		public KeyEvent event;
		/** type of the event. The values are:
		 * <UL>
		 * <LI>SWT.KeyDown
		 * <LI>SWT.KeyUp
		 * </UL>
		 */
		int type;
		/** event processor must set handled to true to break the event handlers chain (this is a return value)*/
		public boolean    handled;
		/** new tooltip text value requested by this event processor (this is a return value) */
		public String     tooltip;
	}
	
	/**
	 * This class provides a default Keyboard event handler for used by the SWTViewer.
	 * This handler manages the focus on the link item and its activation by the Enter key.
	 * <P>
	 * Each doMethod for JScrib Item receives in the parameter event a KeyMouseEvent object.
	 * <P>
	 * This class inherits of DExtensible class. You can extend the default handler by adding
	 * doMethod method if you want to add mouse event action on derived JScrib Item.
	 * 
	 * @see SWTViewer.MouseEventHandler
	 * @see DExtensible
	 */
	public static class KeyEventHandler extends DExtensible
	{
		
		
		public void doMethod(IDLink link, DExtensible ext, Object event)
		{
			ViewerKeyEvent e = (ViewerKeyEvent)event;
			
			if (e.type==SWT.KeyDown) return;
			
			//String method = DLinkUtil.getMethod(link);
			//Point p = new Point(e.cell.getX(), e.cell.getY());
			Point p = new Point();
			TCellUtil.GetGlobalCellXY(e.cell, p);
			org.eclipse.swt.graphics.Point pswt = new org.eclipse.swt.graphics.Point(p.getX(), p.getY());
			pswt = e.viewer.contentsToView(pswt.x, pswt.y);
			pswt = e.viewer.toDisplay(pswt.x, pswt.y-5);
			
			//Point pc = Display.getCurrent().getCursorLocation();
			//System.out.println("p="+p+" pc="+pc+"keycode = "+ e.event.keyCode);		 
			
			
			switch (e.event.keyCode)
			{
			
//			case SWT.TAB:
//				if (prevlink!=null)
//					e.viewer.unhighlightLink(prevlink);
//			    e.viewer.highlightLink(link, p);
//			    prevlink = link;
//			
//			    if (!DLinkUtil.isInternalLink(link)
//					&& (e.viewer.isShowExternalLinkTooltip()))
//			    {   
//				   e.tooltip = link.getTarget();
//			    }
//			    e.handled = true;
//			break;
			case 13: 
			{
				e.viewer.activateLink(link, pswt);
				e.handled = true;
			}
			e.handled = true;
			break;
			
			default:
				e.viewer.closeChildViewer();
			break;
			}
		}
		
		public void doMethod(DGraphic graph, DExtensible ext, Object event)
		{
		    if (graph.getFirstChild() instanceof DGraphic)
		        graph = (DGraphic)graph.getFirstChild();
		    
			ViewerKeyEvent e = (ViewerKeyEvent)event;
			e.handled = true;
			if (e.type==SWT.KeyUp) return;
			e.viewer.closeChildViewer();
		}
		
	}
	
	private String name = "";
	private ViewerRegistry registry = new ViewerRegistry();
	
	private SWTPainter defaultPainterEventHandler      = new SWTPainter();
	private MouseEventHandler defaultMouseEventHandler = new MouseEventHandler();
	private KeyEventHandler defaultKeyEventHandler    = new KeyEventHandler();
	
	private static final class Show{
		public static final int NORMAL = 0;
		public static final int PAGES = 1;
		public static final int ONE_PAGE = 2;
		public static final int TWO_PAGES = 3;
		public static final int FOUR_PAGES = 4;
		public static final int POPUP = 5;
		public static final int POPUP_DETACHED = 6;
		public static final int TOOLTIP = 7;
		private int mode = NORMAL;
		public boolean mode(int nmode){
			switch(nmode){
			case NORMAL:{
				mode=nmode;
				break;
			}
			case PAGES:{
				mode=nmode;
				break;
			}
			case ONE_PAGE:{
				mode=nmode;
				break;
			}
			case TWO_PAGES:{
				mode=nmode;
				break;
			}
			case FOUR_PAGES:{
				mode=nmode;
				break;
			}
			default:{
				return false;
			}
			}
			return true;
		}
		public int mode(){
			return mode;
		}
	}
	private Show mode = new Show();
	
	/*private static final class Orientation{
		public static final int VERTICAL = 0;
		public static final int HORIZONTAL = 1;
		private int data = VERTICAL;
		public boolean orientation(int norientation){
			switch(norientation){
			case VERTICAL:{
				data=norientation;
				break;
			}
			case HORIZONTAL:{
				data=norientation;
				break;
			}
			default:{
				return false;
			}
			}
			return true;
		}
		public int orientation(){
			return data;
		}
	}*/
    
	//private Orientation show_orientation = new Orientation();
	private boolean double_buffer = true;
	
	private IDItem           contentItem;
	private TAbstractCellContainer   contentCell;
	private int top, bottom, left, right;
	private float            zoom = 1.0f;
	private boolean          reversed = false;
	private boolean          maintainRatio = false;
	private int              lastPosition;
	
	/**accessibility*/
	//private String accessibleText = null; 
	
	/** use for the current keyboard focus */ 
	private TCellFocusMark  focus;
	private IDLink prevlink;
	private Selection defaultSelection;
	private Cursor handcursor_;
	private Cursor graphicmovingcursor_;
	
	private boolean showExternalLinkTooltip = false;
	
	/** return hand mouse cursor, create once */
	private Cursor handCursor() 
	{
		if( handcursor_==null )
		{
			handcursor_ = new Cursor(getDisplay(),SWT.CURSOR_HAND);
		}
		return handcursor_;
	}
	private static String[] asc_graphic_moving_cursor_ =
	{
			"                 n              ",
			"                n.n             ",
			"               n...n            ",
			"              n.....n           ",
			"             n.......n          ",
			"            nnnn...nnnn         ",
			"               n...n            ",
			"              n...n             ",
			"     n        n...n      n      ",
			"    nn       n...n       nn     ",
			"   n.n       n...n       n.n    ",
			"  n..nn      n...n      nn..n   ",
			" n.....nn     n.n     nn.....n  ",
			"n........nnn   n   nnn........n ",
			" n..........n nnn n..........n  ",
			"  n..nn......nnnnn......nn..n   ",
			"   n.n nn...n nnn n...nn n.n    ",
			"    nn   nnn   n   nnn   nn     ",
			"     n        n.n        n      ",
			"             n...n              ",
			"             n...n              ",
			"             n...n              ",
			"              n...n             ",
			"              n...n             ",
			"               n...n            ",
			"            nnnn...nnnn         ",
			"             n.......n          ",
			"              n.....n           ",
			"               n...n            ",
			"                n.n             ",
			"                 n              ",
			"                                "
	};
	/** return mouse cursor for graphic moving, create once */
	private Cursor graphicMovingCursor() 
	{
		if( graphicmovingcursor_==null )
		{
			graphicmovingcursor_ = CursorUtil.CursorFromString(asc_graphic_moving_cursor_,15,15);
		}
		return graphicmovingcursor_;
	}
	
	private DefaultProperties defaultProperties;
	private class DefaultProperties{
		public long toolipTimerDelay_ = 500;
		public int topMargin = 0;
		public int bottomMargin = 0;
		public int leftMargin = 0;
		public int rightMargin = 0;
		public DGraphic activeMoveGrachic;
		public int moveGraphicx_ = 0;
		public int moveGraphicy_ = 0;
		
	}
	
	
	
	
	/**
	 * This class provides the status of a range selection in the
	 * SWTViewer. The viewer has one active selection named the 
	 * default selection. You can take it by calling the method
	 * <i>getDefaultSelection()</i>.
	 * 
	 * @see SWTViewer#getDefaultSelection()
	 * @see SWTViewer#resetDefaultSelection()
	 */
	public class Selection
	{
		/** the selection is terminated. The values start and end are valid */ 
		public static final int SELECT_TERMINATED = 0;
		/** the selection is started. Only the value start is valid */ 
		public static final int SELECT_STARTED = 2;
		
		private int state = SELECT_TERMINATED;
		private int startPos = 0;
		private int endPos   = 0;
		
		/** Sets a range in character position for the selection */
		public void setRange(int start, int end)
		{
			if (start <= end)
			{
				startPos = start;
				endPos = end;
			}
			else
			{
				startPos = end;
				endPos = start;
			}
		}
		
		/** resets the selection */
		public void reset()
		{
			startPos = 0;
			endPos = 0;
		}
		
		/** Sets the start position */
		public void setStartPosition(int p)
		{
			startPos = p;
		}
		
		/** Sets the end position */
		public void setEndPosition(int p)
		{
			endPos = p;
		}
		
		/** Returns true if the selection is valid */
		public boolean isValid()
		{
			return  /*((startPos >0) && (endPos >0))
			&&*/ (getEndPosition()-getStartPosition()>0);
		}
		
		/** Returns the range fo the selection */
		public org.eclipse.swt.graphics.Point getRange(){
			return new org.eclipse.swt.graphics.Point(getStartPosition(),getEndPosition());
		}
		
		/** Returns the start position. Checks if start < end */
		public int getStartPosition()
		{
			if (startPos<endPos)
				return startPos;
			else
				return endPos;
		}
		
		/** Returns the end position. Checks if end > start */
		public int getEndPosition()
		{
			if (startPos<endPos)
				return endPos;
			else
				return startPos;
		}
		
		/** Returns the state of the selection. The values are:
		 * <UL>
		 * <LI>Selection.SELECT_TERMINATED
		 * <LI>Selection.SELECT_STARTED
		 * </UL> */
		public int getState()
		{
			return state;
		}
		
	}
	
	/**
	 * <b>Signal :</b> selectionChanged(org.eclipse.tptp.platform.report.ui.internal.SWTViewer)
	 * <br>
	 * Emit when the selection changes in the viewer.
	 */
	public Signal selectionChanged = new Signal("selectionChanged(org.eclipse.tptp.platform.report.ui.internal.SWTViewer)");
	/**
	 * <b>Signal :</b> linkHighlighted(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)
	 * <br>
	 * Emit when the mouse cursor pass through a DLink object.
	 */
	public Signal linkHighlighted = new Signal("linkHighlighted(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)");
	/**
	 * <b>Signal :</b> linkUnlighlighted(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)
	 * <br>
	 * Emit when the mouse cursor exits  a DLink object.
	 */
	public Signal linkUnhighlighted = new Signal("linkUnhighlighted(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)");
	
	/**
	 * <b>Signal :</b> linkActivated(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)
	 * <br>
	 * Emit when the IDlink object is activated by rigth click and it is not resolved itself by
	 * the SWTViewer because it is an external link. 
	 * The slots receive two arguments. The first is the target string and the second is the 
	 * activated IDLink
	 * 
	 * @see org.eclipse.tptp.platform.report.core.internal.DLinkUtil#isInternalLink
	 */
	public Signal linkActivated = new Signal("linkActivated(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)");
	
	/**
	 * <b>Signal :</b> linkError(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)
	 * <br>
	 * Emit when the SWTViewer cannot resolve the link and found the target 
	 * The slots receive two arguments. The first is the target string and the second is the 
	 * IDLink
	 * 
	 * @see org.eclipse.tptp.platform.report.core.internal.DLinkUtil#isInternalLink
	 */
	public Signal linkUnresolved = new Signal("linkError(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)");
	
	/** IImage provider for the viewer (at least used by charts render) */
	private IDIImageProvider iimage_provider;
	
	/** The internal progress monitor */
	private IDProgressMonitor progressMonitor = new ViewerProgressMonitor(this);
	
	/**
	 * This class provides a new widget which is a SWT render for a JScrib Document.
	 * The contentItem is painted in a ScrollView. It tries to adapt the contentItem's width at
	 * the width of the widget. If the contentItem is too big it makes automatically a 
	 * vertical and horizontal scrollbars. <P>
	 * This class used the following default handler to manage paint, cursor and key events:
	 * <b>SWTPainter</b>, <b>MouseEventHandler</b> and <b>KeyEventHandler</b>.
	 * <P>
	 * To sets the contentItem which you can display you have to just call the method
	 * <b>setDocument</b> of the SWTViewer.<P>
	 * Example:
	 * <PRE>
	 * SWTViewer v = new SWTViewer(parent, style);
	 * v.setDocument(my_jcrib_document);
	 * </PRE>
	 *
	 * @see ScrollView
	 * @see SWTPainter
	 * @see SWTViewer.MouseEventHandler
	 * @see SWTViewer.KeyEventHandler
	 */
	public SWTViewer(Composite parent, int style) 
	{
		super (parent,style|SWT.NO_BACKGROUND); 
		defaultProperties = new DefaultProperties();
		iimage_provider = new SWTIImageProvider( getDisplay() );
		tooltip = new ToolTipRedrawer(this);

		setOverviewEnabled( true );
		
		addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e){
				SWTViewer v = (SWTViewer)e.widget;
				v.dispose();
			}
		});
		
//		 Tab managment 
	    getViewControl().addTraverseListener(new TraverseListener() {
	    	public void keyTraversed(TraverseEvent e) 
	        {
	    		switch (e.detail)
				{
	    		  case SWT.TRAVERSE_TAB_NEXT:
	    		  	   SWTViewer.this.nextFocusable();
	    		       startAccessibility();
	    		       break;
	    		  case SWT.TRAVERSE_TAB_PREVIOUS:
	    		  	   SWTViewer.this.previousFocusable();
	    		       startAccessibility();
	    		       break;
	    		  default:		
				}
	    		if (focus==null)
  		          e.doit=true;
	    	}
	    });
		
		getViewControl().getAccessible().addAccessibleControlListener(new AccessibleControlAdapter()
				{
			public void getRole(AccessibleControlEvent e)
			{
				//System.out.println("getRole childID = " + e.childID + "/" + ACC.CHILDID_SELF);
				
				switch (e.childID) {
				case ACC.CHILDID_SELF: 
				{
					e.detail = ACC.ROLE_CLIENT_AREA;
					break;
				}
			    case 0: // Tooltip
			    	e.detail = ACC.ROLE_TOOLTIP;
			    	break;
				default: 
				{
					e.detail = ACC.ROLE_CHECKBUTTON;
				}
				}
			}
			
			
			
			public void getValue(AccessibleControlEvent e)
			{
				//System.out.println("getValue childID = " + e.childID + "/" + ACC.CHILDID_SELF);
				switch (e.childID) {
				case ACC.CHILDID_SELF:
					 e.result = null;
					 break;
				case 0:
				{
					SWTViewer v2 = getChildViewer();
					if (v2!=null)
					    e.result = v2.getContentText(); 
					break;
				}
				default: 
				{
					if (focus!=null)
						e.result = focus.getLinkText();
					else
						e.result = null;
				}
				}
				//System.out.println("getValue e.result = " + e.result);
			}
			
			public void getState(AccessibleControlEvent e)
			{
				//System.out.println("getState childID = " + e.childID + "/" + ACC.CHILDID_SELF);
				e.detail = ACC.STATE_FOCUSABLE;
				
				switch (e.childID)
				{
				  case ACC.CHILDID_SELF: //isFocusControl() && (focus == null))
					   e.detail |=  ACC.STATE_FOCUSED;
		               break;
		          case 0:
		          	   e.detail |= ACC.STATE_SELECTABLE | ACC.STATE_FOCUSED;
		          	   break;
		          default:
		          	   //e.detail |= ACC.STATE_SELECTABLE ;
		               e.detail |= ACC.STATE_FOCUSED;//  | ACC.STATE_SELECTED;
				}
			}
			
			public void getFocus(AccessibleControlEvent e) {
				
				if (getChildViewer()!=null)
				   	e.childID = 1;
				else
				if  (focus != null)
					e.childID =  focus.hashCode();
				else
					e.childID = ACC.CHILDID_NONE;
			}
			
		});
		
		// focus managment
		addFocusListener( new FocusListener() {
			public void focusGained(FocusEvent e) {
				getViewControl().setFocus();
				startAccessibility();
				if (focus==null)
				{
				    SWTViewer.this.nextFocusable();
				    startAccessibility();
				}
			}
			
			public void focusLost(FocusEvent e) {
				
			}
		});
		
	}
	
	/** return the internal IImage provider used by viewer */
	public IDIImageProvider getImageProvider()
	{
	  return iimage_provider;
	}
	
	/** sets the internal IImage provider used by viewer */
	public void setImageProvider(IDIImageProvider img_p)
	{
	   iimage_provider = img_p;
	}
	
	private void initialise()
	{
		resetDefaultSelection();
		this.setContentsPos(0,0);
	}
	
	/** dumps the internal data of the SWTViewer. 
	 * This method is just used for debug session
	 */
	public void dump()
	{
		if (contentCell!=null)
			contentCell.dump(0);
	}
	
	/** dispose all objet used by the viewer */
	public void dispose()
	{
		defaultSelection=null;
		if (contentCell!=null)
		{
			contentCell.dispose();
		    contentCell = null;
		}
		
		if(handcursor_ != null){
			handcursor_.dispose();
		}
		if(graphicmovingcursor_ != null){
			graphicmovingcursor_.dispose();
		}
		
		closeChildViewer();
		
		synchronized(this){
			if(tooltiptimer != null){
				tooltiptimer.cancel();
				tooltiptimer = null;
			}
		}
		iimage_provider.dispose();
	}
	
	/**
	 * Returns the delay used before to open a Tooltip popup.
	 */
	public long getToolipTimerDelay() {
		return defaultProperties.toolipTimerDelay_;
	}
	
	/**
	 * Sets the delay used before to open a Tooltip popup.
	 */
	public void setToolipTimerDelay(long toolipTimerDelay) {
		defaultProperties.toolipTimerDelay_ = toolipTimerDelay;
	}
	
	
	/** 
	 * Emits the signal linkHighlighted_Signal and show the
	 * tooltip if the link has a DLinkUtil.TOOLTIP method
	 */
	public void highlightLink(IDLink link, org.eclipse.swt.graphics.Point p)
	{
		Object[] param1 ={link.getTarget(), (IDLink)link};
		linkHighlighted.emit(param1);
		
		if (DLinkUtil.isPopupLink(link) 
				&& (DLinkUtil.TOOLTIP.equals( DLinkUtil.getMethod(link))))
		{
			IDItem i = resolveLink(link);
			if (i!=null)
				showTooltip(i, p);
			else
				linkUnresolved.emit(new Object[]{link.getTarget(), link});
			
		}
	}
	
	/** 
	 * Hide the tooltip if the link has a DLinkUtil.TOOLTIP method 
	 * and Emits the signal linkUnhighlighted_Signal and 
	 */
	public void unhighlightLink(IDLink link)
	{
		if (DLinkUtil.isPopupLink(link) 
				&& (DLinkUtil.TOOLTIP.equals(DLinkUtil.getMethod(link))))
		{
			hideTooltip();
		}
		
		Object[] param2 ={link.getTarget(), (IDLink)link};
		linkUnhighlighted.emit(param2);
	}
	
	/**
	 * The link acquires the focus. If the link is an internalLink it calls the
	 * DResolverLink object to have the target of the link. It opens a new
	 * window if the link has a  DLinkUtil.isPopupLink method overwise it moves 
	 * the contentItem to show the target of the link. 
	 */
	public void activateLink(IDLink link, org.eclipse.swt.graphics.Point p)
	{
		for (TAbstractCell c = flyingCell; c!=null; c = c.getPrevious())
		{
			if (c instanceof TCellFocusMark)
			{
				if (((TCellFocusMark)c).hasFocus(flyingCell))
				{
					setFocus((TCellFocusMark)c);
					break;
				}
			}
		}
		
		if (DLinkUtil.isInternalLink(link))
		{
			String method = DLinkUtil.getMethod(link);
			if( DLinkUtil.isPopupLink( link ))
			{
				IDItem i = resolveLink(link);
				if (i!=null)
					openNewViewer(i, p, method);
				else
					linkUnresolved.emit(new Object[]{link.getTarget(), link});
			}
			else
			{
				DLinkResolver r = new DLinkResolver(link);
				r.resolve(null);
				IDTaggedItem i = r.getTaggedItem();
				IDItem rib = r.getItemToRebuild();
				if( rib!=null )
				{
					if ( (rib.getRoot() != getContentItem()))
					{
						setContent(rib.getRoot()); 
					}
					else
					{
						//TODO: use getItemToRebuild() to be smarter.
						refresh();
					}
				}
				
				if (i!=null) 
				{
					if ((i.getRoot() != getContentItem()))
						setContent(i.getRoot()); 
					
					scrollTo(i.getTag() );
				} 
				else if ( rib==null )
				{
					Object[] param ={ (String)link.getTarget(), (IDLink)link };
					linkUnresolved.emit(param);
					return;
				}
			}
		}
		else
		{
			// emit signal linkActivated(java.lang.String, org.eclipse.tptp.platform.report.core.internal.IDLink)
			Object[] param ={ (String)link.getTarget(), (IDLink)link };	
			linkActivated.emit(param);
		}
	}
	
	
	private Timer tooltiptimer = null;
	//private boolean cancelledtimer = true;
	private TimerTask tasquette = null;
	boolean runned_ = false;
	private TimerTask getTooltipTask(IDItem i, org.eclipse.swt.graphics.Point pos)
	{
		final IDItem locitem=i;
		final org.eclipse.swt.graphics.Point locpos=pos;
		return new TimerTask()
		{
			boolean cancelled_=false;
			IDItem item_=locitem;
			org.eclipse.swt.graphics.Point pos_=locpos;
			public boolean cancel(){
				cancelled_=super.cancel();
				return cancelled_;
			}
			public void run(){
				if(cancelled_){
					return;
				}
				if(isDisposed()){
					cancel();
					return;
				}
				runned_ = true;
				if (!Display.getDefault().isDisposed())
				{
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							if (pos_==null)
							{
								pos_ = getDisplay().getCursorLocation();
							}
							openNewViewer(item_, pos_, DLinkUtil.TOOLTIP);
						}
					});
				}
			}
		};
	}
	
	/** Shows a tooltip 
	 * @param content is the contents of the tooltip
	 * @param pos is the position of the tooltip. If pos is null the tooltip is shown a the position of the mouse cursor */
	public void showTooltip(IDItem content, org.eclipse.swt.graphics.Point pos)
	{
		//Start the timer
		tasquette = this.getTooltipTask(content, pos);
		if(tooltiptimer == null)
			tooltiptimer = new Timer(true);
		tooltiptimer.schedule(tasquette,100);
		runned_=false;
	}
	
	/**
	 * Hides the opened tooltip
	 */
	public void hideTooltip()
	{
		// Stop the timer
		if(tasquette != null)
		{
			tasquette.cancel();
			tasquette=null;
		}
		closeChildViewer();
		runned_=false;
	}
	
	private IDItem resolveLink(IDLink link)
	{
		DLinkResolver r = new DLinkResolver(link);
		r.resolve(null);
		return r.getTaggedItem();
	}
	
	/**
	 * This class supports tooltip information of the SWTViewer
	 */
	private static class ChildViewer
	{
		ChildViewer(Shell shell, SWTViewer viewer, String method)
		{
			this.shell = shell;
			this.viewer = viewer;
			this.method = method;
		}
		
		public SWTViewer viewer;
		public Shell     shell;
		public String    method;
	}
	
	/** use to show only one popup viewer a one time */
	private ChildViewer childViewer;    
	
	/**
	 * Closes the child viewer if there is one opened
	 */
	protected void closeChildViewer()
	{
		if(childViewer==null) return;
		
		if(!childViewer.shell.isDisposed())
			childViewer.shell.dispose();
		childViewer=null;
	}
	
	/**
	 * Returns the child viewer if it is opened otherwise return null 
	 */
	protected SWTViewer getChildViewer()
	{
		return (childViewer!=null ?  childViewer.viewer: null);
	}
	
	/** 
	 * Opens new viewer in a new Shell window if the method is DLinkUtil.DETACH else it
	 * opens a popup window. 
	 * @param content is the contents of the new window
	 * @param Pos is the coordinates. Can be null, in this case the cursor location has been taken for the coordinates
	 */        
	protected void openNewViewer(IDItem content, org.eclipse.swt.graphics.Point pos, String method)
	{
		closeChildViewer();
		if (content == null) return;
		
		if (!(DLinkUtil.TOOLTIP.equals( method )
				||DLinkUtil.POINTER.equals( method )
				||DLinkUtil.DETACH .equals( method ))) return;
		
		boolean is_win = DLinkUtil.DETACH.equals( method );
		Shell shell = null;
		Color back, fore ;
		Display disp = getDisplay();
		//int w_max, h_max;
		if( is_win )
		{
			//TODO: only one shell for a given link => a map must be used.              
			shell = new Shell( disp, SWT.RESIZE|SWT.MENU);
			shell.setText( getShell().getText() );
			fore = getForeground();
			back = getBackground();
		} 
		else 
		{
			shell = new Shell(getShell(), SWT.ON_TOP);
			fore = disp.getSystemColor(SWT.COLOR_INFO_FOREGROUND);
			back = disp.getSystemColor(SWT.COLOR_INFO_BACKGROUND);
		}
		
		shell.setLayout(new FillLayout());
		shell.setForeground( fore );
		shell.setBackground( back );
		
		SWTViewer popupView = new SWTViewer(shell,SWT.NONE);
		
		if (!is_win)
		    childViewer = new ChildViewer(shell, popupView, method);
		
		popupView.setForeground( fore );
		popupView.setBackground( back );
		popupView.setZoom(zoom, maintainRatio);
		
		popupView.selectionChanged  = selectionChanged;
		popupView.linkHighlighted   = linkHighlighted;
		popupView.linkUnhighlighted = linkUnhighlighted;
		popupView.linkActivated     = linkActivated;
		
		popupView.setShowExternalLinkTooltip(isShowExternalLinkTooltip());
		
		GC ngc = new GC(this);
		SWTGC swtgc = new SWTGC(popupView.getDisplay(), ngc);
		
		int al = (content instanceof DAbstractParagraph)?((DAbstractParagraph)content).getAlignment():DParagraph.LEFT;
		TCellParagraph popupCell = new TCellParagraph(content, null, al);
		popupCell.fillContent(content, swtgc, getImageProvider(), getProgressMonitor(),
				              new DTitleLevel(), new DListLevel(), false);
		popupCell.computeSize(ILayout.NO_SIZE_HINT, ILayout.NO_SIZE_HINT, getZoom(), swtgc, getProgressMonitor());
		popupView.lastPosition = popupCell.layout(getZoom(), 0, getProgressMonitor());
		
		ngc.dispose();
		
		int wdoc = 100, hdoc = 100;
		wdoc=popupCell.getWidth();
		hdoc=popupCell.getHeight();
		
		Rectangle rr = popupView.computeTrim( 0, 0, wdoc, hdoc );
		Rectangle tri2 = shell.computeTrim( 0, 0, rr.width, rr.height );
		int sw=tri2.width-tri2.x, sh=tri2.height-tri2.y;
		
		//location/size outside screen ?
		if (pos==null)
			pos = getDisplay().getCursorLocation();
		
		int x = pos.x;
		int y = pos.y;
		
		if( !is_win )
		{
			//tooltip following mouse musn't take place under mouse cursor ...
			//as it depend on mous cursor shape and cursor hotspot location
			//AND swt doesn't provide method to get them, I try to guess...
			y -= tri2.height;
		}
		
		if( x < 0 ) x=0;
		if( y < 0 ) y=0;
		rr = disp.getBounds();
		if( x+sw > rr.width )
		{
			x=rr.width-sw;
			if( x < 0 ) sw = rr.width;
		} 
		if( y+sh > rr.height)
		{
			y=rr.height-sh;
			if( y < 0 ) sh = rr.height;
		}
		
		shell.setBounds(x,y, sw, sh);
		popupView.setContent(popupCell);
		shell.setVisible(true);
		startAccessibility();
	}
	
	
	/**
	 * Will be implemented in the next version
	 */
	public int getViewMode() 
	{
		return mode.mode();    
	}
	
	/**
	 * Will be implemented in the next version
	 */
	public void setViewMode(int aMode) 
	{
		mode.mode(aMode);    
	}
	
	/**
	 * returns the contentItem that actually shown in the viewer
	 */
	public IDItem getContentItem() 
	{
		return contentItem;    
	}
	
	/**
	 * Sets the contentItem which has to shown in the viewer
	 */
	public void setContent(IDItem rootItem) 
	{
	    if (rootItem instanceof IDDocument)
	        contentItem = ((IDDocument)rootItem).getDocument();
	    else
	        contentItem = rootItem;
	        
		initialise();
		refresh();
	}
	
	/** @deprecated it's replaced by setContent(IDItem) method. 
	 * Sets a document in the viewer
	 */
	public void setDocument(IDDocument document)
	{
		setContent(document);
	}
	
	/**
	 * Sets the contentCell which has to shown in the viewer
	 */
	protected void setContent(TAbstractCellContainer c)
	{
		if (contentCell != null)
			contentCell.dispose();
		
		contentCell = c;
		contentItem = c.getItem();
		resizeContents(c.getWidth(), c.getHeight());	
	}
	
	/**
	 * @deprecated use setContent(TAbstractCellContainer c)
	 * @see SWTViewer#setContent(TAbstractCellContainer)
	 */
	protected void setContent(TCellContainer c)
	{
		setContent((TAbstractCellContainer)c);
	}
	
	/**
	 * Returns the selected text in the viewer in a string 
	 */
	public String getSelectedText()
	{
		if ((contentCell==null) || (!getDefaultSelection().isValid())) return null;
		org.eclipse.swt.graphics.Point pt = getDefaultSelection().getRange();
	 	return TCellUtil.GetTextRange(contentCell,pt.x, pt.y-1);
	}
	
	
	/**
	 * Clears and rebuilds the internal data cell structure 
	 * for the contentItem already setted in the SWTViewer.
	 * This method can take a long time. It is called by
	 * refresh() method. This methods doesn't produce drawing
	 * effect.
	 * 
	 * @see org.eclipse.tptp.platform.report.SWTViewer#refresh()
	 */
	protected void rebuildDoc(boolean contentChanged)
	{
		flyingCell     = null;   
		prevFlyingCell = null; 
		Rectangle area = getClientArea();
		closeChildViewer();
		
		//IDItem focusItem = (focus!=null)?focus.getItem():null;
		
		GC gc = new GC(this);
		SWTGC swtgc = new SWTGC(getDisplay(), gc );
		
		IDProgressMonitor monitor = getProgressMonitor();
		
		monitor.beginTask("Please wait", 0);
			
		if (contentChanged)
		{	
		    
		    
			if (contentCell!=null)
				contentCell.dispose();
			
			int al = (contentItem instanceof DAbstractParagraph)?((DAbstractParagraph)contentItem).getAlignment():DParagraph.LEFT;
			contentCell = new TCellParagraph(contentItem, null, al);
			if (contentCell.getLayout() instanceof IMarginLayout)
			{
			    ((IMarginLayout)contentCell.getLayout()).setMargin(left, right, top, bottom);
			}
			
			contentCell.getLayout().setReversed(reversed);
			
			if (contentItem!=null)
			{
			    monitor.setTotalWorks(1);
			    
				contentCell.fillContent(contentItem, swtgc, getImageProvider(), monitor,
						new DTitleLevel(), new DListLevel(), false);
				
				monitor.worked(monitor.getWorkCount()+1);
			}
			
			
		}
		
		int w;
		if (maintainRatio)
			w = (int)(area.width * zoom);
		else
			w = area.width;
		
		if (contentCell.getLayout() instanceof IMarginLayout)
		{
		    ((IMarginLayout)contentCell.getLayout()).setMargin(left, right, top, bottom);
		}
		//contentCell.setMargin(left, right, top, bottom);
		monitor.setTotalWorks(monitor.getTotalWorks()+2);
		
		contentCell.computeSize(w, ILayout.NO_SIZE_HINT, getZoom(), swtgc, monitor);
		monitor.worked(monitor.getWorkCount()+1);
		
		lastPosition = contentCell.layout(getZoom(), 0, monitor);
		monitor.worked(monitor.getWorkCount()+1);
		
		monitor.done();
		gc.dispose();
		
		computeContentsSize();
		
		
		
		// reset the focus
		focus = null;
		//setFocus(null);
//		if (focusItem!=null)
//		{
//			do 
//			{
//				nextFocusable();
//				if ((focus!=null) && (focus.getItem()==focusItem)) break;
//			}
//			while (focus!=null);
//		}
	}
	
	
	/**
	 * Refreshes the entire contentItem. The text size is recomputed and
	 * the style too. If you want just draw the view because the text
	 * size is not modified prefer call the method redraw.
	 * This method calls rebuildDoc() and after calls redraw()
	 * 
	 * @see org.eclipse.swt.Composite#redraw
	 */
	public void refresh()
	{
		if (isDisposed()) return;
		rebuildDoc(true);
		redraw();
	}
	
	
	/**
	 * Rebuild the contentItem only if the new viewer's width is different that
	 * the older.
	 * @see ScrollView#viewResized()
	 */
	protected void  viewResized()
	{     
		if (contentCell==null) return;
		
		int curr_width = getClientArea().width;
		int need_width = contentCell.getWidth();         
		
		if (( curr_width != need_width ) && (!contentCell.isFixedSize()))
		{
			rebuildDoc(false);
			redraw();
		}
	}
	
	/**
	 * Searches and returns a valid IDLink object for an ITCell when the mouse cursor
	 * is flying on the ITCell.
	 * If it does'nt exist a link for this cell the method returns null.
	 */
	protected IDLink getLinkFromCell( ViewerMouseEvent evt )
	{
	  //get cell location in contents coordinates, I know there are cell.absoluteXY()
	  //but I don't want to allocate a point each time ...
	  
	  int cx = 0, cy = 0;
	  for( TAbstractCell c=evt.cell; c!=null; c=c.getParent() ) { cx+=c.getX(); cy+=c.getY(); }
	  int px =  evt.event.x-cx;
	  int py =  evt.event.y-cy;
	  
	  TAbstractCell cell = evt.cell;
	  
	  IDItem item = cell.getItem();
	  IDLink link=null;
	  if( item instanceof DGraphic )
	  {       
	    link = extractRenderableLink( locatePointInRenderable(px, py,  (DGraphic)item, cell ) );
	  }
	  //parent of item might be a link class ?
	  if(link==null) link = (IDLink)hasParent(item, IDLink.class);
	  
	  return link;
	}
	
	private IDItem hasParent(IDItem item, Class Wanted){
		if(item == null){
			return null;
		}
		for(IDItem iparent = item.getParent();iparent != null;iparent = iparent.getParent()){
			if(Wanted.isInstance(iparent)){
				return iparent;
			}
		}
		return null;
	}
	
	//private IDLink currentPopuped = null;
	//private String tooltip_text = null;  
	/** Use of the JScrib tooltip to avoid Unix platform drawing problem */
	private ToolTipRedrawer tooltip = null;
	
	/** allocated on time for contentsMouse???Event */
	private ViewerMouseEvent me = new ViewerMouseEvent();
	
	/** the current cell where the mouse pointer enter */
	private TAbstractCell flyingCell;
	
	/** allocated on time for keyPress and Release ???Event */
	private ViewerKeyEvent ke = new ViewerKeyEvent();
	
	/** 
	 * Makes a valid ViewerKeyEvent and calls the default key handler event for the viewer
	 * The ViewerKeyEvent is made with a SWT.KeyDown event type.
	 */
	protected void keyPressedEvent( KeyEvent _e )
	{
		try {
			
			ke.handled = false;
			
			//boolean shift = ((_e.stateMask &SWT.SHIFT)!=0);
			boolean ctrl  = ((_e.stateMask &SWT.CTRL)!=0);
			
			//System.out.println("Key pressed = "+_e.keyCode);
			
			switch (_e.keyCode)
			{
			
			case 99:
				if (ctrl) copy();
				break;
				
			case 97:
				if (ctrl) selectAll();
				break;
				
			default:
				break;
			}
			
			
			if (focus == null) return; 	  
			
			ke.viewer  = this;
			ke.cell    = focus;
			ke.event   = _e;
			ke.type    = SWT.KeyDown;
			ke.tooltip = null;
			defaultKeyEventHandler.invokeDoMethod(focus.getItem(), defaultKeyEventHandler, ke);
		}
		finally
		{
			if (!ke.handled)
				super.keyPressedEvent(_e);
			//setToolTipText(me.tooltip);
			tooltip.showToolTip( me.tooltip );
			setFocus();
		}
	}
	
	/** 
	 * Makes a ViewerKeyEvent object and calls the default key handler event for the viewer
	 * The ViewerKeyEvent is made with a SWT.KeyUp event type.
	 */
	protected void keyReleasedEvent( KeyEvent _e )
	{
		try {
			
			ke.handled = false;
			if (focus == null) return; 	  
			
			ke.viewer  = this;
			ke.cell    = focus;
			ke.event   = _e;
			ke.type    = SWT.KeyUp;
			ke.tooltip = null;
			defaultKeyEventHandler.invokeDoMethod(focus.getItem(), defaultKeyEventHandler, ke);
		}
		finally
		{
			if (!ke.handled)
				super.keyReleasedEvent(_e);
			//setToolTipText(me.tooltip);
			tooltip.showToolTip( me.tooltip );
			setFocus();
		}
	}
	
	/** 
	 * Makes a ViewerMouseEvent object and calls the default mouse handler event for the viewer
	 * The ViewerMouseEvent is made with a SWT.MouseDoubleClick event type.
	 */
	protected void contentsMouseDoubleClickEvent( MouseEvent e ) 
	{    
		try {   	
			
			if (flyingCell==null) return;
			
			me.event = e;
			me.type = SWT.MouseDoubleClick;
			me.viewer = this;
			me.handled = false;
			me.cell = flyingCell;
			me.cursor = null;
			me.tooltip = null;
			
			defaultMouseEventHandler.invokeDoMethod(flyingCell.getItem(), defaultMouseEventHandler, me);
		}     
		
		finally {
			
			if (!me.handled)
			{
				super.contentsMouseDownEvent(e);
			}
			setCursor(me.cursor);
			//setToolTipText(me.tooltip);
			tooltip.showToolTip( me.tooltip );
			setFocus();
		}
	}
	
	/** 
	 * Makes a ViewerMouseEvent object and calls the default mouse handler event for the viewer
	 * The ViewerMouseEvent is made with a SWT.MouseDown event type.
	 */
	protected void contentsMouseDownEvent(MouseEvent e) 
	{
		try {   	
			
			if (flyingCell==null) {
				//flyingCell = TCellUtil.MapPointToCell(contentCell)contentCell.mapPointToCell(new Point(e.x, e.y));
 			    flyingCell = TCellUtil.MapPointToCell(contentCell, new Point(e.x, e.y));
				if (flyingCell==null)
				    return;
			}
			
			me.event = e;
			me.type = SWT.MouseDown;
			me.viewer = this;
			me.handled = false;
			me.cell = flyingCell;
			me.cursor = null;
			me.tooltip = null;       
			
			defaultMouseEventHandler.invokeDoMethod(flyingCell.getItem(), defaultMouseEventHandler, me);
			
			if( me.invalidate )
			{
			  refresh();
			}
		}     
		
		finally {
			
			if (!me.handled)
			{
				closeChildViewer();
				super.contentsMouseDownEvent(e);
			}
			setCursor(me.cursor);
			//setToolTipText(me.tooltip);
			tooltip.showToolTip( me.tooltip );
			setFocus();
		}
	}
	
	
	private TAbstractCell prevFlyingCell;
	
	/** 
	 * This method detects when the cursor leaves an ITCell and enters in another. In these cases it calls
	 * twice the default mouse event handler. The first time it is called with a ViewerMouseEvent typed SWT.MouseExit and
	 * a second time with SWT.MouseEnter.<P>
	 * When the cursor stills in the same cell it calls the mouse event handler with a SWT.MouseMove event type.
	 */
	protected void contentsMouseMoveEvent(MouseEvent e) 
	{
		flyingCell = null;
		
		try {
			if (contentCell==null) return;
			
			flyingCell = TCellUtil.MapPointToCell(contentCell, new Point(e.x, e.y));
			
			if (flyingCell == null)
			{
				//System.out.println("No cell detected :((");
				tooltip.hideToolTip();
				return;
			}
			
			if (flyingCell != prevFlyingCell)
			{
				if (prevFlyingCell!=null)
				{
					// Exit from previous Cell
					me.event      = e;
					me.type       = SWT.MouseExit;
					me.viewer     = this;
					me.handled    = false;
					me.cell       = prevFlyingCell;
					me.cursor     = null;
					me.tooltip    = null;
					
					defaultMouseEventHandler.invokeDoMethod(prevFlyingCell.getItem(), defaultMouseEventHandler, me);
				}
				
				if (flyingCell.getItem()==null) 
				{
					//System.out.println("No item for cell :((");
					tooltip.hideToolTip();
					return;
				}
				
				// enter in flying cell
				me.event      = e;
				me.type       = SWT.MouseEnter;
				me.viewer     = this;
				me.handled    = false;
				me.cell       = flyingCell;
				me.cursor     = null;
				me.tooltip    = null;
				
				defaultMouseEventHandler.invokeDoMethod(flyingCell.getItem(), defaultMouseEventHandler, me);
				prevFlyingCell = flyingCell;
				return;
			}
			
			prevFlyingCell = flyingCell;
			
			me.event      = e;
			me.type       = SWT.MouseMove;
			me.viewer     = this;
			me.handled    = false;
			me.cell       = flyingCell;
			me.cursor     = null;
			me.tooltip    = null;
			
			defaultMouseEventHandler.invokeDoMethod(flyingCell.getItem(), defaultMouseEventHandler, me);
			
			if(getDefaultSelection().isValid())
			{
				if(((e.stateMask & SWT.BUTTON1) != 0)||(e.button == 1))
				{
					//Scroll the view
					int dy = contentsToViewY(e.y);
					if(dy<0){
						scrollBy(0,dy);
					}else{
						dy = this.getVisibleHeight() - dy;
						if(dy<0){
							scrollBy(0,-dy);
						}
					}
					me.handled = true;
				}
			}
		}
		
		finally {
			
			if ( !me.handled) 
			{
				//			super.contentsMouseMoveEvent(e);
			}
			
			setCursor(me.cursor);
			
			tooltip.showToolTip( me.tooltip );
			//setFocus();
		}
		
	}
	
	/** 
	 * Makes a ViewerMouseEvent object and calls the default mouse handler event for the viewer
	 * The ViewerMouseEvent is made with a SWT.MouseUp event type.
	 */
	protected void contentsMouseUpEvent(MouseEvent e) 
	{
		if (flyingCell==null) return;
		
		me.event = e;
		me.type = SWT.MouseUp;
		me.viewer = this;
		me.handled = false;
		me.cell = flyingCell;
		me.cursor = null;
		me.tooltip = null;
		
		defaultMouseEventHandler.invokeDoMethod(flyingCell.getItem(), defaultMouseEventHandler, me);
		if (!me.handled)
			super.contentsMouseUpEvent(e);
	}
	
	  /**
	   * Called when the mouse exits the window area
	   * @param e
	   */
	   protected void contentsMouseExit(MouseEvent e)
	   {
	   	if (e.x < 0 || e.y < 0 || e.x >= getContentsWidth() || e.y >= getContentsHeight()) {
		   	//System.out.println(e+" "+e.x+" "+e.y);
			if (flyingCell==null) return;
			
			//	Exit from previous Cell
			me.event      = e;
			me.type       = SWT.MouseExit;
			me.viewer     = this;
			me.handled    = false;
			me.cell       = flyingCell;
			me.cursor     = null;
			me.tooltip    = null;
			
			defaultMouseEventHandler.invokeDoMethod(flyingCell.getItem(), defaultMouseEventHandler, me);
			prevFlyingCell = flyingCell = null;
	   	}
		tooltip.hideToolTip();
	   }
	   
	/**
	 * Changes the properties DGraphic.P_XYZ_PHI and P_XYZ_THETA of the graphic with
	 * the cursor position and redraw the graphic.
	 * 
	 * @see DGraphic
	 * @see SWTViewer#redraw(ITCell)
	 */
	protected void move3DView(ViewerMouseEvent e, DGraphic gph)
	{  
		if( gph==null ) return ;
		
		float phi   = gph.getProperties().get( DGraphic.P_XYZ_PHI, 0.3f );
		float theta = gph.getProperties().get( DGraphic.P_XYZ_THETA, 0.2f );
		int dx = e.event.x -defaultProperties.moveGraphicx_;
		int dy = e.event.y -defaultProperties.moveGraphicy_;
		
		phi   += 0.02f*dy;
		theta -= 0.02f*dx;
		gph.getProperties().store( DGraphic.P_XYZ_PHI, phi);
		gph.getProperties().store( DGraphic.P_XYZ_THETA, theta);
		defaultProperties.moveGraphicx_ = e.event.x;
		defaultProperties.moveGraphicy_ = e.event.y;    
		redraw(e.cell);  
	}
	
	/**
	 * Redraw just on cell of the SWTViewer. Called this method is more
	 * efficient that the method redraw() if just one cell is modified. 
	 * 
	 * @see org.eclipse.swt.Composite#redraw
	 * @see #refresh()
	 */
	public void redraw(TAbstractCell e) 
	{
		
		boolean hs = getDefaultSelection().isValid();
		
		final Rectangle area = getClientArea();
		Rect interarea = Rect.Intersection( area.x,area.y,area.width,area.height,
				                            0,0,e.getWidth(),e.getHeight() );
		if( interarea ==null ) return ;
		
		Image dbuffer = new Image(getDisplay(), e.getWidth(),e.getHeight());//interarea.width, interarea.height);
		GC gcim    = new GC(dbuffer);
		
		gcim.setBackground(getBackground());
		gcim.fillRectangle(0, 0, interarea.getW(), interarea.getH());
		
		Point pigc = new Point(); //0,0);
		TCellUtil.GetGlobalCellXY(e, pigc);
		org.eclipse.swt.graphics.Point pe = new org.eclipse.swt.graphics.Point(pigc.getX(), pigc.getY());
		pe = contentsToView(pe.x, pe.y);  
		
		SWTGC swtgc = new SWTGC(getDisplay(), gcim);
		SWTPainter.PainterEvent p = new SWTPainter.PainterEvent( swtgc, e, 
				 getZoom(), getImageProvider(),
				focus, (focus!=null)?focus.hasFocus(e):false,
						hs, getDefaultSelection().startPos, getDefaultSelection().endPos,
						0, 0,
						interarea.getX(), interarea.getY(), interarea.getW(), interarea.getH(),     
						//redraw all cell...
						interarea.getX(), interarea.getY(), interarea.getW()-interarea.getX()+pigc.getX(), interarea.getH()-interarea.getY()+pigc.getY()); 
		
		defaultPainterEventHandler.drawCell( p );
		
		GC ngc = new GC(getViewControl());
		
		ngc.drawImage(dbuffer,
				0, 0, e.getWidth(),e.getHeight(), //interarea.width, interarea.height, 
				pe.x, pe.y, e.getWidth(),e.getHeight());//interarea.width, interarea.height);
		ngc.dispose();
		gcim.dispose();
		dbuffer.dispose();
		swtgc.dispose();
	
	}
	
	/**
	 * @deprecated use redraw(TAbstractCell)
	 * @see SWTViewer#redraw(TAbstractCell)
	 */
	public void redraw(TCell e) 
	{
	    redraw ((TAbstractCell)e);
	}
	
	/**
	 * Scrolls the contentItem to the item which as the tag \tag
	 */
	public void scrollTo(String tag) 
	{
		if (contentCell == null) return;
		
		TAbstractCell c = TCellUtil.SearchCellNamed(contentCell, tag);
		if (c == contentCell) 
		{
			setContentsPos(0,0);
			return;
		}
		
		if (c == null)
		{
			linkUnresolved.emit(new Object[]{tag, null});
			return;
		}
		scrollTo(c);
	}
	
	/**
	 * Returns an interface on the line at the position lineNumber to access
	 * of the line's cells and line size.
	 * The first line is the line number 0.
	 * Returns null if there is not line at the position lineNumber.
	 */
	public ILineLayout.ILine getLineAtPosition(int lineNumber)
	{
		if (contentCell == null) return null;
		
		TCellParagraph lp = (TCellParagraph)contentCell.getLayout();
		
		int i=0;
		ILineLayout.ILine lastLine = lp.getFirstLine();
		for (ILineLayout.ILine l = lp.getFirstLine(); l!=null; l=l.getNext())
		{
			if (i==lineNumber) break;
			i++;
			lastLine = l;
		}
		return lastLine;
	}
	
	/**
	 * Scrolls the view to the line line.
	 *
	 * @see getLineAtPosition(int lineNumber)
	 */
	public void scrollTo(ILineLayout.ILine line) 
	{
		if (line == null) return;
		
		TAbstractCell c = line.getFirst();
		if (c!=null)
			ensureVisible(c.getX(), c.getY(), c.getWidth(), c.getHeight(), SWT.CENTER);
	}
	
	/**
	 * Selects the line line. 
	 * Deletes the current selection if line is null.
	 * 
	 * @see scrollTo(TLineLayout.ILine line) 
	 */
	public void select(ILineLayout.ILine line)
	{
		if (line!=null && (line.getFirst()!=null))
		{
		    int len = 0;
		    if (line.getLast() instanceof TCellText)
		        len += ((TCellText)line.getLast()).getLen();
		    
			setDefaultSelectionRange(line.getFirst().getPosition(), 
					line.getLast().getPosition()+len-1);
		}
		else
			resetDefaultSelection();
	}
	
	/**
	 * Scrolls the view to ensure visible the ITCell passed in parameter
	 */
	public void scrollTo(TAbstractCell cell) 
	{
		if (cell==null) return;
		
		Point p = new Point(); //0,0);
		TCellUtil.GetGlobalCellXY(cell, p);
		
		if(!isVisible(p.getX(), p.getY(), cell.getWidth(), cell.getHeight()))
			ensureVisible(p.getX(), p.getY(), cell.getWidth(), cell.getHeight(), SWT.CENTER);
	}
	
	/**
	 * @deprecated use scrollTo(TAbstractCell cell)
	 * @see SWTViewer#scrollTo(TAbstractCell)
	 */
	public void scrollTo(TCell cell) 
	{
	    scrollTo((TAbstractCell)cell);
	}
	
	
	/**
	 * Will be implemented in the next version
	 */
	public void search(String regexp) 
	{}
	
	/**
	 * Copies the current selected text in the clipboard
	 * Will be implemented in the next version
	 */
	public void copy() 
	{
		
		String s = getSelectedText();
		if (s==null) return;
		
		Clipboard clip = new Clipboard(getDisplay());
		TextTransfer tr = TextTransfer.getInstance();
		clip.setContents( new Object[]{s}, new Transfer[]{tr});
		
		clip.dispose();
	}
	
	/* do not allocate this every locatePointInGraphic, since it's done when mouse if moving */
	private Rect priv_locatePointInGraphic_rect = new Rect();
	
	/**
	 * Returns the data which is describing what is under the point px,py
	 * @return a IChartLocation or null if nothing can be located.
	 */
	private IRenderLocation locatePointInRenderable( int px, int py, IDRenderable gph, TAbstractCell cell )
	{
		IRender render_ = DRenderRegistry.GetRender( gph );
		IRenderLocation loc = null;
		
		GC gcimg = new GC(this);
		SWTGC igc = new SWTGC( getDisplay(), gcimg );
		try
		{
			priv_locatePointInGraphic_rect.setSize( cell.getWidth(), cell.getHeight() );
			
			TCellData celldata = cell instanceof TCellData ? (TCellData)cell : null ;
			
			loc = render_.locate( px, py, igc, priv_locatePointInGraphic_rect, gph, getZoom(), getImageProvider(), celldata );
		}
		finally
		{
			gcimg.dispose();
			igc.dispose();
		}
		
		return loc;
	}
	
	/**
	 * Convert specified graphic location to a IDLink: located item might implements IDLink,
	 * if not, one of its parent can.
	 * @return a IDLink or null if none.
	 */ 
	public IDLink extractRenderableLink( IRenderLocation loc )
	{
		IDLink link = null;
		if( loc != null ) 
		{
			IDItem item = loc.getItem();
			//link
			if( item instanceof IDLink )
				link = (IDLink)item;
			else
				link = (IDLink)hasParent( item, IDLink.class) ;
		}
		return link;
	}
	
	/**
	 * Selects all the text in the contentItem
	 */
	public void selectAll() 
	{    
		if (contentCell==null) return;
		
		getDefaultSelection().setRange(0, lastPosition);
		redraw();
	}
	
	private void drawProgress(IDProgressMonitor monitor, IGC gc, int x, int y, int w, int h)
    {
	   
	    IBrush oldbrush = gc.setBrush(new SolidBrush(RGBA.BLACK));
	      
	    Rect rect = new Rect(x,y, w-1, h-1);
	    gc.setBrush(new SolidBrush(RGBA.Get(220,220,220)));
	    gc.fillRect(rect);
	    gc.setBrush(new SolidBrush(RGBA.BLACK));
	    gc.drawRect(rect);
	    
	    IFont oldf = gc.setFont(new Font("", 14, Font.BOLD));
	    ISize s = gc.textExtent(monitor.getTaskName());
	    int tx = (w-s.getW()-18-5)/2 + x, ty = (h-s.getH())/2 + y;
	    
	    int sw = s.getW();
	    int sh = s.getH();
	    
	    gc.drawText(monitor.getTaskName(), tx, ty);
	    gc.setFont(oldf);
	    
	    tx += sw + 16;
	    ty = ty + sh/2;

	    gc.fillOval( tx-9,ty-9, 18, 18 );
	    gc.setBrush(new SolidBrush(RGBA.Get(255,210,0)));

	    double prg = ((double)monitor.getWorkCount()/(double)monitor.getTotalWorks())*(Math.PI*2);
	       
	    gc.fillArc( tx,ty,0, 9, 9, prg, Math.PI/2);
	    gc.fillArc( tx,ty,0, 9,9, prg+Math.PI, Math.PI/2);
	            
        gc.setBrush(oldbrush);
    }
	
	/**
	 * Calls the default painter to draw the contents of the view
	 * This method uses double buffering to eliminate flakering
	 * 
	 * @see ScrollView#drawContents()
	 */
	protected void drawContents(GC gc, int clipx, int clipy, int clipw, int cliph) 
	{
		
		Rectangle area = getClientArea();
        
        TAbstractCell cell = contentCell;
        if(cell == null)
        {
           gc.setBackground(getBackground());
           gc.fillRectangle(0, 0, area.width, area.height);
           return;
        }

	    SWTPainter.PainterEvent pe = new SWTPainter.PainterEvent( null, cell, 
				getZoom(), getImageProvider(),
				focus, false,
				getDefaultSelection().isValid(), getDefaultSelection().startPos, getDefaultSelection().endPos,
				//cell in contents coordinates
				cell.getX(), cell.getY(),
				//view area in contents coordinates
				getContentsX(), getContentsY(), area.width, area.height,
				//clip area in contents coordinates
				clipx, clipy, clipw, cliph );
		
		if( double_buffer)
		{
			Image dbuffer = null;
			GC gcim = null;
			SWTGC swtgc = null;
			try{
				
				
				if (getProgressMonitor().isInProgress())
				{
				    int wi = 185+(getVerticalBarWidth()*2);
				    int hi = 35+(getHorizontalBarHeight()*2);
				    
				    dbuffer = new Image(getDisplay(), wi, hi);
					gcim    = new GC(dbuffer);
					
					gcim.setBackground(getBackground());
					gcim.fillRectangle(0, 0, area.width, area.height);
				
					swtgc = new SWTGC(getDisplay(), gcim);
				    drawProgress(getProgressMonitor(), swtgc, getVerticalBarWidth(), getHorizontalBarHeight(), 185, 35);
				    gc.drawImage( dbuffer,(area.width-wi)/2,(area.height-hi)/2);
				}
				else
				{
				   dbuffer = new Image(getDisplay(), area.width, area.height);
				   gcim    = new GC(dbuffer);
					
				   gcim.setBackground(getBackground());
					
				   swtgc = new SWTGC(getDisplay(), gcim);
				   pe.gc = swtgc; 
				    
				   gcim.fillRectangle(0, 0, area.width, area.height);
				
				   defaultPainterEventHandler.drawCell( pe );
				   gc.drawImage( dbuffer,0,0);
				}
				
				
			}
			finally
			{
				//handle-ish resource must be disposed
				if(gcim   !=null) { gcim.dispose(); }
				if(dbuffer!=null) { dbuffer.dispose(); }
				if(swtgc!=null) { swtgc.dispose(); }
				
			}
		}
		else
		{
			gc.setBackground(getBackground());
			gc.fillRectangle(0, 0, area.width, area.height);
			
			SWTGC swtgc = new SWTGC(getDisplay(), gc);
			pe.gc = swtgc;
			defaultPainterEventHandler.drawCell( pe );
			swtgc.dispose();
		}
	}
	
	/**
	 * Sets the margins in pixel for the contentItem
	 * @param t is the top margin
	 * @param b is the botton margin
	 * @param l is the left margin
	 * @param r is the right margin
	 */
	public void setMargins(int _top, int _bottom, int _left, int _right) 
	{
		top    = _top;
		bottom = _bottom;
		left   = _left;
		right  = _right;
	}
	
	/**
	* Sets the selection in character position
	*/
    public void setDefaultSelectionRange(int start, int end) 
    {
   	  getDefaultSelection().setRange(start, end);
   	  redraw();       
    }
   
   /**
	* Disables the current selection
	*/
    public void resetDefaultSelection() 
    { 
   	   getDefaultSelection().reset();
       redraw();
    }
   
   /**
    * Returns the instance of the internal default selection
    */
    public Selection getDefaultSelection() 
    {
   	    if (defaultSelection==null)
   	    	defaultSelection = new Selection();
	    return defaultSelection;
    } 
	
	/**
	 * Returns true if the viewer shows the external link target in a tooltip 
	 * text information.<P>
	 * For example 
	 */
	public boolean isShowExternalLinkTooltip() 
	{
		return showExternalLinkTooltip;
	}
	
	/**
	 * Shows a tooltip text information with the link target for the 
	 * External link only
	 */
	public void setShowExternalLinkTooltip(boolean b) 
	{
		showExternalLinkTooltip = b;
	}
	
	private void computeContentsSize()
	{
		if (contentCell!=null)
		{
			resizeContents((int)(contentCell.getWidth()),
					(int)(contentCell.getHeight()));
		}
	}
	
	/**
	 * Returns the current zoom factor in the viewer
	 */
	public float getZoom() 
	{
		return zoom;
	}
	
	/**
	 * Sets the zoom factor 
	 */
	public void setZoom(float f, boolean mr) 
	{
		maintainRatio = mr;
		zoom = f;
		initialise();
		refresh();
	}
	
	/** 
	 * Pass the focus to the next item
	 * the focusable item can be null if no next focusable
	 * item exist in the contentItem. In this case the focus returns to
	 * the parent widget.
	 */
	public void nextFocusable()
	{
		if (contentCell==null) return;
		
		TAbstractCell start;
		if (focus == null)
			start = contentCell;
		else
			start = focus;
		
		try {
			TCellIterator i = new TCellIterator(start);
			for (TAbstractCell c = i.getNext(); c!=null; c = i.getNext())
			{
				if (c instanceof TCellFocusMark)
				{
					setFocus((TCellFocusMark)c);
					return;
				}
				
			}
			setFocus( null );
//			if (getParent()!=null)
//				getParent().setFocus();
		}
		finally 
		{
			scrollTo(focus);
		}
	}
	
	/** 
	 * Pass the focus to the next item
	 * the focusable item can be null if no previous focusable
	 * item exist in the contentItem
	 */
	public void previousFocusable()
	{
		if (contentCell==null) return;
		
		if (focus==null) return;
		TAbstractCell start = focus;
		
		try {   
			
			TCellIterator i = new TCellIterator(start);
			for (TAbstractCell c = i.getPrevious(); c!=null; c = i.getPrevious())
			{
				if (c instanceof TCellFocusMark)
				{
					setFocus((TCellFocusMark)c);
					return;
				}   
			}
			setFocus( null );
			if (getParent()!=null)
				getParent().setFocus();
		}
		finally
		{
		}
		
	}
	
	
	/**
	 * Returns the internal default paint event handler of the SWTViewer.
	 */
	public SWTPainter getDefaultPainterEventHandler() {
		return defaultPainterEventHandler;
	}
	
	/**
	 * Sets the default paint event handler of the SWTViewer.
	 */
	public void setDefaultPainterEventHandler(SWTPainter painterEventHandler) {
		this.defaultPainterEventHandler = painterEventHandler;
	}
	
	/**
	 * Returns the internal default mouse event handler of the SWTViewer
	 */
	public MouseEventHandler getDefaultMouseEventHandler() {
		return defaultMouseEventHandler;
	}
	
	/**
	 * Sets the default mouse event handler of the SWTViewer.
	 */
	public void setDefaultMouseEventHandler(MouseEventHandler mouseEventHandler) {
		this.defaultMouseEventHandler = mouseEventHandler;
	}
	
	/**
	 * Returns the internal default keyboard event handler of the SWTViewer.
	 */
	public KeyEventHandler getDefaultKeyEventHandler() {
		return defaultKeyEventHandler;
	}
	
	/**
	 * Sets the default keyboard event handler of the SWTViewer.
	 */
	public void setDefaultKeyEventHandler(KeyEventHandler keyEventHandler) {
		this.defaultKeyEventHandler = keyEventHandler;
	}
	
	/**
	 * @return Returns the maintainRatio.
	 */
	public boolean isMaintainRatio() {
		return maintainRatio;
	}
	
	/**
	 * @param maintainRatio The maintainRatio to set.
	 */
	public void setMaintainRatio(boolean maintainRatio) {
		this.maintainRatio = maintainRatio;
	}
	
	/**
	 * Returns true if the viewer's content is reversed (BiDi support).
	 */
	public boolean isReversed() {
		return reversed;
	}
	
	/**
	 * Un/Reverses the viewer's content for BiDi support.
	 */
	public void setReversed(boolean reversed) {
		this.reversed = reversed;
		initialise();
		refresh();
	}
	
	/**
	 * Starts the accessibility 
	 */
	public void startAccessibility()
	{
		getViewControl().getAccessible().setFocus(ACC.CHILDID_SELF);
		
		if (getChildViewer()!=null)
			getViewControl().getAccessible().setFocus(0); 
		else
		if (focus != null) 
			getViewControl().getAccessible().setFocus(focus.hashCode()); 	
	}
	
	/**
	 * Called when the control acquires the focus
	 */
	public boolean setFocus()
	{
		boolean v = getViewControl().setFocus();
		return v;
	}
	/**
	 * Pass the focus to the focus cell mark passed in parameter
	 */
	protected void setFocus( TCellFocusMark cell)
	{
		//System.out.println("setFocus focus = " + cell);
		focus = cell;
		
		if (focus!=null && (focus.getItem() instanceof IDLink))
		{
		  IDLink link = (IDLink)focus.getItem();
		  if (prevlink!=null)
			unhighlightLink(prevlink);
		
		  Point p = new Point();
		  TCellUtil.GetGlobalCellXY(focus, p);
		  org.eclipse.swt.graphics.Point pswt = new org.eclipse.swt.graphics.Point(p.getX(), p.getY());
	      pswt = contentsToView(pswt.x, pswt.y);
		  pswt = toDisplay(pswt.x, pswt.y-5);
		  
	      highlightLink(link, pswt);
	      prevlink = link;
		}
		
		redraw();
	}
	
	/** 
	 * Returns the text content in the SWTViewer
	 */
	public String getContentText()
	{
		if (contentCell!=null)
			return TCellUtil.GetTextRange(contentCell, 0,lastPosition);
		else
			return null;
	}
	
	/** 
	 * Returns the text content in the SWTViewer
	 */
	public String getViewerName() {
		return name;
	}
	
	/**
	 * Set a name to the Viewer and register it in the ViewerRegistry. 
	 * If it is already registered, it updates the key. 
	 * @param n viewer name
	 */
	public void setViewerName(String n) {
		//String oldname = this.name;
		// update the name of the viewer
		this.name = n;
		// update the registry
		this.registry.putViewer(n, this);
		//if (oldname != null) {
		//	this.registry.removeViewer(oldname);
	}
	
	/** 
	 * Returns the ViewerRegistry in which the viewer is registered
	 */
	public ViewerRegistry getViewerRegistry() {
		return registry;
	}
	
	/**
	 * Sets the viewer registry
	 * @param registry
	 */
	public void setViewerRegistry(ViewerRegistry registry) {
		this.registry = registry;
	}
	
	/** @return the internal progress monitor */
	public IDProgressMonitor getProgressMonitor()
	{
	    return progressMonitor;
	}
	
	/** @return the internal progress monitor */
	public void  setProgressMonitor(IDProgressMonitor monitor)
	{
	    progressMonitor = monitor;
	}
	
    /** 
     * This method  returns the IDItem which contains the point x, y.
     * The coordinates are automatically translated to contents coordinates.
     * In fact, this method works with the mouse coordinates directly.
     */
    public IDItem mapPointToIDItem(int x, int y) 
    {
        
        if (contentCell==null) return null;
        
        //bug #106093
        x = viewToContentsX(x);
        y = viewToContentsY(y);
            
        TAbstractCell cell = TCellUtil.MapPointToCell(contentCell, new Point(x, y));
            
        if (cell == null) return null;
            
        if (cell.getItem() instanceof IDRenderable)
        {
           int cx = 0, cy = 0;
           for( TAbstractCell c=cell; c!=null; c=c.getParent() ) { cx+=c.getX(); cy+=c.getY(); }
           int px =  x-cx;
           int py =  y-cy;
           IRenderLocation loc = locatePointInRenderable( px,py, 
                                                          (IDRenderable)cell.getItem(), 
                                                           cell );
           return loc.getItem();
        }
        else
        {
           return cell.getItem();
        }
        
    }
}

