/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ZoomEditorDialog.java,v 1.2 2008/05/23 14:12:20 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 14 janv. 2004
 *
 */
package org.eclipse.tptp.platform.report.ui.dialogs.internal;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Scale;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.platform.report.ReportCorePlugin;
import org.eclipse.tptp.platform.report.signals.internal.Signal;
import org.eclipse.tptp.platform.report.ui.swt.widgets.internal.ScreenPreview;

/**
 * This class provides a dialog to configure a zoom factor. It provides
 * some predefined zoom factor and a custom zoom factor selector. The
 * zoom factor and its impact on the forn is shown in a preview area. 
 * 
 * This class provides a signal named zoomChanged. This signal is emitted
 * every time when the zoom factor changed in the dialog editor.
 * 
 *  @see JScribDialog
 *  @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class ZoomEditorDialog extends JScribDialog 
{

	float   zoomFactor = 1.0f;
	boolean maintainRatio = true;
	boolean fontPreviewEnabled = true;
	int     viewW      = 100;
	int     viewH      = 100;
	int     contentsW  = 100;
	int     contentsH  = 100;
	String  previewText = ReportCorePlugin.translate("PREVIEW_STRING"); //$NON-NLS-1$
	String  previewFontName = ReportCorePlugin.translate("PREVIEW_FONT_NAME"); //$NON-NLS-1$
	int     previewFontSize = 10;
	float   maxZoomFactor = 10.0f;
	float   minZoomFactor = 0.1f;
	
	ZoomFactorWidget fw;
	ZoomPreviewWidget preview;
	
	/**
	 * <b>Signal :</b> zoomChanged(Float f)
	 * <br>
	 * Emit when the zoom factor changed in the zoom editor dialog
	 */
	public Signal zoomChanged = new Signal("zoomChanged(Float)"); //$NON-NLS-1$
	
	/**
	 * <b>Signal :</b> ratioEnabled(Boolean f)
	 * <br>
	 * Emit when the maintain aspect ratio checkbox changed in the zoom editor dialog
	 */
	public Signal ratioEnabled = new Signal("ratioEnabled(Boolean)"); //$NON-NLS-1$
	
 	
	public class ZoomPreviewWidget extends Composite
	{
		private Font font;
		private Text f_text;
		private ScreenPreview s;
		
		public ZoomPreviewWidget(Composite parent, int style)
		{
			super(parent, style|SWT.NO_BACKGROUND);
			setLayout(new FillLayout(SWT.VERTICAL));
			
			Group gb1 = new Group( this, SWT.NONE);
			RowLayout rowLayout1= new RowLayout(SWT.VERTICAL);
			rowLayout1.justify = true;
			gb1.setText(ReportCorePlugin.translate("PREVIEW")); //$NON-NLS-1$
			gb1.setLayout(rowLayout1);
			
			s = new ScreenPreview(gb1, SWT.CENTER);
			
			Group gb2 = new Group( gb1, SWT.NONE);
			gb2.setLayoutData(new RowData(ScreenPreview.SCREEN_W, ScreenPreview.SCREEN_H/2));
			gb2.setLayout( new FillLayout(SWT.VERTICAL) );
			gb2.setText(previewFontSize + ReportCorePlugin.translate("POINT") + previewFontName); //$NON-NLS-1$
			gb2.setEnabled(fontPreviewEnabled);
			
			f_text = new Text(gb2, SWT.MULTI);
			font = new Font(getDisplay(), previewFontName, previewFontSize, SWT.NORMAL);
			f_text.setFont(font);
			f_text.setEnabled(fontPreviewEnabled);
			f_text.setEditable(false);
			f_text.setBackground(getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
			
			f_text.addDisposeListener(new DisposeListener() {
				public void widgetDisposed(DisposeEvent e) {
					if (font != null)
						font.dispose();
				}
			});
			
			if (fontPreviewEnabled)
			   f_text.setText(previewText);
			else
			   f_text.setText(ReportCorePlugin.translate("ZOOM_APPLIED")); //$NON-NLS-1$
			
		}
		
		public void setZoomFactor(Float zoom)
		{
			if (fontPreviewEnabled)
			{	
		   	   Font fn = new Font(getDisplay(), previewFontName, (int)(previewFontSize*zoom.floatValue()), SWT.NORMAL);
		   	   f_text.setFont(fn);
		   	   if (font!=null)
		   		   font.dispose();
		   	   font = fn;
			}
		   	
		   	Rectangle rv = new Rectangle(300,200,viewW, viewH);
		   	Rectangle rc = new Rectangle(2,2,(int)(contentsW*zoomFactor), (int)(contentsH*zoomFactor));
		   	s.setRectangles(rv, rc);
		}
	}
		
	
	public class ZoomFactorWidget extends Composite
	{
	   private Scale custom;
	   private Label label;
		
	   private void createButton(Composite parent, String t, float v)
	   {
	   	final float value = v;
	   	Button b = new Button(parent,SWT.RADIO);
	   	b.setText(t);
	   	b.addSelectionListener(new SelectionAdapter(){
	   		public void widgetSelected(SelectionEvent e) {
	   			zoomChanged.emit(new Float(value));
	   		}
	   	});
	   }
	   
	   public ZoomFactorWidget(Composite parent, int style)
	   {
	   	  super(parent, style);	   	  

	   	  FillLayout l = new FillLayout();
	   	  setLayout(l);
	   	  
	   	  Group gb = new Group( this, SWT.NONE);
	   	  RowLayout rowLayout= new RowLayout(SWT.VERTICAL);
	   	  gb.setLayout( rowLayout );
	   	  gb.setText(ReportCorePlugin.translate("ZOOM_FACTOR")); //$NON-NLS-1$
	   	  
	   	  createButton(gb, ReportCorePlugin.translate("200_PERCENT"), 2.0f); //$NON-NLS-1$
	   	  createButton(gb, ReportCorePlugin.translate("100_PERCENT"), 1.0f); //$NON-NLS-1$
	   	  createButton(gb, ReportCorePlugin.translate("75_PERCENT"), 0.75f); //$NON-NLS-1$
	   	  createButton(gb, ReportCorePlugin.translate("PAGE_WIDTH"), ((float)viewW/(float)contentsW)); //$NON-NLS-1$
	   	  createButton(gb, ReportCorePlugin.translate("ENSURE_VISIBLE"), ((float)viewH/(float)contentsH)); //$NON-NLS-1$
	   	  	   	  
	   	  Composite s = new Composite(gb,SWT.NONE);
	   	  RowLayout pLayout= new RowLayout(SWT.VERTICAL); 
	   	  s.setLayout(pLayout);
	   	  
	   	  label= new Label(s, SWT.LEFT);
	   	  label.setText(ReportCorePlugin.translate("CUSTOM")); //$NON-NLS-1$
	   	  
	   	  custom = new Scale(gb,SWT.HORIZONTAL);
	   	  custom.setMinimum((int)(minZoomFactor*100.0f));
	   	  custom.setMaximum((int)(maxZoomFactor*100.0f));
	   	  custom.setPageIncrement( (int)(((maxZoomFactor-minZoomFactor)/10)*100.0f));
	   	  custom.setSelection((int)zoomFactor*100);
	   	  custom.addSelectionListener(new SelectionAdapter(){
	   	  	public void widgetSelected(SelectionEvent e) {
	   	  		zoomChanged.emit(new Float((float)custom.getSelection()/100.0f));
	   	  	}
	   	  });
	   	  
	   	  Button b_aspectRatio = new Button(gb,SWT.CHECK);
	   	  b_aspectRatio.setSelection(maintainRatio);
	   	  b_aspectRatio.setText(ReportCorePlugin.translate("MAINTAIN_RATIO")); //$NON-NLS-1$
	   	  b_aspectRatio.addSelectionListener(new SelectionAdapter(){
	   	  	public void widgetSelected(SelectionEvent e) {
	   	  		maintainRatio = !maintainRatio;
	   	  		ratioEnabled.emit(new Boolean(maintainRatio));
	   	  	}
	   	  });
	   	  
	   	  zoomChanged.connect(this, "setZoomFactor(Float)"); //$NON-NLS-1$
	   	  ratioEnabled.connect(this, "setRatioEnabled(Boolean)"); //$NON-NLS-1$
	   	  
	   	  setZoomFactor(new Float(zoomFactor));
	   	  custom.setFocus();
	   }
	   
	   public void setZoomFactor(Float v)
	   {
	   	  int percent = (int)(v.floatValue()*100);
	   	  
	   	  custom.setSelection(percent);
	   	  zoomFactor = v.floatValue();
	   	  label.setText(ReportCorePlugin.translate("CUSTOM_AT")+percent+ReportCorePlugin.translate("PERCENT")); //$NON-NLS-1$ //$NON-NLS-2$
	   }
	}
	
	/**
	 * Create a zoom editor 
	 */
	public ZoomEditorDialog(Shell parent)
	{
		super(parent);
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent) 
	{
		Composite composite = (Composite) super.createDialogArea(parent);
		
		GridLayout g = new GridLayout();
		g.numColumns = 2;
		g.horizontalSpacing = 10;
	
		composite.setLayout(g);		
	
		fw = new ZoomFactorWidget(composite, SWT.NONE);
		fw.setZoomFactor(new Float(zoomFactor));
		fw.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.GRAB_HORIZONTAL));
		
		preview = new ZoomPreviewWidget(composite, SWT.NONE);
		zoomChanged.connect(preview, "setZoomFactor(Float)"); //$NON-NLS-1$
		preview.setZoomFactor(new Float(zoomFactor));
		preview.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.GRAB_HORIZONTAL));
	
		getShell().setText(ReportCorePlugin.translate("ZOOM")); //$NON-NLS-1$
		return composite;
	}

	
	/**
	 * Returns the zoom factor configurated in the editor
	 */
	public float getZoomFactor() 
	{
		return zoomFactor;
	}

	/**
	 * Returns the value of the font size and zoomFactor synchronisation
	 * property.
	 */
	public boolean isFontPreviewEnabled() {
		return fontPreviewEnabled;
	}

	/**
	 * Synchronize or not the font size with the zoom factor.
	 * By default this property is enabled in the ZoomEditorDialog class
	 */
	public void setFontPreviewEnabled(boolean fontZoom) {
		this.fontPreviewEnabled = fontZoom;
		
	}
	
	/**
	 * Sets the data for the zoom dialog. This dialog needs the size of
	 * the contents and the size of the view to compute the zoom factor
	 * for the option Page Width and Ensure visible
	 * @param _contentsW is the width in pixel of the contents (document)
	 * @param _contentsH is the height in pixel of the contents.
	 * @param _viewW is the width in pixel of the view.
	 * @param _viewH is the height in pixel of the view.
	 * @param _zoomFactor is the current zoomFactor of the view.
	 */
	public void setData(int _contentsW, int _contentsH, 
			            int _viewW, int _viewH,
			            float _zoomFactor)
	{
	   contentsW  = (int)(_contentsW / _zoomFactor);
	   contentsH  = (int)(_contentsH / _zoomFactor);
	   viewW      = _viewW;
	   viewH      = _viewH;
	   zoomFactor = _zoomFactor;
	}
	 

	

	/**
	 * Sets the text show in the preview area.
	 */
	public void setPreviewText(String sampleText) {
		this.previewText = sampleText;
	}
	
	/**
	 * Sets the font used to show the preview text with zoom factor
	 * applied on the font. 
	 * @param font name is the string name of the font (ex: "Times New Roman").
	 * @param size is the original size of the font in pixel.
	 */
	public void setPreviewFont(String fontName, int size) {
		previewFontName = fontName;
		previewFontSize = size;
	}
	

	/**
	 * Returns the maximum of the zoom factor that the dialog can edit
	 * @return Returns the maxZoomFactor.
	 */
	public float getMaxZoomFactor() {
		return maxZoomFactor;
	}

	/**
	 * Sets the maximum of the zoom factor that the dialog can edit
	 * @param maxZoomFactor The maxZoomFactor to set.
	 */
	public void setMaxZoomFactor(float maxZoomFactor) {
		this.maxZoomFactor = maxZoomFactor;
	}

	/**
	 * Returns the minimum of the zoom factor that the dialog can edit
	 * @return Returns the minZoomfactor.
	 */
	public float getMinZoomFactor() {
		return minZoomFactor;
	}

	/**
	 * Sets the minimum of the zoom factor that the dialog can edit
	 * @param minZoomfactor The minZoomfactor to set.
	 */
	public void setMinZoomFactor(float minZoomfactor) {
		this.minZoomFactor = minZoomfactor;
	}

	/**
	 * @return Returns the maintainRatio.
	 */
	public boolean isMaintainRatio() {
		return maintainRatio;
	}

	/**
	 * @param maintainRatio The maintainRatio to set.
	 */
	public void setMaintainRatio(boolean maintainRatio) {
		this.maintainRatio = maintainRatio;
	}

}
