/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NoOverviewScrollView.java,v 1.4 2008/05/23 14:12:16 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 20 juin 2003
 *
 */
package org.eclipse.tptp.platform.report.ui.swt.widgets.internal;

import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;

/**
 * ScrollView widget provides a scrolling area with on-demand scroll bars
 * 
 * @author Eric Miravete
 * @version 1.0
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class NoOverviewScrollView extends Canvas 
{
   /** Value for scroll bar mode, default is AUTO */
   public static final int AUTO = 0;
   public static final int ALWAYS_OFF = 2;
   public static final int ALWAYS_ON = 1;
   
   private int contents_height_ = 0;
   private int contents_width_ = 0;
   private int contents_x_ = 0;
   private int contents_y_ = 0;
   private int h_scrollbar_mode_ = AUTO;
   private int v_scrollbar_mode_ = AUTO;
   private int h_scrollbar_increment_ = 1;
   private int v_scrollbar_increment_ = 1;
   private boolean auto_scroll_enabled_ = true;
   private int auto_scroll_period_ = 75;
   
   private PaintListener localPaintListener = null;
   private MouseMoveListener localMouseMoveListener = null;
   private MouseListener localMouseListener = null;
   private ControlListener localControlListener = null;
   private KeyListener localKeyListener = null;
   
   /**
    * Create a ScrollView, child of composite c.
    * Both scroll bar have the mode AUTO.
    * Auto scroll feature is enabled using a delay of 250ms.
    * @param c
    * @param style SWT style bits @see SWT
    * @roseuid 3EF324820276
    */
   public NoOverviewScrollView(Composite c, int style) 
   {
    super (c, style|SWT.H_SCROLL|SWT.V_SCROLL);  

    localPaintListener = new PaintListener(){
    public void paintControl(PaintEvent event) {
        Rectangle r = getClientArea();
        int cx = viewToContentsX( r.x );
            int cy = viewToContentsY( r.y );
        drawContents( event.gc, cx, cy, r.width, r.height );
    }
    };
    addPaintListener(localPaintListener);
    
    localMouseMoveListener = new MouseMoveListener(){
    public void mouseMove(MouseEvent e) { 
            int ox = e.x, oy=e.y;
            e.x = viewToContentsX( e.x );
            e.y = viewToContentsY( e.y );
      contentsMouseMoveEvent(e); 
            e.x = ox; e.y = oy; 
       }
    };
    addMouseMoveListener(localMouseMoveListener);
    
    localMouseListener = new MouseListener(){
    public void mouseDoubleClick(MouseEvent e) { 
      int ox = e.x, oy=e.y;
            e.x = viewToContentsX( e.x );
            e.y = viewToContentsY( e.y );
            contentsMouseDoubleClickEvent(e); 
            e.x = ox; e.y = oy; 
    }
    public void mouseDown(MouseEvent e) {
      int ox = e.x, oy=e.y;
            e.x = viewToContentsX( e.x );
            e.y = viewToContentsY( e.y );
           contentsMouseDownEvent(e); 
            e.x = ox; e.y = oy;  
    }
    public void mouseUp(MouseEvent e) {
      int ox = e.x, oy=e.y;
            e.x = viewToContentsX( e.x );
            e.y = viewToContentsY( e.y );
            contentsMouseUpEvent(e); 
            e.x = ox; e.y = oy; 
    }
    };
    addMouseListener(localMouseListener);   
    
    localControlListener = new ControlAdapter() {
      public void controlResized (ControlEvent e) {
         //System.out.println("Scrollview.controlResized");
         viewResizeEvent(e);
      }
    };
    addControlListener(localControlListener);  
    
      localKeyListener =  new KeyListener() {
        public void keyPressed(KeyEvent e) { keyPressedEvent( e ); }
        public void keyReleased(KeyEvent e) { keyReleasedEvent(e); }
      };
      addKeyListener(localKeyListener);  
    
      getVerticalBar().addSelectionListener( new SelectionListener() {
        public void widgetSelected(SelectionEvent e)
        {         
          setContentsPos( contents_x_, getVerticalBar().getSelection() );
        }
        public void widgetDefaultSelected(SelectionEvent e){}
      });
      getHorizontalBar().addSelectionListener( new SelectionListener() {
        public void widgetSelected(SelectionEvent e)
        {
          setContentsPos( getHorizontalBar().getSelection(), contents_y_ );
        }
        public void widgetDefaultSelected(SelectionEvent e){}
      });
      updateScrollBars();
   }
   
   public void dispose()
   {
     if( auto_scroll_!=null )
     {
       auto_scroll_.cancel();
       auto_scroll_=null;
     }
   }
 
   /**
  * Remove the local Listener and add the new listener.
  * 
  * @param nlistener the new listener
  */
   public void replaceControlListener(ControlListener nlistener)
   {
     if(localControlListener != null){
      removeControlListener(localControlListener);
      localControlListener=null;
     }
     addControlListener(nlistener);
   }

   /**
  * Remove the local Listener and add the new listener.
  * 
  * @param nlistener the new listener
  */
   public void replaceKeyListener(KeyListener nlistener)
   {
   if(localKeyListener != null){
    removeKeyListener(localKeyListener);
    localKeyListener=null;
   }
   addKeyListener(nlistener);
   }

   /**
  * Remove the local Listener and add the new listener.
  * 
  * @param nlistener the new listener
  */
   public void replaceMouseListener(MouseListener nlistener)
   {
   if(localMouseListener != null){
    removeMouseListener(localMouseListener);
    localMouseListener=null;
   }
   addMouseListener(nlistener);
   }

   /**
  * Remove the local Listener and add the new listener.
  * 
  * @param nlistener the new listener
  */
   public void replaceMouseMoveListener(MouseMoveListener nlistener)
   {
   if(localMouseMoveListener != null){
    removeMouseMoveListener(localMouseMoveListener);
    localMouseMoveListener=null;
   }
   addMouseMoveListener(nlistener);
   }

   /**
  * Remove the local Listener and add the new listener.
  * 
  * @param nlistener the new listener
  */
   public void replacePaintListener(PaintListener nlistener)
   {
   if(localPaintListener != null){
    removePaintListener(localPaintListener);
    localPaintListener=null;
   }
   addPaintListener(nlistener);
   }
   
   /**
    * Access method for the contentsHeight property.
    * 
    * @return  the current value of the contentsHeight property
    */
   public int getContentsHeight() 
   {
      return contents_height_;    
   }
   
   /**
    * Access method for the contentsWidth property.
    * 
    * @return   the current value of the contentsWidth property
    */
   public int getContentsWidth() 
   {
      return contents_width_;    
   }
   
   /**
    * Access method for the contentsX property.
    * 
    * @return   the current value of the contentsX property
    */
   public int getContentsX() 
   {
      return contents_x_;    
   }
   
   /**
    * Access method for the contentsY property.
    * 
    * @return   the current value of the contentsY property
    */
   public int getContentsY() 
   {
      return contents_y_;    
   }
   
   /**
    * Determines if the dragAutoScroll property is true.
    * 
    * @return <code>true<code> if the dragAutoScroll property is true
    */
   public boolean getDragAutoScroll() 
   {
      return auto_scroll_enabled_;    
   }
   
   /**
    * Sets the value of the dragAutoScroll property.
    * 
    * @param aDragAutoScroll the new value of the dragAutoScroll property
    */
   public void setDragAutoScroll(boolean aDragAutoScroll) 
   {
      auto_scroll_enabled_ = aDragAutoScroll;    
      if( !auto_scroll_enabled_ && (auto_scroll_!=null))
      {
        auto_scroll_.cancel();
        auto_scroll_=null;
      }
   }
   
   /**
    * Change delay (in millisec) used for auto scroll feature.
    * @param _period new period between to auto scroll
    */
   public void setDragAutoScrollPeriod( int _period )
   {
     auto_scroll_period_ = Math.max( 0, _period );
   }
   
   /**
    * Return auto scroll period.
    */
   public int getDragAutoScrollPeriod()
   {
     return auto_scroll_period_;
   }
   
   /**
    * Access method for the hScrollBarMode property.
    * 
    * @return   the current value of the hScrollBarMode property
    */
   public int getHScrollBarMode() 
   {
      return h_scrollbar_mode_;    
   }
   
   /**
    * Sets the value of the hScrollBarMode property.
    * 
    * @param aHScrollBarMode the new value of the hScrollBarMode property
    */
   public void setHScrollBarMode(int aHScrollBarMode) 
   {
      h_scrollbar_mode_ = aHScrollBarMode;    
   }   
   
   /**
    * Access method for the vScrollBarMode property.
    * 
    * @return   the current value of the vScrollBarMode property
    */
   public int getVScrollBarMode() 
   {
      return v_scrollbar_mode_;    
   }
   
   /**
    * Sets the value of the vScrollBarMode property.
    * 
    * @param aVScrollBarMode the new value of the vScrollBarMode property
    */
   public void setVScrollBarMode(int aVScrollBarMode) 
   {
      v_scrollbar_mode_ = aVScrollBarMode;    
   }
   
   /**
    * Return horizontal scroll bar increment, default:1
    */
   public int getHScrollBarIncrement() 
   {
     return h_scrollbar_increment_;
   }
   /**
    * Return vertical scroll bar increment, default:1
    */
   public int getVScrollBarIncrement() 
   {
     return v_scrollbar_increment_;
   }
   /**
    * Change horizontal scroll bar increment, minimum:1.
    * Page increment is always set to visible width.
    */
   public void setHScrollBarIncrement( int _inc ) 
   {
     h_scrollbar_increment_ = Math.max( 1, _inc );
   }
   /**
    * Change vertical scroll bar increment, minimum:1.
    * Page increment is always set to visible height.
    */
   public void setVScrollBarIncrement( int _inc ) 
   {
     v_scrollbar_increment_ = Math.max( 1, _inc );
   }
   
   /**
    * Called when user double on contents area.
    * @param e
    * @roseuid 3EF6CA190300
    */
   protected void contentsMouseDoubleClickEvent( MouseEvent e ) 
   {    
   }
   
   /**
    * Called when mouse is on contents area and button is pressed.
    * @param e
    * @roseuid 3EF6C9B501A7
    */
   protected void contentsMouseDownEvent(MouseEvent e) 
   {    
     mouse_down_x_=e.x;
     mouse_down_y_=e.y;
   }

   /** where mouse down appear on contents area */
   private int mouse_down_x_,mouse_down_y_;
   
   /** TimerTask for auto scroll feature. */
   private static class AutoScroll extends TimerTask
   {
     public int dx_, dy_;
     public NoOverviewScrollView sv_;
     public AutoScroll( NoOverviewScrollView _sv, int _dx, int _dy )
     {
       sv_=_sv; dx_=_dx;dy_=_dy; 
     }
     public void run()
     {
        Display.getDefault().asyncExec(new Runnable() {
          public void run() {
            sv_.scrollBy( dx_, dy_ );
          }
        });
     }
   }
   
   /** Timer for auto_scroll feature */
   private AutoScroll auto_scroll_=null;
   /** TimerTask for auto_scroll feature !=null when auto scroll is running */
   private Timer auto_scroll_timer_=null;

   /**
    * Called when mouse is on contents area and mode.
    * @param _event
    * @roseuid 3EF6D0720338
    */
   protected void contentsMouseMoveEvent(MouseEvent _event) 
   {         
     if( (_event.stateMask&SWT.BUTTON_MASK)!=0 )
     {
       if( !auto_scroll_enabled_ )
       {
         scrollBy( -(_event.x-mouse_down_x_), -(_event.y-mouse_down_y_));
         return;
       }
       
       int sx=0, sy=0;
       
       int v_right = getContentsX()+getVisibleWidth();
       int v_bottom= getContentsY()+getVisibleHeight();
       
       //auto scroll... ?
       if( _event.x < getContentsX() ) {
         sx = -(getContentsX()-_event.x);
         mouse_down_x_=getContentsX();
       } else if ( _event.x > v_right ) {
         sx = _event.x - v_right;
         mouse_down_x_=v_right;
       }
       if( _event.y < getContentsY() ) {
         sy = -(getContentsY()-_event.y);
         mouse_down_y_ = getContentsY();
       } else if ( _event.y > v_bottom ) {
         sy = _event.y - v_bottom;
         mouse_down_y_ = v_bottom;
       }
       
       if( sx!=0 || sy != 0 )
       {
         //start auto scroll...
         if( auto_scroll_==null)
         {
           if( auto_scroll_timer_==null)
           {
             auto_scroll_timer_ = new Timer(true);
           }
           auto_scroll_ = new AutoScroll( this, sx,sy );
           auto_scroll_timer_.schedule( auto_scroll_, 0, auto_scroll_period_ ); 
         } else {
           auto_scroll_.dx_ = sx;
           auto_scroll_.dy_ = sy;         
         }
       } 
       else 
       {   
         if( auto_scroll_ != null )
         {
           auto_scroll_.cancel();
           auto_scroll_ = null;
         }

         scrollBy( -(_event.x-mouse_down_x_), -(_event.y-mouse_down_y_));
       }
     }
   }
   
   /**
    * Called when mouse is on contents area and button is released
    * @param _event
    * @roseuid 3EF6CA010084
    */
   protected void contentsMouseUpEvent(MouseEvent _event) 
   {    
     //reset auto scroll if it's engaged  
     if( auto_scroll_!=null )
     {
       auto_scroll_.cancel();
       auto_scroll_=null;
     }
   }
      
   /**
    * Responsible to draw contents area.  At least rectangle clipX must be redrawn.
    * This rectangle is given in contents coordinates.
    * By default, no paint is produced.
    * @param gc
    * @param clipx
    * @param clipy
    * @param clipw
    * @param cliph
    * @roseuid 3EF323640030
    */
   protected void drawContents(GC gc, int clipx, int clipy, int clipw, int cliph) 
   {    
   }
   
   /**
    * Change the size of the contents area.
    * @param w new width of the area.
    * @param h new height of the area.
    * @roseuid 3EF6AD97017F
    */
   public void resizeContents( int _w, int _h ) 
   {
     if (_w < 0) _w = 0;
     if (_h < 0) _h = 0;
            
       int oldW = contents_width_;
       int oldH = contents_height_;
       
       if( _w==oldW && _h==oldH ) return ;
       
     contents_width_ = _w;
     contents_height_ = _h;
         
       if (oldW > _w) {
           int s = _w;
           _w = oldW;
           oldW = s;
       }
       
       if (oldW < getVisibleWidth())
       {
          if (_w > getVisibleWidth())
          {
             _w = getVisibleWidth();    
          }
          redraw(getContentsX() + oldW, 0, _w - oldW, getVisibleHeight(), true);
       }
       
       if (oldH > _h)
       {
          int s = _h;
          _h = oldH;
          oldH = s;
       }
    
       if (oldH < getVisibleHeight())
       {
         if (_h > getVisibleHeight())
         {
          _h = getVisibleHeight();
         }
         redraw(0, getContentsY()+oldH, getVisibleWidth(), _h - oldH, true);
       }
     updateScrollBars();
   }
   
   /**
    * @param _dx
    * @param _dy
    * @roseuid 3EF6AE43037B
    */
   public void scrollBy( int _dx, int _dy ) 
   {
     setContentsPos( getContentsX()+_dx, getContentsY()+_dy  );
   }
   
   /**
    * Scroll to ensure point(in contents coordinates) is visible.
    */
   public void ensureVisible( int _px, int _py )
   {
     int cx=getContentsX(), cy=getContentsY();
     int right = getContentsX()+getVisibleWidth();
     int bottom= getContentsY()+getVisibleHeight();
     if( _px < getContentsX() )
     {
       cx = _px;
     }
     else if ( _px > right )
     {
       cx = _px-getVisibleWidth();
     }
     if( _py < getContentsY() )
     {
       cy = _py;
     }
     else if ( _py > bottom )
     {
       cy = _py -getVisibleHeight();
     }
     setContentsPos( cx, cy );
   }
   
   /**
    * Make rectangle (_x,_y,_w,_h, in contents coordinates) visible.
    * if rectangle cannot be completly visible, use _align flags.
    * @param _x x contents coordinates of rectangle.
    * @param _y y contents coordinates of rectangle.
    * @param _w width of rectangle.
    * @param _h height of rectangle.
    * @param _align bit or'ed SWT flag like SWT.LEFT,RIGHT,CENTER,TOP,BOTTOM,VERTICAL
    *               used only for bigger rectangle than visible area.
    *               By default CENTER/VERTICAL
    */
   public void ensureVisible( int _x, int _y, int _w, int _h, int _align )
   {
     int cx = getContentsX(), cy = getContentsY();
     int right = getContentsX()+getVisibleWidth();
     int bottom= getContentsY()+getVisibleHeight();
     boolean align_h=false, align_v=false;
     if( _x < getContentsX() )
     {
       cx = _x;
       if( cx+_w > right ) align_h=true;
     }
     else if ( _x+_w > right )
     {
       cx = right-_w;
       if( _x < cx ) align_h=true;
     }
     if( _y < getContentsY() )
     {
       cy = _y;
       if( cy+_h > bottom ) align_h=true;
     }
     else if ( _y+_h > bottom )
     {
       cy = bottom-_h;
       if( _y < cy ) align_v=true;
     }
     
     //compute alignment on visible area horizontally
     if( align_h )
     {
       //use _align flags
         if( (_align&SWT.LEFT)!=0 )
         {
           cx = _x;
         }
         else if ((_align & SWT.RIGHT)!=0 )
         {
           cx = right-_w;
         }
         else //center
         {
           cx = _x+(_w-getVisibleWidth())/2;
         }
     }
     //compute alignment on visible area vertically
     if( align_v )
     {
       //use _align flags
         if( (_align&SWT.TOP)!=0 )
         {
           cy = _y;
         }
         else if ((_align & SWT.BOTTOM)!=0 )
         {
           cy = bottom-_h;
         }
         else //center
         {
           cy = _y+(_h-getVisibleHeight())/2;
         }
     }     
   }
   
   /**
    * @return true if point is visible (expressed in contents coordinates)
    */
   public boolean isVisible( int _px, int _py )
   {
     if( _px < getContentsX() ) return false;
     if( _py < getContentsY() ) return false;
     if( _px > (getContentsX()+getVisibleWidth())) return false;
     if( _py > (getContentsY()+getVisibleHeight())) return false;
     return true;
   }
   /**
    * @return true if rectangle if partially visible.
    */
   public boolean isVisible( int _x, int _y, int _w , int _h )
   {
     if( _x+_w<getContentsX()) return false;
     if( _y+_h<getContentsY()) return false;
     int vr = getContentsX()+getVisibleWidth();
     int vb = getContentsY()+getVisibleHeight();
     if( _x > vr ) return false;
     if( _y > vb ) return false;
     return true;
   }
   /**
    * @return visible part of rectangle, or null if rectangle is not visible.
    *         rectangle is expressed in contents coordinates.
    */
   public Rectangle getVisiblePart( int _x, int _y, int _w , int _h )
   {
     if( _x+_w<getContentsX()) return null;
     if( _y+_h<getContentsY()) return null;
     int vr = getContentsX()+getVisibleWidth();
     int vb = getContentsY()+getVisibleHeight();
     if( _x > vr ) return null;
     if( _y > vb ) return null;
     int rr = _x+_w,
         rb = _y+_h;
     int nl = Math.max( _x, getContentsX() ),
         nt = Math.max( _y, getContentsY() ),
         nr = Math.min( rr, vr ),
         nb = Math.min( rb, vb );
     Rectangle c = new Rectangle( nl, nt, nr-nl, nb-nt );
     return c;
   }
   /**
    * @see getVisiblePart
    */
   public final Rectangle getVisiblePart( Rectangle _r )
   {
     if( _r==null ) return null;
     return getVisiblePart( _r.x, _r.y, _r.width, _r.height );
   }
   
   /**
    * Change top left position of visible area. Check if the given point
    * is inside contents area.
    * @param _x
    * @param _y
    * @roseuid 3EF6AE5D035B
    */
   public void setContentsPos(int _x, int _y) 
   {
     int nx=_x,
         ny=_y;
     if( getVisibleWidth() >= getContentsWidth() )
     {
       nx = 0;
     }
     else
     {
       if( _x<0 )
       {
         nx=0;
       }
       else if ( _x+getVisibleWidth() > getContentsWidth() )
       {
         nx = getContentsWidth()-getVisibleWidth();
       }
     }
     if( getVisibleHeight() >= getContentsHeight() )
     {
       ny = 0;
     }
     else
     {
       if( _y<=0 )
       {
         ny=0;
       }
       else if ( _y+getVisibleHeight() > getContentsHeight() )
       {
         ny = getContentsHeight()-getVisibleHeight();
       }
     }     
     //no move
     if( nx==contents_x_ && ny==contents_y_ )
     {
       return ;
     } 
     contents_x_ = nx;
     contents_y_ = ny;
     updateScrollBars();
//? find smallest area to redraw only them ?
     redraw();     
   }
      
   /**
    * Setup scroll bar using contents, visible and scroll bar mode properties.
    * 
    * @roseuid 3EF6F495009C
    */
   public void updateScrollBars() 
   {
     ScrollBar b = getVerticalBar();
   b.setMinimum( 0 );
   b.setMaximum( getContentsHeight() );
   b.setThumb( getVisibleHeight() );
   b.setPageIncrement( getVisibleHeight() );
   b.setIncrement( v_scrollbar_increment_ );       
   b.setSelection( getContentsY() );
     if( getContentsHeight() <= getVisibleHeight() )
     {
       //hide bar
       switch( v_scrollbar_mode_)
       {
         case AUTO      : if( b.isVisible() ) b.setVisible( false ); break;
         case ALWAYS_OFF: if( b.isVisible() ) b.setVisible( false ); break;
         case ALWAYS_ON : if(!b.isVisible() ) b.setVisible( true  ); break;
       }
     } else {
       switch( v_scrollbar_mode_)
       {
         case AUTO      : if(!b.isVisible() ) b.setVisible( true  ); break;
         case ALWAYS_OFF: if( b.isVisible() ) b.setVisible( false ); break;
         case ALWAYS_ON : if(!b.isVisible() ) b.setVisible( true  ); break;
       } 
     }
     
     b = getHorizontalBar();
   b.setMaximum( getContentsWidth() );
   b.setThumb( getVisibleWidth() );
   b.setSelection( getContentsX() );
   b.setPageIncrement( getVisibleWidth() );
   b.setIncrement( h_scrollbar_increment_ );
     if( getContentsWidth() <= getVisibleWidth() )
     {
       //hide bar
       switch( h_scrollbar_mode_)
       {
         case AUTO      : if( b.isVisible() ) b.setVisible( false ); break;
         case ALWAYS_OFF: if( b.isVisible() ) b.setVisible( false ); break;
         case ALWAYS_ON : if(!b.isVisible() ) b.setVisible( true  ); break;
       } 
     } else {
     //  b.setMaximum( getContentsWidth()-getVisibleWidth());
       switch( h_scrollbar_mode_)
       {
         case AUTO      : if(!b.isVisible() ) b.setVisible( true  ); break;
         case ALWAYS_OFF: if( b.isVisible() ) b.setVisible( false ); break;
         case ALWAYS_ON : if(!b.isVisible() ) b.setVisible( true  ); break;
       } 
     }
   }
   
   /**
    * Transform (x,y) point in widget coordinates to contents coordinates.
    * @param x
    * @param y
    * @return org.eclipse.swt.graphics.Point
    * @roseuid 3EF6CD9E026F
    */
   public final Point viewToContents( int _x, int _y ) 
   {
     Point p = new Point( viewToContentsX(_x), viewToContentsY(_y) );
     return p;
   }
   public int viewToContentsX( int _x )
   {
     return contents_x_+_x;
   }
   public int viewToContentsY( int _y )
   {
     return contents_y_+_y;
   }
   
   /**
    * Transform (x,y) point from contents coordinates, to widget coordinates.
    * @see contentsToViewX, contentsToViewY
    * @param x
    * @param y
    * @return org.eclipse.swt.graphics.Point
    * @roseuid 3EF6CE530085
    */
   public final Point contentsToView( int _x, int _y )
   {
     Point p = new Point( contentsToViewX(_x), contentsToViewY(_y) );
     return p;
   }
   /**
    * Transform X axis coordinates from contents to widgets.
    * @param _x contents coordinate to transform
    * @return coordinate in widget area
    */
   public int contentsToViewX( int _x )
   {
     return _x-contents_x_;
   }
   /**
    * Transform Y axis coordinates from contents to widgets.
    * @param _y contents coordinate to transform
    * @return coordinate in widget area
    */
   public int contentsToViewY( int _y )
   {
     return _y-contents_y_;
   }
   
   /**
    * @return int
    * @roseuid 3EF6F4FC00B8
    */
   public int getVisibleHeight() 
   {
     Rectangle r = getClientArea();
     return r.height ;    
   }
   
   /**
    * @return int
    * @roseuid 3EF6F50703CB
    */
   public int getVisibleWidth() 
   {
    Rectangle r = getClientArea();
//ou uniquement la partie contentsWidth si < r.width     
    return r.width;    
   }
   
   protected void keyPressedEvent( KeyEvent _e )
   {
     switch( _e.keyCode )
     {
       case SWT.ARROW_UP: scrollBy( 0, -getVisibleHeight() );
         break;
       case SWT.ARROW_DOWN: scrollBy( 0, +getVisibleHeight() );
         break;
       case SWT.ARROW_LEFT: scrollBy( -getVisibleWidth(), 0 );
         break;
       case SWT.ARROW_RIGHT: scrollBy( +getVisibleWidth(), 0 );
         break;
     }
   }
   
   protected void keyReleasedEvent( KeyEvent _e )
   {
   }
   
   /**
    * Called when ScrollView is resized.
    * @param _event
    * @roseuid 3EF6F56900F1
    */
   protected void viewResizeEvent( ControlEvent _event ) 
   {
     int ocx = getContentsX(), ocy = getContentsY();
     //scroll contents x,y if visible area bigger than contents...     
     setContentsPos( ocx, ocy );     
     //if no scroll done, scroll bar must be updated.
     if( ocx==getContentsX() && ocy==getContentsY() )
     {
       updateScrollBars();
     }
   }
   
   public static void main( String[] argv )
   {
     Display disp = Display.getDefault();
     Shell shell = new Shell(disp, SWT.DIALOG_TRIM|SWT.RESIZE );
     final Image img = new Image( null, "C:\\manometer.png");
     NoOverviewScrollView sv = new NoOverviewScrollView( shell, SWT.NO_BACKGROUND ){
       public void drawContents( GC _gc, int _x, int _y, int _w, int _h )
       {
         Rectangle r = getClientArea();
         _gc.setBackground( new Color(null,255,255,255));
         _gc.fillRectangle( r.x,r.y,r.width,r.height);
         _gc.setForeground( new Color(null,255,0,0));
         //croix rouge en 0,0,max,max (contents area)
         Point cpl = contentsToView( 0,0 );
         Point cpr = contentsToView( getContentsWidth(), getContentsHeight() ); 
         _gc.drawLine( cpl.x, cpl.y, cpr.x,cpr.y);
         _gc.drawLine( cpl.x,cpr.y,cpr.x,cpl.y);
         _gc.setForeground( new Color(null,0,255,0));

         _gc.drawLine( 0,0,r.width,r.height);
         _gc.drawLine( 0,r.height,r.width,0);
         
         _gc.setForeground( new Color(null,0,0,255));
         _gc.setBackground( new Color(null,0,0,255));
         if( isVisible( 100,200,100,50))
         {
           _gc.fillRectangle( contentsToViewX(100), contentsToViewY(200), 100,50);
         }
         Rectangle rimg = img.getBounds();
         if( isVisible( 500,500,rimg.width,rimg.height) )
         {
           _gc.drawImage( img, contentsToViewX(500), contentsToViewY(500) );
         }
       }
     };
     sv.resizeContents( 1000, 1000 );
     shell.setLayout( new FillLayout() );
     shell.setSize( 400, 300 );
     shell.open();
      while (!shell.isDisposed())
        if (!disp.readAndDispatch())
          disp.sleep();
   }
}
