/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionStatisticPage2.java,v 1.11 2010/06/30 20:45:50 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.adapter.internal;

import java.security.AccessController;
import java.security.PrivilegedAction;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.trace.ui.TraceViewer;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.internal.ExecutionStatisticView;
import org.eclipse.hyades.trace.views.internal.MultiLevelStatisticView;
import org.eclipse.hyades.trace.views.internal.TraceUIPlugin;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.PlatformUI;

public class ExecutionStatisticPage2 extends MultiLevelStatisticMethodPage {
    
    ExecutionStatisticTabItem[] _items;
    CTabFolder _tabFolder;
    
    static final int EXECUTION_SUMMARY_TAB_INDEX = 0; 
    static final int EXECUTION_DETAILS_TAB_INDEX = 1;
    static final int AGGREGATED_EXECUTION_STATISTICS_TAB_INDEX = 2; 
    static final int METHOD_DETAILS_TAB_INDEX = 3;
    static final int METHOD_INVOCATION_TAB_INDEX = 4;
    static final int NO_ACTIVE_TAB_INDEX = -2;        

    private int activeTabIndex = NO_ACTIVE_TAB_INDEX;
    
    private MethodDetailsTab _methodDetailsTab;
    
    public MethodDetailsTab getMethodDetailsTab() {
		return _methodDetailsTab;
	}
    
    public ExecutionStatisticPage2(EObject mofObject, TraceViewer _viewer) {
        super(mofObject, _viewer);
        
        _methodDetailsTab = new MethodDetailsTab(this);
        
        _items = new ExecutionStatisticTabItem[] {
                new ExecutionSummaryTab(this),
                new ExecutionStatisticTab(this),
                new AggregatedExecutionStatisticsTab(this),
                _methodDetailsTab,
                new SinglePatternTab(this)
        };
    }

    protected MultiLevelStatisticView displayData(Composite parent) {
        return null;
    }
    
    public void createControl(Composite parent) {

    	PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, TraceUIPlugin.getPluginId()+".clst0000");
    	
        GridData gd= new GridData();
        gd.horizontalAlignment= GridData.FILL;
        gd.grabExcessHorizontalSpace= true;
        gd.verticalAlignment= GridData.FILL;
        gd.grabExcessVerticalSpace= true;
        
        _tabFolder = new CTabFolder(parent, SWT.BOTTOM | SWT.FLAT);
        _tabFolder.setLayoutData(gd);
        _tabFolder.setFont(parent.getFont());
        _tabFolder.setBackground(parent.getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
        _tabFolder.setForeground(parent.getDisplay().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND));
        _tabFolder.setSelectionBackground(
        		new Color[] { parent.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND) }, new int[0]);

        _tabFolder.addSelectionListener(new SelectionListener() {
            public void widgetDefaultSelected(SelectionEvent e) {
                tabItemSelected();
            }
            public void widgetSelected(SelectionEvent e) {
                tabItemSelected();
            }
        });
        
        for (int i = 0; i < _items.length; i++) {
        	if (_items[i].isEmpty()) {
        		continue;
        	}
            _items[i].createControl(_tabFolder);

			CTabItem tabItem = new CTabItem(_tabFolder, SWT.NONE);
			tabItem.setText(_items[i].getViewTitle());
			tabItem.setControl(_items[i].getControl());
			tabItem.setData(_items[i]);
		}
        selectTabItem(EXECUTION_SUMMARY_TAB_INDEX);
    }
    
    public Action percentMode() {
        return ((ExecutionStatisticViewer2) getTraceViewer()).percentMode();
    }
    
    public Action deltaColumns() {
        return ((ExecutionStatisticViewer2) getTraceViewer()).deltaColumns();
    }   
    
    /**
     * @return the Method Level action
     */
    public Action getInstanceLevel() {
        return ((ExecutionStatisticViewer2) _viewer).getInstanceLevel();
    }
    
    protected ExecutionStatisticTabItem getSelectedItem() {
        if (_tabFolder == null) {
            return null; 
        }
        CTabItem selection = _tabFolder.getSelection();
        return (ExecutionStatisticTabItem) ((selection != null) ? selection.getData() : null);
    }
    
    public ExecutionStatisticViewer2 getViewer() {
        return (ExecutionStatisticViewer2)_viewer;
    }
    
    public void updateButtons(int level)
    {
        super.updateButtons(level);
        
        getInstanceLevel().setChecked(ExecutionStatisticView.INSTANCE_LEVEL == level);
        deltaColumns().setEnabled(ExecutionStatisticView.INSTANCE_LEVEL != level);
    }
    
    public boolean isEmpty() {
        return !PerftraceUtil.hasExecutionInfo(getMOFObject());
    }
    
    public Control getControl() {
        return _tabFolder;
    }
    
    public void refreshPage() {
        if (_tabFolder != null) {
            for (int i = 0; i < _tabFolder.getItemCount(); i++) {
            	((ExecutionStatisticTabItem)_tabFolder.getItem(i).getData()).refresh();
            }
        }
    }   

    public void setFocus() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.setFocus();
        }
    }   

	public void selectionChanged() {
		if (_tabFolder != null) {
            for (int i = 0; i < _tabFolder.getItemCount(); i++) {
            	((ExecutionStatisticTabItem)_tabFolder.getItem(i).getData()).selectionChanged();
            }
		}
	}  

    public void fillContextMenu(IMenuManager menu) {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.fillContextMenu(menu);
        }
    }

    public void dispose() {
		super.dispose();
		
		_mofObject = null;  
		
		if(fContextMenu != null)
		   fContextMenu.dispose();
	   
		disposed = true;
    	
        if (_items != null) {
            for (int i = 0; i < _items.length; i++) {
                _items[i].dispose();
            }
        }
        _tabFolder.dispose();
       
    }

    public MultiLevelStatisticView getView() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            return selItem.getStatisticView();
        }
        return null;
    }   

	public void update(boolean newPage) {
		if (newPage) {
            for (int i = 0; i < _items.length; i++) {
                _items[i].update(newPage);
            }
		} else if (_tabFolder != null) {
            _items[getActiveTabIndex()].update(newPage);
		}
	}

	protected void setActiveTabIndex(int index){
        if (index >= 0
            && index < _items.length) {
            
            activeTabIndex = index;
            return;
        }
        activeTabIndex = NO_ACTIVE_TAB_INDEX;
    }    
    
    protected int getActiveTabIndex(){
        return activeTabIndex;
    }    
    
    protected void selectTabItem(int index) {
		if (_tabFolder != null && index >= 0 && index < _items.length) {
			for (int tab = 0; tab < _tabFolder.getItemCount(); tab++) {
				if (_tabFolder.getItem(tab).getData() == _items[index]) {
					_tabFolder.setSelection(tab);
					activeTabIndex = index;
		            tabItemSelected();
				}
			}
		}
    }
    
	public void showMethodDetailsTab() {
		selectTabItem(METHOD_DETAILS_TAB_INDEX);
	}

	public void showExecutionStatisticTab() {
		selectTabItem(EXECUTION_DETAILS_TAB_INDEX);
	}   
    
	public void showMethodInvocationTab() {
		selectTabItem(METHOD_INVOCATION_TAB_INDEX);
	}   
	
    public void showAggregatedExecutionStatistic() {
		selectTabItem(AGGREGATED_EXECUTION_STATISTICS_TAB_INDEX);
    }

    public void showExecutionSummaryTab() {
		selectTabItem(EXECUTION_SUMMARY_TAB_INDEX);
    }

	public void setCheckedLevels(int newLevel) {
		for (int i = 0; i < _items.length; i++) {
			_items[i].setCheckedLevels(newLevel);
		}
	}

	public AggregatedExecutionStatisticsTab getAggregatedExecutionStatisticsTab() {
        return (AggregatedExecutionStatisticsTab)_items[AGGREGATED_EXECUTION_STATISTICS_TAB_INDEX];
    }

    public void runShowPercentAction() {
        for (int i = 0; i < _items.length; i++) {
            _items[i].runShowPercentAction();
        }
    }

    public void runDeltaColumnsAction() {
        for (int i = 0; i < _items.length; i++) {
            _items[i].runDeltaColumnsAction();
        }
    }
    
    public void runNextAction() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.runNextAction();
        }
    }

    public void runPreviousAction() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.runPreviousAction();
        }
    }

    public void runHomeAction() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.runHomeAction();
        }
    }

    public void runShowCallerAction() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.runShowCallerAction();
        }
    }

    public void runShowCalleeAction() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.runShowCalleeAction();
        }
    }

    public void runStdSelectAction() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.runStdSelectAction();
        }
    }

    public void runZoomInAction() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.runZoomInAction();
        }
    }

    public void runZoomOutAction() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
            selItem.runZoomOutAction();
        }
    }
    
    protected void tabItemSelected() {
    	final ExecutionStatisticTabItem selItem = getSelectedItem();
        if (selItem != null) {
        	if(isVisible())
        		propagateHiddenToItem(activeTabIndex);
        	
        	activeTabIndex = NO_ACTIVE_TAB_INDEX;
        	for (int i = 0; i < _items.length; i++) {
        		if (_items[i] == selItem) {
        			activeTabIndex = i;
        			break;
        		}
        	}
        	
        	if(isVisible())
        		propagateVisibleToItem(activeTabIndex);

        	getViewer().setFilterActionEnabled(selItem.isFilteringEnabled());
            getViewer().enableSupportedActions(selItem);
            selItem.tabItemSelected();

            AccessController.doPrivileged(new PrivilegedAction() {
            	public Object run() {
            		System.setProperty("ShowFilterAction", Boolean.toString(selItem.isFilteringEnabled()));
            		return null;
            	}
            });
        } else {
            getViewer().enableSupportedActions(null);
        }
    }    

	public Object[] getViewerControls() {
        ExecutionStatisticTabItem selItem = getSelectedItem();
		return selItem != null ? selItem.getViewerControls() : null;
	}

	protected void handleHidden() {
		propagateHiddenToItem(activeTabIndex);
	}

	protected void handleVisible() {
		propagateVisibleToItem(activeTabIndex);
	}
	
	protected void propagateVisibleToItem(int index){
		if(index == NO_ACTIVE_TAB_INDEX)
			return;
		
		ExecutionStatisticTabItem item = _items[index];
		item.setVisible();
	}

	protected void propagateHiddenToItem(int index){
		if(index == NO_ACTIVE_TAB_INDEX)
			return;
		
		ExecutionStatisticTabItem item = _items[index];
		item.setHidden();
	}

}
