/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MultiLevelStatisticViewer.java,v 1.21 2006/10/10 20:01:22 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.adapter.internal;

import java.util.Enumeration;
import java.util.Vector;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.FilterTraceViewer;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.views.actions.internal.ContextSelectionMenuListener;
import org.eclipse.hyades.trace.views.internal.MultiLevelStatisticView;
import org.eclipse.hyades.trace.views.internal.TracePluginImages;
import org.eclipse.hyades.trace.views.internal.TraceUIMessages;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.trace.views.util.internal.OpenSource;
import org.eclipse.hyades.ui.filters.IFilterScopes;
import org.eclipse.hyades.ui.provisional.context.ContextManager;
import org.eclipse.hyades.ui.provisional.context.IContextAttributes;
import org.eclipse.hyades.ui.provisional.context.IContextLabelFormatProvider;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.part.IPage;

public abstract class MultiLevelStatisticViewer extends FilterTraceViewer implements IContextViewer {
	
	protected Action _packageLevel;
	protected Action _classLevel;
	protected Action _openSource;   
	protected Action _chooseColumns;   
	protected Action _sortByColumn;
	protected ContextSelectionMenuListener _contextSelectionMenuListener;
	
	protected String _context;
	protected boolean _contextChanged;

	public MultiLevelStatisticViewer() {
		super();
		_contextChanged = false;
	}

	public abstract String getViewTitle();	
	public abstract TraceViewerPage createPage(EObject mofObject);	

	public void setCheckedLevels(MultiLevelStatisticPage page, int newLevel)
	{
		if (page.getView().getLevel() == newLevel)
		{
			page.getView().updateButtons();
		}
		else
		{
			page.getView().setLevel(newLevel);
			page.getView().updateButtons();
			page.getView().redrawTable();
		}
	}
	
	/**
	 * return the selected agent in the Profiling monitor view
	 * called by addViewPage() -handelProfileEvent() - refreshVisiblePage()and
	 * removePage() of the parent class (TraceViewer)
	 */
	public EObject getObjectToView(EObject selObject) {
		if (selObject == null)
			return selObject;

		if (selObject instanceof TRCProcessProxy) {
			int nbOfAgents = 0;
			TRCAgentProxy a = null;
			EList agents = ((TRCProcessProxy) selObject).getAgentProxies();
			for (int idx = 0; idx < agents.size(); idx++) {
				TRCAgentProxy agent = (TRCAgentProxy) agents.get(idx);
				if (agent == null || agent.eIsProxy())
					continue;
				if (agent
					.getType()
					.equals(HyadesConstants.PROFILE_AGENT_TYPE)) {
					nbOfAgents++;
					a = agent;
				}
			}
			if (nbOfAgents == 1) {
				return a;
			}
		}
		return selObject;
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose
	 */
	public void dispose() {
		Enumeration pages = _pages.elements();
		while (pages.hasMoreElements()) {
			IPage page = (IPage) pages.nextElement();

			if (page instanceof MultiLevelStatisticPage)
				 ((MultiLevelStatisticPage) page).dispose();
		}

		setActionsNull();

		if (fContextMenu != null)
			fContextMenu.dispose();

		fContextMenu = null;
		super.dispose();
	}

	protected void setActionsNull()
	{
		_classLevel = null;
		_packageLevel = null;
		_openSource = null;		
	}
	
	/**
	 * set the focus on the current page
	 */
	public void setFocus() {
		MultiLevelStatisticPage page = (MultiLevelStatisticPage) getCurrentPage();
		if (page != null)
			page.setFocus();
		ContextUpdaterHelper.setCurrentFocusContextLanguage(getSelectedContext());
	}
	
	protected void getLevelActions(Vector actions)
	{
		_packageLevel = new Action(IContextAttributes.SHOW_PACKAGE_LEVEL) {
			public void run() {
				MultiLevelStatisticPage page = (MultiLevelStatisticPage) getCurrentPage();

				if(page == null)
				{
				  revertChecked(_packageLevel);
				  return;
				}

				setCheckedLevels(page, MultiLevelStatisticView.PACKAGE_LEVEL);
			}
			public int getStyle() {
				return IAction.AS_CHECK_BOX;
			}
		};

//		_packageLevel.setText(strPackageLevel);
//		_packageLevel.setImageDescriptor(packageLevelLP.getIcon());
//		TracePluginImages.setImageDescriptors(
//			_packageLevel,
//			TracePluginImages.T_LCL,
//			TracePluginImages.IMG_PACKAGE);
//		_packageLevel.setDescription(strPackageLevel);
//		_packageLevel.setToolTipText(strPackageLevel);
		_packageLevel.setEnabled(true);
		_packageLevel.setChecked(true);


		_classLevel = new Action(IContextAttributes.SHOW_CLASS_LEVEL) {
			public void run() {
				MultiLevelStatisticPage page = (MultiLevelStatisticPage) getCurrentPage();

				if(page == null)
				{
				  revertChecked(_classLevel);
				  return;
				}
				
				setCheckedLevels(page, MultiLevelStatisticView.CLASS_LEVEL);
			}

			public int getStyle() {
				return IAction.AS_CHECK_BOX;
			}
		};

//		_classLevel.setText(strClassLevel);
//		TracePluginImages.setImageDescriptors(
//			_classLevel,
//			TracePluginImages.T_LCL,
//			TracePluginImages.IMG_CLASS);
//		_classLevel.setImageDescriptor(classLevelLP.getIcon());
//		_classLevel.setDescription(strClassLevel);
//		_classLevel.setToolTipText(strClassLevel);
		_classLevel.setEnabled(true);
		
		actions.addElement(new Separator());		
		actions.addElement(_packageLevel);
		actions.addElement(_classLevel);
	}
	
	protected void getActions(Vector actions)
	{
		String openSourceStr = TraceUIMessages._74;
		_openSource = new Action(openSourceStr)
		{
			public void run()
			{
				MultiLevelStatisticPage page = (MultiLevelStatisticPage)getCurrentPage();

				if(page == null)
				  return;

				OpenSource.openSource(UIPlugin.getDefault().getSelectionModel(page.getMOFObject()).getFirstElement());
			}
		};

		_openSource.setText(openSourceStr);
		TracePluginImages.setImageDescriptors(_openSource,TracePluginImages.T_TOOL,TracePluginImages.IMG_SOURCE);	
		_openSource.setDescription(openSourceStr);   
		_openSource.setToolTipText(openSourceStr);
		_openSource.setEnabled(false);


		String strChooseColumns = TraceUIMessages._56;
		_chooseColumns = new Action(strChooseColumns) {
			public void run() {
				MultiLevelStatisticPage page = (MultiLevelStatisticPage)getCurrentPage();
				if (page == null) {
					return;
				}
				MultiLevelStatisticView view = page.getView();
				view.getChooseColumnsAction(view.getColumnDataList(), view.getColumnsPreferencesKey()).run();
			}
		};
		_chooseColumns.setText(strChooseColumns);
		TracePluginImages.setImageDescriptors(_chooseColumns, TracePluginImages.T_LCL, TracePluginImages.IMG_CHOOSECOLUMNS);
		_chooseColumns.setDescription(strChooseColumns);
		_chooseColumns.setToolTipText(strChooseColumns);
		_chooseColumns.setEnabled(true);

		String strSortByColumn = TraceUIMessages._58;
		_sortByColumn = new Action(strSortByColumn) {
			public void run() {
				MultiLevelStatisticPage page = (MultiLevelStatisticPage)getCurrentPage();
				if (page == null) {
					return;
				}
				MultiLevelStatisticView view = page.getView();
				view.getSortByColumnAction().run();
			}
		};
		
		_sortByColumn.setText(strSortByColumn);
		TracePluginImages.setImageDescriptors(_sortByColumn, TracePluginImages.T_LCL, TracePluginImages.IMG_SORTCOLUMNS);
		_sortByColumn.setDescription(strSortByColumn);
		_sortByColumn.setToolTipText(strSortByColumn);
		_sortByColumn.setEnabled(true);
		
		IMenuManager menum = getViewSite().getActionBars().getMenuManager();
		menum.add(_chooseColumns);
		menum.add(_sortByColumn);
//		menum.add(new Separator());
		
		MenuManager submenu = new MenuManager(TraceUIMessages._222);
		submenu.addMenuListener(getContextSelectionMenuListener());
		submenu.add(new Action(){});
		
		menum.add(submenu);
//		menum.add(_printColumns);
		
		actions.addElement(_openSource);
		actions.addElement(new Separator());
	}
	
	/**
	 * Shows the given page
	 */
	public void showPage(IPage page) {
		super.showPage(page);

		getContextSelectionMenuListener().setViewer(this);
	}

	/**
	 * Three action buttons are defined:
	 * -packageLevel: display all the tree (package/class/method)
	 * -classlevel: display only the class tree
	 * -method level: display only the methods
	 */
	public void makeActions() {
		if (isInitializedMenu())
			return; //create the toolbar only once

		super.makeActions();		
		
		initializedMenu(true);

		Vector actions = new Vector();
		
		getLevelActions(actions);
		updateContext();
		actions.addElement(new Separator());		
		getActions(actions);
		
		IToolBarManager tbm = getViewSite().getActionBars().getToolBarManager();

		//tbm.removeAll();

		for (int i=0; i < actions.size(); i++)
		{
			if (actions.elementAt(i) instanceof IContributionItem)
				tbm.add((IContributionItem)actions.elementAt(i));
			else if(actions.elementAt(i) instanceof IAction)
				tbm.add((IAction)actions.elementAt(i));
		}


		// require update because toolbar control has been created by this point,
		// but manager does not update it automatically once it has been created
		getViewSite().getActionBars().updateActionBars();
	}

	/**
	 * @return the class level action
	 */
	public Action getClassLevel() {
		return _classLevel;
	}

	/**
	 * @return the package level action
	 */
	public Action getPackageLevel() {
		return _packageLevel;
	}

	public Action openSource() {
		return _openSource;
	}
	
	protected void revertChecked(Action action)
	{
		action.setChecked(!action.isChecked());
	}	
	
	public String[] getFilterScope()
	{
		String[] scope = new String[1];
		scope[0] = IFilterScopes.PROFILING_AGENT_FILTER_SCOPE;
		return scope;
	}	
	
	public void filterUpdated()
	{
		TraceViewerPage page = (TraceViewerPage)getCurrentPage();
		if (page != null) {
			if (fPartVisible)
				page.refreshPage();
			else
				fForceRefresh = true;
		}
	}	
	
	protected void setActionProperties(Action action, String context, String attributeId)
	{
		String strLevel = null;
		Image icon = null;
		
		IContextLabelFormatProvider clfp = ContextManager.getContextLabelFormatProvider(context, attributeId, IContextLabelFormatProvider.MODE_VIEW_ACTION);
		if (clfp!=null)
		{
			strLevel = clfp.getDisplayStringFromElement(attributeId, null, IContextLabelFormatProvider.MODE_VIEW_ACTION);		
			icon = clfp.getDisplayImageByElement(attributeId, null, IContextLabelFormatProvider.MODE_VIEW_ACTION);
		}
		else
		{
			strLevel = attributeId;
		}
		
		action.setText(strLevel);
		action.setImageDescriptor(ImageDescriptor.createFromImage(icon));
		action.setDescription(strLevel);
		action.setToolTipText(strLevel);		
	}
	
	public ContextSelectionMenuListener getContextSelectionMenuListener()
	{
		if (_contextSelectionMenuListener == null)
			_contextSelectionMenuListener = new ContextSelectionMenuListener();
		
		return _contextSelectionMenuListener;
	}
	
	public void updateContext()
	{
		setActionProperties(_packageLevel, _context, IContextAttributes.SHOW_PACKAGE_LEVEL);
		setActionProperties(_classLevel, _context, IContextAttributes.SHOW_CLASS_LEVEL);	
		ContextUpdaterHelper.setCurrentFocusContextLanguage(getSelectedContext());
		_contextChanged = false;
	}
	
	/**
	 * 
	 * @return control containing the data to be exported
	 */
    public Object[] getViewerControls()
    {
    	if(getCurrentPage() != null && getCurrentPage() instanceof MultiLevelStatisticPage)
    	{
    		return new Object[] {((MultiLevelStatisticPage)getCurrentPage()).getView().getTree()};
    	}
    	return null;
    }
    
	/**
	 * 
	 * @return viewers associated with this view, if applicable
	 * can be a TreeViewer, TableTreeViewer or null if the view is not using a viewer
	 */
    public Viewer[] getExportViewer()
    {
    	if(getCurrentPage() != null && getCurrentPage() instanceof MultiLevelStatisticPage)
    	{
    		return new Viewer[] {((MultiLevelStatisticPage)getCurrentPage()).getView().getTreeViewer()};
    	}
    	return null;
    }
    
	public String getSelectedContext()
	{
		return _context;
	}
	
	public boolean contextChanged()
	{
		return _contextChanged;
	}
	
	public void setSelectedContext(String context)
	{
		setSelectedContext(context, true);
	}
	
	public void setSelectedContext(String context, boolean refresh)
	{
		if (context == null)
			_contextChanged = _context != null;
		else
			_contextChanged = !context.equals(_context);
		
		_context = context;
		
		if (_contextChanged && refresh && getCurrentPage() != null)
			((TraceViewerPage)getCurrentPage()).refreshPage();
	}
}
