/*******************************************************************************
 * Copyright (c) 2008 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Intel Corporation - Initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.views.internal;

import org.eclipse.hyades.trace.ui.IViewSelectionChangedListener;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.tptp.platform.common.ui.trace.internal.TraceUIManager;

public abstract class AbstractLazyHandler implements IViewSelectionChangedListener {
	private boolean isVisible;
	private boolean lazyDisabled;
	private boolean lazyMode;

	protected static final int LAZY_UPDATE = 1; 
	protected static final int LAZY_UPDATE_BUTTONS = 1 << 1;
	protected static final int LAZY_HANDLE_SELECTION_EVENT = 1 << 2;
	protected static final int LAZY_HANDLE_VIEWER_SELECTION_EVENT = 1 << 3;
	protected static final int LAZY_UPDATE_WITH_REFRESH = LAZY_UPDATE | 1 << 4; 
	protected static final int LAZY_UPDATE_UI = LAZY_UPDATE | 1 << 4; 
	protected static final int LAZY_REFRESH = 1 << 5; 
	
	protected int lazyOperations;
	protected Object lazySelectionEventSource;
	
	public AbstractLazyHandler(boolean lazyMode) {
		this.lazyMode = lazyMode;
	}

	protected final boolean checkSetLazyViewerSelectionOperation(ViewSelectionChangedEvent event){
		Object source = event.getSource();
		if(!checkSetLazyOperation(LAZY_HANDLE_VIEWER_SELECTION_EVENT)){
			return false;
		}
		
		addPostponedSelectionEventSource(source);
		
		return true;
	}
	
	protected void addPostponedSelectionEventSource(Object source){
		if(isHandleSelectionEventNeeded(source))
			lazySelectionEventSource = source;
		else if(lazySelectionEventSource == null || !isHandleSelectionEventNeeded(lazySelectionEventSource))
			lazySelectionEventSource = source;
	}
	
	protected boolean isHandleSelectionEventNeeded(Object source){
		return source != this;
	}
	
	protected Object nextPostponedSelectionEventSource(){
		if(lazySelectionEventSource == null)
			return null;
		
		Object o = lazySelectionEventSource;
		lazySelectionEventSource = null;
		return o;
	}

	
	protected boolean lazyHandlesOpType(int opType){
		return true;
	}
	
	protected final boolean checkSetLazyOperation(int opType){
		if(!lazyMode || !lazyHandlesOpType
				(opType))
			return false;
		
		if(lazyDisabled || isVisible()){
			lazyOperations &= (~opType);
			return false;
		}
		
		lazyOperations |= opType;
		return true;
	}
	
	public boolean isLazyMode(){
		return lazyMode;
	}
	
	public void setLazyMode(boolean lazy){
		if(lazyMode == lazy)
			return;
		
		if(!lazy)
			processLazyPostponedOperations();
		
		lazyMode = lazy;
	}

	protected final void processLazyPostponedOperations(){
		boolean oldLazy = lazyDisabled;
		lazyDisabled = true;
		try {
			doProcessLazyPostponedOperations();
		} finally {
			lazyDisabled = oldLazy;
		}
	}
	
	protected void doProcessLazyPostponedOperations(){
		int processedOps = 0;
		
		if((lazyOperations & LAZY_UPDATE) != 0){
			int updateOp = lazyOperations & (~LAZY_UPDATE_WITH_REFRESH);
			lazyOperations &= (~LAZY_UPDATE_WITH_REFRESH);
			doUpdate((updateOp & LAZY_UPDATE_WITH_REFRESH) ==  LAZY_UPDATE_WITH_REFRESH, true, processedOps);
			processedOps |= updateOp;
		}

		if((lazyOperations & LAZY_UPDATE_UI) != 0){
			lazyOperations &= (~LAZY_UPDATE_UI);
			doUpdateUI(true, processedOps);
			processedOps |= LAZY_UPDATE_UI;
		}

		if((lazyOperations & LAZY_REFRESH) != 0){
			lazyOperations &= (~LAZY_REFRESH);
			doUpdateUI(true, processedOps);
			processedOps |= LAZY_REFRESH;
		}

		if((lazyOperations & LAZY_UPDATE_BUTTONS) != 0){
			lazyOperations &= (~LAZY_UPDATE_BUTTONS);
			doUpdateButtons(true, processedOps);
			processedOps |= LAZY_UPDATE_BUTTONS;
		}
		
		if((lazyOperations & LAZY_HANDLE_VIEWER_SELECTION_EVENT) != 0){
			lazyOperations &= (~LAZY_HANDLE_VIEWER_SELECTION_EVENT);
			
			for(Object src = nextPostponedSelectionEventSource(); src != null; src = nextPostponedSelectionEventSource()){
				ViewSelectionChangedEvent event = TraceUIManager.getTraceUIManager().getViewSelectionChangedEvent();
				event.setSource(src);
				doHandleViewSelectionChangedEvent(event, true, processedOps);
			}
			
			processedOps |= LAZY_HANDLE_VIEWER_SELECTION_EVENT;
		}

		if((lazyOperations & LAZY_HANDLE_SELECTION_EVENT) != 0){
			lazyOperations &= (~LAZY_HANDLE_SELECTION_EVENT);
			doHandleSelectionEvent(true, processedOps);
			processedOps |= LAZY_HANDLE_SELECTION_EVENT;
		}

		lazyOperations = 0;
	}
	

	protected final boolean isVisible(){
		return isVisible;
	}

	/**
	 * this method is called by the framework to notify the view that
	 * it has become visible
	 * 
	 * @see #setHidden()
	 */
	public final void setVisible(){
		if(isVisible)
			return;
	
		isVisible = true;
	
		handleVisible();
	}
	
	/**
	 * this method is called by the framework to notify the view that
	 * it has become hidden
	 * 
	 * @see #setVisible()
	 */
	public final void setHidden(){
		if(!isVisible)
			return;
		
		isVisible = false;
		handleHidden();
	}
	
	protected void handleVisible(){
		processLazyPostponedOperations();
	}
	
	protected void handleHidden(){
		
	}

	protected final boolean checkSetLazyUpdateOperation(boolean refresh){
		if(!checkSetLazyOperation(refresh ? LAZY_UPDATE_WITH_REFRESH : LAZY_UPDATE)){
			return false;
		}
		
		return true;
	}

	public void update() {
		update(true);
	}

	public void update(boolean refresh) {
		if(checkSetLazyUpdateOperation(refresh))
			return;
		
		doUpdate(refresh, false, 0);
	}
	
	protected void updateButtons() {
		if(checkSetLazyOperation(LAZY_UPDATE_BUTTONS))
			return;
		
		doUpdateButtons(false, 0);
	}
	
	protected void handleSelectionEvent() {
		if(checkSetLazyOperation(LAZY_HANDLE_SELECTION_EVENT))
			return;
		
		doHandleSelectionEvent(false, 0);
	}

	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event){
		if(checkSetLazyViewerSelectionOperation(event))
			return;
		
		doHandleViewSelectionChangedEvent(event, false, 0);
	}
	
	public void updateUI(){
		if(checkSetLazyOperation(LAZY_UPDATE_UI))
			return;
		
		doUpdateUI(false, 0);
	}

	public void refresh(){
		if(checkSetLazyOperation(LAZY_REFRESH))
			return;
		
		doRefresh(false, 0);
	}
	protected abstract void doUpdate(boolean refresh, boolean isPostponedOperation, int processedOperations);
	
	protected abstract void doUpdateButtons(boolean isPostponedOperation, int processedOperations);

	protected abstract void doHandleSelectionEvent(boolean isPostponedOperation, int processedOperations);
	
	protected abstract void doHandleViewSelectionChangedEvent(ViewSelectionChangedEvent event, boolean isPostponedOperation, int processedOperations);

	protected abstract void doUpdateUI(boolean isPostponedOperation, int processedOperations);

	protected abstract void doRefresh(boolean isPostponedOperation, int processedOperations);

}
