/**********************************************************************
 * Copyright (c) 2006, 2010 Intel Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AggregatedCallStack.java,v 1.6 2010/06/30 20:45:51 jwest Exp $
 * 
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.trace.ui.ITraceSelection;
import org.eclipse.hyades.trace.ui.TraceViewer;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.hyades.trace.views.adapter.internal.AggregatedExecutionStatisticsTab;
import org.eclipse.hyades.trace.views.adapter.internal.ExecutionStatisticPage2;
import org.eclipse.hyades.trace.views.adapter.internal.ExecutionStatisticViewer2;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.AggregatedCumulativeTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.MethodQualifiedNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.PerThreadPercentColumnLabel;
import org.eclipse.hyades.trace.views.util.internal.AggregatedInvocation;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.hyades.ui.provisional.context.IContextAttributes;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;

public class AggregatedCallStack extends StatisticView {

	public class CallStackInfoProvider {
		protected AggregatedInvocation _aggregatedInv;
		protected int _selItem = 0;

		public void setAggregatedInvocation(AggregatedInvocation aggregatedInv) {
			if (_aggregatedInv != aggregatedInv) {
				_aggregatedInv = aggregatedInv;
				_selItem = 0;
			}
		}
		
		public int getInvocationCount() {
			return (_aggregatedInv != null) ? _aggregatedInv.getAllInvocations().size() : 0;
		}
		
		public boolean selectNext() {
			int count = getInvocationCount();
			if (_selItem < count - 1) {
				_selItem ++;
				return true;
			} else if (count > 1) {
				_selItem = 0;
				return true;
			}
			return false;
		}

		public boolean selectPrevious() {
			if (_selItem > 0) {
				_selItem --;
				return true;
			} else {
				int count = getInvocationCount();
				if (_selItem == 0 && count > 1) {
					_selItem = count - 1;
					return true;
				}
			}
			return false;
		}
		
		public int getSelectedIndex() {
			return _selItem;
		}

		public AggregatedInvocation.Item getSelectedInvocation() {
			if (_aggregatedInv == null) {
				return null;
			}
        	List allInvocations = 
        		_aggregatedInv.getAllInvocations();
        	
            if (_selItem >= 0 && _selItem < allInvocations.size()) { 
            	return (AggregatedInvocation.Item) allInvocations.get(_selItem);
            }
			return null;
		}
		
        public void dispose() {
        	_aggregatedInv = null;
        }

		public Object getAggregatedInvocation() {
			return _aggregatedInv;
		}
    
	}
	
	public class CallStacksContentProvider implements ITreeContentProvider {

		public CallStacksContentProvider() {
		}
		
        public Object[] getElements(Object inputElement) {
        	AggregatedInvocation.Item invocation = _infoProvider.getSelectedInvocation();
            ArrayList stack = new ArrayList();
            
            while (invocation != null) {
                stack.add(invocation);
                
                AggregatedInvocation.Item invokedBy = invocation.getInvokedBy();
                if (invokedBy == null) {
                	break;
                }
                invocation = invokedBy;                
            }
            return stack.toArray();
        }

        public Object[] getChildren(Object element) {
            return new Object[0];            
        }

        public Object getParent(Object element) {
			return null;
		}

		public void inputChanged(Viewer viewer,
                                 Object oldInput, Object newInput) {
        }

        public boolean hasChildren(Object element) {
        	return false;
        }
        public void dispose() {
        }
	}

	public class CallStacksCellLabelProvider extends StatisticCellLabelProvider {
 		public CallStacksCellLabelProvider(ColumnData colData) {
			super(colData);
 		}
		
		public void update(ViewerCell cell) {
			visualIndex = cell.getVisualIndex();
			cell.setText(((CallStacksLabelProvider)getTableLabelProvider()).getColumnText(cell.getElement(),visualIndex));
			cell.setImage(((CallStacksLabelProvider)getTableLabelProvider()).getColumnImage(cell.getElement(),visualIndex));
		}
	}
	
    public class CallStacksLabelProvider extends LabelProvider
        implements ITableLabelProvider {
        
        protected StatisticView viewer;
    
    	public CallStacksLabelProvider(StatisticView viewer) {
            super();
    		this.viewer = viewer;
    	}
    
    	public String getColumnText(Object obj, int col) {
    		if (obj == null) {
    			return "";
    		}
    		StatisticTableColumnInfo info
                = StatisticTableColumnInfo.getStatisticTableColumnInfo(viewer.getTree()
                                                                       .getColumn(col));                
            int pos = info.getColumnData().getInitalPos();
            
    		if (pos == 0) { // method name
        		return getElementColumnText(obj, _methodNameCol, info.isDeltaColumn());
    		} else if (pos == 1) { //percent per thread
				return getElementColumnText(obj, _percentCol, false);
    		} else if (pos == 2) { // cumulative time
    			return getElementColumnText(obj, _cumulativeTimeCol, info.isDeltaColumn());
    		}
    		return "";
    	}
    
    	public Image getColumnImage(Object obj, int col) {
            StatisticTableColumnInfo info
                = StatisticTableColumnInfo.getStatisticTableColumnInfo(viewer.getTree()
                                                                       .getColumn(col));
            int pos = info.getColumnData().getInitalPos();                
            if (obj != null && pos == 0) {
                return getElementColumnImage(obj, _methodNameCol, info.isDeltaColumn());
            }                
    		return null;
    	}
    }

    public class CallStacksSorter extends StatisticSorter {
    

        public void sort(Viewer viewer, Object[] elements) {
            // Sorting is disabled
        }
    	public int compare(Viewer viewer,
                           Object _e1, Object _e2) {            
            return 0;
    	}
        
        public void setSortedColumn(TreeColumn newSortColumn) {
            // Sorting is disabled
        }
        
        public void setSortedColumn(int sortSequence, TreeColumn newSortColumn) {
            // Sorting is disabled
        }
    }
    
    protected CallStackInfoProvider _infoProvider;

    protected ColumnLabelAdapter _methodNameCol;
    protected ColumnLabelAdapter _percentCol;
    protected ColumnLabelAdapter _cumulativeTimeCol;
    protected Button _prevButton;
    protected Button _nextButton;
    protected Label _stackCountInfo;

    public AggregatedCallStack(Composite parent, 
     			               AggregatedExecutionStatisticsTab tabItem,
    	                       boolean initializeContent) {
    	super(true, parent, tabItem.getPage(), initializeContent);
    	
    	_infoProvider = new CallStackInfoProvider();
        createColumnsLabelProviders();
	}

    public AggregatedCallStack(Composite parent,
    							AggregatedExecutionStatisticsTab tabItem) {
    	this(parent, tabItem, true);
	}

    public void createColumnsLabelProviders() {
		_methodNameCol = new MethodQualifiedNameColumnLabel();
		_percentCol = new PerThreadPercentColumnLabel();
		_cumulativeTimeCol = new AggregatedCumulativeTimeColumnLabel();
	}
    
    public Object getSelectedElement() {
        if (_page instanceof ExecutionStatisticPage2) {
            return ((ExecutionStatisticPage2)_page).getAggregatedExecutionStatisticsTab().getView().getCurrentSelection();
        }
        
        ITraceSelection selModel = 
            UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());        
        if(selModel.size() > 0)
        {
            return selModel.getFirstElement();
        }
        return null;
    }
    
    public String getDefaultColumnsTemplate() {
		return IContextAttributes.METHOD_QUALIFIED_NAME + ":0:" 
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:500,"
			+ IContextAttributes.METHOD_PERCENT_PER_THREAD	+ ":1:"				
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:120,"
			+ IContextAttributes.METHOD_CUMULATIVE_TIME + ":2:"
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:120";
	}

    public String getColumnsPreferencesKey() {
		return "AggrCallStack";
	}

    public IContentProvider getContentProvider() {
    	return new CallStacksContentProvider();
    }
   
    public LabelProvider getTableLabelProvider() {
        return new CallStacksLabelProvider(this);
    }

    public Tree getTree() {
        return (Tree) getTreeViewer().getControl();
        
    }

    protected void doHandleViewSelectionChangedEvent(ViewSelectionChangedEvent event, boolean isPostponedOperation, int processedOperations) {
        Object  source = event.getSource();
        if(source!=this){     
            handleSelectionEvent();
        }
    }

    
    protected void doUpdate(boolean refresh, boolean isPostponedOperation, int processedOperations)
    {
        _methodNameCol.resetMap();
        _percentCol.resetMap();
        _cumulativeTimeCol.resetMap();
        super.doUpdate(refresh, isPostponedOperation, processedOperations);
    }

    protected void doUpdateButtons(boolean isPostponedOperation, int processedOperations) {
    }

    public void updateModelSelection() {
        ISelection selection = getTreeViewer().getSelection();
        if(selection != null && !selection.isEmpty())
        {
            Object sel = ((IStructuredSelection)selection).getFirstElement();            
            notifyViewSelectionChanged(this,sel);
        }
    }
    
    protected Composite createControl(Composite parent,
                                      ArrayList cols) {
        _viewContainer = _toolkit.createComposite(parent, SWT.FLAT);
        _viewContainer.setLayout(new GridLayout(1, false));
        _viewContainer.setLayoutData(GridUtil.createFill());
        
        // Create CallStack table
        Composite callStackTable = _toolkit.createComposite(_viewContainer,
                                                            SWT.FLAT);
        callStackTable.setLayout(new GridLayout(1, false));
        callStackTable.setLayoutData(GridUtil.createFill());
                    
        _viewComp = _toolkit.createComposite(callStackTable, SWT.FLAT);
        _viewComp.setLayout(new GridLayout(1, false));
        _viewComp.setLayoutData(GridUtil.createFill());
    
        _dataPane = _toolkit.createViewForm(_viewComp);
        _dataPane.setContent(createTableViewer(_dataPane, cols));
        _dataPane.setLayoutData(GridUtil.createFill());
    
        callStackTable.addControlListener(this);
        
        
        // Create buttons panel
        Composite buttonsPanel = _toolkit.createComposite(_viewContainer);        
        GridLayout gridLayout = new GridLayout(3, false);
        gridLayout.marginWidth  = 0;
        gridLayout.marginHeight = 0;
        buttonsPanel.setLayoutData(GridUtil.createHorizontalFill());
        buttonsPanel.setLayout(gridLayout);
        
        _prevButton = _toolkit.createButton(buttonsPanel, " < " + TraceUIMessages._271 + " ", SWT.FLAT);
        _prevButton.setLayoutData(new GridData(SWT.LEFT, SWT.FILL, false, false));
        _prevButton.addSelectionListener(new SelectionAdapter(){
            public void widgetSelected(SelectionEvent e) {
            	if (_infoProvider.selectPrevious()) {
                    updateColumns();
            	}
            } 
        });
        
        _nextButton = _toolkit.createButton(buttonsPanel, " " + TraceUIMessages._272 + " > ", SWT.FLAT);
        _nextButton.setLayoutData(new GridData(SWT.LEFT, SWT.FILL, false, false));
        _nextButton.addSelectionListener(new SelectionAdapter(){
            public void widgetSelected(SelectionEvent e) {
            	if (_infoProvider.selectNext()) {
                    updateColumns();
            	}
            } 
        });
            
        _stackCountInfo = _toolkit.createLabel(buttonsPanel, "");
        _stackCountInfo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        
        return _viewContainer;
    }
    
    protected Composite createTree(Composite parent, int options) {
    	Tree tree = _toolkit.createTree(parent, options);
    	
     	tree.addListener(SWT.EraseItem, 
     			AggregatedExecutionStatistic.getBarGraphPainter(
     					tree, 1)); // 1 = percent per thread		
    	
    	return tree;
    }

    protected StructuredViewer createTreeViewer(Composite tree) {        
        TreeViewer tv = getTreeViewer((Tree)tree);
        tv.addDoubleClickListener(new IDoubleClickListener() {
            public void doubleClick(DoubleClickEvent event) {
                ISelection selection = event.getSelection();
    
                if (selection != null && !selection.isEmpty()) {
                    Object node = ((IStructuredSelection)selection).getFirstElement();
                   
                   if(node instanceof TRCFullMethodInvocation)
                         openSourceForSelection(selection);
                   }                
            }
    
        });
        return (StructuredViewer) tv;
    }

    protected TreeViewer getTreeViewer(Tree tree)
    {
        return new TreeViewer(tree);
    }

    protected String getContextHelpId() {
        //TODO put correct help ID
    	return TraceUIPlugin.getPluginId() + ".mthid0000";
    }

    protected StatisticSorter getViewerSorterInstance() {
		return new CallStacksSorter();
	}

	protected String getViewTypeStr()
    {
        return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.AGGREGATED_CALL_STACKS_VIEW;
    }

    protected StatisticFilter getFilterInstance() {
		return new StatisticFilter();
	}

	protected void doHandleSelectionEvent(boolean isPostponedOperation, int processedOperations) {
        //Do nothing
    }

    protected void openSourceForSelection(ISelection selection)
    {
        if (selection != null) {
            TraceViewerPage page = getTraceViewerPage();
            if (page != null) {
                final TraceViewer traceViewer = page.getTraceViewer();
                if (traceViewer instanceof ExecutionStatisticViewer2) {
                    if (traceViewer != null) {
                        BusyIndicator.showWhile(Display.getDefault(), new Runnable() {
                            public void run() {
                                ((ExecutionStatisticViewer2)traceViewer).openSource().run();
                            }
                        });
                    }
                }
            }
        }
    }

    protected void updateColumns() {
        updateUI(getColumnDataList());

        String label = "";
        int count = _infoProvider.getInvocationCount();
        if (count > 1) {        	
        	label = NLS.bind(TraceUIMessages._273, new Integer(_infoProvider.getSelectedIndex() + 1), new Integer(count));
        }
    	_stackCountInfo.setText(label);
    }

    private Control createTableViewer(final Composite parent,
                                      final ArrayList cols) {
        Composite tree = createTree(parent, SWT.SINGLE | SWT.FULL_SELECTION | SWT.FLAT);
    
        // Create the TableTree Viewer
        _viewer = createTreeViewer(tree);
    
        for (int idx = 0; idx < cols.size(); idx++) {
            ColumnData data = (ColumnData) cols.get(idx);
            data.setContext(getContext());
            if (data.visible()) {
                new StatisticTableColumnInfo(this, data, _deltaColumns);
            }
        }
    
        Tree tre = getTree();
        tre.setLinesVisible(true);
        tre.setHeaderVisible(true);
        tre.setLayout(getLayout());
        tre.setLayoutData(new GridData(GridData.FILL_BOTH));
    
        // add the content provider
        _viewer.setContentProvider(getContentProvider());
    
        // add the label provider
        _viewer.setLabelProvider(getTableLabelProvider());
    
        if (tree instanceof Tree) {
            ((Tree) tree).addSelectionListener(this.getSelectionListener());
        }
        
        UIPlugin.getDefault().addViewSelectionChangedListener(this);
        
        return _viewer.getControl();
    }

	public void updateCallStackInfoProvider() {
        Object selection = getSelectedElement();
        if (selection instanceof AggregatedInvocation) {
            _infoProvider.setAggregatedInvocation((AggregatedInvocation)selection);
        }

        boolean enabled = _infoProvider.getInvocationCount() > 1;
        _prevButton.setEnabled(enabled);
        _nextButton.setEnabled(enabled);
        updateColumns();
	}
	
	protected void firstTimeUpdate()
	{
		super.firstTimeUpdate();
        redrawTable();
	}	

	public void redrawTable() {
		getTree().setRedraw(false);

		ArrayList list = ColumnData.createColumnData(getColumnsPreferencesKey(), getDefaultColumnsTemplate());
		resetColumns(list);
		_currentColumns = list;
		getTree().setRedraw(true);
		
		refresh();
	}
	
	public void dispose() {
		super.dispose();

	    _infoProvider.dispose();
	    _prevButton.dispose();
	    _nextButton.dispose();
	    _stackCountInfo.dispose();

	    _infoProvider = null;

	    _methodNameCol = null;
	    _percentCol = null;
	    _cumulativeTimeCol = null;
	    
	    _prevButton = null;
	    _nextButton = null;
	    _stackCountInfo = null;
	}

	public CellLabelProvider getCellLabelProvider(ColumnData colData) {
		return new CallStacksCellLabelProvider(colData);
	}
	
}
