/**********************************************************************
 * Copyright (c) 2008, 2009 Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AggregatedExecutionStatistic.java,v 1.10 2009/08/28 20:42:22 ewchan Exp $
 * 
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Pattern;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.trace.ui.ITraceSelection;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.actions.internal.OpenMethodDetailsAction;
import org.eclipse.hyades.trace.views.actions.internal.OpenMethodInvocationAction;
import org.eclipse.hyades.trace.views.adapter.internal.AggregatedExecutionStatisticsTab;
import org.eclipse.hyades.trace.views.adapter.internal.ExecutionStatisticPage2;
import org.eclipse.hyades.trace.views.adapter.internal.ExecutionStatisticViewer2;
import org.eclipse.hyades.trace.views.adapter.internal.IContextViewer;
import org.eclipse.hyades.trace.views.adapter.internal.TraceConstants;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.AggregatedAvgTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.AggregatedCallsColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.AggregatedCumulativeTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.AggregatedMaxTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.AggregatedMinTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnDisplayInfo;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.PerThreadPercentColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ThreadMethodInvocationNameColumnLabel;
import org.eclipse.hyades.trace.views.util.internal.AggregatedInvocation;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.hyades.ui.provisional.context.ContextManager;
import org.eclipse.hyades.ui.provisional.context.IContextAttributes;
import org.eclipse.hyades.ui.provisional.context.IContextLanguage;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.ColumnViewerToolTipSupport;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.graphics.Region;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;

public class AggregatedExecutionStatistic extends MultiLevelStatisticView
{    
	private int _drawMode = TraceUIPlugin.getDefault().getPreferenceStore().getInt(TraceConstants.TIME_OPTION);
	
	protected ContextInfoContainer _contextInfo;
    
	public class ExecutionStatisticFilter extends StatisticFilter {

		public ExecutionStatisticFilter() {
			super();
		}

		public boolean select(final Viewer viewer, final Object parent,
                              final Object element) {
			
			boolean flag = true;
			
			if(!(element instanceof TRCThread))
			  return true;
			  
			String compareText = "";
		   compareText = ((TRCThread)element).getName();
		   if(compareText.equals(""))
			 compareText = TraceUIMessages._87;
			
			if (_noPattern)
				return true;
			
			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {
					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
		
	}
	
	public class ExecutionStatisticSorter extends StatisticSorter {

		public ExecutionStatisticSorter() {
			super();
		}

		public int compare(final Viewer viewer,
                           final Object e1, Object e2) {
			
			if(e1 instanceof TRCThread && e2 instanceof TRCThread)
			{
				if(_pos == 0) // name
					return _sortSequence * compareElements(e1, e2, _threadNameCol, false);
				else if(_pos == 1)//percent per thread: sort threads by cumulative time
					return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, false);
				else if(_pos == 2)//cumulative time
					return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, false);
			}
			if(e1 instanceof AggregatedInvocation && e2 instanceof AggregatedInvocation)
			{
				if(_pos == 0) //name
					return _sortSequence * compareElements(e1, e2, _threadNameCol, false);
				else if(_pos == 1) //percent per thread
					return _sortSequence * compareElements(e1, e2, _percentCol, false);				
				else if(_pos == 2) //cumulative time
					return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, false);
				else if(_pos == 3) //min time
					return _sortSequence * compareElements(e1, e2, _minTimeCol, false);
				else if(_pos == 4) //average time
					return _sortSequence * compareElements(e1, e2, _avgTimeCol, false);
				else if(_pos == 5) //max time
					return _sortSequence * compareElements(e1, e2, _maxTimeCol, false);
				else if(_pos == 6) //calls count
					return _sortSequence * compareElements(e1, e2, _callsCol, false);
			}
			
			return 0;
		}
		
	}

    public class ExecutionStatisticContentProvider implements
            ITreeContentProvider {
        
        public void dispose() {
        }
        
        public Object getParent(final Object element) {
        	if (element instanceof AggregatedInvocation) {
        		AggregatedInvocation parent = ((AggregatedInvocation)element).getParent();
        		if (parent != null) {
        			return parent;
        		}
        		return ((AggregatedInvocation)element).getThread();
        	}
            return null;
        }
        
        public void inputChanged(final Viewer viewer, final Object oldInput,
                final Object newInput) {
        }
        
        public Object[] getElements(final Object inputElement) {
            return PerftraceUtil.getAllThreads(_page.getMOFObject());
        }
        
        public Object[] getChildren(final Object element) {
        	
        	if (element instanceof TRCThread) {
        		return AggregatedInvocation.getThreadCalls((TRCThread) element).toArray();
        	}
        	if (element instanceof AggregatedInvocation) {
        		return ((AggregatedInvocation)element).getNestedCalls().toArray();
        	}
        	return null;
        }
        
        public boolean hasChildren(final Object element) {
            
            if (element instanceof TRCThread) {                
                return ((TRCThread)element).getInitialInvocations().size() > 0;
            }
            if (element instanceof AggregatedInvocation) {
            	return ((AggregatedInvocation)element).getNestedCalls().size() > 0;
            }
            return false;
            
        }
    }
  
    public class ExecutionStatisticCellLabelProvider extends StatisticCellLabelProvider {
 		public ExecutionStatisticCellLabelProvider(ColumnData colData) {
			super(colData);
 		}
		
		public void update(ViewerCell cell) {
			visualIndex = cell.getVisualIndex();
			cell.setText(((ExecutionStatisticLabelProvider)getTableLabelProvider()).getColumnText(cell.getElement(),visualIndex));
			cell.setImage(((ExecutionStatisticLabelProvider)getTableLabelProvider()).getColumnImage(cell.getElement(),visualIndex));
			
		}
	}
    
    
	public class ExecutionStatisticLabelProvider extends LabelProvider
        implements ITableLabelProvider
	{
		protected StatisticView _viewer;
		
		public ExecutionStatisticLabelProvider(final StatisticView viewer) {
			super();
			_viewer = viewer;
		}
		
		public Image getColumnImage(final Object obj, final int col) {
			
			StatisticTableColumnInfo info =
                StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree()
                                                                     .getColumn(col));
			int pos = info.getColumnData().getInitalPos();
			if (pos == 0) {
				return getElementColumnImage(obj, _threadNameCol, false);
			}
	        return null;
		}

		public String getColumnText(final Object obj, final int col) {
			
			StatisticTableColumnInfo info =
                StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree()
                                                                     .getColumn(col));
			int pos = info.getColumnData().getInitalPos();			
			switch(pos)
			{					
				case 0:
					return getElementColumnText(obj, _threadNameCol, false);
				case 1: //percent per thread
					return getElementColumnText(obj, _percentCol, false);
				case 2://cumulative time
					return getElementColumnText(obj, _cumulativeTimeCol, false);
				case 3: //min time
					return getElementColumnText(obj, _minTimeCol, false);						
				case 4: //average time
					return getElementColumnText(obj, _avgTimeCol, false);
				case 5: //max time
					return getElementColumnText(obj, _maxTimeCol, false);
				case 6: //calls count
					return getElementColumnText(obj, _callsCol, false);
			}
			return "";
		}
	}
    	
	class AggregatedExecutionTreeViewer extends MultiLevelStatisticTreeViewer { //TreeViewer {
		public AggregatedExecutionTreeViewer(final Composite parent) {
			super(parent);
		}

		public AggregatedExecutionTreeViewer(final Tree tree) {
			super(tree);
		}

		public void expandItem(final TreeItem item) {
			item.setExpanded(true);
			createChildren(item);

		}

	}
	
	protected static Listener getBarGraphPainter(final Tree tree, final int columnNumber) {
     	return new Listener() {
     		private ColumnLabelAdapter columnLabel = new PerThreadPercentColumnLabel();
     		
     		public void handleEvent(Event event) {
    			event.gc.fillRectangle(event.x, event.y, event.width, event.height);

     			if ((event.detail & SWT.SELECTED) != 0) {
    				GC gc = event.gc;
     				gc.setAdvanced(true);
     				if (gc.getAdvanced()) {
     					int columnCount = tree.getColumnCount();
     					if (event.index == columnCount - 1 || columnCount == 0) {
     						Rectangle area = tree.getClientArea();
     						int width = area.x + area.width - event.x;
     						if (width > 0) {
     							Region region = new Region();
     							gc.getClipping(region);
     							region.add(event.x, event.y, width, event.height); 
     							gc.setClipping(region);
     							region.dispose();
     						}
     					}
     					Color foreground = gc.getForeground();
     					Color background = gc.getBackground();
     					gc.setForeground(event.display.getSystemColor(SWT.COLOR_LIST_SELECTION));
     					gc.fillRectangle(0, event.y, 100000, event.height);
     					gc.setForeground(foreground);
     					gc.setBackground(background);
     					event.detail &= ~SWT.SELECTED;
     				}
     			}
 
 				TreeColumn column = tree.getColumn(event.index);
 				StatisticTableColumnInfo info = 
 					StatisticTableColumnInfo.getStatisticTableColumnInfo(column);
 	            int pos = info.getColumnData().getInitalPos();			
     			
     			if (pos == columnNumber) {
 					TreeItem item = (TreeItem)event.item;
 					Double value = columnLabel.comparableDoubleValue(item.getData(), null);
 					int barWidth = (value != null) ? (int)Math.round(value.doubleValue() * (double)column.getWidth()) : 0;
 
 					GC gc = event.gc;
     				Display display = event.display;
     				Color foreground = gc.getForeground();
     				Color background = gc.getBackground();
     				
     				Color sysBack = display.getSystemColor(SWT.COLOR_RED);
     				Color sysFore = display.getSystemColor(SWT.COLOR_YELLOW);
     				Color fore = new Color(display,
     						(sysBack.getRed() + 3 * sysFore.getRed()) / 4,
     						(sysBack.getGreen() + 3 * sysFore.getGreen()) / 4, 
     						(sysBack.getBlue() + 3 * sysFore.getBlue()) / 4);
     				Color back = new Color(display,
     						(3 * sysBack.getRed() + sysFore.getRed()) / 4, 
     						(3 * sysBack.getGreen() + sysFore.getGreen()) / 4, 
     						(3 * sysBack.getBlue() + sysFore.getBlue()) / 4);
     				
     				
     				gc.setAdvanced(true);
     				if (gc.getAdvanced()) {
     					gc.setAlpha(220);								
     				}
     				gc.setBackground(back);
     				gc.setForeground(fore);
     				gc.fillGradientRectangle(event.x, event.y, barWidth, event.height, false);
 
     				gc.setForeground(foreground);
     				gc.setBackground(background);
     				back.dispose();
     				fore.dispose();
     			}
     		}
     	};
	}
	
    protected Composite createTree(final Composite parent, final int options) {
     	final Tree tree = _toolkit.createTree(parent, options);
     	
     	tree.addListener(SWT.EraseItem, 
     			getBarGraphPainter(tree, 1)); // 1 = percent per thread		
     	
 		return tree;
	}

    protected AggregatedExecutionStatisticsTab _tabItem;
    
	protected ColumnLabelAdapter _threadNameCol;
	protected ColumnLabelAdapter _cumulativeTimeCol;
	protected ColumnLabelAdapter _minTimeCol;
	protected ColumnLabelAdapter _avgTimeCol;
	protected ColumnLabelAdapter _maxTimeCol;
	protected ColumnLabelAdapter _callsCol;
	protected ColumnLabelAdapter _percentCol;
    
    protected Object lastSelectedObject;
	
 	public AggregatedExecutionStatistic(Composite parent, AggregatedExecutionStatisticsTab tabItem) {
 		super(true, parent, tabItem.getPage());
 		
 		_tabItem = tabItem;
		
		_viewerFilter = new ExecutionStatisticFilter();		
		
		createColumnsLabelProviders();
	}
	
    protected static boolean isAccessMethod(final TRCMethodInvocation inv) {
        Pattern p = Pattern.compile("access\\$(\\p{Digit})?");
        return p.matcher(((TRCMethod)inv.getMethod()).getName()).matches();
    }
    
    protected String getContextHelpId()
	{
        //TODO fix help ID
        return TraceUIPlugin.getPluginId()+".exef0008";
	}
	
	public void createColumnsLabelProviders()
	{
		_threadNameCol = new ThreadMethodInvocationNameColumnLabel();
		_percentCol = new PerThreadPercentColumnLabel();
		_cumulativeTimeCol = new AggregatedCumulativeTimeColumnLabel();
		_minTimeCol = new AggregatedMinTimeColumnLabel();
		_maxTimeCol = new AggregatedMaxTimeColumnLabel();
		_avgTimeCol = new AggregatedAvgTimeColumnLabel();
		_callsCol = new AggregatedCallsColumnLabel();
	}
	
	/**
	 * getClassColumnsPerferencesKey returns the string key used to find the
	 * column preferences information in the preferences store.
	 * @return java.lang.String key name
	 */
	public String getColumnsPreferencesKey() {
		// Note this string is not to be translated and must be changed whenever the 
		// default columns template is changed by adding, removing or renaming columns.
		// Changing the version will result in the default preferences being used.
        
		return "AggrExec1_"; 
		
	}
	
	public String getDefaultColumnsTemplate() {
		//Class Columns Data
		
		String executionColumn =
			      IContextAttributes.THREAD_NAME			+ ":0:"
			      + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
				+ IContextAttributes.METHOD_PERCENT_PER_THREAD	+ ":1:"				
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:120,"
				+ IContextAttributes.METHOD_CUMULATIVE_TIME	+ ":2:" 
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:150,"
			    + IContextAttributes.METHOD_MIN_TIME		+ ":3:"				
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
				+ IContextAttributes.METHOD_AVG_TIME		+ ":4:"				
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
				+ IContextAttributes.METHOD_MAX_TIME		+ ":5:"				
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
				+ IContextAttributes.METHOD_CALLS			+ ":6:"				
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:50";
					
		return executionColumn;
	}

	public Tree getTree() {
		return (Tree) getTreeViewer().getControl();
		
	}

	protected TreeViewer getTreeViewer(final Tree tree)
	{
		return new AggregatedExecutionTreeViewer(tree);
	}

    public void expandToSelectedElement(final boolean methodInvOnly) {
        ITraceSelection model =
            UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
        if (model.size() > 0) {
            Object selection = model.getFirstElement();
            if (selection != null) {
                if (methodInvOnly && !(selection instanceof TRCMethodInvocation)) {
                    return;
                }
                select(selection);
                
                if (selection instanceof EObject)
                    updateStatusContext(ContextManager.getContextLanguage(ContextUpdaterHelper.getContext((EObject)selection)));
            }
        }
    }
    
    private void select(final Object selection) {        
        Tree tree = (Tree) getTreeViewer().getControl();
        ArrayList selectedItems = new ArrayList();
        if (selection instanceof TRCMethodInvocation) {
            searchMethodInvocation(tree,
                                   getItemsRoot(tree),
                                   (TRCMethodInvocation)selection,
                                   selectedItems);            
        } else if (selection instanceof TRCMethod) {
            searchMethod(tree,
                         getItemsRoot(tree),
                         (TRCMethod)selection,  
                         selectedItems);            
        }
        
        if (selectedItems.isEmpty()) {
            tree.deselectAll();
            lastSelectedObject = null;
            return;
        }
        
        TreeItem[] treeSelection = new TreeItem[]{(TreeItem)selectedItems.get(0)};
        if (treeSelection[0] != null) {
	        lastSelectedObject = treeSelection[0].getData();
	        tree.setSelection(treeSelection);
        } else {
            lastSelectedObject = null;
            tree.deselectAll();
        }
    }  
    
    private void searchMethod(final Tree tree, final TreeItem item,
                              final TRCMethod method,
                              final ArrayList selectedItems) {
        
        Object[] invocations = method.getInvocations().toArray();
        for (int i = 0; i < invocations.length; i++) {
            searchMethodInvocation(tree, item,
                                  (TRCMethodInvocation)invocations[i],
                                   selectedItems);
            
        }
    }

    protected void searchMethodUsages(TreeItem[] items,
                                      TRCMethod method,
                                      ArrayList selectedItems) {

        TreeItem item;        
        AggregatedExecutionTreeViewer treeViewr =
            ((AggregatedExecutionTreeViewer)getTreeViewer());           
        for (int j = 0; j < items.length; j++) {
            item = items[j];

            if (item != null && !item.isDisposed()){
                Object itemData = getItemModelData(item);
                
                if (itemData instanceof TRCThread) {
                    treeViewr.expandItem(item);
                    searchMethodUsages(item.getItems(), method, selectedItems);
                    break;
                }
                if (itemData instanceof AggregatedInvocation) {                    
                	AggregatedInvocation aggrInv = (AggregatedInvocation)itemData;
                	if (aggrInv.getMethod() == method ) {
                       selectedItems.add(item);
                   }
                   treeViewr.expandItem(item);
                   searchMethodUsages(item.getItems(), method, selectedItems);
                }                
            }
        }
    }

    private void searchMethodInvocation(final Tree tree, final TreeItem item,
                                        final TRCMethodInvocation inv,
                                        final ArrayList selectedItems) {
        
        TreeItem [] threadItems = item != null ? item.getItems() : tree.getItems();        
        
        TreeItem threadItem = searchItem(threadItems, inv.getThread());
        if (threadItem != null){
            TRCMethodInvocation[] callStack = getFilteredCallStack(inv);
            
            AggregatedExecutionTreeViewer treeViewr =
                ((AggregatedExecutionTreeViewer)getTreeViewer());           
            treeViewr.expandItem(threadItem);          
            
            TreeItem aggrInvItem;
            TreeItem area = threadItem;
            
            for (int level = callStack.length -1 ; level >= 0 ; level--) {
                aggrInvItem = searchItem(area.getItems(), callStack[level]);
                if (aggrInvItem == null) {
                    break;
                }
                treeViewr.expandItem(aggrInvItem);
                area = aggrInvItem;
            }
            if (!selectedItems.contains(area)){
                selectedItems.add(area);
            }
            return;
        }
        if (!selectedItems.contains(item)){
            selectedItems.add(item);
        }
    }


    private TRCMethodInvocation[] getFilteredCallStack(final TRCMethodInvocation invocation) {
        ArrayList callStack = new ArrayList();
        TRCMethodInvocation inv = invocation;
        while (inv != null) {
            if (!isAccessMethod(inv)) {
                callStack.add(inv);
            }
            inv =  inv.getInvokedBy();
        }
        TRCMethodInvocation[] stack = new TRCMethodInvocation[callStack.size()];
        for (int i = 0; i < callStack.size(); i++) {
            stack[i] = (TRCMethodInvocation)callStack.get(i);
        }
        return stack;
    }

    protected void doHandleSelectionEvent(boolean isPostponedOperation, int processedOperations) {
    	_tabItem.updateCallStacks();
	}
	
	protected TreeItem getItemsRoot(final Tree tree)
	{
		return null;
	}

	protected TreeItem searchItem(TreeItem[] items, Object obj) {
		TreeItem item;
        
		for (int j = 0; j < items.length; j++) {
			item = items[j];

			if (item != null && !item.isDisposed()){
                
                Object itemData = getItemModelData(item);

                if (!(itemData instanceof AggregatedInvocation)) {
                    continue;
                }
                AggregatedInvocation aggrInv = (AggregatedInvocation)itemData;
                if (obj instanceof TRCMethodInvocation
                    && aggrInv.getMethod().equals(((TRCMethodInvocation)obj).getMethod())) {
                    return item;
                }
            }
		}
		return null;
	}		

	protected Object getItemModelData(final TreeItem item)
	{
		return item.getData();
	}
	

	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(final IMenuManager menu) {
        AggregatedExecutionTreeViewer treeViewer =
            ((AggregatedExecutionTreeViewer)getTreeViewer());           
        ISelection sel = treeViewer.getSelection();
        if(sel != null)
        {
        	//If a method is selected then we want to display related options in the popup menu: feature 256736
        	Object obj = ((IStructuredSelection)sel).getFirstElement();
        	if(obj != null && obj instanceof AggregatedInvocation)
        	{
        		menu.add(fSeparator);

        		OpenMethodInvocationAction act = new OpenMethodInvocationAction(TraceUIPlugin.getString("29"));
        		TracePluginImages.setImageDescriptors(act,TracePluginImages.T_TOOL,TracePluginImages.IMG_METHINVO);        		
        		menu.add(act);

        		OpenMethodDetailsAction act1 = new OpenMethodDetailsAction(TraceUIPlugin.getString("28"));
        		TracePluginImages.setImageDescriptors(act1,TracePluginImages.T_TOOL,TracePluginImages.IMG_METHINVODETAILS);
        		menu.add(act1);        		        		
        	}
        }
		menu.add(fSeparator);
		menu.add(getUpdateAction());		
		menu.add(fSeparator);
 		menu.add(getChooseColumnsAction(getColumnDataList(),
                 getColumnsPreferencesKey()));
 		menu.add(((ExecutionStatisticViewer2)_page.getTraceViewer()).openSource());
	}


	protected void doUpdateButtons(boolean isPostponedOperation, int processedOperations) {
	}

	protected void updateDetailsPane() {
		int selCount = getTree().getSelectionCount();
		if (selCount != 1) {
			return;
		}

		Item item = getTree().getSelection()[0];
		Object itemData = item.getData();

		if (itemData == null) {
			return;
		}
		if (itemData != null && itemData instanceof TreeItem) {
			itemData = ((TreeItem) itemData).getData();
		}
	}

	public void updateModelSelection() {
		
		ISelection selection = getTreeViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			
			notifyViewSelectionChanged(this,sel);
		}
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new ExecutionStatisticSorter();
	}
	
	protected void firstTimeUpdate()
	{
		super.firstTimeUpdate();
		_viewerSorter.setSortedColumn(-1, getTree().getColumn(updateTableGetColumnNumber()));
        redrawTable();
	}	
	
	protected int updateTableGetColumnNumber() {
		return 1; // percent per thread;
	}	
	
	public void dispose()
	{
		super.dispose();
	}

	public IContentProvider getContentProvider()
	{
		return new ExecutionStatisticContentProvider();
	}

	public CellLabelProvider getCellLabelProvider(ColumnData colData) {
		return new ExecutionStatisticCellLabelProvider(colData);
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new ExecutionStatisticLabelProvider(this);
	}
	
    public void notifyViewSelectionChanged(Object source, Object selection) {

        if (selection instanceof AggregatedInvocation) {
            List invocations = ((AggregatedInvocation)selection).getAllInvocations();
            for (Iterator iter = invocations.iterator(); iter.hasNext();) {
            	AggregatedInvocation.Item item = (AggregatedInvocation.Item)iter.next();
                UIPlugin.getDefault().getSelectionModel(_page.getMOFObject()).add(item.getInvocation());
            }
        } else {
            UIPlugin.getDefault().getSelectionModel(_page.getMOFObject()).add(selection);
        }
        lastSelectedObject  = selection;
            
        ViewSelectionChangedEvent event = UIPlugin.getDefault().getViewSelectionChangedEvent();
        event.setSource(source);
        UIPlugin.getDefault().notifyViewSelectionChangedListener(event);
    }
    
	protected void doHandleViewSelectionChangedEvent(final ViewSelectionChangedEvent event, boolean isPostponedOperation, int processedOperations)
	{
		Object  source = event.getSource();

		if(source!=this){	  
			handleSelectionEvent();
		}
		else
		{
			ITraceSelection model =
				UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
			if (model.size() > 0) {
				Object sel = model.getFirstElement();
				if (sel != null && sel instanceof EObject)
						updateStatusContext(ContextManager.getContextLanguage(ContextUpdaterHelper.getContext((EObject)sel)));
			}
		}
	}
	
	public boolean isEmpty()
	{
		if (PerftraceUtil.getAllThreads(_page.getMOFObject()).length > 0)
			return false;
		return true;
	}
	
    protected String getViewTypeStr()
    {
    	return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.AGGREGATED_EXECUTION_STATS_VIEW;
    }			
	/**
	 * @return Returns the _drawMode.
	 */
	public int getDrawMode() {
		return _drawMode;
	}
	/**
	 * @param mode The _drawMode to set.
	 */
	public void setDrawMode(final int mode) {
		_drawMode = mode;
	}
	
	protected void showPercentUpdate()
	{
		if(isShowPercent())
		{
			// Updated to use totalCumulativeTime here and in ColumnDisplayInfo
			// _maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
			// if (_maxTime==0)_maxTime=1;		
			_totalCumulativeTime = PerftraceUtil.getTotalCumulativeTime(_page.getMOFObject());
			if(_totalCumulativeTime==0)_totalCumulativeTime=1;
		}		
	}
	
	protected ColumnDisplayInfo getColumnDisplayInfo(final ColumnLabelAdapter col,
                                                     final boolean isDeltaColumn)
	{
		if (col == _cumulativeTimeCol)
			return ContextUpdaterHelper.updateCumulTime(col, isDeltaColumn, isShowPercent(), getDrawMode(), _totalCumulativeTime);
		else if (col == _callsCol)
			return ContextUpdaterHelper.updateCalls(col, isDeltaColumn, isShowPercent(), true, _totalCalls);	
		else 
			return super.getColumnDisplayInfo(col, isDeltaColumn);
	}	
	
	protected Composite createControl(final Composite parent,
                                      final ArrayList cols) {
		final Composite vc = super.createControl(parent, cols);
		
		_contextInfo = new ContextInfoContainer();
		_contextInfo.setViewer((IContextViewer)_page.getTraceViewer());
		_contextInfo.createControl(vc);
		_contextInfo.addContextInfoContainerListener(
				new IContextInfoContainerListener()
				{
					public void visibilityChanged(boolean isVisible)
					{
						vc.layout(true, true);
					}
				});

		return vc;
	}
	
	protected void updateStatusContext(final IContextLanguage language)
	{
		if (_contextInfo != null)
			_contextInfo.updateStatusContext(language);
	}
	
	public void doUpdate(boolean refresh, boolean isPostponedOperation, int processedOperations) {
		resetColumns();
		
        if (_page instanceof ExecutionStatisticPage2){   
             if (_contextInfo != null) {
                 _contextInfo.setMOFObject(_page.getMOFObject());
             }
        }
		super.doUpdate(refresh, isPostponedOperation, processedOperations);
	}
	
	protected void resetColumns() {
		_threadNameCol.resetMap();
		_cumulativeTimeCol.resetMap();
		_callsCol.resetMap();
		_avgTimeCol.resetMap();
		_minTimeCol.resetMap();
		_maxTimeCol.resetMap();
		_percentCol.resetMap();
	}
    
	protected StructuredViewer createTreeViewer(final Composite tree) {
        
        TreeViewer tv = getTreeViewer((Tree) tree);
        tv.addDoubleClickListener(new IDoubleClickListener() {
            public void doubleClick(DoubleClickEvent event) {
                ISelection selection = event.getSelection();

                if (selection != null && !selection.isEmpty()) {
                    TreeViewer tv = (TreeViewer)getTreeViewer();
                    Object node = ((IStructuredSelection)selection).getFirstElement();
                    
                    if(tv.isExpandable(node))
                    {
                        tv.setExpandedState(node, !tv.getExpandedState(node));
                    }
                }
            }
        });
        
        ColumnViewerToolTipSupport.enableFor(tv);
        
        return (StructuredViewer) tv;
    }

    public Object getLastSelectedObject() {
        return lastSelectedObject;
    }
    
 	protected String getDefaultColumnsTemplateClassLevel() {
 		return null;
 	}
 
 	protected String getDefaultColumnsTemplatePackageLevel() {
 		return null;
 	}
 
 	protected StatisticFilter getFilterInstance() {
 		return _viewerFilter;
 	}
 
 	public int getLevel() {
 		return 0;
 	}
 
 	public boolean isEmptyUpdate() {
 		return false;
 	}
 
 	public void setLevel(int i) {
 		// ignore
 	}
}
