/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CoverageStatisticView.java,v 1.28 2008/03/10 16:37:19 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.views.adapter.internal.CoverageStatisticPage;
import org.eclipse.hyades.trace.views.adapter.internal.MultiLevelStatisticViewer;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.CallsColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ClassNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnDisplayInfo;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.MethodNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.MethodsHitColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.MethodsMissedColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.MethodsPercentHitColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.PackageNameColumnLabel;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.ColumnExtensionValue;
import org.eclipse.hyades.trace.views.util.internal.Coverage;
import org.eclipse.hyades.trace.views.util.internal.CoverageAnalysis;
import org.eclipse.hyades.trace.views.util.internal.CoverageFolder;
import org.eclipse.hyades.trace.views.util.internal.CoverageUtil;
import org.eclipse.hyades.trace.views.util.internal.IColumnExtension;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.hyades.ui.provisional.context.IContextAttributes;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

/*
 * This view contains the coverage statistics data.  This data concerns itself
 * with which methods were called or not.  It has the package, class, and
 * method levels.
 * 
 * It was modified from earlier to extend a MultiLevelStatisticMethodView base
 * class that other views with similar properties extend.
 * The functionality is more or less the same as it was before.  One change is that
 * each level is now stored in a preference string.  So reloading levels preserves
 * the previous settings.
 * 
 * Concerning the Calls functionality.  In checking to see if the calls column
 * is meaningfull.  The current solution is that we check to see if calls column
 * is meaningfull, and if it isn't we fill the Calls cells blank.  So they don't
 * contain non-meaningfull data.
 */
public class CoverageStatisticView extends MultiLevelStatisticMethodView{
	 
	/**
	 * Use this TRCOption key with value "true" to say that calls column is useless.
	 * It won't appear if this flag is found in at least one of the TRCConfiguration of the runs belonging
	 * to the selected node in the profiling monitor. 
	 */
	public static final String CALLS_NOT_COMPUTED = "COVERAGE_STATISTICS.CALLS_NOT_COMPUTED";
	
	private static final int _numberOfColumns[] = { 7, 8, 9 };
	
	private static int _level = DEFAULT_LEVEL;		
	
	private boolean _isCallsMeaningfull;
	
	protected ColumnLabelAdapter _packageNameCol;
	protected ColumnLabelAdapter _classNameCol;	
	protected ColumnLabelAdapter _methodNameCol;
	
	protected ColumnLabelAdapter _callsCol;
	protected ColumnLabelAdapter _methodsMissedCol;
	protected ColumnLabelAdapter _methodsHitCol;
	protected ColumnLabelAdapter _methodsPercentHitCol;	
	
	
	public CoverageStatisticView(Composite parent, TraceViewerPage page){
		super(parent, page);
		
		createColumnsLabelProviders();
		_isCallsMeaningfull = true;
	}
	
	protected String getContextHelpId()
	{
		return TraceUIPlugin.getPluginId()+".ccsv0000";
	}
	
	public void createColumnsLabelProviders()
	{
		_packageNameCol = new PackageNameColumnLabel();
		_classNameCol = new ClassNameColumnLabel();
		_methodNameCol = new MethodNameColumnLabel();		
		
		_callsCol = new CallsColumnLabel();
		_methodsMissedCol = new MethodsMissedColumnLabel();
		_methodsHitCol = new MethodsHitColumnLabel();
		_methodsPercentHitCol = new MethodsPercentHitColumnLabel();	
	}
	
	public class CoverageStatisticContentProvider extends MultiLevelStatisticContentProvider
	{
		public Object getParent(Object element) {
			return null;
		}

		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {
			_isCallsMeaningfull = isCallsMeaningful();
		}

		/**
		 * return the children of the coverage element
		 */
		public Object[] getChildren(Object element) {
			ArrayList childs = ((Coverage)element).getChilds();
			if (childs == null) {
				return null;
			}
			return childs.toArray();
		}

		/**
		 * test if the coverage element has children
		 */
		public boolean hasChildren(Object element) {
			return (getChildren(element) != null);
		}
		
		/**
		 * return the list of elements to display in the table tree
		 */
		public Object[] getElements(Object inputElement) {

			ArrayList tmpList = new ArrayList();
			tmpList.clear();

			CoverageAnalysis covAnalysis =
				((CoverageStatisticPage) _page).getCovAnalysis();

			if (covAnalysis.getRoot() != null) {
				switch (getLevel()) {
					case 1 : // Package level
						tmpList.add(covAnalysis.getRoot());
						return tmpList.toArray();

					case 2 : // Class level
						tmpList.add(covAnalysis.getClassLevel());
						return tmpList.toArray();

					case 3 : // Method level
						tmpList.add(covAnalysis.getMethodLevel());
						return tmpList.toArray();

					default :
						return tmpList.toArray();
				}

			}
			/* When we "relaunch" a process the active page have no content until we make
				a "refreh views" */
			else { //No elements to display
				return tmpList.toArray();
			}
		}
	}
	
	public class CoverageStatisticCellLabelProvider  extends StatisticCellLabelProvider {
 		public CoverageStatisticCellLabelProvider(ColumnData colData) {
			super(colData);
 		}
		
		public void update(ViewerCell cell) {
			visualIndex = cell.getVisualIndex();
			cell.setText(((CoverageStatisticLabelProvider)getTableLabelProvider()).getColumnText(cell.getElement(),visualIndex));
			cell.setImage(((CoverageStatisticLabelProvider)getTableLabelProvider()).getColumnImage(cell.getElement(),visualIndex));
		}
 	}
	
	
	public class CoverageStatisticLabelProvider extends MultiLevelStatisticLabelProvider
	{
		public CoverageStatisticLabelProvider(StatisticView viewer) {
			super(viewer);
		}
		
		/**
		 *  return for the coverage element to display the associated image 
		 */
		public Image getColumnImage(Object obj, int col) {

			StatisticTableColumnInfo info =
				StatisticTableColumnInfo.getStatisticTableColumnInfo(
					_viewer.getTree().getColumn(col));
			int pos = info.getColumnData().getInitalPos();
			Coverage cov = (Coverage) obj;

			switch (pos) {
				case 0 :
					if (cov.getTraceObj() instanceof TRCPackage)
						return getElementColumnImage(cov, _packageNameCol, false);					
					else if (cov.getTraceObj() instanceof TRCClass)
						return getElementColumnImage(cov, _classNameCol, false);
					else if (cov.getTraceObj() instanceof TRCMethod)
						return getElementColumnImage(cov, _methodNameCol, false);
				case 1 : //Class names column
					if (cov.getTraceObj() instanceof TRCMethod)
						return getElementColumnImage(cov, _classNameCol, false);
					else
						return null;
				case 2 : // Package column
					if (cov.getTraceObj() instanceof TRCMethod)
						return getElementColumnImage(cov, _packageNameCol, false);
					else if (cov.getTraceObj() instanceof TRCClass)
						return getElementColumnImage(cov, _packageNameCol, false);
					else
						return null;
			}
			return null;
		}
		
		
		public String getColumnText(Object obj, int col) {

			StatisticTableColumnInfo info =
				StatisticTableColumnInfo.getStatisticTableColumnInfo(
					_viewer.getTree().getColumn(col));
			int pos = info.getColumnData().getInitalPos();
			Coverage cov = (Coverage) obj;

			switch (pos) {
				case 0 : // packages / classes and methods names
					if (cov.getTraceObj() instanceof TRCPackage) {
						return getElementColumnText(cov, _packageNameCol, false);
					} else if (cov.getTraceObj() instanceof TRCClass) {
						return getElementColumnText(cov, _classNameCol, false);
					}
					if (cov.getTraceObj() instanceof TRCMethod) {
						return getElementColumnText(cov, _methodNameCol, false);
					} else {
						return TraceUIMessages._128;
					}

				case 1 : // class names
					if (cov.getTraceObj() instanceof TRCMethod)
						return getElementColumnText(cov, _classNameCol, false);
					else
						return "";

				case 2 : // package
					if (cov.getTraceObj() instanceof TRCMethod)
						return getElementColumnText(cov, _packageNameCol, false);
					else if (cov.getTraceObj() instanceof TRCClass)
						return getElementColumnText(cov, _packageNameCol, false);
					else
						return "";
				case 3 : //Calls
					return getElementColumnText(cov, _callsCol, false);					
				case 4 : //Method missed
					return getElementColumnText(cov, _methodsMissedCol, false);
				case 5 : //method hits
					return getElementColumnText(cov, _methodsHitCol, false);
				case 6 : //%Method hit
					return getElementColumnText(cov, _methodsPercentHitCol, false);
			}
			//if column extended
			if (_listOfColumExtension.size() != 0) {
				ColumnExtensionValue elt =
					(ColumnExtensionValue) _listOfColumExtension.get(
						pos - _numberOfColumns[_level-DEFAULT_LEVEL]);
				return (
					(IColumnExtension)
						(elt.getClassOfColumnExtension())).getColumnText(
					obj,
					elt.getInitialColumnPos());
			}

			return "";
		}
	}

	public class CoverageStatisticSorter extends MultiLevelStatisticSorter{
		public int compare(Viewer viewer, Object e1, Object e2) {

			Coverage cov1 = (Coverage) e1;
			Coverage cov2 = (Coverage) e2;

			switch (_pos) {
				case 0 : //item names
					if (cov1.getTraceObj() instanceof TRCPackage) {
						return _sortSequence * compareElements(e1, e2, _packageNameCol, false);
					} else if (cov1.getTraceObj() instanceof TRCClass) {
						return _sortSequence * compareElements(e1, e2, _classNameCol, false);
					}
					if (cov1.getTraceObj() instanceof TRCMethod) {
						return _sortSequence * compareElements(e1, e2, _methodNameCol, false);
					} else {
						return 0;
					}

				case 1 : //Class names

					if ((cov1.getTraceObj() instanceof TRCMethod)
						&& (cov2.getTraceObj() instanceof TRCMethod)) {
						return _sortSequence * compareElements(e1, e2, _classNameCol, false);
					} else
						return 0;
				case 2 : //Package
					if (cov1.getTraceObj() instanceof TRCMethod
						&& cov2.getTraceObj() instanceof TRCMethod) {
						return _sortSequence * compareElements(e1, e2, _packageNameCol, false);
					} else if (cov1.getTraceObj() instanceof TRCClass
							&& cov2.getTraceObj() instanceof TRCClass) {
						return _sortSequence * compareElements(e1, e2, _packageNameCol, false);
					} else
						return 0;
				case 3 : //Calls
					return _sortSequence * compareElements(e1, e2, _callsCol, false);
				case 4 : //Methods missed
					return _sortSequence * compareElements(e1, e2, _methodsMissedCol, false);
				case 5 : //Methods hit
					return _sortSequence * compareElements(e1, e2, _methodsHitCol, false);
				case 6 : //% methods hit
					return _sortSequence * compareElements(e1, e2, _methodsPercentHitCol, false);
			}
			//	 if column extended
			if (_listOfColumExtension.size() != 0) {
				ColumnExtensionValue elt =
					(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns[_level-DEFAULT_LEVEL]);
				return (int) _sortSequence * ((IColumnExtension)
							(elt.getClassOfColumnExtension())).compare(
								elt.getInitialColumnPos(), e1, e2);
			}

			return 0;
		}
	}

	public class CoverageStatisticFilter extends StatisticFilter {

		public CoverageStatisticFilter() {
			super();
		}

		/**
		 * filter the elements according to the pattern
		 */
		public boolean select(Viewer viewer, Object parent, Object element) {

			Coverage cov = (Coverage) element;
			boolean flag = true;
			String compareText = "";

			switch (getLevel()) {
				case PACKAGE_LEVEL : //Package level
					if (!(cov.getTraceObj() instanceof TRCPackage))
						return true;
					if (((TRCPackage) cov.getTraceObj()).getName().length()==0)
						compareText =
							TraceUIMessages._87;
					else
						compareText =
							((TRCPackage) cov.getTraceObj()).getName();
					break;
				case CLASS_LEVEL : //class level
					if (!(cov.getTraceObj() instanceof TRCClass))
						return true;
					compareText = ((TRCClass) cov.getTraceObj()).getName();
					break;

				case METHOD_LEVEL : //method level
					if (!(cov.getTraceObj() instanceof TRCMethod))
						return true;
					compareText = ((TRCMethod) cov.getTraceObj()).getName();
					break;
			}

			if (_noPattern)
				return true;

			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {

					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
	}
	
	public boolean isEmptyUpdate()
	{
		return !CoverageUtil.isCoverageSupport(_page.getMOFObject());
		
//		CoverageAnalysis covAnalysis =
//			((CoverageStatisticPage) _page).getCovAnalysis();
//
//		return (covAnalysis.getRoot() == null ||
//				covAnalysis.getRoot().getMethods() == 0);	
		
		
	}
	
	protected String getDefaultColumnsTemplatePackageLevel()
	{
		return TraceUIMessages._131
		+ ":0:"
		+ String.valueOf(
			ColumnData.NONDELETABLE
				| ColumnData.IS_VISIBLE
				| ColumnData.NONMOVABLE)
		+ ":221,"
		+ IContextAttributes.CLASS_NAME
		+ ":1:0"
		+ ":100,"
		+ IContextAttributes.PACKAGE_NAME
		+ ":2:0"
		+ ":100,"
		+ IContextAttributes.PACKAGE_CALLS
		+ ":3:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:120,"
		+ IContextAttributes.COVERAGE_METHOD_MISSED
		+ ":4:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:120,"
		+ IContextAttributes.COVERAGE_METHOD_HIT
		+ ":5:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:120,"
		+ IContextAttributes.COVERAGE_METHOD_PERC_HIT
		+ ":6:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:120";
	}

	protected String getDefaultColumnsTemplateClassLevel()
	{
		return TraceUIMessages._131
		+ ":0:"
		+ String.valueOf(
			ColumnData.NONDELETABLE
				| ColumnData.IS_VISIBLE
				| ColumnData.NONMOVABLE)
		+ ":200,"
		+ IContextAttributes.CLASS_NAME
		+ ":1:0"
		+ ":100,"
		+ IContextAttributes.PACKAGE_NAME
		+ ":2:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":150,"
		+ IContextAttributes.CLASS_CALLS
		+ ":3:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:60,"		
		+ IContextAttributes.COVERAGE_METHOD_MISSED
		+ ":4:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:100,"
		+ IContextAttributes.COVERAGE_METHOD_HIT
		+ ":5:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:100,"
		+ IContextAttributes.COVERAGE_METHOD_PERC_HIT
		+ ":6:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:100";
	}
	
	protected String getDefaultColumnsTemplateMethodLevel()
	{
		return TraceUIMessages._131
		+ ":0:"
		+ String.valueOf(
			ColumnData.NONDELETABLE
				| ColumnData.IS_VISIBLE
				| ColumnData.NONMOVABLE)
		+ ":200,"
		+ IContextAttributes.CLASS_NAME
		+ ":1:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":100,"
		+ IContextAttributes.PACKAGE_NAME
		+ ":2:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":100,"
		+ IContextAttributes.METHOD_CALLS
		+ ":3:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:45,"		
		+ IContextAttributes.COVERAGE_METHOD_MISSED
		+ ":4:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:90,"
		+ IContextAttributes.COVERAGE_METHOD_HIT
		+ ":5:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:90,"
		+ IContextAttributes.COVERAGE_METHOD_PERC_HIT
		+ ":6:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:90";
	}
	
	public String getDefaultColumnsTemplate()
	{	
		String columns = super.getDefaultColumnsTemplate();
		
		if (columns == null)
			columns = "";
		
		return columns + getColumnsTemplateExtensions(_numberOfColumns[_level-DEFAULT_LEVEL],"org.eclipse.hyades.trace.views.adapter.internal.CoverageStatisticViewer",true);
	}	

	
	public String getColumnsPreferencesKey()
	{
		setLevel(getLevel());
		
		if (getLevel() == CoverageStatisticView.PACKAGE_LEVEL)
			return PREFERENCE_KEY_PREFIX + "CovStatsPack60";
		else if (getLevel() == CoverageStatisticView.CLASS_LEVEL)
			return PREFERENCE_KEY_PREFIX + "CovStatsClass60";
		else if (getLevel() == CoverageStatisticView.METHOD_LEVEL)
			return PREFERENCE_KEY_PREFIX + "CovStatsMethod60";
		
		return "";
	}
	
	private boolean isCallsMeaningful() {
		CoverageAnalysis covAnalysis = ((CoverageStatisticPage) _page).getCovAnalysis();
		CoverageFolder root = covAnalysis.getRoot();
		if (root == null) {
			return true;
		}
		ArrayList traceObjects = root.getTraceObjects();
		for (Iterator i = traceObjects.iterator(); i.hasNext(); ) {
			Object o = i.next();
			if (o instanceof TRCAgentProxy) {
				TRCAgentProxy agentProxy = (TRCAgentProxy)o;
				if (!isCallsMeaningful(agentProxy)) {
					return false;
				}
			} else if (o instanceof TRCNode) {
				TRCNode node = (TRCNode)o;
				if (!isCallsMeaningful(node)) {
					return false;
				}
			} else if (o instanceof TRCMonitor) {
				TRCMonitor monitor = (TRCMonitor)o;
				if (!isCallsMeaningful(monitor)) {
					return false;
				}
			}
		}
		return true;
	}
	
	private boolean isCallsMeaningful(TRCMonitor monitor) {
		EList nodes = monitor.getNodes();
		for (Iterator j = nodes.iterator(); j.hasNext(); ) {
			TRCNode node = (TRCNode)j.next();
			if (!isCallsMeaningful(node)) {
				return false;
			}
		}
		return true;
	}

	private boolean isCallsMeaningful(TRCNode node) {
		EList processProxies = node.getProcessProxies();
		for (Iterator j = processProxies.iterator(); j.hasNext(); ) {
			TRCProcessProxy processProxy = (TRCProcessProxy)j.next();
			EList agentProxies = processProxy.getAgentProxies();
			for (Iterator k = agentProxies.iterator(); k.hasNext(); ) {
				TRCAgentProxy agentProxy = (TRCAgentProxy)k.next();
				if (!isCallsMeaningful(agentProxy)) {
					return false;
				}
			}
		}
		return true;
	}

	private boolean isCallsMeaningful(TRCAgentProxy agentProxy) {
		EList configs = agentProxy.getConfigurations();
		for (Iterator j = configs.iterator(); j.hasNext(); ) {
			TRCConfiguration config = (TRCConfiguration)j.next();
			EList options = config.getOptions();
			for (Iterator k = options.iterator(); k.hasNext(); ) {
				TRCOption option = (TRCOption)k.next();
				if (CALLS_NOT_COMPUTED.equals(option.getKey()) &&
					option.getValue().equalsIgnoreCase("true")) {
					return false;
				}
			}
		}
		return true;
	}

	public IContentProvider getContentProvider()
	{
		return new CoverageStatisticContentProvider();
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new CoverageStatisticLabelProvider(this);
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new CoverageStatisticSorter();
	}
	
	protected StatisticFilter getFilterInstance()
	{
		return new CoverageStatisticFilter();
	}	
	
	protected String getViewTypeStr()
	{
		return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.COVERAGE_STATS_VIEW;
	}	
	
	/**
	 * Update the "model selection" when an element is selected in the table
	 */
	public void updateModelSelection() {
	
		ISelection selection = getTreeViewer().getSelection();
		if (selection != null && !selection.isEmpty()) {
			Object sel = ((IStructuredSelection) selection).getFirstElement();
			EObject obj = ((Coverage)sel).getTraceObj();
			
			notifyViewSelectionChanged(this,obj);
		}
	}
	
	/**
	 * expand the fisrt element of the table which is "Summary"
	 */
	protected void expandFirstElement() {
		Tree tree = ((TreeViewer) getTreeViewer()).getTree();
		TreeItem[] items = tree.getItems();

		if (items.length != 0)
			 ((MultiLevelStatisticTreeViewer) getTreeViewer()).expandItem(items[0]);
	}	
	
	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {
		super.menuAboutToShow(menu);
		menu.add(fSeparator);		
		menu.add(((MultiLevelStatisticViewer)getTraceViewerPage().getTraceViewer()).openSource());
	}	
	
	/**
	 * @param i:the level of information to display (package/class or method)
	 */
	public void setLevel(int i) {
		_level = i;
	}	
	
	public int getLevel()
	{
		return _level;
	}	
	
	protected Object getItemModelData(TreeItem item)
	{
		if (item.getData()!= null)
			return ((Coverage)item.getData()).getTraceObj();
		else
			return null;
	}	

	protected TreeItem getItemsRoot(Tree tree)
	{
		if (tree.getItemCount() > 0)
			return tree.getItems()[0];
		else
			return null;
	}
	
	protected void showPercentUpdate()
	{
		super.showPercentUpdate();
	}
	
	protected ColumnDisplayInfo getColumnDisplayInfo(ColumnLabelAdapter col, boolean isDeltaColumn)
	{
		if (col == _packageNameCol)
			return ContextUpdaterHelper.updatePackageName(col, _page.getMOFObject());
		else if (col == _callsCol)
			return ContextUpdaterHelper.updateCalls(col, false, false, _isCallsMeaningfull, _totalCalls);
		else
			return super.getColumnDisplayInfo(col, false);
	}
	
//	public IContextLabelFormatProvider getContextFormatter(Object object, String attributeId)
//	{
//		return super.getContextFormatter(((Coverage)object).getTraceObj(), attributeId);
//	}
	
	public void update()
	{
		_packageNameCol.resetMap();
		_classNameCol.resetMap();
		_methodNameCol.resetMap();		
		
		_callsCol.resetMap();
		_methodsMissedCol.resetMap();
		_methodsHitCol.resetMap();
		_methodsPercentHitCol.resetMap();		
		
		super.update();
	}

	public CellLabelProvider getCellLabelProvider(ColumnData colData) {
		return new CoverageStatisticCellLabelProvider(colData);
	}
}
