/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionDetails.java,v 1.6 2008/06/27 17:59:25 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import org.eclipse.hyades.trace.ui.IViewSelectionChangedListener;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.hyades.trace.views.adapter.internal.PatternPage;
import org.eclipse.hyades.trace.views.util.internal.OpenSource;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ColumnViewerToolTipSupport;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Sash;

public class ExecutionDetails 
{
	protected PatternPage _page;	
	protected Sash hSash;
	
	protected Composite _viewContainer;
	protected Pattern _executionGraph;
	protected ExecutionStatistic _executionStatistic;
	
	protected Separator fSeparator;

	/* Constants */
	static final int SASH_WIDTH = 3;

	protected String tmpString;

    public ExecutionDetails(Composite parent, PatternPage page) {
    	
    	fSeparator = new Separator();
    	_page = page;		
    	_viewContainer = new Composite(parent, SWT.NULL);
    	
    	_executionGraph = new Pattern (_viewContainer, _page, true);
    	
    	_executionStatistic = new ExecutionStatistic(_viewContainer, _page);
    	
    	TreeViewer tv = (TreeViewer)(_executionStatistic.getTreeViewer());
    	ColumnViewerToolTipSupport.enableFor(tv);
    	
    	

    	
    	UIPlugin.getDefault().addViewSelectionChangedListener(new IViewSelectionChangedListener() {
    		public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event){
				selectionChanged();
    		}
    	});
	
		/* Create the sashes */
		hSash = new Sash (_viewContainer, SWT.HORIZONTAL);
		
		/* Add the listeners */
		hSash.addSelectionListener (new SelectionAdapter () {
			public void widgetSelected (SelectionEvent event) {
				if (event.detail != SWT.DRAG) {
					hSash.setBounds (event.x, event.y, event.width, event.height);
					layout ();
				}
			}
		});
		_viewContainer.addControlListener (new ControlAdapter () {
			public void controlResized (ControlEvent event) {
				shellResized ();
			}
		});
	}

	/**
	 * Layout the list and text widgets according to the new
	 * positions of the sashes..events.SelectionEvent
	 */
	void layout () {
		
		Rectangle tabCompositeBounds = _viewContainer.getClientArea ();
		Rectangle hSashBounds = hSash.getBounds ();
		
		Control comp = _executionGraph.getControl();
		comp.setBounds (0, 0, tabCompositeBounds.width, hSashBounds.y);		
		_executionStatistic.getControl().setBounds (0, hSashBounds.y + hSashBounds.height, tabCompositeBounds.width, tabCompositeBounds.height - (hSashBounds.y + hSashBounds.height));	
	}
    
	/**
	 * Handle the shell resized event.
	 */
	void shellResized () {
	
		/* Get the client area for the shell */
		Rectangle tabFolderPageBounds = _viewContainer.getClientArea ();
		
		/*
		* Make list 1 half the width and half the height of the tab leaving room for the sash.
		* Place list 1 in the top left quadrant of the tab.
		*/
		Control control = _executionGraph.getControl();
		Rectangle list1Bounds = new Rectangle (0, 0, tabFolderPageBounds.width, 3*(tabFolderPageBounds.height - SASH_WIDTH) / 4);
		control.setBounds (list1Bounds);
		
		/*
		* Make the text area the full width and half the height of the tab leaving room for the sash.
		* Place the text area in the bottom half of the tab.
		*/
		_executionStatistic.getControl().setBounds (0, list1Bounds.height + SASH_WIDTH, tabFolderPageBounds.width, tabFolderPageBounds.height - (list1Bounds.height + SASH_WIDTH));
	
		/* Position the sashes */
		hSash.setBounds (0, list1Bounds.height, tabFolderPageBounds.width, SASH_WIDTH);
	}
	
	public Control getControl() {
		return _viewContainer;
	}
	
	/*
	 * This method handles a selection event
	 */
	public void refresh() {
		_executionGraph.refresh();
		_executionStatistic.refresh();

	}
	
	/*
	 * This method handles a selection event
	 */
	public void selectionChanged() {
		
		_executionStatistic.selectionChanged();
		_executionGraph.selectionChanged();		
	}
	
	
	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {
		
		if(_executionGraph.getCanvas() != null 
			&& _executionGraph.getCanvas().getControl() != null
			&& _executionGraph.getCanvas().getControl().isFocusControl())
		{
			_executionGraph.menuAboutToShow(menu);
		}
		else
		{
			addExecutionStatisticMenu(menu);
		}
	}

	protected void addExecutionStatisticMenu(IMenuManager menu)
	{
		menu.add(fSeparator);

		menu.add(_executionStatistic.getUpdateAction());
		
		menu.add(fSeparator);
		menu.add(_executionStatistic.getChooseColumnsAction(_executionStatistic.getColumnDataList(), _executionStatistic.getColumnsPreferencesKey()));		
	}
	
	public void updateButtons() {
		
		_page.percentMode().setChecked(_executionStatistic.isShowPercent());		
	}

	public void updateModelSelection() {
		
		ISelection selection = _executionStatistic.getTreeViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			
			_executionStatistic.notifyViewSelectionChanged(this,sel);
		}
		
	}
	
	public void dispose() {
		
		_executionStatistic.dispose();
		_executionGraph.dispose();
	}

  protected void postUpdateEvents()
	{
	}		
	
	public boolean isEmpty()
	{
		return _executionGraph.isEmpty();
	}
	
    protected String getViewTypeStr()
    {
    	return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.METHOD_INVOCATION_STATS_VIEW;
    }		
	
	public void update() {
		_executionGraph.updateUI();
	}
	
	public void update(boolean redraw) {
		
		_executionGraph.updateUI();
		
		if(redraw)
			_executionStatistic.updateUI(_executionStatistic.getColumnDataList());

	}
	
	public void initialize() {
		_executionGraph.initialize();
	} 
	
	public JCanvas getCanvas() {
		return _executionGraph.getCanvas();
	}
	
	public void updateUI() {
	   _executionGraph.updateUI();
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (1/11/2001 5:07:13 PM)
	 */
	public void zoomOut(boolean zoom) {
		
		_executionGraph.zoomOut(zoom);
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (1/11/2001 5:07:13 PM)
	 */
	public void zoomIn(boolean zoom) {
		_executionGraph.zoomIn(zoom);
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (8/21/2001 6:31:49 PM)
	 */
	public void openSource() {
	
		OpenSource.openSource(UIPlugin.getDefault().getSelectionModel(_page.getMOFObject()).getFirstElement());
	
	}

	/**
	 * @return Returns the _invocationStatistic.
	 */
	public ExecutionStatistic getExecutionStatistic() {
		return _executionStatistic;
	}
	/**
	 * @return Returns the _invocationGraph.
	 */
	public Pattern getExecutionGraph() {
		return _executionGraph;
	}
}