/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionStatistic.java,v 1.40 2008/05/27 15:25:35 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.trace.ui.ITraceSelection;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.adapter.internal.IContextViewer;
import org.eclipse.hyades.trace.views.adapter.internal.PatternPage;
import org.eclipse.hyades.trace.views.adapter.internal.TraceConstants;
import org.eclipse.hyades.trace.views.internal.fragment.ContentProviderNewLazyAdaptor;
import org.eclipse.hyades.trace.views.internal.fragment.FragmentedTreeViewer;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnDisplayInfo;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.CumulativeTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.InstanceNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.MethodInvocationEntryTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.MethodNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ThreadNameColumnLabel;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.ITimeChangedListener;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.hyades.ui.provisional.context.ContextManager;
import org.eclipse.hyades.ui.provisional.context.IContextAttributes;
import org.eclipse.hyades.ui.provisional.context.IContextLanguage;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;

public class ExecutionStatistic extends StatisticView
	                              implements ITimeChangedListener

{
    protected String tmpString;
    
	private int _drawMode = TraceUIPlugin.getDefault().getPreferenceStore().getInt(TraceConstants.TIME_OPTION);
	
	protected ContextInfoContainer _contextInfo;
	
	/**
	 * 
	 *
	 */
	public class ExecutionStatisticFilter extends StatisticFilter {

		public ExecutionStatisticFilter() {
			super();
		}

		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			
			if(!(element instanceof TRCThread))
			  return true;
			  
			String compareText = "";
		   compareText = ((TRCThread)element).getName();
		   if(compareText.equals(""))
			 compareText = TraceUIMessages._87;
			
			if (_noPattern)
				return true;
			
			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {
					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
		
	}
	
	public class ExecutionStatisticSorter extends StatisticSorter {

		public ExecutionStatisticSorter() {
			super();
		}

		public int compare(Viewer viewer, Object e1, Object e2) {
			
			if(e1 instanceof TRCThread && e2 instanceof TRCThread)
			{
				if(_pos == 0)
					return _sortSequence * compareElements(e1, e2, _threadNameCol, false);
			}
			
			if(e1 instanceof TRCFullMethodInvocation && e2 instanceof TRCFullMethodInvocation)
			{
				if(_pos == 0) //name
					return _sortSequence * compareElements(e1, e2, _methodNameCol, false);
				else if(_pos == 1) //instances
					return _sortSequence * compareElements(e1, e2, _instanceNameCol, false);
				else if(_pos == 2)//base time
					return _sortSequence * compareElements(e1, e2, _methodInvocationEntryTimeCol, false);
				else if(_pos == 3) //cumulative time
					return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, _info.isDeltaColumn());
			}
			
			return 0;
		}
		
	}
	
	/**
	 * 
	 */
	public class ExecutionStatisticContentProvider implements ITreeContentProvider {
		
		public void dispose() {
		}
		
		public Object getParent(Object element) {
//			if(element instanceof TRCThread){
//				return null;//_page.getMOFObject();
//			}
			if(element instanceof TRCMethodInvocation){
				TRCMethodInvocation mi = ((TRCMethodInvocation)element);
				TRCMethodInvocation parent = mi.getInvokedBy();
				if(parent == null)
					return mi.getThread();
				return parent;
			}
			return null;
		}
		
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
		
		public Object[] getElements(Object inputElement)
		{
			return PerftraceUtil.getAllThreads(_page.getMOFObject());
		}
				
		public Object[] getChildren(Object element)
		{
			tmpList.clear();
			
			if(element instanceof TRCThread)
				return ((TRCThread)element).getInitialInvocations().toArray();
			
			TRCMethodInvocation inv = (TRCMethodInvocation)element;
			Object[] segments = inv.getInvokes().toArray();
			for(int idx=0; idx<segments.length; idx++)
			{
				tmpList.add(((TRCMethodInvocation)segments[idx]));
			}
			
			return tmpList.toArray(); 	
		}
	
		public boolean hasChildren(Object element) {
			if (element instanceof TRCThread)
				return true;

			return (((TRCMethodInvocation)element).getInvokes().size() > 0);
		}
		
		public Object[] search(TRCMethod method, TRCMethodInvocation inv){
			Object[] threads = getElements(_page.getMOFObject());
			List list = new ArrayList();
			for(int i = 0; i < threads.length; i++){
				if(search(threads[i], method, inv, list)){
					Object[] result = new Object[list.size() + 1];
					result[0] = threads[i];
					for(int k = 1; k < result.length; k++){
						result[k] = list.get(result.length - k - 1);
					}
					return result;
				}
			}
			return null;
		}
		
		private boolean search(Object parent, TRCMethod method, TRCMethodInvocation inv, List result){
			Object[] children = getChildren(parent);
			Object child;
			for(int i = 0; i < children.length; i++){
				child = children[i];
				if(inv != null && inv == child){
					result.add(child);
					return true;
				}
				if(child instanceof TRCMethodInvocation && ((TRCMethodInvocation)child).getMethod() == method){
					result.add(child);
					return true;
				}
				if(search(child,method, inv, result)){
					result.add(child);
					return true;
				}
			}
			return false;
		}
	}

	public class ExecutionStatisticCellLableProvider  extends StatisticCellLabelProvider {
		public ExecutionStatisticCellLableProvider(ColumnData colData) {
			super(colData);
 		}
		
		public void update(ViewerCell cell) {
			visualIndex = cell.getVisualIndex();
			cell.setText(((ITableLabelProvider)getTableLabelProvider()).getColumnText(cell.getElement(),visualIndex));
			cell.setImage(((ITableLabelProvider)getTableLabelProvider()).getColumnImage(cell.getElement(),visualIndex));
		}
 	}
	
	public class ExecutionStatisticLableProvider extends LabelProvider implements ITableLabelProvider
	{
		protected StatisticView _viewer;
		
		public ExecutionStatisticLableProvider(StatisticView viewer) {
			super();
			_viewer = viewer;
		}
		
		public Image getColumnImage(Object obj, int col) {
				return null;
		}

		public String getColumnText(Object obj, int col) {
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCThread)	
			{
				if(pos == 0)//thread name
					return getElementColumnText(obj, _threadNameCol, info.isDeltaColumn());		  
			}
			
			if(obj instanceof TRCMethodInvocation)
			{
				switch(pos)
				{
					case 0://method name
						return getElementColumnText(obj, _methodNameCol, info.isDeltaColumn());
					case 1://instance name
						return getElementColumnText(obj, _instanceNameCol, info.isDeltaColumn());						
					case 2://base time
						return getElementColumnText(obj, _methodInvocationEntryTimeCol, info.isDeltaColumn());
					case 3: //cumulative time
						return getElementColumnText(obj, _cumulativeTimeCol, info.isDeltaColumn());
				}
			}
			return "";
		}
	}
    	
	class ExecutionTreeViewer extends FragmentedTreeViewer {
		public ExecutionTreeViewer(Composite parent) {
			super(parent);
		}

		public ExecutionTreeViewer(Tree tree) {
			super(tree);
		}

		public void expandItem(TreeItem item) {
			item.setExpanded(true);
			createChildren(item);

		}

	}
	protected Composite createTree(Composite parent, int options) {
		return _toolkit.createTree(parent, options | SWT.VIRTUAL);
	}

	protected ColumnLabelAdapter _threadNameCol;
	protected ColumnLabelAdapter _methodNameCol;
	protected ColumnLabelAdapter _instanceNameCol;
	protected ColumnLabelAdapter _methodInvocationEntryTimeCol;
	protected ColumnLabelAdapter _cumulativeTimeCol;
	
	public ExecutionStatistic(Composite parent, TraceViewerPage page) {
		super(true, parent, page);
		
		TraceUIPlugin.getDefault().addTimeChangedEventListener(this);
		_viewerFilter = new ExecutionStatisticFilter();		
		
		createColumnsLabelProviders();
	}
	
	protected String getContextHelpId()
	{
		return TraceUIPlugin.getPluginId()+".exef0008";
	}
	
	public void createColumnsLabelProviders()
	{
		_threadNameCol = new ThreadNameColumnLabel();
		_methodNameCol = new MethodNameColumnLabel();
		_instanceNameCol = new InstanceNameColumnLabel();
		_methodInvocationEntryTimeCol = new MethodInvocationEntryTimeColumnLabel();
		_cumulativeTimeCol = new CumulativeTimeColumnLabel();
	}
	
	/**
	 * getClassColumnsPerferencesKey returns the string key used to find the
	 * column preferences information in the preferences store.
	 * @return java.lang.String key name
	 */
	public String getColumnsPreferencesKey() {
		// Note this string is not to be translated and must be changed whenever the 
		// default columns template is changed by adding, removing or renaming columns.
		// Changing the version will result in the default preferences being used.
		return "Exec51"; 
		
	}
	
	public String getDefaultColumnsTemplate() {
		//Class Columns Data
		
		String executionColumn =
			      IContextAttributes.THREAD_NAME			+ ":0:"
			      + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
				+ IContextAttributes.OBJECT_NAME 		+ ":1:"
			  		+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:200,"					 	
				+ IContextAttributes.METHOD_START_TIME				+ ":2:" 
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"
			    + IContextAttributes.METHOD_CUMULATIVE_TIME		+ ":3:"				
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100";
					
		return executionColumn;
	}

/*
	protected StatisticList createStatisticList() {
		return new ThreadStatisticList(this);
	}
*/
	public Tree getTree() {
		return (Tree) getTreeViewer().getControl();
		
	}

	protected TreeViewer getTreeViewer(Tree tree)
	{
		return new ExecutionTreeViewer(tree);
	}

	protected void doHandleSelectionEvent(boolean isPostponedOperation, int processedOperations) {
		if (_page == null) {
			return;
		}
		
		ITraceSelection model =
			UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() > 0) {
			Object sel = model.getFirstElement();
			if (sel != null && sel instanceof EObject) {
				select(sel);
				updateStatusContext(ContextManager.getContextLanguage(ContextUpdaterHelper.getContext((EObject)sel)));
			}
		}

	}

	/**
	 * select the element (in the coverage table) which is selected in the other 
	 * statistics views
	 */
	private void select(Object obj) {
		TRCClass cls = null;
		TRCPackage pack = null;
		TRCMethod meth = null;
		TRCObject trcObj = null;

		if(adaptor == null){
			Tree tree = (Tree) getTreeViewer().getControl();
	
			//Summary element
			TreeItem[] items = tree.getItems();
			if (items.length != 0) // if we have data
				{
				//((MultiLevelStatisticTreeViewer) getTableViewer()).expandItem(items[0]);
	
				if (obj instanceof TRCPackage)
					pack = (TRCPackage) obj;
				else if (obj instanceof TRCClass) {
					cls = (TRCClass) obj;
					pack = cls.getPackage();
				} else if (obj instanceof TRCMethod) {
					meth = (TRCMethod) obj;
					cls = meth.getDefiningClass();
					pack = cls.getPackage();
				}
				else if (obj instanceof TRCObject)
				{
					cls = ((TRCObject)obj).getIsA();
					pack = cls.getPackage();
					trcObj = (TRCObject)obj;
				}
				else if(obj instanceof TRCMethodInvocation)
				{
					meth = ((TRCMethodInvocation) obj).getMethod();
				}
				
				searchInLevel(tree,
							getItemsRoot(tree),
							pack,
							cls,
							meth,
							trcObj);
			}
		} else {
			if(obj instanceof TRCMethod){
				IStructuredSelection curSel = (IStructuredSelection)_viewer.getSelection();
				Object sel[] = curSel.toArray();
				if(sel.length != 0){
					for(int i = sel.length-1; i >= 0; i--){
						Object s = sel[i];
						if(s instanceof TRCMethodInvocation && ((TRCMethodInvocation)s).getMethod() == obj){
							return;
						}
					}
				}
				TRCMethod method = (TRCMethod)obj;
				ExecutionStatisticContentProvider p = (ExecutionStatisticContentProvider)adaptor.getWrappedContentProvider();
				Object []path = p.search(method, null);
				if(path != null){
					for(int i = 0; i < path.length; i++){
						Object ve = adaptor.toViewerElement(path[i]);
						if(ve == null)
							return;
						path[i] = ve;
					}
					TreeSelection newSel = new TreeSelection(new TreePath(path));
					_viewer.setSelection(newSel, true);
				}
			} else if (obj instanceof TRCMethodInvocation){
				IStructuredSelection curSel = (IStructuredSelection)_viewer.getSelection();
				Object sel[] = curSel.toArray();
				if(sel.length != 0){
					for(int i = sel.length-1; i >= 0; i--){
						Object s = sel[i];
						if(s == obj){
							return;
						}
					}
				}
				TRCMethodInvocation inv = (TRCMethodInvocation)obj;
				ExecutionStatisticContentProvider p = (ExecutionStatisticContentProvider)adaptor.getWrappedContentProvider();
				Object []path = p.search(null, inv);
				if(path != null){
					for(int i = 0; i < path.length; i++){
						Object ve = adaptor.toViewerElement(path[i]);
						if(ve == null)
							return;
						path[i] = ve;
					}
					TreeSelection newSel = new TreeSelection(new TreePath(path));
					_viewer.setSelection(newSel, true);
				}
			}
		}
	}
	
	protected TreeItem getItemsRoot(Tree tree)
	{
		return null;
	}

	protected TreeItem searchItem(TreeItem[] items, Object obj) {
		TreeItem item = null;
		for (int j = 0; j < items.length; j++) {
			item = items[j];

			if (item!=null && !item.isDisposed()) {
				if (item.getData() != null &&
						obj == getItemModelData(item))
					return item;
				
				if (item.getData() != null &&
						item.getData() instanceof TRCMethodInvocation &&
						obj instanceof TRCMethod &&
						((TRCMethodInvocation)item.getData()).getMethod().equals(obj))
					return item;
			}
			
			TreeItem item1 = searchItem(item.getItems(),obj);
			if (item1!=null) {
				return item1;
			}
		}
		return null;
	}		

	protected Object getItemModelData(TreeItem item)
	{
		return adaptor != null ? ContentProviderNewLazyAdaptor.getObject(item.getData()) : item.getData();
	}
	
	protected void searchInLevel(Tree tree,
			TreeItem item,
			TRCPackage pack,
			TRCClass cls,
			TRCMethod meth,
			TRCObject trcObj)
	{
		TreeItem[] objItems = null;
		
		if (item != null)
			objItems = item.getItems();
		else
			objItems = tree.getItems();			

		TreeItem objItem = searchItem(objItems, trcObj);
		if (objItem == null) {
			objItem = searchItem(objItems, meth);
		}
		if (objItem == null) {
			objItem = searchItem(objItems, cls);
		}
		if (objItem == null) {
			objItem = searchItem(objItems, pack);
		}

		if (objItem != null) {
			tree.setSelection(new TreeItem[] { objItem });
			return;
		}
		else if (item != null)
		{
			tree.setSelection(new TreeItem[] { item });		
			return;			
		}

		tree.deselectAll();
	}
	
	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {
		menu.add(fSeparator);

		menu.add(getUpdateAction());
		
		menu.add(fSeparator);
		menu.add(getChooseColumnsAction(getColumnDataList(), getColumnsPreferencesKey()));
	}


	protected void doUpdateButtons(boolean isPostponedOperation, int processedOperations) {
	}

	protected void updateDetailsPane() {
		int selCount = getTree().getSelectionCount();
		if (selCount != 1) {
			return;
		}

		Item item = getTree().getSelection()[0];
		Object itemData = item.getData();

		if (itemData == null) {
			return;
		}
		if (itemData != null && itemData instanceof TreeItem) {
			itemData = ((TreeItem) itemData).getData();
		}
	}

	public void updateModelSelection() {
		
		ISelection selection = getTreeViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			if(adaptor != null)
				sel = ContentProviderNewLazyAdaptor.getObject(sel);
			
			notifyViewSelectionChanged(this,sel);
		}
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new ExecutionStatisticSorter();
	}
	
	protected void firstTimeUpdate()
	{
		super.firstTimeUpdate();
		handleTimeChangedEvent();		
	}	
	
	protected int updateTableGetColumnNumber()
	{
		//pick third column as sorting column unless it is disabled.
		if (getTree().getColumnCount()<3) return 0;
		else {
			return 2;
		}
	}	
	
  public void handleTimeChangedEvent()
  {		
  	    Action base = ((PatternPage)getTraceViewerPage()).baseTime();
  	    Action raw = ((PatternPage)getTraceViewerPage()).rawTime();
	    _drawMode = TraceUIPlugin.getDefault().getPreferenceStore().getInt(TraceConstants.TIME_OPTION);
	    if(base != null)
	      base.setChecked(_drawMode == TraceConstants.COMPENSATED_TIME);
	    if(raw != null)
	      raw.setChecked(_drawMode == TraceConstants.RAW_TIME);
	    
	    String oldKey;
	    String newKey;
	    if (_drawMode == TraceConstants.COMPENSATED_TIME)
	    	newKey = IContextAttributes.METHOD_CUMULATIVE_TIME;
	    else
	    	newKey = IContextAttributes.METHOD_RAW_CUMULATIVE_TIME;
	    
	    if (newKey.equals(IContextAttributes.METHOD_CUMULATIVE_TIME))
	    	oldKey = IContextAttributes.METHOD_RAW_CUMULATIVE_TIME;
	    else
	    	oldKey = IContextAttributes.METHOD_CUMULATIVE_TIME;
	    
	
		Tree tree = getTree();	
		for (int idx = tree.getColumnCount(); idx > 0; idx--)
		{
			TreeColumn col = tree.getColumn(idx - 1);
			StatisticTableColumnInfo staticTblColInfo = (StatisticTableColumnInfo)(col.getData());
			String columnKey = staticTblColInfo.getColumnData().key();	
			if(columnKey.equals(oldKey))
			{
				staticTblColInfo.getColumnData().key(newKey);
				String newName = staticTblColInfo.getColumnData().name();
			    if(staticTblColInfo.isSortColumn())
			    {
			    	newName = col.getText().substring(0, 1) + newName; 
			    }			   	
 			   	col.setText(newName);
			}
		}
	
		getTreeViewer().refresh();
  }

	public void dispose()
	{
		super.dispose();
	    TraceUIPlugin.getDefault().removeTimeChangedEventListener(this);
		
	}

	public IContentProvider getContentProvider()
	{
		return new ExecutionStatisticContentProvider();
	}

	public LabelProvider getTableLabelProvider()
	{
		return adaptor != null ? wrappedLp : new ExecutionStatisticLableProvider(this);
	}
	
	protected void doHandleViewSelectionChangedEvent(ViewSelectionChangedEvent event, boolean isPostponedOperation, int processedOperations)
	{
		Object  source = event.getSource();
		if(source!=this){	  
			handleSelectionEvent();
		}
		else
		{
			ITraceSelection model =
				UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
			if (model.size() > 0) {
				Object sel = model.getFirstElement();
				if (sel != null && sel instanceof EObject)
						updateStatusContext(ContextManager.getContextLanguage(ContextUpdaterHelper.getContext((EObject)sel)));
			}
		}

	}
	
	public boolean isEmpty()
	{
		if (PerftraceUtil.getAllThreads(_page.getMOFObject()).length > 0)
			return false;
		return true;
	}
	
    protected String getViewTypeStr()
    {
    	return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.EXECUTION_STATS_VIEW;
    }			
	/**
	 * @return Returns the _drawMode.
	 */
	public int getDrawMode() {
		return _drawMode;
	}
	/**
	 * @param mode The _drawMode to set.
	 */
	public void setDrawMode(int mode) {
		_drawMode = mode;
	}
	
	protected void showPercentUpdate()
	{
		if(isShowPercent())
		{
			// Updated to use totalCumulativeTime here and in ColumnDisplayInfo
			// _maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
			// if (_maxTime==0)_maxTime=1;		
			_totalCumulativeTime = PerftraceUtil.getTotalCumulativeTime(_page.getMOFObject());
			if(_totalCumulativeTime==0)_totalCumulativeTime=1;
		}		
	}
	
	protected ColumnDisplayInfo getColumnDisplayInfo(ColumnLabelAdapter col, boolean isDeltaColumn)
	{
		if (col == _cumulativeTimeCol)
			return ContextUpdaterHelper.updateCumulTime(col, isDeltaColumn, isShowPercent(), getDrawMode(), _totalCumulativeTime);
		else
			return super.getColumnDisplayInfo(col, isDeltaColumn);
	}	
	
	protected Composite createControl(Composite parent, ArrayList cols) {
		final Composite vc = super.createControl(parent, cols);
		
		_contextInfo = new ContextInfoContainer();
		_contextInfo.setViewer((IContextViewer)_page.getTraceViewer());
		_contextInfo.createControl(vc);
		_contextInfo.addContextInfoContainerListener(
				new IContextInfoContainerListener()
				{
					public void visibilityChanged(boolean isVisible)
					{
						vc.layout(true, true);
					}
				});

		return vc;
	}
	
	protected void updateStatusContext(IContextLanguage language)
	{
		if (_contextInfo != null)
			_contextInfo.updateStatusContext(language);
	}
	
	protected void doUpdate(boolean refresh, boolean isPostponedOperation, int processedOperations) {
		resetColumns();
		
		if(_page instanceof PatternPage)
		{
			if (_contextInfo != null)
				_contextInfo.setMOFObject(_page.getMOFObject());
		}
		
		super.doUpdate(refresh, isPostponedOperation, processedOperations);
	}
	
	protected void resetColumns() {
		_threadNameCol.resetMap();
		_methodNameCol.resetMap();
		_instanceNameCol.resetMap();
		_methodInvocationEntryTimeCol.resetMap();
		_cumulativeTimeCol.resetMap();
	}

	public CellLabelProvider getCellLabelProvider(ColumnData colData) {
		return new ExecutionStatisticCellLableProvider(colData);
	}
}