/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionStatisticView.java,v 1.34 2008/06/04 23:56:01 jkubasta Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCMethodImpl;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl.TRCClassSnapshot;
import org.eclipse.hyades.models.trace.impl.TRCMethodImpl.TRCMethodSnapshot;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl.TRCPackageSnapshot;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.adapter.internal.DynamicFilter;
import org.eclipse.hyades.trace.views.adapter.internal.ExecutionStatisticPage2;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.AvgBaseTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.BaseTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.CPUTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.CallsColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ClassNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnDisplayInfo;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.CumulativeTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.InheritedBaseTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.InheritedCallsColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.InheritedCumulativeTimeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.InstanceNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.MethodNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.NewItemColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.PackageNameColumnLabel;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.ColumnExtensionValue;
import org.eclipse.hyades.trace.views.util.internal.IColumnExtension;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.hyades.ui.provisional.context.IContextAttributes;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

/*
 * This view contains execution statistics data.  That is time related data for
 * the package level, class level, method level, and instance level.
 */
public class ExecutionStatisticView extends MultiLevelStatisticMethodView{
	
	private static final int _numberOfColumns[] = { 20, 20, 20, 20 };
	
	private static int _level = DEFAULT_LEVEL;
	
	public static final int INSTANCE_LEVEL = 4;	
	
	protected double _totalPackageCPUtime = 0;
	
	private TRCPackageSnapshot packageSnapshot = new TRCPackageImpl.TRCPackageSnapshot();
	private TRCClassSnapshot classSnapshot = new TRCClassImpl.TRCClassSnapshot();
	private TRCMethodSnapshot methodSnapshot = new TRCMethodImpl.TRCMethodSnapshot();		
	
	protected ColumnLabelAdapter _packageNameCol;
	protected ColumnLabelAdapter _classNameCol;	
	protected ColumnLabelAdapter _methodNameCol;
	protected ColumnLabelAdapter _instanceNameCol;	
	
	protected ColumnLabelAdapter _newItemCol;
	
	protected ColumnLabelAdapter _baseTimeCol;
	protected ColumnLabelAdapter _avgBaseTimeCol;
	protected ColumnLabelAdapter _inheritedBaseTimeCol;
	protected ColumnLabelAdapter _cumulativeTimeCol;
	protected ColumnLabelAdapter _inheritedCumulativeTimeCol;
	protected ColumnLabelAdapter _callsCol;
	protected ColumnLabelAdapter _inheritedCallsCol;						
	protected ColumnLabelAdapter _CPUTimeCol;	

	public ExecutionStatisticView(Composite parent, TraceViewerPage page){
		this(parent, page, true);
	}

	public ExecutionStatisticView(Composite parent, TraceViewerPage page, boolean initializeContent){
		super(true, parent, page, initializeContent);
		createColumnsLabelProviders();
	}
	
	protected String getContextHelpId()
	{
		return TraceUIPlugin.getPluginId()+".clst0000";
	}
	
	public void createColumnsLabelProviders()
	{
		_packageNameCol = new PackageNameColumnLabel();
		_classNameCol = new ClassNameColumnLabel();
		_methodNameCol = new MethodNameColumnLabel();		
		_instanceNameCol = new InstanceNameColumnLabel();		
		
		_newItemCol = new NewItemColumnLabel();
		ContextUpdaterHelper.addSnapshots(_newItemCol, packageSnapshot, classSnapshot, methodSnapshot);
		_baseTimeCol = new BaseTimeColumnLabel();
		ContextUpdaterHelper.addSnapshots(_baseTimeCol, packageSnapshot, classSnapshot, methodSnapshot);		
		_avgBaseTimeCol = new AvgBaseTimeColumnLabel();
		_inheritedBaseTimeCol = new InheritedBaseTimeColumnLabel();
		ContextUpdaterHelper.addSnapshots(_inheritedBaseTimeCol, packageSnapshot, classSnapshot, null);		
		_cumulativeTimeCol = new CumulativeTimeColumnLabel();
		ContextUpdaterHelper.addSnapshots(_cumulativeTimeCol, packageSnapshot, classSnapshot, methodSnapshot);
		_inheritedCumulativeTimeCol = new InheritedCumulativeTimeColumnLabel();
		ContextUpdaterHelper.addSnapshots(_inheritedCumulativeTimeCol, packageSnapshot, classSnapshot, null);				
		_callsCol = new CallsColumnLabel();
		ContextUpdaterHelper.addSnapshots(_callsCol, packageSnapshot, classSnapshot, methodSnapshot);		
		_inheritedCallsCol = new InheritedCallsColumnLabel();
		ContextUpdaterHelper.addSnapshots(_inheritedCallsCol, packageSnapshot, classSnapshot, null);		
		_CPUTimeCol = new CPUTimeColumnLabel();
		ContextUpdaterHelper.addSnapshots(_CPUTimeCol, packageSnapshot, classSnapshot, methodSnapshot);		
	}

	protected int getClassPosForLevel(int pos)
	{
        if (getLevel()== ExecutionStatisticView.PACKAGE_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 3;
           		case 3:
           			return 4;
           		case 4:
           			return 5;
           		case 5:
           			return 6;
           		case 6:
           			return 7;
           		case 7:
           			return 8;
           		case 8:
           			return 9;
           		case 9:
           			return 10;
			}
        }
        else if (getLevel()== ExecutionStatisticView.INSTANCE_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 2;
           		case 3:
           			return 3;
           		case 4:
           			return 6;
           		case 5:
           			return 8;
           		case 6:
           			return 10;
			}
        }
        return pos;
	}
	
	protected int getMethodPosForLevel(int pos)
	{
        if (getLevel()== ExecutionStatisticView.PACKAGE_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 4;
           		case 3:
           			return 5;
           		case 4:
           			return -1;
           		case 5:
           			return 6;
           		case 6:
           			return -1;
           		case 7:
           			return 7;
           		case 8:
           			return -1; 
           		case 9:
           			return 8;
			}
        }
        else if (getLevel()== ExecutionStatisticView.CLASS_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 3;
           		case 3:
           			return 4;           			
           		case 4:
           			return 5;
           		case 5:
           			return -1;
           		case 6:
           			return 6;
           		case 7:
           			return -1;
           		case 8:
           			return 7;
           		case 9:
           			return -1;
           		case 10:
           			return 8;
			}
        }       
        return pos;
	}		
	
	public class ExecutionStatisticContentProvider extends MultiLevelStatisticContentProvider
	{
		public Object[] getChildren(Object element)
		{
			tmpList.clear();
			
			if(element instanceof TRCPackage)
			{
				return FilteringUtil.getFilteredClasses(getCurrentFilter(), (TRCPackage)element).toArray();
			}
			else if (element instanceof TRCClass)
			{
				if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				{
						Object[] elements = FilteringUtil.getFilteredFullTraceObjects(getCurrentFilter(), (TRCClass)element).toArray();
						
						if (elements.length > 0)
							return elements;
						else
							return FilteringUtil.getFilteredHeapObjects(getCurrentFilter(), (TRCClass)element).toArray();
				}
				else
				{
					return FilteringUtil.getFilteredMethods(getCurrentFilter(), (TRCClass)element).toArray();
				}
			}
			else
				return tmpList.toArray(); 	
		}
		
		public Object getParent(Object element) {
			
			return null;
		}		

		public boolean hasChildren(Object element) {
			if (FilteringUtil.USE_ENGINE_FOR_HAS_CHILDREN)
				return getChildren(element).length > 0;
			else if (element instanceof TRCPackage)
				return true;
			else if (element instanceof TRCClass)
				return true;
			else
				return false;
		}
		
		/**
		 * return the list of elements to display in the table tree
		 */
		public Object[] getElements(Object inputElement) {
			SimpleSearchQuery currentFilter = getCurrentFilter();
			currentFilter = updateDynamicFilters(currentFilter, _page.getMOFObject());
			if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
			{
				return FilteringUtil.getFilteredPackages(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();
			}
			else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
			{
					return FilteringUtil.getFilteredClasses(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();
			}
			else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
			{
				return FilteringUtil.getFilteredMethods(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();
			}
			else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
			{
				return FilteringUtil.getFilteredClasses(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();
			}
			else
			{
				return new ArrayList().toArray();
			}
		}

		/**
		 * updates filters which depend on contents, if any
		 */
		private SimpleSearchQuery updateDynamicFilters(SimpleSearchQuery filter, EObject object) {
			if (!DynamicFilter.containsDynamicFilters(filter)) {
				return filter;
			}
			return DynamicFilter.updateDynamicFilters(filter, object);
		}
	}
	
	public class ExecutionStatisticCellLabelProvider extends StatisticCellLabelProvider {
 		public ExecutionStatisticCellLabelProvider(ColumnData colData) {
			super(colData);
 		}
		
		public void update(ViewerCell cell) {
			visualIndex = cell.getVisualIndex();
			cell.setText(((ExecutionStatisticLabelProvider)getTableLabelProvider()).getColumnText(cell.getElement(),visualIndex));
			cell.setImage(((ExecutionStatisticLabelProvider)getTableLabelProvider()).getColumnImage(cell.getElement(),visualIndex));
		}
 	}
	
	public class ExecutionStatisticLabelProvider extends MultiLevelStatisticLabelProvider
	{
		public ExecutionStatisticLabelProvider(StatisticView viewer) {
			super(viewer);
		}
		
		protected String getColumnTextPackageLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCPackage)	
			{
				switch(pos)
				{
					case 0://package name
						return getElementColumnText(obj, _packageNameCol, info.isDeltaColumn());
					case 1:
						return getElementColumnText(obj, _newItemCol, info.isDeltaColumn());
					case 2: //base time
						return getElementColumnText(obj, _baseTimeCol, info.isDeltaColumn());
					case 3: //average base time
						return getElementColumnText(obj, _avgBaseTimeCol, info.isDeltaColumn());
					case 4: //inherited base time
						return getElementColumnText(obj, _inheritedBaseTimeCol, info.isDeltaColumn());
					case 5: //cumulative time
						return getElementColumnText(obj, _cumulativeTimeCol, info.isDeltaColumn());
					case 6: //inherited cumulative time
						return getElementColumnText(obj, _inheritedCumulativeTimeCol, info.isDeltaColumn());
					case 7: //calls
						return getElementColumnText(obj, _callsCol, info.isDeltaColumn());
					case 8: //inherited calls
						return getElementColumnText(obj, _inheritedCallsCol, info.isDeltaColumn());						
					case 9: //cpu time
						return getElementColumnText(obj, _CPUTimeCol, info.isDeltaColumn());
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt = (ColumnExtensionValue) _listOfColumExtension.get(pos - _numberOfColumns[_level-DEFAULT_LEVEL]);
					return ((IColumnExtension)(elt.getClassOfColumnExtension())).getColumnText(obj,elt.getInitialColumnPos());
				}
			}
			else
			{
				return getColumnTextClassLevel(obj, col);
			}
            							
			return "";
		}
		
		protected String getColumnTextClassLevel(Object obj, int col){
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = getClassPosForLevel(info.getColumnData().getInitalPos());			
            
			if (obj instanceof TRCClass) {
				switch (pos) {
					case 0 : //class name
						return getElementColumnText(obj, _classNameCol, info.isDeltaColumn());
					case 1:
						return getElementColumnText(obj, _newItemCol, info.isDeltaColumn());
					case 2 : //package name
						return getElementColumnText(obj, _packageNameCol, info.isDeltaColumn());
					case 3: //base time
						return getElementColumnText(obj, _baseTimeCol, info.isDeltaColumn());
					case 4: //average base time
						return getElementColumnText(obj, _avgBaseTimeCol, info.isDeltaColumn());
					case 5: //inherited base time
						return getElementColumnText(obj, _inheritedBaseTimeCol, info.isDeltaColumn());
					case 6: //cumulative time
						return getElementColumnText(obj, _cumulativeTimeCol, info.isDeltaColumn());
					case 7: //inherited cumulative time
						return getElementColumnText(obj, _inheritedCumulativeTimeCol, info.isDeltaColumn());
					case 8: //calls
						return getElementColumnText(obj, _callsCol, info.isDeltaColumn());
					case 9: //inherited calls
						return getElementColumnText(obj, _inheritedCallsCol, info.isDeltaColumn());						
					case 10: //cpu time
						return getElementColumnText(obj, _CPUTimeCol, info.isDeltaColumn());
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt = (ColumnExtensionValue) _listOfColumExtension.get(pos - _numberOfColumns[_level-DEFAULT_LEVEL]);
					return ((IColumnExtension)(elt.getClassOfColumnExtension())).getColumnText(obj,elt.getInitialColumnPos());
				}
			}
			else
			{
				return getColumnTextMethodLevel(obj, col);
			}

			return "";
		}

		protected String getColumnTextMethodLevel(Object obj, int col){
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = getMethodPosForLevel(info.getColumnData().getInitalPos());			
			
            //TRCMethod
			switch(pos)
			{				
				case 0://method name
					return getElementColumnText(obj, _methodNameCol, info.isDeltaColumn());
				case 1:
					return getElementColumnText(obj, _newItemCol, info.isDeltaColumn());
				case 2: //class name
					return getElementColumnText(obj, _classNameCol, info.isDeltaColumn());
				case 3: //package name
					return getElementColumnText(obj, _packageNameCol, info.isDeltaColumn());
				case 4: //base time
					return getElementColumnText(obj, _baseTimeCol, info.isDeltaColumn());
				case 5: //average base time
					return getElementColumnText(obj, _avgBaseTimeCol, info.isDeltaColumn());
				case 6: //cumulative time
					return getElementColumnText(obj, _cumulativeTimeCol, info.isDeltaColumn());
				case 7: //calls
					return getElementColumnText(obj, _callsCol, info.isDeltaColumn());
				case 8: //cpu time
					return getElementColumnText(obj, _CPUTimeCol, info.isDeltaColumn());
			}
			// if column extended
			if (_listOfColumExtension.size() != 0) {
				ColumnExtensionValue elt = (ColumnExtensionValue) _listOfColumExtension.get(pos - _numberOfColumns[_level-DEFAULT_LEVEL]);
				return ((IColumnExtension)(elt.getClassOfColumnExtension())).getColumnText(obj,elt.getInitialColumnPos());
			}
            							
			return "";
		}
		
		protected String getColumnTextInstanceLevel(Object obj, int col){
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCClass)	
			{
				return getColumnTextClassLevel(obj, col);
			}
			else
			{
				if(info.isDeltaColumn())
				   return "";

				switch(pos)
				{
					case 0://class name
						return getElementColumnText(obj, _instanceNameCol, info.isDeltaColumn());
					case 1: //new item
						return getElementColumnText(obj, _newItemCol, info.isDeltaColumn());
					case 2: //package name
						return getElementColumnText(obj, _packageNameCol, info.isDeltaColumn());
					case 3: //base time
						return getElementColumnText(obj, _baseTimeCol, info.isDeltaColumn());
					case 4: //cumulative time
						return getElementColumnText(obj, _cumulativeTimeCol, info.isDeltaColumn());
					case 5: //calls
						return getElementColumnText(obj, _callsCol, info.isDeltaColumn());
					case 6: //cpu time
						return getElementColumnText(obj, _CPUTimeCol, info.isDeltaColumn());
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt = (ColumnExtensionValue) _listOfColumExtension.get(pos - _numberOfColumns[_level-DEFAULT_LEVEL]);
					return ((IColumnExtension)(elt.getClassOfColumnExtension())).getColumnText(obj,elt.getInitialColumnPos());
				}
			}
			return "";
		}
		
		public String getColumnText(Object obj, int col) {
			if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
				return getColumnTextPackageLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
				return getColumnTextClassLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
				return getColumnTextMethodLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				return getColumnTextInstanceLevel(obj, col);
			else
				return "";
		}
		
		protected Image getColumnImagePackageLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;

			if(obj instanceof TRCPackage)	
			{
				switch(pos)
				{
					case 0://package name
						return getElementColumnImage(obj, _packageNameCol, info.isDeltaColumn());
					case 1:
						return getElementColumnImage(obj, _newItemCol, info.isDeltaColumn());
					case 2: //base time
						return getElementColumnImage(obj, _baseTimeCol, info.isDeltaColumn());
					case 3: //average base time
						return getElementColumnImage(obj, _avgBaseTimeCol, info.isDeltaColumn());
					case 4: //inherited base time
						return getElementColumnImage(obj, _inheritedBaseTimeCol, info.isDeltaColumn());
					case 5: //cumulative time
						return getElementColumnImage(obj, _cumulativeTimeCol, info.isDeltaColumn());
					case 6: //inherited cumulative time
						return getElementColumnImage(obj, _inheritedCumulativeTimeCol, info.isDeltaColumn());
					case 7: //calls
						return getElementColumnImage(obj, _callsCol, info.isDeltaColumn());
					case 8: //inherited calls
						return getElementColumnImage(obj, _inheritedCallsCol, info.isDeltaColumn());						
					case 9: //cpu time
						return getElementColumnImage(obj, _CPUTimeCol, info.isDeltaColumn());
				}				  				  				 				   				  
			}
			else
			{
				return getColumnImageClassLevel(obj, col);
			}
      							
			return null;
		}
		
		protected Image getColumnImageClassLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = getClassPosForLevel(info.getColumnData().getInitalPos());			
			
			if(info.isDeltaColumn())
			   return null;

			if(obj instanceof TRCClass)	
			{
				switch(pos)
				{
					case 0 : //class name
						return getElementColumnImage(obj, _classNameCol, info.isDeltaColumn());
					case 1:
						return getElementColumnImage(obj, _newItemCol, info.isDeltaColumn());
					case 2 : //package name
						return getElementColumnImage(obj, _packageNameCol, info.isDeltaColumn());
					case 3: //base time
						return getElementColumnImage(obj, _baseTimeCol, info.isDeltaColumn());
					case 4: //average base time
						return getElementColumnImage(obj, _avgBaseTimeCol, info.isDeltaColumn());
					case 5: //inherited base time
						return getElementColumnImage(obj, _inheritedBaseTimeCol, info.isDeltaColumn());
					case 6: //cumulative time
						return getElementColumnImage(obj, _cumulativeTimeCol, info.isDeltaColumn());
					case 7: //inherited cumulative time
						return getElementColumnImage(obj, _inheritedCumulativeTimeCol, info.isDeltaColumn());
					case 8: //calls
						return getElementColumnImage(obj, _callsCol, info.isDeltaColumn());
					case 9: //inherited calls
						return getElementColumnImage(obj, _inheritedCallsCol, info.isDeltaColumn());						
					case 10: //cpu time
						return getElementColumnImage(obj, _CPUTimeCol, info.isDeltaColumn());
				}				  				  				 				   				  
			}
			else
			{
				return getColumnImageMethodLevel(obj, col);
			}
                  
			return null;		
		}
		
		protected Image getColumnImageMethodLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = getMethodPosForLevel(info.getColumnData().getInitalPos());			
			
			if(info.isDeltaColumn())
			   return null;

			switch(pos)
			{
				case 0://method name
					return getElementColumnImage(obj, _methodNameCol, info.isDeltaColumn());
				case 1:
					return getElementColumnImage(obj, _newItemCol, info.isDeltaColumn());
				case 2: //class name
					return getElementColumnImage(obj, _classNameCol, info.isDeltaColumn());
				case 3: //package name
					return getElementColumnImage(obj, _packageNameCol, info.isDeltaColumn());
				case 4: //base time
					return getElementColumnImage(obj, _baseTimeCol, info.isDeltaColumn());
				case 5: //average base time
					return getElementColumnImage(obj, _avgBaseTimeCol, info.isDeltaColumn());
				case 6: //cumulative time
					return getElementColumnImage(obj, _cumulativeTimeCol, info.isDeltaColumn());
				case 7: //calls
					return getElementColumnImage(obj, _callsCol, info.isDeltaColumn());
				case 8: //cpu time
					return getElementColumnImage(obj, _CPUTimeCol, info.isDeltaColumn());
			}
      							
			return null;
		}
		
		protected Image getColumnImageInstanceLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			   
			if(obj instanceof TRCClass)	
			{
				return getColumnImageClassLevel(obj, col);
			}
			else
			{
				switch(pos)
				{
					case 0://class name
						return getElementColumnImage(obj, _instanceNameCol, info.isDeltaColumn());
					case 1: //new item
						return getElementColumnImage(obj, _newItemCol, info.isDeltaColumn());
					case 2: //package name
						return getElementColumnImage(obj, _packageNameCol, info.isDeltaColumn());
					case 3: //base time
						return getElementColumnImage(obj, _baseTimeCol, info.isDeltaColumn());
					case 4: //cumulative time
						return getElementColumnImage(obj, _cumulativeTimeCol, info.isDeltaColumn());
					case 5: //calls
						return getElementColumnImage(obj, _callsCol, info.isDeltaColumn());
					case 6: //cpu time
						return getElementColumnImage(obj, _CPUTimeCol, info.isDeltaColumn());
				}	
			}

			return null;
			
		}
		
		public Image getColumnImage(Object obj, int col) {
			if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
				return getColumnImagePackageLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
				return getColumnImageClassLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
				return getColumnImageMethodLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				return getColumnImageInstanceLevel(obj, col);
			return null;
		}
	}

	public class ExecutionStatisticSorter extends MultiLevelStatisticSorter{
		
		protected int comparePackageLevel(Viewer viewer, Object e1, Object e2)
		{
 			if(e1 instanceof TRCPackage && e2 instanceof TRCPackage)	
			{
				switch(_pos)
				{
					case 0://package name
					  return _sortSequence * compareElements(e1, e2, _packageNameCol, _info.isDeltaColumn());
					case 1 : // new items
						  return _sortSequence * compareElements(e1, e2, _newItemCol, _info.isDeltaColumn());
					case 2: //base time
						  return _sortSequence * compareElements(e1, e2, _baseTimeCol, _info.isDeltaColumn());
					case 3: //average base time
						  return _sortSequence * compareElements(e1, e2, _avgBaseTimeCol, _info.isDeltaColumn());						
					case 4: //inherited base time
						  return _sortSequence * compareElements(e1, e2, _inheritedBaseTimeCol, _info.isDeltaColumn());
					case 5: //cumulative time
						  return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, _info.isDeltaColumn());	
					case 6: //inherited cumulative time
						  return _sortSequence * compareElements(e1, e2, _inheritedCumulativeTimeCol, _info.isDeltaColumn());
					case 7: //calls
						  return _sortSequence * compareElements(e1, e2, _callsCol, _info.isDeltaColumn());
					case 8: //inherited calls
						  return _sortSequence * compareElements(e1, e2, _inheritedCallsCol, _info.isDeltaColumn());	
					case 9: //cpu time
						  return _sortSequence * compareElements(e1, e2, _CPUTimeCol, _info.isDeltaColumn());							
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt =
						(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns[_level-DEFAULT_LEVEL]);
					return (int) _sortSequence * ((IColumnExtension)
								(elt.getClassOfColumnExtension())).compare(
									elt.getInitialColumnPos(), e1, e2);
				}
			}
 			else
 			{
 				return compareClassLevel(viewer, e1, e2);
 			}

			return 0;
		}
		
		protected int compareClassLevel(Viewer viewer, Object e1, Object e2)
		{
			int pos = getClassPosForLevel(_pos);
            
			if (e1 instanceof TRCClass && e2 instanceof TRCClass) {
					
				switch (pos) {
					case 0 : //class name
						return _sortSequence * compareElements(e1, e2, _classNameCol, _info.isDeltaColumn());
					case 1 : // new items
						return _sortSequence * compareElements(e1, e2, _newItemCol, _info.isDeltaColumn());
					case 2 :	//package name
						return _sortSequence * compareElements(e1, e2, _packageNameCol, _info.isDeltaColumn());
					case 3: //base time
						  return _sortSequence * compareElements(e1, e2, _baseTimeCol, _info.isDeltaColumn());
					case 4: //average base time
						  return _sortSequence * compareElements(e1, e2, _avgBaseTimeCol, _info.isDeltaColumn());						
					case 5: //inherited base time
						  return _sortSequence * compareElements(e1, e2, _inheritedBaseTimeCol, _info.isDeltaColumn());
					case 6: //cumulative time
						  return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, _info.isDeltaColumn());	
					case 7: //inherited cumulative time
						  return _sortSequence * compareElements(e1, e2, _inheritedCumulativeTimeCol, _info.isDeltaColumn());
					case 8: //calls
						  return _sortSequence * compareElements(e1, e2, _callsCol, _info.isDeltaColumn());
					case 9: //inherited calls
						  return _sortSequence * compareElements(e1, e2, _inheritedCallsCol, _info.isDeltaColumn());	
					case 10: //cpu time
						  return _sortSequence * compareElements(e1, e2, _CPUTimeCol, _info.isDeltaColumn());							
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt =
						(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns[_level-DEFAULT_LEVEL]);
					return (int) _sortSequence * ((IColumnExtension)
								(elt.getClassOfColumnExtension())).compare(
									elt.getInitialColumnPos(), e1, e2);
				}
			} 
			else
			{
				return compareMethodLevel(viewer, e1, e2);
			}
			
			return 0;
		}

		protected int compareMethodLevel(Viewer viewer, Object e1, Object e2)
		{
			int pos = getMethodPosForLevel(_pos);
			
 			if(e1 instanceof TRCMethod && e2 instanceof TRCMethod)	
			{
				switch(pos)
				{
					case 0://method name
						return _sortSequence * compareElements(e1, e2, _methodNameCol, _info.isDeltaColumn());
					case 1 : // new items
						return _sortSequence * compareElements(e1, e2, _newItemCol, _info.isDeltaColumn());	  
					case 2://class name
						return _sortSequence * compareElements(e1, e2, _classNameCol, _info.isDeltaColumn());
					case 3://package name
						return _sortSequence * compareElements(e1, e2, _packageNameCol, _info.isDeltaColumn());
					case 4: //base time
						return _sortSequence * compareElements(e1, e2, _baseTimeCol, _info.isDeltaColumn());
					case 5: //average base time
						return _sortSequence * compareElements(e1, e2, _avgBaseTimeCol, _info.isDeltaColumn());
					case 6: //cumulative time
						return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, _info.isDeltaColumn());					
					case 7: //calls
						return _sortSequence * compareElements(e1, e2, _callsCol, _info.isDeltaColumn());			
					case 8: //cpu time
						return _sortSequence * compareElements(e1, e2, _CPUTimeCol, _info.isDeltaColumn());	
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt =
						(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns[_level-DEFAULT_LEVEL]);
					return (int) _sortSequence * ((IColumnExtension)
								(elt.getClassOfColumnExtension())).compare(
									elt.getInitialColumnPos(), e1, e2);
				}
			}
			return 0;
		}
		
		protected int compareInstanceLevel(Viewer viewer, Object e1, Object e2)
		{
 			if(e1 instanceof TRCClass)	
			{
 				return compareClassLevel(viewer, e1, e2);
			}
			else
			{				 
				switch(_pos)
				{
					case 0://instance name
						return _sortSequence * compareElements(e1, e2, _instanceNameCol, _info.isDeltaColumn());
					case 1: //new item
						return _sortSequence * compareElements(e1, e2, _newItemCol, _info.isDeltaColumn());
					case 2: //package
						return _sortSequence * compareElements(e1, e2, _packageNameCol, _info.isDeltaColumn()); 
					case 3: //base time
						return _sortSequence * compareElements(e1, e2, _baseTimeCol, _info.isDeltaColumn());
					case 4: //cumulative time
						return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, _info.isDeltaColumn());
					case 5: //calls
						return _sortSequence * compareElements(e1, e2, _callsCol, _info.isDeltaColumn());
					case 6: //cpu time
						return _sortSequence * compareElements(e1, e2, _CPUTimeCol, _info.isDeltaColumn());
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt =
						(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns[_level-DEFAULT_LEVEL]);
					return (int) _sortSequence * ((IColumnExtension)
								(elt.getClassOfColumnExtension())).compare(
									elt.getInitialColumnPos(), e1, e2);
				}
			}
			
			return 0;
		
		}
		
		public int compare(Viewer viewer, Object e1, Object e2) {
				
			if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
				return comparePackageLevel(viewer, e1, e2);
			else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
				return compareClassLevel(viewer, e1, e2);
			else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
				return compareMethodLevel(viewer, e1, e2);
			else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				return compareInstanceLevel(viewer, e1, e2);			
			
			return 0;
		}
	}

	public class ExecutionStatisticFilter extends StatisticFilter{
		
		public ExecutionStatisticFilter() {
			super();
		}
		
		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			String compareText = "";

			switch (getLevel()) {
				case PACKAGE_LEVEL : //Package level
					if (!(element instanceof TRCPackage))
						return true;
					if (((TRCPackage) element).getName().length() == 0)
						compareText =
							TraceUIMessages._87;
					else
						compareText =
							((TRCPackage) element).getName();
					break;
				case CLASS_LEVEL : //class level
					if (!(element instanceof TRCClass))
						return true;
					compareText = ((TRCClass) element).getName();
					break;

				case METHOD_LEVEL : //method level
					if (!(element instanceof TRCMethod))
						return true;
					compareText = ((TRCMethod) element).getName();
					break;
					
				case INSTANCE_LEVEL : // instance level
					if(element instanceof TRCClass)   
						compareText = ((TRCClass)element).getName();					
					break;
			}

			if (_noPattern)
				return true;

			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {

					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());

				}
			}
			return flag;
		}
	}
	
	public boolean isEmptyUpdate()
	{
		return PerftraceUtil.getTotalCalls(_page.getMOFObject()) <= 0.0;
	}
	
	protected String getDefaultColumnsTemplatePackageLevel()
	{
		return IContextAttributes.PACKAGE_NAME 			+ ":0:" 
		      	+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
			+ ""				+ ":1:"
			    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":20,"				
			+ IContextAttributes.PACKAGE_BASE_TIME				+ ":2:"
			  	+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE) + ":right:100,"
		    + IContextAttributes.PACKAGE_AVG_BASE_TIME		+ ":3:" 
				+ String.valueOf(ColumnData.IS_VISIBLE ) + ":right:100,"			  
			+ IContextAttributes.PACKAGE_INH_BASE_TIME	+ ":4:"
			   	+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"			
			+ IContextAttributes.PACKAGE_CUMULATIVE_TIME 		+ ":5:"
			   	+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE) + ":right:100,"
			+ IContextAttributes.PACKAGE_INH_CUMULATIVE_TIME 	+ ":6:"
			   	+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"			
			+ IContextAttributes.PACKAGE_CALLS 					+ ":7:"
			   	+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE) + ":right:50,"
			+ IContextAttributes.PACKAGE_INH_CALLS		+ ":8:"
			  	+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"
		    + IContextAttributes.PACKAGE_CPU_TIME   	+ ":9:"
			  	+ String.valueOf(ColumnData.SUPPORTS_DELTA ) + ":right:100";
	}

	protected String getDefaultColumnsTemplateClassLevel()
	{
		return IContextAttributes.CLASS_NAME
		+ ":0:"
		+ String.valueOf(
			ColumnData.NONDELETABLE
				| ColumnData.IS_VISIBLE
				| ColumnData.NONMOVABLE)
		+ ":200,"
		+ ""
		+ ":1:"
	    + String.valueOf(ColumnData.NONDELETABLE
	    					| ColumnData.IS_VISIBLE
							| ColumnData.NONMOVABLE
							| ColumnData.NONRESIZABLE
							| ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS)
		+ ":20,"		
		+ IContextAttributes.PACKAGE_NAME
		+ ":2:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":left:100,"
		+ IContextAttributes.CLASS_BASE_TIME
		+ ":3:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE)				
		+ ":right:100,"
		+ IContextAttributes.CLASS_AVG_BASE_TIME
		+ ":4:" 
		+ String.valueOf(ColumnData.IS_VISIBLE )
		+ ":right:100,"			
		+ IContextAttributes.CLASS_INH_BASE_TIME
		+ ":5:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA)								
		+ ":right:100,"
		+ IContextAttributes.CLASS_CUMULATIVE_TIME
		+ ":6:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE)								
		+ ":right:100,"
		+ IContextAttributes.CLASS_INH_CUMULATIVE_TIME
		+ ":7:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA)				
		+ ":right:100,"				
		+ IContextAttributes.CLASS_CALLS
		+ ":8:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE)								
		+ ":right:50,"
		+ IContextAttributes.CLASS_INH_CALLS
		+ ":9:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA)								
		+ ":right:100,"
		+ IContextAttributes.CLASS_CPU_TIME
		+ ":10:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA)				
		+ ":right:100";
	}
	
	protected String getDefaultColumnsTemplateMethodLevel()
	{
		return IContextAttributes.METHOD_NAME		+ ":0:"
		    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
		+ ""			+ ":1:"
		    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":20,"		
		+ IContextAttributes.CLASS_NAME				+ ":2:"
			+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:100,"
		+ IContextAttributes.PACKAGE_NAME			+ ":3:"
			+ String.valueOf(ColumnData.IS_VISIBLE)+ ":left:100,"					
		+ IContextAttributes.METHOD_BASE_TIME				+ ":4:" 
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
		+ IContextAttributes.METHOD_AVG_BASE_TIME		+ ":5:" 
			+ String.valueOf(ColumnData.IS_VISIBLE ) + ":right:100,"
		+ IContextAttributes.METHOD_CUMULATIVE_TIME		+ ":6:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
		+ IContextAttributes.METHOD_CALLS					+ ":7:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:55,"
		+ IContextAttributes.METHOD_CPU_TIME	+ ":8:"
			+ String.valueOf(ColumnData.SUPPORTS_DELTA )		+ ":right:100";
	}
	
	protected String getDefaultColumnsTemplateInstanceLevel()
	{
		return IContextAttributes.CLASS_NAME 			+ ":0:" 
		      	+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
			+ ""				+ ":1:"
			    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":20,"		
		   	+ IContextAttributes.PACKAGE_NAME			+ ":2:"
		   		+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:100,"
			+ IContextAttributes.OBJECT_BASE_TIME				+ ":3:"
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"
			+ IContextAttributes.OBJECT_CUMULATIVE_TIME 		+ ":4:"
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"
			+ IContextAttributes.OBJECT_CALLS			 		+ ":5:"
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:50,"
			+ IContextAttributes.OBJECT_CPU_TIME	+ ":6:"
				+ "0:right:100";
	}	
	
	public String getDefaultColumnsTemplate()
	{	
		String columns = super.getDefaultColumnsTemplate();
		
		if (columns == null)
		{
			if (getLevel() == INSTANCE_LEVEL)
				columns = getDefaultColumnsTemplateInstanceLevel();
			else
				columns = "";
		}
		
		return columns += getColumnsTemplateExtensions(_numberOfColumns[_level-DEFAULT_LEVEL],"org.eclipse.hyades.trace.views.adapter.internal.ExecutionStatisticViewer2",false);
	}	
	
	/**
	 * @param i:the level of information to display (package/class or method)
	 */
	public void setLevel(int i) {
		_level = i;
	}	
	
	public int getLevel()
	{
		return _level;
	}

	public String getColumnsPreferencesKey()
	{
		setLevel(getLevel());

		if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
			return PREFERENCE_KEY_PREFIX + "ExecStatsPack60";
		else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
			return PREFERENCE_KEY_PREFIX + "ExecStatsClass60";
		else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
			return PREFERENCE_KEY_PREFIX + "ExecStatsMethod60";
		else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
			return PREFERENCE_KEY_PREFIX + "ExecStatsInstance60";
		
		return "";
	}
	
	public IContentProvider getContentProvider()
	{
		return new ExecutionStatisticContentProvider();
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new ExecutionStatisticLabelProvider(this);
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new ExecutionStatisticSorter();
	}
	
	protected StatisticFilter getFilterInstance()
	{
		return new ExecutionStatisticFilter();
	}
	
	protected String getViewTypeStr()
	{
		return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.EXECUTION_STATS_VIEW2;
	}
	
	protected void doUpdateButtons(boolean isPostponedOperation, int processedOperations) {
		super.doUpdateButtons(isPostponedOperation, processedOperations);
			
		((ExecutionStatisticPage2) getTraceViewerPage()).percentMode().setChecked(isShowPercent());
		if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
			((ExecutionStatisticPage2) getTraceViewerPage()).deltaColumns().setChecked(false);
		else
			((ExecutionStatisticPage2) getTraceViewerPage()).deltaColumns().setChecked(showingDeltaColumns());		
	}		
	
	protected void showPercentUpdate()
	{
		if(isShowPercent())
		{
			_totalCalls = PerftraceUtil.getTotalCalls(_page.getMOFObject());
			if (_totalCalls==0)_totalCalls=1;			
			_maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
			if (_maxTime==0)_maxTime=1;
			_totalPackagesCumulativeTime = PerftraceUtil.getTotalPackagesCumulativeTime(_page.getMOFObject());
			if (_totalPackagesCumulativeTime==0)_totalPackagesCumulativeTime=1;
			_totalBaseTime = PerftraceUtil.getTotalBaseTime(_page.getMOFObject());
			if (_totalBaseTime==0)_totalBaseTime=1;
			_totalCumulativeTime = PerftraceUtil.getTotalCumulativeTime(_page.getMOFObject());
			if(_totalCumulativeTime==0)_totalCumulativeTime=1;
			_totalInheritedBaseTime = PerftraceUtil.getTotalInheritedBaseTime(_page.getMOFObject());
			if(_totalInheritedBaseTime==0)_totalInheritedBaseTime=1;
			_totalInheritedCumulativeTime = PerftraceUtil.getTotalInheritedCumulativeTime(_page.getMOFObject());
			if(_totalInheritedCumulativeTime==0)_totalInheritedCumulativeTime=1;
			_totalPackageCPUtime = getTotalPackageCPUtime();
			if (_totalPackageCPUtime==0) _totalPackageCPUtime=1;
		}		
	}
	
	protected ColumnDisplayInfo getColumnDisplayInfo(ColumnLabelAdapter col, boolean isDeltaColumn)
	{
		if (col ==_packageNameCol)
			return ContextUpdaterHelper.updatePackageName(col, _page.getMOFObject());
		else if (col ==_baseTimeCol)
			return ContextUpdaterHelper.updateBaseTime(col, isDeltaColumn, isShowPercent(), _totalBaseTime);
		else if (col == _avgBaseTimeCol)
			return ContextUpdaterHelper.updateAvgBaseTime(col, isShowPercent(), _totalBaseTime);			
		else if (col == _inheritedBaseTimeCol)
			return ContextUpdaterHelper.updateInhBaseTime(col, isDeltaColumn, isShowPercent(), _totalInheritedBaseTime);
		else if (col == _cumulativeTimeCol)
			return ContextUpdaterHelper.updateCumulTime(col, isDeltaColumn, isShowPercent(), 0, _totalPackagesCumulativeTime);			
		else if (col == _inheritedCumulativeTimeCol)
			return ContextUpdaterHelper.updateInhCumulTime(col, isDeltaColumn, isShowPercent(), _totalInheritedCumulativeTime);
		else if (col == _callsCol)
			return ContextUpdaterHelper.updateCalls(col, isDeltaColumn, isShowPercent(), true, _totalCalls);
		else if (col == _inheritedCallsCol)
			return ContextUpdaterHelper.updateInhCalls(col, isDeltaColumn, isShowPercent(), _totalCalls);			
		else if (col == _CPUTimeCol)
			return ContextUpdaterHelper.updateCPUTime(col, isDeltaColumn, isShowPercent(), _totalPackageCPUtime);
		else
			return super.getColumnDisplayInfo(col, isDeltaColumn);

//		if (   attributeId.equals(IContextAttributes.BASE_TIME)
//				|| attributeId.equals(IContextAttributes.AVERAGE_BASE_TIME) 
//				|| attributeId.equals(IContextAttributes.INHERITED_BASE_TIME)
//				|| attributeId.equals(IContextAttributes.CUMULATIVE_TIME)
//				|| attributeId.equals(IContextAttributes.INHERITED_CUMULATIVE_TIME)
//				|| attributeId.equals(IContextAttributes.CALLS)
//				|| attributeId.equals(IContextAttributes.INHERITED_CALLS)
//				|| attributeId.equals(IContextAttributes.CUMULATIVE_CPU_TIME)
//				|| attributeId.equals(IContextAttributes.NEW_ITEM)
//			)
//		{
//			ContextUpdaterHelper.addSnapshots(col, packageSnapshot, classSnapshot, methodSnapshot);
//		}
	}
	
	protected double getTotalPackageCPUtime()
	{
		double totalPackageCPUtime = 0;
		List packagelist = null; 
		
		packagelist = FilteringUtil.getFilteredPackages(getCurrentFilter(), FilteringUtil.getProcessList(_page.getMOFObject()));
		
		for (int i=0;i<packagelist.size();i++)
		{
			TRCPackage pack = (TRCPackage)(packagelist.get(i));
			totalPackageCPUtime += pack.getTotalCpuTime();
		}
		
		return totalPackageCPUtime;
	}
	
	protected void searchInLevel(Tree tree,
			TreeItem item,
			TRCPackage pack,
			TRCClass cls,
			TRCMethod meth,
			TRCObject trcObj)
	{
		
		super.searchInLevel(tree, item, pack, cls, meth, trcObj);
		
		switch (getLevel()) {
			case INSTANCE_LEVEL :
				if (cls != null)
					searchInClassLevel(
							tree,
							item,
							cls,
							null,
							trcObj);
				break;
		}
	}	
	
	protected void firstTimeUpdate() {
		super.firstTimeUpdate();
		redrawTable();
	}
	
	protected void doUpdate(boolean refresh, boolean isPostponedOperation, int processedOperations)
	{
		_packageNameCol.resetMap();
		_classNameCol.resetMap();
		_methodNameCol.resetMap();	
		_instanceNameCol.resetMap();
		_baseTimeCol.resetMap();
		_avgBaseTimeCol.resetMap();
		_inheritedBaseTimeCol.resetMap();
		_cumulativeTimeCol.resetMap();
		_inheritedCumulativeTimeCol.resetMap();
		_callsCol.resetMap();
		_inheritedCallsCol.resetMap();
		_CPUTimeCol.resetMap();
		
		super.doUpdate(refresh, isPostponedOperation, processedOperations);
	}
	
	public CellLabelProvider getCellLabelProvider(ColumnData colData) {
		return new ExecutionStatisticCellLabelProvider(colData);
	}
}
