/***************************************************************************************************************************************************************************************************************************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others. All rights reserved. This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html $Id: FilteringUtil.java,v 1.3 2010/07/27 02:06:57 jwest Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 **************************************************************************************************************************************************************************************************************************************************************************************/
package org.eclipse.hyades.trace.views.internal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.hierarchy.AbstractTRCProcess;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByElement;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.ResultEntry;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.util.internal.HierarchyXMIQueryEngine;
import org.eclipse.hyades.models.hierarchy.util.internal.QueryUtils;
import org.eclipse.hyades.models.trace.TRCLanguageElement;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TracePackage;
import org.eclipse.hyades.models.trace.impl.TracePackageImpl;
import org.eclipse.hyades.trace.ui.HyadesConstants;
public class FilteringUtil {
	public static final boolean USE_ENGINE_FOR_HAS_CHILDREN = true;
	
	private static final boolean USE_NEW_IN_CONDITIONS = true;
	
	
	public static List<AbstractTRCProcess> getProcessList(EObject object) {
		List<AbstractTRCProcess> processList = new ArrayList<AbstractTRCProcess>();
		
		if(object instanceof TRCMonitor)
		{
			for (int i = 0; i < ((TRCMonitor)object).getNodes().size(); i++)
				processList.addAll(getProcessList((TRCNode)((TRCMonitor)object).getNodes().get(i)));
		}
		if(object instanceof TRCNode)
		{
			for (int i = 0; i < ((TRCNode)object).getProcessProxies().size(); i++)
				processList.addAll(getProcessList((TRCProcessProxy)((TRCNode)object).getProcessProxies().get(i)));
		}
		if(object instanceof TRCProcessProxy)
		{
			for (int i = 0; i < ((TRCProcessProxy)object).getAgentProxies().size(); i++)
				processList.addAll(getProcessList((TRCAgentProxy)((TRCProcessProxy)object).getAgentProxies().get(i)));
		}			
		if(object instanceof TRCAgentProxy)
		{
			if (HyadesConstants.PROFILE_AGENT_TYPE.equals(((TRCAgentProxy)object).getType()))
				processList.add(((TRCAgentProxy)object).getAgent().getProcess());
		}
		if(object instanceof TRCProcess)
		{
			processList.add((TRCProcess)object);
		}
		
		return processList;
	}		
	
	private static List<TRCLanguageElement> getFilteredElements(SimpleSearchQuery filter, EObject[] source, EClass outputEClass) {
		if (source == null)
			return new ArrayList<TRCLanguageElement>();

		if (source.length == 0)
			return new ArrayList<TRCLanguageElement>();
		
		if (filter == null)
			return new ArrayList<TRCLanguageElement>();
		
		filter.getSources().clear();
		
		Resource resource = null;
		
		for (int i = 0; i < source.length; i++)
		{
			if (source[i] instanceof EObject)
			{
				filter.getSources().add(EcoreUtil.getURI((EObject)source[i]).toString());
				
				if (resource == null)
					resource = ((EObject)source[i]).eResource(); 
			}
		}
		
		if (resource == null)
			return new ArrayList<TRCLanguageElement>();
		
		filter.getOutputElements().clear();
		SimpleOperand outputElement = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		outputElement.setType(outputEClass);
		filter.getOutputElements().add(outputElement);
		
		HierarchyXMIQueryEngine engine = new HierarchyXMIQueryEngine();
		QueryResult result = engine.executeQuery(filter, resource.getResourceSet(),Collections.EMPTY_LIST);
		if(result==null)
			return new ArrayList<TRCLanguageElement>();
		
		List<TRCLanguageElement> r = (List<TRCLanguageElement>)((ResultEntry)result.getResultEntries().get(0)).getValue();
		return r;
	}	
	
	private static SimpleSearchQuery getEmptyFilter()
	{
		SimpleSearchQuery filter = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		filter.setName("EmptyFilter");
		LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		logicalExpression.setOperator(LogicalOperators.AND_LITERAL);
		filter.setWhereExpression(logicalExpression);

		return filter;
	}
	
	public static List<TRCPackage> getFilteredPackages(SimpleSearchQuery filter, List source) {
		EObject[] sourceEObjects = new EObject[source.size()];
		source.toArray(sourceEObjects);

		return getFilteredPackages(filter, sourceEObjects);
	}
	
	public static List<TRCPackage> getFilteredPackages(SimpleSearchQuery filter, EObject source) {
		EObject[] sourceEObjects = new EObject[1];
		sourceEObjects[0] = source;
		
		return getFilteredPackages(filter, sourceEObjects);
	}	
	
	public static List<TRCPackage> getFilteredPackages(SimpleSearchQuery filter, EObject[] source) {
		if (filter == null)
			filter = getEmptyFilter();
		else
			filter = EcoreUtil.copy(filter);
		
		addInConditions((LogicalExpression)filter.getWhereExpression(), source, TracePackage.eINSTANCE.getTRCPackage());
		
		List allPackages = getFilteredElements(filter, source, TracePackage.eINSTANCE.getTRCPackage());
		List<TRCPackage> list = new ArrayList<TRCPackage>();
		for(int i=0; i < allPackages.size(); i++)
		{
			TRCPackage pack = (TRCPackage)allPackages.get(i);
			if(pack.getClasses().size() > 0)
			   list.add(pack);
		}
		return list;  
	}
	
	public static List getFilteredClasses(SimpleSearchQuery filter, List source) {
		EObject[] sourceEObjects = new EObject[source.size()];
		source.toArray(sourceEObjects);

		return getFilteredClasses(filter, sourceEObjects);
	}
	
	public static List<TRCLanguageElement> getFilteredClasses(SimpleSearchQuery filter, EObject source) {
		EObject[] sourceEObjects = new EObject[1];
		sourceEObjects[0] = source;
		
		return getFilteredClasses(filter, sourceEObjects);
	}	
		
	public static List<TRCLanguageElement> getFilteredClasses(SimpleSearchQuery filter, EObject[] source) {
		if (filter == null)
			filter = getEmptyFilter();
		else
		{		
			filter = EcoreUtil.copy(filter);			
			replaceOrderByConditionToClasses(filter);
		}
		addInConditions((LogicalExpression)filter.getWhereExpression(), source, TracePackage.eINSTANCE.getTRCClass());

		return getFilteredElements(filter, source, TracePackage.eINSTANCE.getTRCClass());  
	}
	
	public static List<TRCLanguageElement> getFilteredMethods(SimpleSearchQuery filter, List source) {
		EObject[] sourceEObjects = new EObject[source.size()];
		source.toArray(sourceEObjects);

		return getFilteredMethods(filter, sourceEObjects);
	}
	
	public static List<TRCLanguageElement> getFilteredMethods(SimpleSearchQuery filter, EObject source) {
		EObject[] sourceEObjects = new EObject[1];
		sourceEObjects[0] = source;
		
		return getFilteredMethods(filter, sourceEObjects);
	}	
		
	public static List<TRCLanguageElement> getFilteredMethods (SimpleSearchQuery filter, EObject[] source) {
		if (filter == null)
			filter = getEmptyFilter();
		else
		{		
			filter = EcoreUtil.copy(filter);
			replaceOrderByConditionToMethods(filter);
		}
		addInConditions((LogicalExpression)filter.getWhereExpression(), source, TracePackage.eINSTANCE.getTRCMethod());
		
		return getFilteredElements(filter, source, TracePackage.eINSTANCE.getTRCMethod());  
	}
	
	
	public static List<TRCLanguageElement> getFilteredObjects(SimpleSearchQuery filter, List source) {
		EObject[] sourceEObjects = new EObject[source.size()];
		source.toArray(sourceEObjects);

		return getFilteredObjects(filter, sourceEObjects);
	}
	
	public static List<TRCLanguageElement> getFilteredObjects(SimpleSearchQuery filter, EObject source) {
		EObject[] sourceEObjects = new EObject[1];
		sourceEObjects[0] = source;
		
		return getFilteredObjects(filter, sourceEObjects);
	}		
	
	public static List<TRCLanguageElement> getFilteredObjects (SimpleSearchQuery filter, EObject[] source) {
		if (filter == null)
			filter = getEmptyFilter();
		else
		{
			filter = EcoreUtil.copy(filter);			
			replaceOrderByConditionToFullTraceObjects(filter);
		}
		addInConditions((LogicalExpression)filter.getWhereExpression(), source, TracePackage.eINSTANCE.getTRCObject());
		
		Set<EClass> removeClasses = new HashSet<EClass>();
		removeClasses.add(TracePackage.eINSTANCE.getTRCMethod());
//		removeClasses.add(TracePackage.eINSTANCE.getTRCPackage());
		filter = QueryUtils.removeClasses(filter,removeClasses);
		
		return getFilteredElements(filter, source, TracePackage.eINSTANCE.getTRCObject());  
	}
	
	public static List<TRCLanguageElement> getFilteredFullTraceObjects(SimpleSearchQuery filter, List source) {
		EObject[] sourceEObjects = new EObject[source.size()];
		source.toArray(sourceEObjects);

		return getFilteredFullTraceObjects(filter, sourceEObjects);
	}
	
	public static List<TRCLanguageElement> getFilteredFullTraceObjects(SimpleSearchQuery filter, EObject source) {
		EObject[] sourceEObjects = new EObject[1];
		sourceEObjects[0] = source;
		
		return getFilteredFullTraceObjects(filter, sourceEObjects);
	}	
	
	public static List<TRCLanguageElement> getFilteredFullTraceObjects (SimpleSearchQuery filter, EObject[] source) {	
		if (filter == null)
			filter = getEmptyFilter();
		else
		{		
			filter = EcoreUtil.copy(filter);			
			replaceOrderByConditionToFullTraceObjects(filter);
		}
		addInConditions((LogicalExpression)filter.getWhereExpression(), source, TracePackage.eINSTANCE.getTRCFullTraceObject());
		Set<EClass> removeClasses = new HashSet<EClass>();
		removeClasses.add(TracePackage.eINSTANCE.getTRCMethod());
//		removeClasses.add(TracePackage.eINSTANCE.getTRCPackage());
		filter = QueryUtils.removeClasses(filter,removeClasses);
		
		return getFilteredElements(filter, source, TracePackage.eINSTANCE.getTRCFullTraceObject());  
	}

	public static List<TRCLanguageElement> getFilteredHeapObjects(SimpleSearchQuery filter, List source, boolean inClassObjects) {
		EObject[] sourceEObjects = new EObject[source.size()];
		source.toArray(sourceEObjects);

		return getFilteredHeapObjects(filter, sourceEObjects, inClassObjects);
	}
	
	public static List<TRCLanguageElement> getFilteredHeapObjects(SimpleSearchQuery filter, EObject source, boolean inClassObjects) {
		EObject[] sourceEObjects = new EObject[1];
		sourceEObjects[0] = source;
		
		return getFilteredHeapObjects(filter, sourceEObjects, inClassObjects);
	}	
	
	
	public static List<TRCLanguageElement> getFilteredHeapObjects(SimpleSearchQuery filter, List source) {
		EObject[] sourceEObjects = new EObject[source.size()];
		source.toArray(sourceEObjects);

		return getFilteredHeapObjects(filter, sourceEObjects);
	}
	
	public static List<TRCLanguageElement> getFilteredHeapObjects(SimpleSearchQuery filter, EObject source) {
		EObject[] sourceEObjects = new EObject[1];
		sourceEObjects[0] = source;
		
		return getFilteredHeapObjects(filter, sourceEObjects);
	}
	
	public static List<TRCLanguageElement> getFilteredHeapObjects (SimpleSearchQuery filter, EObject[] source) {
		return getFilteredHeapObjects(filter, source, false);
	}
	
	public static List<TRCLanguageElement> getFilteredHeapObjects (SimpleSearchQuery filter, EObject[] source, boolean inClassObjects) {	
		if (filter == null)
			filter = getEmptyFilter();
		else
		{		
			filter = EcoreUtil.copy(filter);
			replaceOrderByConditionToHeapObjects(filter);
		}
		addInConditions((LogicalExpression)filter.getWhereExpression(), source, TracePackage.eINSTANCE.getTRCHeapObject(), inClassObjects, false);
		Set<EClass> removeClasses = new HashSet<EClass>();
		removeClasses.add(TracePackage.eINSTANCE.getTRCMethod());
//		removeClasses.add(TracePackage.eINSTANCE.getTRCPackage());
		filter = QueryUtils.removeClasses(filter,removeClasses);
		
		return getFilteredElements(filter, source, TracePackage.eINSTANCE.getTRCHeapObject());  
	}
	
	public static List<TRCLanguageElement> getFilteredObjectReferences(SimpleSearchQuery filter, List source, boolean owner) {
		EObject[] sourceEObjects = new EObject[source.size()];
		source.toArray(sourceEObjects);

		return getFilteredObjectReferences(filter, sourceEObjects, owner);
	}
	
	public static List<TRCLanguageElement> getFilteredObjectReferences(SimpleSearchQuery filter, EObject source, boolean owner) {
		EObject[] sourceEObjects = new EObject[1];
		sourceEObjects[0] = source;
		
		return getFilteredObjectReferences(filter, sourceEObjects, owner);
	}	
	
	public static List<TRCLanguageElement> getFilteredObjectReferences (SimpleSearchQuery filter, EObject[] source, boolean owner) {	
		if (filter == null)
			filter = getEmptyFilter();
		else
		{		
			filter = EcoreUtil.copy(filter);
			replaceOrderByConditionToHeapObjects(filter);
		}
		addInConditions((LogicalExpression)filter.getWhereExpression(), source, TracePackage.eINSTANCE.getTRCObjectReference(), false, owner);
		Set<EClass> removeClasses = new HashSet<EClass>();
//		getClassesToRemoveAboveClass(TracePackage.eINSTANCE.getTRCMethod(), removeClasses);
//		removeClasses.add(TracePackage.eINSTANCE.getTRCPackage());
		filter = QueryUtils.removeClasses(filter,removeClasses);
		
		return getFilteredElements(filter, source, TracePackage.eINSTANCE.getTRCObjectReference());  
	}		
	
	private static void addClassInPackageCondition(LogicalExpression logicalExpression) {
		addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCClass(), TracePackage.eINSTANCE.getTRCPackage_Classes());
	}	

	private static void addMethodInClassCondition(LogicalExpression logicalExpression) {
		addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCMethod(), TracePackage.eINSTANCE.getTRCClass_Methods());
	}	
	
	private static void addObjectInClassCondition(LogicalExpression logicalExpression) {
		addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCObject(), TracePackage.eINSTANCE.getTRCClass_Objects());
	}
	
	private static void addFullTraceObjectInClassCondition(LogicalExpression logicalExpression) {
		addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCFullTraceObject(), TracePackage.eINSTANCE.getTRCClass_Objects());
	}
	
	private static void addHeapObjectInClassCondition(LogicalExpression logicalExpression, boolean inClassObjects, boolean fromObject) {
		if (inClassObjects)
			addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCHeapObject(), TracePackage.eINSTANCE.getTRCClass_ClassObjects());			
		else
			if(fromObject)
				addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCClass(), TracePackage.eINSTANCE.getTRCObject_IsA());
			else
				addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCHeapObject(), TracePackage.eINSTANCE.getTRCClass_Objects());
			
	}
	
	private static void addObjectReferenceInHeapObjectCondition(LogicalExpression logicalExpression, boolean owner) {
		if (owner)
			addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCObjectReference(), TracePackage.eINSTANCE.getTRCHeapObject_RefOwner());
		else
			addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCObjectReference(), TracePackage.eINSTANCE.getTRCHeapObject_RefTarget());
	}			
	
	private static void addInCondition(LogicalExpression logicalExpression, EClass left, EReference right)
	{
		BinaryExpression binaryExpression = ExtensionsFactory.eINSTANCE.createBinaryExpression();

		SimpleOperand lhs = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		lhs.setType(left);
		
		binaryExpression.setOperator(RelationalOperators.IN_LITERAL);

		SimpleOperand rhs = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		rhs.setFeature(right);
		
		binaryExpression.setLeftOperand(lhs);
		binaryExpression.getRightOperands().add(rhs);
		
		logicalExpression.getArguments().add(binaryExpression);
	}
	
	private static void addInConditions(LogicalExpression logicalExpression, EObject[] source, EClass outputEClass) {
		addInConditions(logicalExpression, source, outputEClass, false, false);
	}
	
	private static boolean mapHasHigherLevel(Map cp, EClass outputEClass)
	{
		EClass parent = getParent(outputEClass);
		
		if (parent == null)
			return false;
		
		if (cp.containsKey(parent))
			return true;
		else
			return mapHasHigherLevel(cp, parent);
	}
	
	private static void addInConditionsUpTree(List<EClass> outputToAdd, Map cp, EClass outputEClass) {
		if (mapHasHigherLevel(cp, outputEClass))
		{
			addToList(outputToAdd, outputEClass);
			addInConditionsUpTree(outputToAdd, cp, getParent(outputEClass));
		}
	}

	
	private static boolean isInParent(EClass parent, EClass child)
	{
		if (parent.equals(child))
			return true;

		child = getParent(child);
		
		if (child == null)
			return false;
		
		return isInParent(parent, child);
	}

	private static void addInConditionsUpTreeToClass(List<EClass> outputToAdd, Map cp, EClass outputEClass, EClass toEClass, boolean addOutputEClass) {
		
		if (outputEClass == null || (toEClass != null && isInParent(outputEClass, toEClass)))
			return;
		
		if (addOutputEClass || cp.containsKey(outputEClass))
		{
			addToList(outputToAdd, outputEClass);
			addInConditionsUpTreeToClass(outputToAdd, cp, getParent(outputEClass), toEClass, true);
			return;
		}

		addInConditionsUpTreeToClass(outputToAdd, cp, getParent(outputEClass), toEClass, false);
	}	

	private static void addToList(List<EClass> list, EClass toAdd)
	{
		if (!list.contains(toAdd))
		{
			if (toAdd.equals(TracePackage.eINSTANCE.getTRCFullTraceObject())
				|| toAdd.equals(TracePackage.eINSTANCE.getTRCHeapObject()))
			{
				if (!list.contains(TracePackage.eINSTANCE.getTRCObject()))
					list.add(toAdd);
			}
			else if (toAdd.equals(TracePackage.eINSTANCE.getTRCObject()))
			{
				if (list.contains(TracePackage.eINSTANCE.getTRCFullTraceObject()))
					list.remove(TracePackage.eINSTANCE.getTRCFullTraceObject());
				
				if (list.contains(TracePackage.eINSTANCE.getTRCHeapObject()))
					list.remove(TracePackage.eINSTANCE.getTRCHeapObject());
				
				list.add(toAdd);
			}
			else
			{
				list.add(toAdd);
			}
		}
	}

	
	private static void addInConditionsFromLeaves(List<EClass> outputToAdd, LogicalExpression logicalExpression, EClass outputEClass, boolean inClassObjects, boolean owner)
	{
		Map cp = QueryUtils.getClassPredicatesIndex(logicalExpression.getSearchQuery());

		addInConditionsUpTreeToClass(outputToAdd, cp, TracePackage.eINSTANCE.getTRCMethod(), outputEClass, false);
		addInConditionsUpTreeToClass(outputToAdd, cp, TracePackage.eINSTANCE.getTRCObject(), outputEClass, false);
		addInConditionsUpTreeToClass(outputToAdd, cp, TracePackage.eINSTANCE.getTRCFullTraceObject(), outputEClass, false);
		addInConditionsUpTreeToClass(outputToAdd, cp, TracePackage.eINSTANCE.getTRCObjectReference(), outputEClass, false);		

		for (int i = 0; i < outputToAdd.size(); i++)
		{
			addInConditionForOutput(logicalExpression, outputToAdd.get(i), inClassObjects, owner, outputEClass.equals(TracePackage.eINSTANCE.getTRCObjectReference()));
		}		
	}
	
	private static void addInConditionsNew(LogicalExpression logicalExpression, EObject[] source, EClass outputEClass, boolean inClassObjects, boolean owner) {

		Map cp = QueryUtils.getClassPredicatesIndex(logicalExpression.getSearchQuery());		
		
		List<EClass> outputToAdd = new ArrayList<EClass>();
		
		addInConditionsFromLeaves(outputToAdd, logicalExpression, outputEClass, inClassObjects, owner);
		
		int j = outputToAdd.size();
		
		if (cp.containsKey(TracePackage.eINSTANCE.getTRCThread()))
		{
			if (outputEClass.equals(TracePackage.eINSTANCE.getTRCFullTraceObject()))
				addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCThread(), TracePackage.eINSTANCE.getTRCTraceObject_Thread());
			else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCClass())
					|| outputEClass.equals(TracePackage.eINSTANCE.getTRCPackage())
					|| outputEClass.equals(TracePackage.eINSTANCE.getTRCMethod()))
				addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCThread(), TracePackage.eINSTANCE.getTRCClass_LoadedBy());
			
			if (outputEClass.equals(TracePackage.eINSTANCE.getTRCPackage()))
				addToList(outputToAdd, TracePackage.eINSTANCE.getTRCClass());
		}
		
		if (outputEClass.equals(TracePackage.eINSTANCE.getTRCFullTraceObject())
			|| outputEClass.equals(TracePackage.eINSTANCE.getTRCHeapObject())
			|| outputEClass.equals(TracePackage.eINSTANCE.getTRCObject())
			|| outputEClass.equals(TracePackage.eINSTANCE.getTRCObjectReference()))
		{
			addToList(outputToAdd, outputEClass);
		}
			
		
		addInConditionsUpTree(outputToAdd, cp, outputEClass);

		for (int i = j; i < outputToAdd.size(); i++)
		{
			addInConditionForOutput(logicalExpression, outputToAdd.get(i), inClassObjects, owner, outputEClass.equals(TracePackage.eINSTANCE.getTRCObjectReference()));
		}
	}	
	
	
	private static EClass getParent(EClass outputEClass)
	{
		if (outputEClass.equals(TracePackage.eINSTANCE.getTRCMethod()))
			return TracePackage.eINSTANCE.getTRCClass();
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCFullTraceObject()))
			return TracePackage.eINSTANCE.getTRCClass();
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCHeapObject()))
			return TracePackage.eINSTANCE.getTRCClass();
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCObject()))
			return TracePackage.eINSTANCE.getTRCClass();
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCObjectReference()))
			return TracePackage.eINSTANCE.getTRCHeapObject();						
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCClass()))
			return TracePackage.eINSTANCE.getTRCPackage();
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCPackage()))
			return TracePackage.eINSTANCE.getTRCProcess();
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCProcess()))
			return HierarchyPackage.eINSTANCE.getTRCAgent();
		else if (outputEClass.equals(HierarchyPackage.eINSTANCE.getTRCAgent()))
			return HierarchyPackage.eINSTANCE.getTRCAgentProxy();		
		else if (outputEClass.equals(HierarchyPackage.eINSTANCE.getTRCAgentProxy()))
			return HierarchyPackage.eINSTANCE.getTRCProcessProxy();
		else if (outputEClass.equals(HierarchyPackage.eINSTANCE.getTRCProcessProxy()))
			return HierarchyPackage.eINSTANCE.getTRCNode();		
		
		return null;
	}
	
	private static void addInConditionForOutput(LogicalExpression logicalExpression, EClass outputEClass, boolean inClassObjects, boolean owner, boolean fromObj)
	{
		if (outputEClass.equals(TracePackage.eINSTANCE.getTRCMethod()))
			addMethodInClassCondition(logicalExpression);
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCFullTraceObject()))
			addFullTraceObjectInClassCondition(logicalExpression);
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCHeapObject()))
			addHeapObjectInClassCondition(logicalExpression, inClassObjects, fromObj);
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCObject()))
			addObjectInClassCondition(logicalExpression);
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCObjectReference()))
			addObjectReferenceInHeapObjectCondition(logicalExpression, owner);
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCClass()))
			addClassInPackageCondition(logicalExpression);
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCPackage()))
			addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCPackage(), TracePackage.eINSTANCE.getTRCProcess_Packages());
		else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCProcess()))
			addInCondition(logicalExpression, TracePackage.eINSTANCE.getTRCProcess(), HierarchyPackage.eINSTANCE.getTRCAgent_Process());
		else if (outputEClass.equals(HierarchyPackage.eINSTANCE.getTRCAgent()))
			addInCondition(logicalExpression, HierarchyPackage.eINSTANCE.getTRCAgentProxy(),HierarchyPackage.eINSTANCE.getTRCAgent_AgentProxy());
		else if (outputEClass.equals(HierarchyPackage.eINSTANCE.getTRCAgentProxy()))
			addInCondition(logicalExpression, HierarchyPackage.eINSTANCE.getTRCProcessProxy(),HierarchyPackage.eINSTANCE.getTRCAgentProxy_ProcessProxy());
		else if (outputEClass.equals(HierarchyPackage.eINSTANCE.getTRCProcessProxy()))
			addInCondition(logicalExpression, HierarchyPackage.eINSTANCE.getTRCNode(),HierarchyPackage.eINSTANCE.getTRCProcessProxy_Node());
	}
	
	
	private static void addInConditions(LogicalExpression logicalExpression, EObject[] source, EClass outputEClass, boolean inClassObjects, boolean owner) {
		
		if (USE_NEW_IN_CONDITIONS)
			addInConditionsNew(logicalExpression, source, outputEClass, inClassObjects, owner);
		else
		{
		
			Map cp = QueryUtils.getClassPredicatesIndex(logicalExpression.getSearchQuery());
			if (outputEClass.equals(TracePackage.eINSTANCE.getTRCMethod()))
			{
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCPackage()))
				{
					addClassInPackageCondition(logicalExpression);
					addMethodInClassCondition(logicalExpression);
				}
				else if(cp.containsKey(TracePackage.eINSTANCE.getTRCClass()))
					addMethodInClassCondition(logicalExpression);
			}
			else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCFullTraceObject()))
			{
	//			addObjectInClassCondition(logicalExpression);	
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCMethod()))
					addMethodInClassCondition(logicalExpression);
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCPackage()))
					addClassInPackageCondition(logicalExpression);
				addFullTraceObjectInClassCondition(logicalExpression);
	//			addHeapObjectInClassCondition(logicalExpression);	
			}
			else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCHeapObject()))
			{
	//			addObjectInClassCondition(logicalExpression);	
	//			addFullTraceObjectInClassCondition(logicalExpression);
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCMethod()))
					addMethodInClassCondition(logicalExpression);
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCPackage()))
					addClassInPackageCondition(logicalExpression);
				addHeapObjectInClassCondition(logicalExpression, inClassObjects, false);	
			}
			else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCObjectReference()))
			{
	//			addObjectInClassCondition(logicalExpression);	
	//			addFullTraceObjectInClassCondition(logicalExpression);
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCMethod()))
					addMethodInClassCondition(logicalExpression);
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCPackage()))
					addClassInPackageCondition(logicalExpression);
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCClass()))
					addHeapObjectInClassCondition(logicalExpression, inClassObjects, false);	
				
				addObjectReferenceInHeapObjectCondition(logicalExpression, owner);
			}				
			else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCObject()))
			{
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCMethod()))
					addMethodInClassCondition(logicalExpression);
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCPackage()))
					addClassInPackageCondition(logicalExpression);
				addObjectInClassCondition(logicalExpression);	
	//			addFullTraceObjectInClassCondition(logicalExpression);
	//			addHeapObjectInClassCondition(logicalExpression);	
			}
			else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCClass()))
			{
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCMethod()))
					addMethodInClassCondition(logicalExpression);
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCPackage()))
					addClassInPackageCondition(logicalExpression);
	//			addObjectInClassCondition(logicalExpression);	
	//			addFullTraceObjectInClassCondition(logicalExpression);
	//			addHeapObjectInClassCondition(logicalExpression);	
			}
			else if (outputEClass.equals(TracePackage.eINSTANCE.getTRCPackage()))
			{
				if(cp.containsKey(TracePackage.eINSTANCE.getTRCMethod()))
				{
					addClassInPackageCondition(logicalExpression);
					addMethodInClassCondition(logicalExpression);
				}
				else if(cp.containsKey(TracePackage.eINSTANCE.getTRCClass()))
					addClassInPackageCondition(logicalExpression);
	//			addObjectInClassCondition(logicalExpression);	
	//			addFullTraceObjectInClassCondition(logicalExpression);
	//			addHeapObjectInClassCondition(logicalExpression);				
			}
		}
	}	

	private static void replaceOrderByConditionToClasses(SimpleSearchQuery filter)
	{
		if (filter.getOrderByExpresions().size() == 1)
		{
			OrderByElement orderByElement = (OrderByElement)filter.getOrderByExpresions().get(0);
			SimpleOperand operand = (SimpleOperand)orderByElement.getOperand();			
			if (TracePackageImpl.init().getTRCPackage_BaseTime().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCClass_BaseTime());
			else if (TracePackageImpl.init().getTRCPackage_CumulativeTime().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCClass_CumulativeTime());
			else if (TracePackageImpl.init().getTRCPackage_Calls().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCClass_Calls());
			else if (TracePackageImpl.init().getTRCPackage_TotalSize().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCClass_TotalSize());
			else if (TracePackageImpl.init().getTRCPackage_CollectedSize().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCClass_CollectedSize());
			else if (TracePackageImpl.init().getTRCPackage_TotalInstances().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCClass_TotalInstances());
			else if (TracePackageImpl.init().getTRCPackage_CollectedInstances().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCClass_CollectedInstances());			
			else
				filter.getOrderByExpresions().clear();			
		}
	}

	private static void replaceOrderByConditionToMethods(SimpleSearchQuery filter)
	{
		if (filter.getOrderByExpresions().size() == 1)
		{
			OrderByElement orderByElement = (OrderByElement)filter.getOrderByExpresions().get(0);
			SimpleOperand operand = (SimpleOperand)orderByElement.getOperand();			
			if (TracePackageImpl.init().getTRCPackage_BaseTime().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCMethod_BaseTime());
			else if (TracePackageImpl.init().getTRCPackage_CumulativeTime().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCMethod_CumulativeTime());
			else if (TracePackageImpl.init().getTRCPackage_Calls().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCMethod_Calls());
			else
				filter.getOrderByExpresions().clear();
		}
	}	
	
	private static void replaceOrderByConditionToFullTraceObjects(SimpleSearchQuery filter)
	{
		if (filter.getOrderByExpresions().size() == 1)
		{
			OrderByElement orderByElement = (OrderByElement)filter.getOrderByExpresions().get(0);
			SimpleOperand operand = (SimpleOperand)orderByElement.getOperand();			
			if (TracePackageImpl.init().getTRCPackage_BaseTime().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCFullTraceObject_BaseTime());
			else if (TracePackageImpl.init().getTRCPackage_CumulativeTime().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCFullTraceObject_CumulativeTime());
			else if (TracePackageImpl.init().getTRCPackage_Calls().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCFullTraceObject_Calls());
			else
				replaceOrderByConditionToObjects(filter);
		}
	}
	
	private static void replaceOrderByConditionToHeapObjects(SimpleSearchQuery filter)
	{
		replaceOrderByConditionToObjects(filter);
	}	

	private static void replaceOrderByConditionToObjects(SimpleSearchQuery filter)
	{
		if (filter.getOrderByExpresions().size() == 1)
		{
			OrderByElement orderByElement = (OrderByElement)filter.getOrderByExpresions().get(0);
			SimpleOperand operand = (SimpleOperand)orderByElement.getOperand();			
			if (TracePackageImpl.init().getTRCPackage_TotalSize().equals(operand.getFeature()))
				operand.setFeature(TracePackageImpl.init().getTRCObject_Size());
			else
				filter.getOrderByExpresions().clear();
		}
	}	
}