/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Mapping.java,v 1.1 2005/07/22 14:49:34 hleung Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;


import java.io.*;

import org.eclipse.swt.graphics.*;

// Mapping:  Represents the mapping of data values to visual attributes.
//
//           A Mapping ties together a data classification and a rendering
//           scheme (for now, just a ColorScheme).  Most of the work of a
//           mapping is delegated to the Classification, which implements the
//           bucketing of the data.  The Classification is visible to the caller
//           of Mapping, and it has the methods for setting up the
//           classification scheme.
//
//           Mappings fall into two broad categories: mappings of
//           quantitative information, and mappings of categorical information.
//           We are modelling this distinction informally in Mapping and
//           ColorScheme, but a little more formally in Classification, where
//           the interface is different based on the type of data.  We are
//           leaving it to the caller to mix and match classifications and
//           color schemes appropriately when building a mapping.
//
public class Mapping
	implements Serializable
{
	/**
	 * 
	 */
	private static final long serialVersionUID = 3545799887989061432L;
	// --- Public constants ---
	// Describe the shape we will be rendering in the legend.
	// Eventually this needs to be more general.
	public static final int SHAPE_RECTANGLE = 0;
	public static final int SHAPE_LINE = 1;

	// --- Internal representation ---
	private String          _title;          // Optional title for legend
	private String          _description;    // Optional user comment
	private ColorScheme     _colorScheme;    // The set of colors
	private Classification  _classification; // The current classification strategy
	private int             _shape;          // SHAPE_RECTANGLE, etc.
   
	private boolean         _isLongValue=true;

	// --- Constructors ---
	public Mapping(Classification classification, ColorScheme colorScheme,
				   String title, int shape) {
		setClassification(classification);
		setColorScheme(colorScheme);
		setTitle(title);
		setShape(shape);
	}
	// --- Accessor/updater methods ---
	public Classification classification() {
		return _classification;
	}
	// Get the color associated with an object
	// Note:  assumes a quantitative classification
	public Color color (long value) {
	   QuantitativeClassification classification =
								  (QuantitativeClassification)_classification;
	   int bucket = classification.bucket(value);
	   //System.out.println("Mapping color: " + value + ", " + bucket + ", " + classification.numBuckets());
	   return _colorScheme.color(bucket, classification.numBuckets());
	}
	// --- Public methods for using the Mapping ---
	// Get the color associated with an object
	// Note:  assumes a categorical classification
	public Color color (Object value) {
		CategoricalClassification classification =
			(CategoricalClassification)_classification;

		int bucket = classification.bucket(value);
		return _colorScheme.color(bucket, classification.numBuckets());
	}
	public ColorScheme colorScheme() {
		return _colorScheme;
	}
	// Optional comment to be displayed in the mapping editor
	public String description() {
		return _description;
	}
/**
 * Insert the method's description here.
 * Creation date: (4/1/2001 1:32:13 PM)
 * @return boolean
 */
public boolean isLongValue() {
	return _isLongValue;
}
/**
 * Insert the method's description here.
 * Creation date: (4/1/2001 1:30:53 PM)
 * @param value boolean
 */
public void isLongValue(boolean value)
{
	_isLongValue = value;
}
	// --- Loading/saving the mapping ---
	// Load from file: creates a new Mapping object
	public static Mapping load (String filename) {
		return null;
	}
	// Save to file: saves a Mapping object to a file
	public void save (String filename) {
	}
	public void setClassification(Classification c) {
		_classification = c;
	}
	public void setColorScheme(ColorScheme colorScheme) {
		_colorScheme = colorScheme;
	}
	public void setDescription(String description) {
		_description = description;
	}
	public void setShape(int shape) {
		_shape = shape;
	}
	public void setTitle (String title) {
		_title = title;
	}
	// The type of shape we are rendering: SHAPE_RECTANGLE, SHAPE_LINE
	public int shape() {
		return _shape;
	}
	// The title which appears in the legend
	public String title() {
		return _title;
	}
}
