/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MemoryStatisticView.java,v 1.37 2010/04/20 19:56:14 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCObjectValue;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl.TRCClassSnapshot;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl.TRCPackageSnapshot;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.adapter.internal.DynamicFilter;
import org.eclipse.hyades.trace.views.adapter.internal.MemoryStatisticPage;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ActiveSizeColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ClassNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.CollectedInstancesColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnDisplayInfo;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.InstanceNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.LiveInstancesColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.NewItemColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.PackageNameColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.TotalInstancesColumnLabel;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.TotalSizeColumnLabel;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.ColumnExtensionValue;
import org.eclipse.hyades.trace.views.util.internal.IColumnExtension;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.hyades.ui.provisional.context.IContextAttributes;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

/*
 * This view contains memory statistics data for
 * the package level, class level, and instance level.
 */
public class MemoryStatisticView extends MultiLevelStatisticView{

	private static final int _numberOfColumns[] = { 20, 20, 20 };
	
	private static int _level = DEFAULT_LEVEL;	
	
	public static final int INSTANCE_LEVEL = 3;		

	private TRCPackageSnapshot packageSnapshot = new TRCPackageImpl.TRCPackageSnapshot();
	private TRCClassSnapshot classSnapshot = new TRCClassImpl.TRCClassSnapshot();		
	
	protected ColumnLabelAdapter _packageNameCol;
	protected ColumnLabelAdapter _classNameCol;
	protected ColumnLabelAdapter _instanceNameCol;	
	
	protected ColumnLabelAdapter _newItemCol;
	protected ColumnLabelAdapter _totalInstancesCol;
	protected ColumnLabelAdapter _liveInstancesCol;
	protected ColumnLabelAdapter _collectedInstancesCol;
	protected ColumnLabelAdapter _totalSizeCol;
	protected ColumnLabelAdapter _activeSizeCol;
	
	public MemoryStatisticView(Composite parent, TraceViewerPage page){
		super(true, parent, page);
		
		createColumnsLabelProviders();
	}
	
	protected String getContextHelpId()
	{
		return TraceUIPlugin.getPluginId()+".stvw0001";
	}
	
	public void createColumnsLabelProviders()
	{
		_packageNameCol = new PackageNameColumnLabel();
		_classNameCol = new ClassNameColumnLabel();
		_instanceNameCol = new InstanceNameColumnLabel();		
		
		_newItemCol = new NewItemColumnLabel();
		ContextUpdaterHelper.addSnapshots(_newItemCol, packageSnapshot, classSnapshot, null);
		_totalInstancesCol = new TotalInstancesColumnLabel();
		ContextUpdaterHelper.addSnapshots(_totalInstancesCol, packageSnapshot, classSnapshot, null);
		_liveInstancesCol = new LiveInstancesColumnLabel();
		ContextUpdaterHelper.addSnapshots(_liveInstancesCol, packageSnapshot, classSnapshot, null);		
		_collectedInstancesCol = new CollectedInstancesColumnLabel();
		ContextUpdaterHelper.addSnapshots(_collectedInstancesCol, packageSnapshot, classSnapshot, null);
		_totalSizeCol = new TotalSizeColumnLabel();
		ContextUpdaterHelper.addSnapshots(_totalSizeCol, packageSnapshot, classSnapshot, null);		
		_activeSizeCol = new ActiveSizeColumnLabel();
		ContextUpdaterHelper.addSnapshots(_activeSizeCol, packageSnapshot, classSnapshot, null);		
	}
	
	protected int getClassPosForLevel(int pos)
	{
        if (getLevel()== MemoryStatisticView.PACKAGE_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 3;
           		case 3:
           			return 4;
           		case 4:
           			return 5;
           		case 5:
           			return 6;
           		case 6:
           			return 7;
			}
        }
        else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
        {
           	switch (pos)
			{
           		case 3:
           			return 5;
           		case 4:
           			return 6;
           		case 5:
           			return 7;
           		case 6:
           			return -1;
           		case 7:
           			return -1;           			
			}        	
        }
        return pos;
	}
	
	
	public class MemoryStatisticContentProvider extends MultiLevelStatisticContentProvider
	{
		
		public Object[] getChildren(Object element)
		{
			tmpList.clear();
			
			if(element instanceof TRCPackage)
			{
				return FilteringUtil.getFilteredClasses(getCurrentFilter(), (TRCPackage)element).toArray();
			}
			else if (element instanceof TRCClass && getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
			{
				Object[] elements = FilteringUtil.getFilteredFullTraceObjects(getCurrentFilter(), (TRCClass)element).toArray();
				
				if (elements.length > 0)
					return elements;
				else
					return FilteringUtil.getFilteredHeapObjects(getCurrentFilter(), (TRCClass)element).toArray();
			}
			else
				return tmpList.toArray(); 	
		}
		
		public Object getParent(Object element) {
			return null;
		}		

		public boolean hasChildren(Object element) {
			
			if (FilteringUtil.USE_ENGINE_FOR_HAS_CHILDREN)
				return getChildren(element).length > 0;
			else if (element instanceof TRCPackage)
				return true;
			else if (element instanceof TRCClass
					&& getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
				return true;
			else
				return false;
		}
		
		/**
		 * return the list of elements to display in the table tree
		 */
		public Object[] getElements(Object inputElement) {
			SimpleSearchQuery currentFilter = getCurrentFilter();
			currentFilter = updateDynamicFilters(currentFilter, _page.getMOFObject());
			if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
			{
				return FilteringUtil.getFilteredPackages(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();					
			}
			else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
			{
				return FilteringUtil.getFilteredClasses(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();
			}
			else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
			{
				return FilteringUtil.getFilteredClasses(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();
			}
			else
			{
				return new ArrayList().toArray();
			}
		}

		/**
		 * updates filters which depend on contents, if any
		 */
		private SimpleSearchQuery updateDynamicFilters(SimpleSearchQuery filter, EObject object) {
			if (!DynamicFilter.containsDynamicFilters(filter)) {
				return filter;
			}
			return DynamicFilter.updateDynamicFilters(filter, object);
		}
	}
	
	public class MemoryStatisticCellLabelProvider extends StatisticCellLabelProvider {
 		public MemoryStatisticCellLabelProvider(ColumnData colData) {
			super(colData);
 		}
		
		public void update(ViewerCell cell) {
			visualIndex = cell.getVisualIndex();
			cell.setText(((MemoryStatisticLabelProvider)getTableLabelProvider()).getColumnText(cell.getElement(),visualIndex));
			cell.setImage(((MemoryStatisticLabelProvider)getTableLabelProvider()).getColumnImage(cell.getElement(),visualIndex));
		}
 	}
	
	
	public class MemoryStatisticLabelProvider extends MultiLevelStatisticLabelProvider
	{
		public MemoryStatisticLabelProvider(StatisticView viewer) {
			super(viewer);
		}
		
		protected String getColumnTextPackageLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCPackage)	
			{
				switch(pos)
				{
					case 0://package name
						return getElementColumnText(obj, _packageNameCol, info.isDeltaColumn());
					case 1: //new item column
						return getElementColumnText(obj, _newItemCol, info.isDeltaColumn());					
					case 2: //total inst
						return getElementColumnText(obj, _totalInstancesCol, info.isDeltaColumn());
					case 3: //live inst
						return getElementColumnText(obj, _liveInstancesCol, info.isDeltaColumn());
					case 4://collected
						return getElementColumnText(obj, _collectedInstancesCol, info.isDeltaColumn());					
					case 5: //total size
						return getElementColumnText(obj, _totalSizeCol, info.isDeltaColumn());	
					case 6: //active size
						return getElementColumnText(obj, _activeSizeCol, info.isDeltaColumn());	
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt = (ColumnExtensionValue) _listOfColumExtension.get(pos - _numberOfColumns[_level-DEFAULT_LEVEL]);
					return ((IColumnExtension)(elt.getClassOfColumnExtension())).getColumnText(obj,elt.getInitialColumnPos());
				}
			}
			else
			{
				return getColumnTextClassLevel(obj, col);
			}
            							
			return "";
		}
		
		protected String getColumnTextClassLevel(Object obj, int col){
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = getClassPosForLevel(info.getColumnData().getInitalPos());			
            
			if (obj instanceof TRCClass) {
				
				switch (pos) {
					case 0 : //class name
						return getElementColumnText(obj, _classNameCol, info.isDeltaColumn());
					case 1: //new item column
						return getElementColumnText(obj, _newItemCol, info.isDeltaColumn());
					case 2 :
						return getElementColumnText(obj, _packageNameCol, info.isDeltaColumn());
					case 3: //total inst
						return getElementColumnText(obj, _totalInstancesCol, info.isDeltaColumn());
					case 4: //live inst
						return getElementColumnText(obj, _liveInstancesCol, info.isDeltaColumn());
					case 5://collected
						return getElementColumnText(obj, _collectedInstancesCol, info.isDeltaColumn());					
					case 6: //total size
						return getElementColumnText(obj, _totalSizeCol, info.isDeltaColumn());	
					case 7: //active size
						return getElementColumnText(obj, _activeSizeCol, info.isDeltaColumn());	
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt = (ColumnExtensionValue) _listOfColumExtension.get(pos - _numberOfColumns[_level-DEFAULT_LEVEL]);
					return ((IColumnExtension)(elt.getClassOfColumnExtension())).getColumnText(obj,elt.getInitialColumnPos());
				}
			}

			// Allow the String value to be displayed for TRCObjectValue
			if (obj instanceof TRCObjectValue) {
				// class name
				if(pos == 0)
					return ((TRCObjectValue)obj).getStringValue();
			}

			return "";
		}

		protected String getColumnTextInstanceLevel(Object obj, int col){
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCClass)	
			{
				return getColumnTextClassLevel(obj, col);
			}
			else
			{
				switch(pos)
				{
					case 0://class name
						return getElementColumnText(obj, _instanceNameCol, info.isDeltaColumn());	
					case 1:
						return getElementColumnText(obj, _newItemCol, info.isDeltaColumn());	
					case 2: //package name
						return getElementColumnText(obj, _packageNameCol, info.isDeltaColumn());	
					case 3://collected
						return getElementColumnText(obj, _collectedInstancesCol, info.isDeltaColumn());
					case 4: //total size
						return getElementColumnText(obj, _totalSizeCol, info.isDeltaColumn());
					case 5: //active size
						return getElementColumnText(obj, _activeSizeCol, info.isDeltaColumn());
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt = (ColumnExtensionValue) _listOfColumExtension.get(pos - _numberOfColumns[_level-DEFAULT_LEVEL]);
					return ((IColumnExtension)(elt.getClassOfColumnExtension())).getColumnText(obj,elt.getInitialColumnPos());
				}
			}
			return "";
		}
		
		public String getColumnText(Object obj, int col) {
			
			if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
				return getColumnTextPackageLevel(obj, col);
			else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
				return getColumnTextClassLevel(obj, col);
			else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
				return getColumnTextInstanceLevel(obj, col);
			else
				return "";
		}
		
		protected Image getColumnImagePackageLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			
			if(obj instanceof TRCPackage)	
			{
				switch(pos)
				{
					case 0://package name
						return getElementColumnImage(obj, _packageNameCol, info.isDeltaColumn());
					case 1: //new item column
						return getElementColumnImage(obj, _newItemCol, info.isDeltaColumn());					
					case 2: //total inst
						return getElementColumnImage(obj, _totalInstancesCol, info.isDeltaColumn());
					case 3: //live inst
						return getElementColumnImage(obj, _liveInstancesCol, info.isDeltaColumn());
					case 4://collected
						return getElementColumnImage(obj, _collectedInstancesCol, info.isDeltaColumn());					
					case 5: //total size
						return getElementColumnImage(obj, _totalSizeCol, info.isDeltaColumn());	
					case 6: //active size
						return getElementColumnImage(obj, _activeSizeCol, info.isDeltaColumn());	
				}				  				  				 				   				  
			}
			else
			{
				return getColumnImageClassLevel(obj, col);
			}
                  
			return null;
		}
		
		protected Image getColumnImageClassLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = getClassPosForLevel(info.getColumnData().getInitalPos());			
			
			if(info.isDeltaColumn())
			   return null;
			
			if(obj instanceof TRCClass)	
			{
				switch(pos)
				{
					case 0 : //class name
						return getElementColumnImage(obj, _classNameCol, info.isDeltaColumn());
					case 1: //new item column
						return getElementColumnImage(obj, _newItemCol, info.isDeltaColumn());
					case 2 :
						return getElementColumnImage(obj, _packageNameCol, info.isDeltaColumn());
					case 3: //total inst
						return getElementColumnImage(obj, _totalInstancesCol, info.isDeltaColumn());
					case 4: //live inst
						return getElementColumnImage(obj, _liveInstancesCol, info.isDeltaColumn());
					case 5://collected
						return getElementColumnImage(obj, _collectedInstancesCol, info.isDeltaColumn());					
					case 6: //total size
						return getElementColumnImage(obj, _totalSizeCol, info.isDeltaColumn());	
					case 7: //active size
						return getElementColumnImage(obj, _activeSizeCol, info.isDeltaColumn());	
				}				  				  				 				   				  
			}
			else
			{
				return null;
			}
      							
			return null;		
		}
		
		protected Image getColumnImageInstanceLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			   
			if(obj instanceof TRCClass)	
			{
				return getColumnImageClassLevel(obj, col);
			}
			else
			{
				switch(pos)
				{
					case 0://class name
						return getElementColumnImage(obj, _instanceNameCol, info.isDeltaColumn());	
					case 1:
						return getElementColumnImage(obj, _newItemCol, info.isDeltaColumn());	
					case 2: //package name
						return getElementColumnImage(obj, _packageNameCol, info.isDeltaColumn());
					case 3://collected
						return getElementColumnImage(obj, _collectedInstancesCol, info.isDeltaColumn());
					case 4: //total size
						return getElementColumnImage(obj, _totalSizeCol, info.isDeltaColumn());
					case 5: //active size
						return getElementColumnImage(obj, _activeSizeCol, info.isDeltaColumn());
				}	
			}
			return null;
		}
		
		public Image getColumnImage(Object obj, int col) {
			if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
				return getColumnImagePackageLevel(obj, col);
			else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
				return getColumnImageClassLevel(obj, col);
			else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
				return getColumnImageInstanceLevel(obj, col);			
			else
				return null;
		}
	}

	public class MemoryStatisticSorter extends MultiLevelStatisticSorter{
		
		protected int comparePackageLevel(Viewer viewer, Object e1, Object e2) {
			
 			if(e1 instanceof TRCPackage && e2 instanceof TRCPackage)	
			{
				switch(_pos)
				{
					case 0://package name
						return _sortSequence * compareElements(e1, e2, _packageNameCol, _info.isDeltaColumn());
					case 1 : // new items
						return _sortSequence * compareElements(e1, e2, _newItemCol, _info.isDeltaColumn());						
					case 2: //total inst
						return _sortSequence * compareElements(e1, e2, _totalInstancesCol, _info.isDeltaColumn());						
					case 3: //live inst
						return _sortSequence * compareElements(e1, e2, _liveInstancesCol, _info.isDeltaColumn());
					case 4://collected
						return _sortSequence * compareElements(e1, e2, _collectedInstancesCol, _info.isDeltaColumn());						
					case 5: //total size
						return _sortSequence * compareElements(e1, e2, _totalSizeCol, _info.isDeltaColumn());
					case 6: //active size
						return _sortSequence * compareElements(e1, e2, _activeSizeCol, _info.isDeltaColumn());						
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt =
						(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns[_level-DEFAULT_LEVEL]);
					return (int) _sortSequence * ((IColumnExtension)
								(elt.getClassOfColumnExtension())).compare(
									elt.getInitialColumnPos(), e1, e2);
				}
			}
 			else
 			{
 				return compareClassLevel(viewer, e1, e2);
 			}
 			return 0;
		}

		protected int compareClassLevel(Viewer viewer, Object e1, Object e2) {
			
			int pos = getClassPosForLevel(_pos);
			
			if (e1 instanceof TRCClass && e2 instanceof TRCClass) {
				switch (pos) {
					case 0 : //class name
						return _sortSequence * compareElements(e1, e2, _classNameCol, _info.isDeltaColumn());
					case 1 : // new items
						return _sortSequence * compareElements(e1, e2, _newItemCol, _info.isDeltaColumn());						
					case 2 :	//package name
						return _sortSequence * compareElements(e1, e2, _packageNameCol, _info.isDeltaColumn());						
					case 3: //total inst
						return _sortSequence * compareElements(e1, e2, _totalInstancesCol, _info.isDeltaColumn());						
					case 4: //live inst
						return _sortSequence * compareElements(e1, e2, _liveInstancesCol, _info.isDeltaColumn());
					case 5://collected
						return _sortSequence * compareElements(e1, e2, _collectedInstancesCol, _info.isDeltaColumn());						
					case 6: //total size
						return _sortSequence * compareElements(e1, e2, _totalSizeCol, _info.isDeltaColumn());
					case 7: //active size
						return _sortSequence * compareElements(e1, e2, _activeSizeCol, _info.isDeltaColumn());										
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt =
						(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns[_level-DEFAULT_LEVEL]);
					return (int) _sortSequence * ((IColumnExtension)
								(elt.getClassOfColumnExtension())).compare(
									elt.getInitialColumnPos(), e1, e2);
				}
			}
			return 0;
		}
		
		protected int compareInstanceLevel(Viewer viewer, Object e1, Object e2)
		{
 			if(e1 instanceof TRCClass)	
			{
 				return compareClassLevel(viewer, e1, e2);
			}
			else
			{				 
				switch(_pos)
				{
					case 0://instance name
						return _sortSequence * compareElements(e1, e2, _instanceNameCol, _info.isDeltaColumn());
					case 1: //new item
						return _sortSequence * compareElements(e1, e2, _newItemCol, _info.isDeltaColumn());
					case 2: //package
						return _sortSequence * compareElements(e1, e2, _packageNameCol, _info.isDeltaColumn());
					case 3://collected
						return _sortSequence * compareElements(e1, e2, _collectedInstancesCol, _info.isDeltaColumn());					
					case 4: //total size
						return _sortSequence * compareElements(e1, e2, _totalSizeCol, _info.isDeltaColumn());
					case 5: //active size
						return _sortSequence * compareElements(e1, e2, _activeSizeCol, _info.isDeltaColumn());
				}
				// if column extended
				if (_listOfColumExtension.size() != 0) {
					ColumnExtensionValue elt =
						(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns[_level-DEFAULT_LEVEL]);
					return (int) _sortSequence * ((IColumnExtension)
								(elt.getClassOfColumnExtension())).compare(
									elt.getInitialColumnPos(), e1, e2);
				}
			}
			return 0;
		}
		
		public int compare(Viewer viewer, Object e1, Object e2) {
			
			if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
				return comparePackageLevel(viewer, e1, e2);
			else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
				return compareClassLevel(viewer, e1, e2);
			else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
				return compareInstanceLevel(viewer, e1, e2);
			
			return 0;
		}
	}

	public class MemoryStatisticFilter extends StatisticFilter{
		
		public MemoryStatisticFilter() {
			super();
		}
		
		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			String compareText = "";

			switch (getLevel()) {
				case PACKAGE_LEVEL : //Package level
					if (!(element instanceof TRCPackage))
						return true;
					if (((TRCPackage) element).getName().length() == 0)
						compareText =
							TraceUIMessages._87;
					else
						compareText =
							((TRCPackage) element).getName();
					break;
				case CLASS_LEVEL : //class level
					if (!(element instanceof TRCClass))
						return true;
					compareText = ((TRCClass) element).getName();
					break;
				case INSTANCE_LEVEL : //instance level
					if (!(element instanceof TRCClass))
						return true;
					compareText = ((TRCClass) element).getName();
					break;					
			}

			if (_noPattern)
				return true;

			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {

					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
	}
	
	public String getDefaultColumnsTemplate()
	{	
		String columns = super.getDefaultColumnsTemplate();
		
		if (columns == null)
		{
			if (getLevel() == INSTANCE_LEVEL)
				columns = getDefaultColumnsTemplateInstanceLevel();
			else
				columns = "";
		}
	
		return columns += getColumnsTemplateExtensions(_numberOfColumns[_level-DEFAULT_LEVEL],"org.eclipse.hyades.trace.views.adapter.internal.MemoryStatisticViewer",false);
	}		
	
	protected String getDefaultColumnsTemplatePackageLevel()
	{
		return IContextAttributes.PACKAGE_NAME	+ ":0:" 
      		+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
   		+  ""			+ ":1:"
		    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":20,"		
		+ IContextAttributes.PACKAGE_TOTAL_INST	+ ":2:" 
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
		+ IContextAttributes.PACKAGE_LIVE_INST		+ ":3:"
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
		+ IContextAttributes.PACKAGE_COLLECTED_INST			+ ":4:"
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"			    
		+ IContextAttributes.PACKAGE_TOTAL_SIZE				+ ":5:"
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"						    
		+ IContextAttributes.PACKAGE_ACTIVE_SIZE	 	+ ":6:"
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100";
	}

	protected String getDefaultColumnsTemplateClassLevel()
	{
		return IContextAttributes.CLASS_NAME
			+ ":0:"
			+ String.valueOf(
				ColumnData.NONDELETABLE
					| ColumnData.IS_VISIBLE
					| ColumnData.NONMOVABLE)
			+ ":200,"
			+ ""
			+ ":1:"
		    + String.valueOf(ColumnData.NONDELETABLE
		    					| ColumnData.IS_VISIBLE
								| ColumnData.NONMOVABLE
								| ColumnData.NONRESIZABLE
								| ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS)
			+ ":20,"					
			+ IContextAttributes.PACKAGE_NAME
			+ ":2:"
			+ String.valueOf(ColumnData.IS_VISIBLE)
			+ ":left:100,"
			+ IContextAttributes.CLASS_TOTAL_INST
			+ ":3:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100,"
			+ IContextAttributes.CLASS_LIVE_INST
			+ ":4:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100,"
			+ IContextAttributes.CLASS_COLLECTED_INST
			+ ":5:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100,"
			+ IContextAttributes.CLASS_TOTAL_SIZE
			+ ":6:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100,"
			+ IContextAttributes.CLASS_ACTIVE_SIZE
			+ ":7:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100";			
	}
	
	protected String getDefaultColumnsTemplateInstanceLevel()
	{
		return IContextAttributes.CLASS_NAME		+ ":0:" 
      		+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
      	+ ""				+ ":1:"
			+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":20,"		
		+ IContextAttributes.PACKAGE_NAME 			+ ":2:"
   			+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:100,"
		+ IContextAttributes.OBJECT_COLLECTED				+ ":3:"
		    + String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"			
		+ IContextAttributes.OBJECT_SIZE		 			+ ":4:"  
			+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"				
		+ IContextAttributes.OBJECT_ACTIVE_SIZE		 	+ ":5:"  
			+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100";	
	}
	
	public boolean isEmptyUpdate()
	{
		return PerftraceUtil.getTotalSize(_page.getMOFObject()) <= 0;
	}
	
	public IContentProvider getContentProvider()
	{
		return new MemoryStatisticContentProvider();
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new MemoryStatisticLabelProvider(this);
	}
	
	public CellLabelProvider getCellLabelProvider(ColumnData colData) {
		return new MemoryStatisticCellLabelProvider(colData);
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new MemoryStatisticSorter();
	}
	
	protected StatisticFilter getFilterInstance()
	{
		return new MemoryStatisticFilter();
	}	
	
	public String getColumnsPreferencesKey()
	{
		setLevel(getLevel());		

		if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
			return PREFERENCE_KEY_PREFIX + "MemStatsPack60";
		else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
			return PREFERENCE_KEY_PREFIX + "MemStatsClass60";
		else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
			return PREFERENCE_KEY_PREFIX + "MemStatsInstance60";	
		
		return "";
	}
	
	protected String getViewTypeStr()
	{
		return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.MEMORY_STATS_VIEW;
	}	
	
	/**
	 * @param i:the level of information to display (package/class or method)
	 */
	public void setLevel(int i) {
		_level = i;
	}	
	
	public int getLevel()
	{
		return _level;
	}	
	
	protected void doUpdateButtons(boolean isPostponedOperation, int processedOperations) {
		super.doUpdateButtons(isPostponedOperation, processedOperations);
			
		TraceViewerPage page = getTraceViewerPage();
		if (!(page instanceof MemoryStatisticPage)) {
			return;
		}
		MemoryStatisticPage memoryPage = (MemoryStatisticPage) page;
		memoryPage.percentMode().setChecked(isShowPercent());
		if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL) {
			memoryPage.deltaColumns().setChecked(false);
		} else {
			memoryPage.deltaColumns().setChecked(showingDeltaColumns());
		}
	}	
	
	protected void showPercentUpdate()
	{
		if(isShowPercent())
		{
			_totalInst = PerftraceUtil.getTotalInstances(_page.getMOFObject());
			if (_totalInst==0)_totalInst=1;			
			_activeInst = PerftraceUtil.getActiveInstances(_page.getMOFObject());
			if (_activeInst==0)_activeInst=1;
			_totalSize = PerftraceUtil.getTotalSize(_page.getMOFObject());
			if (_totalSize==0)_totalSize=1;
			_activeSize = PerftraceUtil.getActiveSize(_page.getMOFObject());
			if (_activeSize==0)_activeSize=1;
			// added for bugzilla 147893
			_collectedInst = PerftraceUtil.getTotalCollectedInstances(_page.getMOFObject());
			if (_collectedInst==0)_collectedInst=1;
		}		
	}	
	
	protected ColumnDisplayInfo getColumnDisplayInfo(ColumnLabelAdapter col, boolean isDeltaColumn)
	{
		if (col == _packageNameCol)
			return ContextUpdaterHelper.updatePackageName(col, _page.getMOFObject());
		else if (col == _totalInstancesCol)
			return ContextUpdaterHelper.updateTotalInstances(col, isDeltaColumn, isShowPercent(), _totalInst);
		else if (col == _liveInstancesCol)
			return ContextUpdaterHelper.updateLiveInstances(col, isDeltaColumn, isShowPercent(), _activeInst);			
		else if (col == _collectedInstancesCol)
			//value changed from '1' to totalCollectedInstances so the correct numebr is being divided
			// by to get the percentage ( bugzilla_#147893)
			return ContextUpdaterHelper.updateCollectedInst(col, isDeltaColumn, isShowPercent(), _collectedInst);
		else if (col == _totalSizeCol)
			return ContextUpdaterHelper.updateTotalSize(col, isDeltaColumn, isShowPercent(), _totalSize);			
		else if (col == _activeSizeCol)
			return ContextUpdaterHelper.updateActiveSize(col, isDeltaColumn, isShowPercent(), _activeSize);
		else
			return super.getColumnDisplayInfo(col, isDeltaColumn);

//		if (   attributeId.equals(IContextAttributes.TOTAL_INSTANCES)
//				|| attributeId.equals(IContextAttributes.LIVE_INSTANCES) 
//				|| attributeId.equals(IContextAttributes.COLLECTED)
//				|| attributeId.equals(IContextAttributes.SIZE)
//				|| attributeId.equals(IContextAttributes.ACTIVE_SIZE)
//				|| attributeId.equals(IContextAttributes.NEW_ITEM)				
//			)
//		{
//			ContextUpdaterHelper.addSnapshots(col, packageSnapshot, classSnapshot, null);
//		}
		
	}
	
	
	
	protected void searchInLevel(Tree tree,
			TreeItem item,
			TRCPackage pack,
			TRCClass cls,
			TRCMethod meth,
			TRCObject trcObj)
	{
		
		super.searchInLevel(tree, item, pack, cls, meth, trcObj);
		
		switch (getLevel()) {
			case INSTANCE_LEVEL :
				if (cls != null)
					searchInClassLevel(
							tree,
							item,
							cls,
							null,
							trcObj);
				break;
		}
	}		
	
	protected void doUpdate(boolean refresh, boolean isPostponedOperation, int processedOperations)
	{
		_packageNameCol.resetMap();
		_classNameCol.resetMap();
		_instanceNameCol.resetMap();
		_totalInstancesCol.resetMap();
		_liveInstancesCol.resetMap();
		_collectedInstancesCol.resetMap();
		_totalSizeCol.resetMap();
		_activeSizeCol.resetMap();
		
		super.doUpdate(refresh, isPostponedOperation, processedOperations);

	}
}
