/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MethodDetails.java,v 1.15 2010/06/30 20:45:51 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;


import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.trace.ui.ITraceSelection;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.views.adapter.internal.IContextViewer;
import org.eclipse.hyades.trace.views.adapter.internal.MethodDetailsTab;
import org.eclipse.hyades.trace.views.util.internal.OpenSource;
import org.eclipse.hyades.ui.provisional.context.IContextLanguage;
import org.eclipse.hyades.ui.util.HyadesFormToolkit;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.ManagedForm;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;

public class MethodDetails
{
	protected ContextInfoContainer _contextInfo;
	
	protected MethodDetailsTab _tabItem;	
	
	protected MethodDetailsStatistic _parentStatistic;
	protected MethodDetailsStatistic _childrenStatistic;
	protected MethodDetailsStatistic _methodStatistic;
	protected Label _parentLabel;
	protected Label _childrenLabel;
	
	protected Separator fSeparator;
	protected String tmpString;

	protected Composite viewContainer;
	protected ManagedForm mform;
	protected ScrolledForm _form;	
	protected TRCMethod _currentSelection = null;
	protected HyadesFormToolkit _toolkit;
	protected Label _info;
	
	private TableWrapData _visibleLayoutData;
	private TableWrapData _nonVisibleLayoutData;

    /** Informs the method details page that the currently selected method of the page should be updated */
    boolean _updateCurrentSelection = false;
	
	
    public MethodDetails(Composite parent, MethodDetailsTab tabItem) {
    	
    	_toolkit = new HyadesFormToolkit(parent.getDisplay());
    	fSeparator = new Separator();
    	_tabItem = tabItem;		
    	
		_form = _toolkit.createScrolledForm(parent);
		mform = new ManagedForm(_toolkit, _form);
		
		BusyIndicator.showWhile(parent.getDisplay(), new Runnable() {
			public void run() {
				createFormContent(mform);
			}
		});
	}

	public Control getControl() {
		return _form;
	}
	
	public void refresh() {
		update(true);
	}
	
	/*
	 * This method handles a selection event
	 */
	public void selectionChanged() {
	}
	
	
	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {
		
		if(_parentStatistic.getTree().isFocusControl())
		{
			_parentStatistic.menuAboutToShow(menu);
		}
		else if(_childrenStatistic.getTree().isFocusControl())
		{
			_childrenStatistic.menuAboutToShow(menu);
		}
		else if(_methodStatistic.getTree().isFocusControl())
		{
			_methodStatistic.menuAboutToShow(menu);
		}
		
	}

	public void updateButtons() {
		
		_tabItem.percentMode().setChecked(_methodStatistic.isShowPercent());		
	}

	public void updateModelSelection() {
		
		ISelection selection = null;
		if(_parentStatistic.getTree().isFocusControl())
		{
			selection = _parentStatistic.getTreeViewer().getSelection();
		}
		else if(_childrenStatistic.getTree().isFocusControl())
		{
			selection = _childrenStatistic.getTreeViewer().getSelection();
		}
		else if(_methodStatistic.getTree().isFocusControl())
		{
			selection = _methodStatistic.getTreeViewer().getSelection();
		}
		
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			
			_methodStatistic.notifyViewSelectionChanged(this,sel);
		}
		
		
	}
	
	public void dispose() {
		
		_currentSelection = null;
		
		_parentStatistic.dispose();
		_childrenStatistic.dispose();
		_methodStatistic.dispose();
		mform.dispose();
		
	}

  protected void postUpdateEvents()
	{
	}		
	
	public boolean isEmpty()
	{
		return false;		
	}
	
    protected String getViewTypeStr()
    {
    	return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.METHOD_INVOCATION_STATS_VIEW;
    }
    
    /** Inform the method details page that the currently selected method of the page should be updated */
	public void updateCurrentSelection() {
		_updateCurrentSelection = true;
	}
	
	public void update(boolean redraw) {
		
		IContextViewer contextViewer = _tabItem.getContextViewer();
		if (contextViewer == null) {
			return;
		}
		
		contextViewer.getContextSelectionMenuListener().setMOFObject(_tabItem.getMOFObject());
		if (_contextInfo!=null) 
			_contextInfo.setMOFObject(_tabItem.getMOFObject());
		
		if(redraw)
		{
			ITraceSelection selModel = UIPlugin.getDefault().getSelectionModel(_tabItem.getMOFObject());
			Object selection = selModel.getFirstElement();

			// On data update, we only change the currently selected method if we have been informed that a method update
			// should occur. Changing the currently selected method should only occur on double clicks of one of the elements in the table.
			if(_updateCurrentSelection) {
				_updateCurrentSelection = false;
			
				_currentSelection = null;
				if (selection instanceof TRCMethod)
					_currentSelection = (TRCMethod)selection;
				else if(selection instanceof TRCMethodInvocation)
					_currentSelection = ((TRCMethodInvocation)selection).getMethod();
			}
			
			_childrenStatistic.updateUI(_childrenStatistic.getColumnDataList());			
			_parentStatistic.updateUI(_parentStatistic.getColumnDataList());
			_methodStatistic.updateUI(_methodStatistic.getColumnDataList());	
			
			if (contextViewer.contextChanged())
				contextViewer.updateContext();
			
			_tabItem.updateTitle();
			
			if(_currentSelection == null)
			{
				_info.setText(TraceUIMessages._171);
				return;
			}
			
			if(_currentSelection.getInvocations().size() == 0)
				_info.setText(TraceUIMessages._177);
			else
				_info.setText("");
		}
		
	}
	
	/**
	 * Insert the method's description here.
	 * Creation date: (8/21/2001 6:31:49 PM)
	 */
	public void openSource() {
	
		OpenSource.openSource(UIPlugin.getDefault().getSelectionModel(_tabItem.getMOFObject()).getFirstElement());
	
	}

	/**
	 * @return Returns the _calleeStatistic.
	 */
	public MethodDetailsStatistic getChildrenStatistic() {
		return _childrenStatistic;
	}
	/**
	 * @return Returns the _parentStatistic.
	 */
	public MethodDetailsStatistic getParentStatistic() {
		return _parentStatistic;
	}
	
	public void initialize() {
		_tabItem.updateTitle();
	} 
	
	/**
	 * @return Returns the _methodStatistic.
	 */
	public MethodDetailsStatistic getMethodStatistic() {
		return _methodStatistic;
	}
	/**
	 * @return Returns the _currentSelection.
	 */
	public TRCMethod getCurrentSelection() {
		return _currentSelection;
	}

	/**
	 * 
	 */
	public void runShowPercentAction() {
		getMethodStatistic().getShowPercentAction().run();	
		getParentStatistic().getShowPercentAction().run();		
		getChildrenStatistic().getShowPercentAction().run();		
		
	}
	
	private void fillBody(IManagedForm managedForm) {
		Composite body = managedForm.getForm().getBody();
		TableWrapLayout layout = new TableWrapLayout();
		
		layout.bottomMargin = 5;
		layout.topMargin = 5;
		layout.leftMargin = 5;
		layout.rightMargin = 5;
		
		layout.numColumns = 1;
		layout.verticalSpacing = 10;
		layout.horizontalSpacing = 10;
		body.setLayout(layout);
		
		_info = _toolkit.createLabel(body, "", SWT.WRAP);
		_info.setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_RED));
		
		// sections
		createContextSection(managedForm, body);
		createMehodSection(managedForm, body);
		createParentSection(managedForm, body);
		createChildrenSection(managedForm, body);
	}
	
	public void updateStatusContext(IContextLanguage language)
	{
		if (_contextInfo!=null) _contextInfo.updateStatusContext(language);
	}

	protected void createFormContent(IManagedForm managedForm) {
		_form = managedForm.getForm();

		_form.setText(TraceUIPlugin.getString("7")); //$NON-NLS-1$
		fillBody(managedForm);
		_form.redraw();
		managedForm.refresh();
	}
	
	/**
	 * @param managedForm
	 * @param body
	 * @param toolkit
	 */
	private void createParentSection(IManagedForm managedForm, Composite parent) {
		
		Section section = _toolkit.createSection(parent, Section.TITLE_BAR);
		section.setText(TraceUIMessages._174); //$NON-NLS-1$
		section.marginWidth = 3;
		section.marginHeight = 3;
		
   	    _parentStatistic = new MethodDetailsStatistic (section, _tabItem, false);
    	_parentStatistic.setMethodDetailsType(MethodDetailsStatistic.PARENT_DETAILS);
    	_parentStatistic.setContainer(this);
    	_parentStatistic.createContent(section);
    	section.setClient(_parentStatistic.getControl());

        GridLayout layout = new GridLayout();
        TableWrapData data = new TableWrapData();
        data.grabVertical = true;
        data.grabHorizontal = true;
        data.heightHint = 200;
        section.setLayout(layout);
        section.setLayoutData(data);
    	
	}

	/**
	 * @param managedForm
	 * @param body
	 * @param toolkit
	 */
	private void createChildrenSection(IManagedForm managedForm, Composite parent) {
		
		Section section = _toolkit.createSection(parent, Section.TITLE_BAR);
		section.setText(TraceUIMessages._175); //$NON-NLS-1$

    	_childrenStatistic = new MethodDetailsStatistic(section, _tabItem, false);
    	_childrenStatistic.setMethodDetailsType(MethodDetailsStatistic.CHILDREN_DETAILS);
    	_childrenStatistic.setContainer(this);
    	_childrenStatistic.createContent(section); 
        section.setClient(_childrenStatistic.getControl());
        
        GridLayout layout = new GridLayout();
        TableWrapData data = new TableWrapData();
        data.grabVertical = true;
        data.grabHorizontal = true;        
         data.heightHint = 250;
         section.setLayout(layout);
         section.setLayoutData(data);
		
	}

	/**
	 * @param managedForm
	 * @param body
	 * @param toolkit
	 */
	private void createMehodSection(IManagedForm managedForm, Composite parent) {
		
		Section section = _toolkit.createSection(parent, Section.TITLE_BAR);
		section.setText(TraceUIMessages._176); //$NON-NLS-1$					
		
    	_methodStatistic = new MethodDetailsStatistic(section, _tabItem, false);
    	_methodStatistic.setMethodDetailsType(MethodDetailsStatistic.METHOD_DETAILS);
    	_methodStatistic.setContainer(this);
    	_methodStatistic.createContent(section);
    	section.setClient(_methodStatistic.getControl());
		
	}
	
	/**
	 * @param managedForm
	 * @param body
	 * @param toolkit
	 */
	private void createContextSection(IManagedForm managedForm, Composite parent) {
		
		IContextViewer contextViewer = _tabItem.getContextViewer();
		if (contextViewer != null)
		{
			final Section section = _toolkit.createSection(parent, Section.TITLE_BAR);
			section.setText(TraceUIMessages._222); //$NON-NLS-1$	
			_contextInfo = new ContextInfoContainer();
			_contextInfo.setViewer(contextViewer);
			section.setClient(_contextInfo.createControl(section));
			_contextInfo.addContextInfoContainerListener(
				new IContextInfoContainerListener()
				{
					public void visibilityChanged(boolean isVisible)
					{
						section.setLayoutData(isVisible?getContextSectionVisibleLayoutData():getContextSectionNonVisibleLayoutData());
						section.setVisible(isVisible);
						section.getParent().layout(true, true);
					}
				});
		}
	}
	
	
	
	private TableWrapData getContextSectionVisibleLayoutData()
	{
		if (_visibleLayoutData == null)
		{
			_visibleLayoutData = new TableWrapData();
		}
		
		return _visibleLayoutData;
	}
	
	private TableWrapData getContextSectionNonVisibleLayoutData()
	{
		if (_nonVisibleLayoutData == null)
		{
			_nonVisibleLayoutData = new TableWrapData();
			((TableWrapData)_nonVisibleLayoutData).heightHint = 0;
		}
		
		return _nonVisibleLayoutData;
	}
	
}