/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MethodDetailsStatistic.java,v 1.32 2008/06/04 23:56:01 jkubasta Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.hyades.models.trace.TRCAggregatedMethodInvocation;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.adapter.internal.MethodDetailsTab;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.MethodCallDetails;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.hyades.ui.provisional.context.IContextAttributes;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;

/*
 * This view contains execution statistics data.  That is time related data for
 * the package level, class level, method level, and instance level.
 */
public class MethodDetailsStatistic extends ExecutionStatisticView{
	
	protected MethodDetailsTab _tabItem;

	public class MethodStatisticFilter extends StatisticFilter{
		
		public MethodStatisticFilter() {
			super();
		}
		
		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			String compareText = "";
	
			if (element instanceof TRCMethod)
				compareText = ((TRCMethod) element).getName();
			else if(element instanceof MethodCallDetails)
				compareText = ((MethodCallDetails)element).getMethod().getName();
	
			if (_noPattern)
				return true;
	
			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {
	
					String str1 = (String) _textList.get(k);
	
					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}
	
					compareText = compareText.substring(index + str1.length());
	
				}
			}
			return flag;
		}
	}

	public static final int PARENT_DETAILS = 0;
	public static final int METHOD_DETAILS = 1;	
	public static final int CHILDREN_DETAILS = 2;
	
	private static int _level = METHOD_LEVEL;
	
	protected int _methodDetailsType = METHOD_DETAILS;
	protected MethodDetails _container;
	
	public MethodDetailsStatistic(Composite parent, MethodDetailsTab tabItem, boolean initializeContent){		
		super(parent, tabItem.getPage(), initializeContent);
		_tabItem = tabItem;
	}
	
	protected String getContextHelpId()
	{
		return TraceUIPlugin.getPluginId()+".mthid0000";
	}
	
	public MethodDetailsStatistic(Composite parent, MethodDetailsTab tabItem){		
		this(parent, tabItem, true);
	}
	
	public String getDefaultColumnsTemplate()
	{
		StringBuffer buffer = new StringBuffer();
		
		switch(_methodDetailsType)
		{
			case CHILDREN_DETAILS:
				buffer.append(TraceUIMessages._173);
				break;
			case PARENT_DETAILS:
				buffer.append(TraceUIMessages._172);
				break;
			case METHOD_DETAILS:
			    buffer.append(IContextAttributes.METHOD_CALLS);
				break;
		}
		
		buffer.append(":0:")
		.append(String.valueOf(ColumnData.IS_VISIBLE)).append(":right:80,")
		
		.append(IContextAttributes.METHOD_NAME).append(":1:")
		.append(String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE)).append(":left:200,")
	    .append("").append(":2:")
	    .append(String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS)).append(":20,")		
	    .append(IContextAttributes.CLASS_NAME).append(":3:")
		.append(String.valueOf(ColumnData.IS_VISIBLE)).append(":left:100,")
	    .append(IContextAttributes.PACKAGE_NAME).append(":4:")
		.append(String.valueOf(ColumnData.IS_VISIBLE)).append(":left:100,")					
	    .append(IContextAttributes.METHOD_BASE_TIME).append(":5:") 
        .append(String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)).append(":right:100,")
	    .append(IContextAttributes.METHOD_AVG_BASE_TIME).append(":6:") 
		.append(String.valueOf(ColumnData.IS_VISIBLE )).append(":right:100,")
	    .append(IContextAttributes.METHOD_CUMULATIVE_TIME).append(":7:");
		
		if(getMethodDetailsType() != METHOD_DETAILS)
		{
			buffer.append(String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)).append(":right:100,")
			
		    .append(IContextAttributes.METHOD_CALLS).append(":8:")
			.append(String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)).append(":right:55,")
			
			.append(IContextAttributes.METHOD_CPU_TIME).append(":9:")
			.append(String.valueOf(ColumnData.IS_VISIBLE)).append(":right:100");
			
		}
		else
		{
			buffer.append(String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)).append(":right:100,")
			.append(IContextAttributes.METHOD_CPU_TIME).append(":8:")
			.append(String.valueOf(ColumnData.IS_VISIBLE)).append(":right:100");
			
		}
		
		return buffer.toString();
	}
	
	public String getColumnsPreferencesKey()
	{
		String key ="";
		switch(_methodDetailsType)
		{
			case CHILDREN_DETAILS:
				key = "MDet_Child49";
				break;
			case PARENT_DETAILS:
				key = "MDet_Parent49";
				break;
			case METHOD_DETAILS:
				key = "MDet_Method49";
				break;
		}
		
		return key;
	}
	
	protected void doUpdateButtons(boolean isPostponedOperation, int processedOperations) {
		_tabItem.percentMode().setChecked(isShowPercent());
	}	
	/**
	 * @return Returns the _methodDetailsType.
	 */
	public int getMethodDetailsType() {
		return _methodDetailsType;
	}
	/**
	 * @param detailsType The _methodDetailsType to set.
	 */
	public void setMethodDetailsType(int detailsType) {
		_methodDetailsType = detailsType;
	}
	
	public class MethodStatisticContentProvider extends MultiLevelStatisticContentProvider
	{
		public Object[] getChildren(Object element)
		{
			tmpList.clear();			
			return tmpList.toArray(); 	
		}
		
		public Object getParent(Object element) {
			
			return null;
		}		

		public boolean hasChildren(Object element) {
			return false;
		}
		
		/**
		 * return the list of elements to display in the table tree
		 */
		public Object[] getElements(Object inputElement) {
			tmpList.clear();
			TRCMethod method = _tabItem.getCurrentSelection();

			if(method != null)
			{
				switch(_methodDetailsType)
				{
					case CHILDREN_DETAILS:
						return getChildrenMethods(method);
					case PARENT_DETAILS:
						return getParentMethods(method);
					case METHOD_DETAILS:
						tmpList.add(method);
						break;
				}
			}
			
			return tmpList.toArray();
									
		}
		
	      private HashMap tmpMap = new HashMap();
		
	      protected Object[] getObjectArray() {
	          Object[] ret = tmpList.toArray();

	          tmpList.clear();

	          return ret;
	      }
	      
		   protected Object[] getChildrenMethods(Object object)
		   {
		       tmpList.clear();
		       tmpMap.clear();
		       
		       TRCMethod method = null;
		       if(object instanceof TRCMethod)
		       	  method = (TRCMethod) object;
		       else if(object instanceof TRCMethodInvocation)
		       	  method = ((TRCMethodInvocation) object).getMethod();
		       
		       if(method == null)
		       	 return getObjectArray();
		       
		       Iterator iterator = method.getInvocations().iterator();
		       while(iterator.hasNext())
		       {
		       	  TRCMethodInvocation inv = (TRCMethodInvocation) iterator.next();
		       	  
		       	  Iterator children = inv.getInvokes().iterator();
		       	  while(children.hasNext())
		       	  {
		       		 TRCMethodInvocation child_inv = (TRCMethodInvocation)children.next();
		       	  	 TRCMethod m = child_inv.getMethod();
		       	  	 
				  	 int count = 1;
				  	 if (child_inv instanceof TRCAggregatedMethodInvocation) {
				  		 count = ((TRCAggregatedMethodInvocation) child_inv).getCount();
				  	 }		     

		    	  	 Object value = tmpMap.get(m);
		    	  	 if(value != null) {	
		        	  	 tmpMap.put(m, new Integer(((Integer)value).intValue() + count));
		    	  	 }
		    	  	 else {
		    	  	 	 tmpMap.put(m, new Integer(count));
		    	  	 }
		       	  }
		       }
		       
		       Iterator methods = tmpMap.keySet().iterator();
		       while(methods.hasNext())
		       {
		       	    TRCMethod m = (TRCMethod)methods.next();
		       		tmpList.add(new MethodCallDetails(m, ((Integer)tmpMap.get(m)).intValue()));
		       }
		       
		       tmpMap.clear();
		       
		       return getObjectArray();
		   }
		   
		   protected Object[] getParentMethods(Object object)
		   {
		       tmpList.clear();
		       tmpMap.clear();
		       
		       TRCMethod method = null;
		       if(object instanceof TRCMethod)
		       	  method = (TRCMethod) object;
		       else if(object instanceof TRCMethodInvocation)
		       	  method = ((TRCMethodInvocation) object).getMethod();
		       
		       if(method == null)
		       	 return getObjectArray();
		       
			   int methodCalls = method.getCalls();
			   int parents = 0;
			   
		       Iterator iterator = method.getInvocations().iterator();
		       while(iterator.hasNext())
		       {
					 TRCMethodInvocation inv = (TRCMethodInvocation) iterator.next();
					
					 if(inv.getInvokedBy() == null)
					   continue;
					   
					 TRCMethod m = inv.getInvokedBy().getMethod();
					 
					 int count = 1;
					 if (inv instanceof TRCAggregatedMethodInvocation) {
						 count = ((TRCAggregatedMethodInvocation) inv).getCount();
					 }
					 
					 parents += count;					 
					 
					 Object value = tmpMap.get(m);
					 if(value != null)   
					 {	
					  	 tmpMap.put(m, new Integer(((Integer)value).intValue() + count));
					 }
					 else {
					 	 tmpMap.put(m, new Integer(count));
					 }
		       }
		         
		       Iterator methods = tmpMap.keySet().iterator();
		       while(methods.hasNext())
		       {
		       	    TRCMethod m = (TRCMethod)methods.next();
		       		tmpList.add(new MethodCallDetails(m, ((Integer)tmpMap.get(m)).intValue()));
		       }
		       
		       /* if the method invocations count isn't equal to 
		       sum of parent methods invocations calling the method 
		       than we add notification  message */
		       if (methodCalls > parents) tmpList.add(new MethodCallDetails(null, methodCalls-parents));
		    	   
		       tmpMap.clear();
		       return getObjectArray();
		   }
		
	}
	
	public class MethodStatisticCellLabelProvider extends StatisticCellLabelProvider {
 		public MethodStatisticCellLabelProvider(ColumnData colData) {
			super(colData);
 		}
		
		public void update(ViewerCell cell) {
			visualIndex = cell.getVisualIndex();
			cell.setText(((MethodStatisticLabelProvider)getTableLabelProvider()).getColumnText(cell.getElement(),visualIndex));
			cell.setImage(((MethodStatisticLabelProvider)getTableLabelProvider()).getColumnImage(cell.getElement(),visualIndex));
		}
 	}
	
	public class MethodStatisticLabelProvider extends MultiLevelStatisticLabelProvider
	{
		public MethodStatisticLabelProvider(StatisticView viewer) {
			super(viewer);
		}
		
		public String getColumnText(Object obj, int col) {
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = getMethodPosForLevel(info.getColumnData().getInitalPos());			
			
            TRCMethod m = null;
            if(obj instanceof TRCMethod)
			   m = (TRCMethod)obj;
            else if(obj instanceof MethodCallDetails)
            {
               MethodCallDetails methodDetails = (MethodCallDetails)obj;
               m = methodDetails.getMethod();
               if (m == null) {
            	   if (pos == 0) {
            		   int calls = methodDetails.getCalls();
            		   return isShowPercent() ? 
            				   TString.formatAsPercentage((double)calls/_totalCalls) :
            				   String.valueOf(calls);
            	   }
            	   else if (pos == 1)
            		   return TraceUIMessages._274;
               }
            }
            
            if(m == null)
            	return "";
            
			TRCMethod selection = _container.getCurrentSelection();
			int nb = 0;
            
            //TRCMethod
			switch(pos)
			{	
			   case 0: //calls
			   	
				switch(_methodDetailsType)
				{
					case CHILDREN_DETAILS:
						//get nb of invocations called by the selected method
						if (obj instanceof MethodCallDetails) {
							nb = ((MethodCallDetails) obj).getCalls();
						}
						// MUST FIX: consider to delete this 'else if' (because of unnecessary recalculation)
						else if(selection != null) {
							Iterator i = m.getInvocations().iterator();
							while(i.hasNext())
							{
								TRCMethodInvocation childMi = (TRCMethodInvocation)i.next();
								TRCMethodInvocation mi = childMi.getInvokedBy();
								if(mi != null && mi.getMethod().equals(selection)) {
									if (childMi instanceof TRCAggregatedMethodInvocation) {
										nb += ((TRCAggregatedMethodInvocation) childMi).getCount();
									}
									else {
										nb++;
									}
								}
								
							}
							
						}
						
						if(isShowPercent())
							return TString.formatAsPercentage((double)nb/_totalCalls);
						
						return String.valueOf(nb);
						//return getElementColumnText(m, _callsCol, info.isDeltaColumn());
					case PARENT_DETAILS:
						//get nb of invocations calling the selected method
						if (obj instanceof MethodCallDetails) {
							nb = ((MethodCallDetails) obj).getCalls();
						} 
						// MUST FIX: consider to delete this 'else if' (because of unnecessary recalculation)
						else if (selection != null) {
							Iterator i = selection.getInvocations().iterator();
							while(i.hasNext())
							{
							   TRCMethodInvocation mi = ((TRCMethodInvocation)i.next());
																
								if (mi != null
										&& mi.getInvokedBy() != null
										&& mi.getInvokedBy().getMethod().equals(m)) {
									if (mi instanceof TRCAggregatedMethodInvocation) {
										nb += ((TRCAggregatedMethodInvocation) mi).getCount();
									}
									else {
										nb++;
									}
								}
								
							}
							
						}
						if(isShowPercent())
							return TString.formatAsPercentage((double)nb/_totalCalls);
						
						return String.valueOf(nb);
					case METHOD_DETAILS:
						return getElementColumnText(m, _callsCol, info.isDeltaColumn());
				}
			   				
				case 1://method name
					return getElementColumnText(m, _methodNameCol, info.isDeltaColumn());
				case 2:
					return "";					
				case 3: //class name
					return getElementColumnText(m, _classNameCol, info.isDeltaColumn());
				case 4: //package name
					return getElementColumnText(m, _packageNameCol, info.isDeltaColumn());
				case 5: //base time
					return getElementColumnText(m, _baseTimeCol, info.isDeltaColumn());
				case 6: //average base time
					return getElementColumnText(m, _avgBaseTimeCol, info.isDeltaColumn());
				case 7: //cumulative time
					return getElementColumnText(m, _cumulativeTimeCol, info.isDeltaColumn());
				case 8:
					
					switch(_methodDetailsType)
					{
						case PARENT_DETAILS:
							return getElementColumnText(m, _callsCol, info.isDeltaColumn());				
						case CHILDREN_DETAILS:
							return getElementColumnText(m, _callsCol, info.isDeltaColumn());
					}
				case 9:
					return getElementColumnText(m, _CPUTimeCol, info.isDeltaColumn());
			}				  				  				 				   				  
            							
			return "";
		}
		
		public Image getColumnImage(Object obj, int col) {
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTree().getColumn(col));
            int pos = getMethodPosForLevel(info.getColumnData().getInitalPos());			
			
			if(info.isDeltaColumn())
			   return null;
			
            TRCMethod m = null;
            if(obj instanceof TRCMethod)
			   m = (TRCMethod)obj;
            else if(obj instanceof MethodCallDetails)
            {
               m = ((MethodCallDetails)obj).getMethod();
            }
			
			// if m==null it means we deal with "N/A.." item (see bug231565)
            if (m == null) return null;
            
			switch(pos)
			{
				case 0: //calls
					return getElementColumnImage(m, _callsCol, info.isDeltaColumn());
				case 1: //method
					return null;
				case 2: //new item column
					return getElementColumnImage(m, _newItemCol, info.isDeltaColumn());	
				case 3: //class
					return getElementColumnImage(m, _classNameCol, info.isDeltaColumn());
				case 4: //package
					return getElementColumnImage(m, _packageNameCol, info.isDeltaColumn());
				case 5: //base time
					return getElementColumnImage(m, _baseTimeCol, info.isDeltaColumn());
				case 6: //average base time
					return getElementColumnImage(m, _avgBaseTimeCol, info.isDeltaColumn());
				case 7: //cumulative time
					return getElementColumnImage(m, _cumulativeTimeCol, info.isDeltaColumn());
				case 9: //cpu time
					return getElementColumnImage(m, _CPUTimeCol, info.isDeltaColumn());
			}

			return null;
		}
	}
	
	public class MethodStatisticSorter extends MultiLevelStatisticSorter{
		
		public int compare(Viewer viewer, Object _e1, Object _e2) {
			int pos = getMethodPosForLevel(_pos);
			
			TRCMethod e1 = null;
			TRCMethod e2 = null;
			 
			if (_e1 instanceof MethodCallDetails) {
				e1 = ((MethodCallDetails) _e1).getMethod();
			} else if (_e1 instanceof TRCMethod) {
				e1 = (TRCMethod) _e1;
			}
			if (_e2 instanceof MethodCallDetails) {
				e2 = ((MethodCallDetails) _e2).getMethod();
			} else if (_e2 instanceof TRCMethod) {
				e2 = (TRCMethod) _e2;
			}

			if (e1 != null && e2 != null) 
			{
				switch(pos)
				{
				   case 0: //calls
					   
					TRCMethod selection = _container.getCurrentSelection();
					int nb1 = 0;
					int nb2=0;
					   
				   	 if(_methodDetailsType == METHOD_DETAILS)
							return _sortSequence * compareElements(e1, e2, _callsCol, _info.isDeltaColumn());
				   	 else if(_methodDetailsType == CHILDREN_DETAILS)
				   	 {				   		 
							//get nb of invocations called by the selected method
				   		 	
						if (_e1 instanceof MethodCallDetails) {
							nb1 = ((MethodCallDetails) _e1).getCalls();
						}
						// MUST FIX: consider to delete this 'else if' (because of unnecessary recalculation)
						else if (selection != null) {
							Iterator i1 = e1.getInvocations().iterator();
							while(i1.hasNext())
							{
								TRCMethodInvocation mi = ((TRCMethodInvocation)i1.next()).getInvokedBy();
								if(mi != null && mi.getMethod().equals(selection)) {
									if (mi instanceof TRCAggregatedMethodInvocation) {
										nb1 += ((TRCAggregatedMethodInvocation) mi).getCount();
									}
									else {
										nb1++;
									}
								}
								
							}
							
						}
						if (_e2 instanceof MethodCallDetails) {
							nb2 = ((MethodCallDetails) _e2).getCalls();
						}
						// MUST FIX: consider to delete this 'else if' (because of unnecessary recalculation)
						else if (selection != null) {
							Iterator i2 = e2.getInvocations().iterator();
							while(i2.hasNext())
							{
								TRCMethodInvocation mi = ((TRCMethodInvocation)i2.next()).getInvokedBy();
								if(mi != null && mi.getMethod().equals(selection)) {
									if (mi instanceof TRCAggregatedMethodInvocation) {
										nb2 += ((TRCAggregatedMethodInvocation) mi).getCount();
									}
									else {
										nb2++;
									}
								}
								
							}
							
						}
							
							return _sortSequence * (nb1-nb2);
				   	 }
				   	 else if(_methodDetailsType == PARENT_DETAILS)
				   	 {				   		 
							//get nb of invocations calling the selected method
						if (_e1 instanceof MethodCallDetails) {
							nb1 = ((MethodCallDetails) _e1).getCalls();
						}
						// MUST FIX: consider to delete this 'else if' (because of unnecessary recalculation)
						else if (selection != null) {
							Iterator i = selection.getInvocations().iterator();
							while(i.hasNext())
							{
							   TRCMethodInvocation mi = ((TRCMethodInvocation)i.next());
																
								if (mi != null && mi.getInvokedBy() != null
											&& mi.getInvokedBy().getMethod().equals(e1)) {
									if (mi instanceof TRCAggregatedMethodInvocation) {
										nb1 += ((TRCAggregatedMethodInvocation) mi).getCount();
									}
									else {
										nb1++;
									}
								}
							}
						}
						if (_e2 instanceof MethodCallDetails) {
							nb2 = ((MethodCallDetails) _e2).getCalls();
						}
						// MUST FIX: consider to delete this 'else if' (because of unnecessary recalculation)
						else if (selection != null) {
							Iterator i = selection.getInvocations().iterator();
							while (i.hasNext()) {
								TRCMethodInvocation mi = ((TRCMethodInvocation) i.next());
								if (mi != null && mi.getInvokedBy() != null
											&& mi.getInvokedBy().getMethod().equals(e2)) {
									if (mi instanceof TRCAggregatedMethodInvocation) {
										nb2 += ((TRCAggregatedMethodInvocation) mi).getCount();
									}
									else {
										nb2++;
									}
								}
								
							}
							
						}
							
							return _sortSequence * (nb1-nb2);
				   	 }
				   	 
				   	 
					case 1://method name
						return _sortSequence * compareElements(e1, e2, _methodNameCol, _info.isDeltaColumn());
					case 2 : // new items
						return _sortSequence * compareElements(e1, e2, _newItemCol, _info.isDeltaColumn());						
					case 3://class name
						return _sortSequence * compareElements(e1, e2, _classNameCol, _info.isDeltaColumn());
					case 4://package name
						return _sortSequence * compareElements(e1, e2, _packageNameCol, _info.isDeltaColumn());						
					case 5: //base time
						return _sortSequence * compareElements(e1, e2, _baseTimeCol, _info.isDeltaColumn());						
					case 6: //average base time
						return _sortSequence * compareElements(e1, e2, _avgBaseTimeCol, _info.isDeltaColumn());						
					case 7: //cumulative time
						return _sortSequence * compareElements(e1, e2, _cumulativeTimeCol, _info.isDeltaColumn());						
					case 8: //calls
						return _sortSequence * compareElements(e1, e2, _callsCol, _info.isDeltaColumn());	
					case 9: //cpu time
						return _sortSequence * compareElements(e1, e2, _CPUTimeCol, _info.isDeltaColumn());							
				}				  				  				 				   				  
			}
			return 0;
		  }
	}
	
	public IContentProvider getContentProvider()
	{
		return new MethodStatisticContentProvider();
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new MethodStatisticLabelProvider(this);
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new MethodStatisticSorter();
	}
	
	protected StatisticFilter getFilterInstance()
	{
		return new MethodStatisticFilter();
	}

	public String printTitle()
	{
		final String newLine = System.getProperties().getProperty("line.separator");
		StringBuffer buffer = new StringBuffer();
		
		buffer.append("<caption align=top><B>" + newLine);
		buffer.append(_tabItem.getContentDescription()).append(newLine);
		buffer.append("</B></caption>").append(newLine);
        return buffer.toString();		
	}
	
	protected void showPercentUpdate()
	{
		super.showPercentUpdate();
		
	}

	/**
	 * 
	 * @return
	 */
	public MethodDetails getContainer() {
		return _container;
	}

	/**
	 * 
	 * @param _container
	 */
	public void setContainer(MethodDetails _container) {
		this._container = _container;
	}
	
	/**
	 * @param i:the level of information to display (package/class or method)
	 */
	public void setLevel(int i) {
		_level = i;
	}	
	
	public int getLevel()
	{
		return _level;
	}	
	
	protected void firstTimeUpdate() {
		super.firstTimeUpdate();
		redrawTable();
	}
	
	public CellLabelProvider getCellLabelProvider(ColumnData colData) {
		return new MethodStatisticCellLabelProvider(colData);
	}
}
