/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MultiLevelStatisticView.java,v 1.18 2010/06/30 20:45:50 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.trace.ui.ITraceSelection;
import org.eclipse.hyades.trace.ui.TraceViewer;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.adapter.internal.IContextViewer;
import org.eclipse.hyades.trace.views.adapter.internal.MultiLevelStatisticPage;
import org.eclipse.hyades.trace.views.adapter.internal.MultiLevelStatisticViewer;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.ColumnExtensionValue;
import org.eclipse.hyades.trace.views.util.internal.IColumnExtension;
import org.eclipse.hyades.trace.views.util.internal.MethodCallDetails;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.hyades.ui.provisional.context.ContextManager;
import org.eclipse.hyades.ui.provisional.context.IContextLanguage;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;


public abstract class MultiLevelStatisticView extends StatisticView {
	
	
	protected ContextInfoContainer _contextInfo;
	
	//information can be displayed in three levels:
	// 1: package/class/method level (default)
	// 2: class/method level 
	// 3: method level
	public static final int PACKAGE_LEVEL = 1;
	public static final int CLASS_LEVEL = 2;
	public static final int DEFAULT_LEVEL = PACKAGE_LEVEL;
	
	public static String PREFERENCE_KEY_PREFIX = "org.eclipse.hyades.trace.views.multilevel.";
	
	private boolean _isEmpty;

	/**
	 *  Add a sorter to the coverage view
	 */
	public abstract class MultiLevelStatisticSorter extends StatisticSorter {

		public MultiLevelStatisticSorter() {
			super();
		}

		/**
		 * compare between too coverage elements
		 */
		public abstract int compare(Viewer viewer, Object e1, Object e2);
	}

	/**
	 * Gives the table tree viewer the mlserage elements to display 
	 */
	public abstract class MultiLevelStatisticContentProvider implements ITreeContentProvider {

		public void dispose() {
		}

		public Object getParent(Object element) {

			return null;
		}

		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {
		}

		/**
		 * return the list of elements to display in the table tree
		 */
		public abstract Object[] getElements(Object inputElement);

		/**
		 * return the children of the coverage element
		 */
		public Object[] getChildren(Object element) {
			return new ArrayList().toArray();
		}

		/**
		 * test if the coverage element has children
		 */
		public boolean hasChildren(Object element) {
			return (getChildren(element) != null);
		}
	}

	/**
	 * provides the table tree viewer: text and images
	 * for each coverage element to display
	 */
	public abstract class MultiLevelStatisticLabelProvider
		extends LabelProvider
		implements ITableLabelProvider {
		protected StatisticView _viewer;

		public MultiLevelStatisticLabelProvider(StatisticView viewer) {
			super();
			_viewer = viewer;
		}

		/**
		 * return for the mlserage element to display the corresponding text
		 */
		public abstract String getColumnText(Object obj, int col);
	}

	/**
	 * Implements a table tree viewer	
	 */
	class MultiLevelStatisticTreeViewer extends TreeViewer {
		public MultiLevelStatisticTreeViewer(Composite parent) {
			super(parent);
		}

		public MultiLevelStatisticTreeViewer(Tree tree) {
			super(tree);
		}

		public void expandItem(TreeItem item) {
			item.setExpanded(true);
			createChildren(item);
		}
		
	    /**
	     * Returns the current selection for this provider.
	     * 
	     * @return the current selection
	     */
	    public ISelection getSelection()
	    {
	    	ISelection sel = super.getSelection();
	    	if(sel != null && !sel.isEmpty())
	    	{
	    		Object obj = ((IStructuredSelection)sel).getFirstElement();
	    		if(obj instanceof MethodCallDetails){
	    			TRCMethod m = ((MethodCallDetails)obj).getMethod();
	    			// if m==null it means "N/A.." item is selected (see bug231565)
	    			if(m == null){
	    				return null;
	    			}
	    			return new StructuredSelection(m);
	    		}
	    	}
	    	return sel;
	    }
		
	}

	public MultiLevelStatisticView(Composite parent, TraceViewerPage page, boolean initializeContent){
		this(false, parent, page, initializeContent);		
	}
	
	public MultiLevelStatisticView(Composite parent, TraceViewerPage page) {
		this(false, parent, page, true);
	}
	
	public MultiLevelStatisticView(boolean lazyMode, Composite parent, TraceViewerPage page, boolean initializeContent){
		super(lazyMode, parent, page, initializeContent);
		_viewerFilter = getFilterInstance();		
	}
	
	public MultiLevelStatisticView(boolean lazyMode, Composite parent, TraceViewerPage page) {
		this(lazyMode, parent, page, true);
	}

	public boolean isEmpty()
	{
		return _isEmpty;
	}
	
	public abstract boolean isEmptyUpdate();
	public abstract String getColumnsPreferencesKey();

	protected abstract String getDefaultColumnsTemplatePackageLevel();
	protected abstract String getDefaultColumnsTemplateClassLevel();
	
	protected abstract StatisticFilter getFilterInstance();
	public abstract IContentProvider getContentProvider();	
	public abstract LabelProvider getTableLabelProvider();	
	protected abstract StatisticSorter getViewerSorterInstance();
	
	public abstract int getLevel();
	public abstract void setLevel(int i);	
	
    protected abstract String getViewTypeStr();	
	
	/**
	 * Create the table tree widget
	 */
	protected Composite createTree(Composite parent, int options) {
		return _toolkit.createTree(parent, options);
	}

	protected TreeViewer getTreeViewer(Tree tree)
	{
		return new MultiLevelStatisticTreeViewer(tree);
	}
	
	protected void openSourceForSelection(ISelection selection)
	{
		if (selection != null) {
			TraceViewerPage page = getTraceViewerPage();
			if (page != null) {
				TraceViewer traceViewer = page.getTraceViewer();
				if (traceViewer instanceof MultiLevelStatisticViewer) {
					final MultiLevelStatisticViewer viewer = (MultiLevelStatisticViewer)page.getTraceViewer();
					if (viewer != null) {
						BusyIndicator.showWhile(Display.getDefault(), new Runnable() {
							public void run() {
								viewer.openSource().run();
							}
						});
					}
				}
			}
		}
	}

	/**
	 * return the table widget
	 */
	public Tree getTree() {
		return (Tree) getTreeViewer().getControl();

	}
	
	/**
	 * Try to see if the selected object is candidate for openSource
	 */
	protected boolean canOpenSourceOnThisObject() {
		ITraceSelection _model =
			UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (_model.size() == 0 || _model.getFirstElement() == null) {
			return false;
		}
		Object firstElement = _model.getFirstElement();
		if (firstElement instanceof TRCClass &&
			((TRCClass)firstElement).getSourceInfo() != null &&
			((TRCClass)firstElement).getSourceInfo().getLocation().length() > 0) {
			return true;
		} else if (firstElement instanceof TRCMethod) {
			return true;
		}
		return false;
	}
	
	public String getColumnsTemplateExtensions(int strColumnNumbers,String matchView,boolean nullViewValue)
	{
		String columnName = "";
		String pos = "";
		String style = "";
		String width = "";
		String align = "";
		String viewId ="";
		int colNumber = strColumnNumbers;
		
		boolean extendColumn = false;
		
		_templateOfColumExtension = "";
		_listOfColumExtension.clear();

		IExtensionPoint extPoint = Platform.getExtensionRegistry()
		     .getExtensionPoint(TraceUIPlugin.PLUGIN_ID, "columnExtension");

		if (extPoint != null) {
			IConfigurationElement[] members =
			extPoint.getConfigurationElements();

			if (members.length != 0) {
				//For each column
				for (int i = 0; i < members.length; i++) {
					columnName = members[i].getAttribute("name");
					pos = members[i].getAttribute("pos");
					style =	members[i].getAttribute("style");
					width = members[i].getAttribute("width");
					align = members[i].getAttribute("align");
					
					IConfigurationElement[] applyToView = members[i].getChildren("applyTo");
					
					extendColumn = false;
					if (applyToView!=null) {
						for (int k = 0; k < applyToView.length; k++) {
							viewId = applyToView[k].getAttribute("viewId");
							if (viewId.equalsIgnoreCase(matchView)) {
								extendColumn = true;
								break;
							}
						}
					}
					
					if (extendColumn || 
							(nullViewValue && ((applyToView==null) || applyToView.length==0)))
					{
						_templateOfColumExtension +=
						","
						+ columnName
						+ ":"
						+ colNumber
						+ ":"
						+ style
						+ ":"
						+ align
						+ ":"
						+ width;
						colNumber++;
	
						try {
							ColumnExtensionValue columnVal = new ColumnExtensionValue();
							columnVal.setClassOfColumnExtension(
								(IColumnExtension) members[i].createExecutableExtension("class"));
							columnVal.setInitialColumnPos((new Integer(pos)).intValue());
							_listOfColumExtension.add(columnVal);
						} catch (CoreException e) {
							System.err.println(e);
						}
					}
				}
			}
		}
		return _templateOfColumExtension;
	}
	public String getDefaultColumnsTemplate()
	{
		String columns = null;

		if (getLevel() == 0
			|| getLevel() == PACKAGE_LEVEL) {
			//In the package level the class names column and the package column are hidden				
			columns = getDefaultColumnsTemplatePackageLevel();

		} else if (
			getLevel() == CLASS_LEVEL) {
			//In the class level the class names column is hidden
			columns = getDefaultColumnsTemplateClassLevel();
		}
		
		return columns;
	}	

	/**
	 * The same element must be selected in all the statistics views
	 */
	protected void doHandleSelectionEvent(boolean isPostponedOperation, int processedOperations) {
		ITraceSelection model =
			UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() > 0) {
			Object sel = model.getFirstElement();
			if (sel != null) {
				select(sel);
				
				if (sel instanceof EObject)
					updateStatusContext(ContextManager.getContextLanguage(ContextUpdaterHelper.getContext((EObject)sel)));
			}
		}
	}

	/**
	 * select the element (in the coverage table) which is selected in the other 
	 * statistics views
	 */
	private void select(Object obj) {
		TRCClass cls = null;
		TRCPackage pack = null;
		TRCMethod meth = null;
		TRCObject trcObj = null;

		Tree tree = (Tree) getTreeViewer().getControl();

		//Summary element
		TreeItem[] items = tree.getItems();
		if (items.length != 0) // if we have data
			{
			//((MultiLevelStatisticTreeViewer) getTableViewer()).expandItem(items[0]);

			if (obj instanceof TRCPackage)
				pack = (TRCPackage) obj;
			else if (obj instanceof TRCClass) {
				cls = (TRCClass) obj;
				pack = cls.getPackage();
			} else if (obj instanceof TRCMethod) {
				meth = (TRCMethod) obj;
				cls = meth.getDefiningClass();
				if (null != cls)
					pack = cls.getPackage();
			}
			else if (obj instanceof TRCObject)
			{
				cls = ((TRCObject)obj).getIsA();
				pack = cls.getPackage();
				trcObj = (TRCObject)obj;
			}
			else if(obj instanceof TRCMethodInvocation)
			{
				cls = PerftraceUtil.getClass((TRCMethodInvocation)obj);
				if (null != cls)
					pack = cls.getPackage();
			}
			
			searchInLevel(tree,
						getItemsRoot(tree),
						pack,
						cls,
						meth,
						trcObj);
		}
	}
	
	protected TreeItem getItemsRoot(Tree tree)
	{
		return null;
	}

	/**
	 * search the element to select in the package level table
	 */
	protected void searchInPackageLevel(
		Tree tree,
		TreeItem item,
		TRCPackage pack,
		TRCClass cls,
		TRCMethod meth,
		TRCObject trcObj) {
		TreeItem pacItem = null;

		TreeItem[] pacItems = null;
		
		if (item != null)
			pacItems = item.getItems();
		else
			pacItems = tree.getItems();

		pacItem = searchItem(pacItems, pack);

		if (pacItem != null) {
			if (cls == null) { //Package selected
				tree.setSelection(new TreeItem[] { pacItem });
				return;
			}
			
			((MultiLevelStatisticTreeViewer)getTreeViewer()).expandItem(pacItem);

			//TRCClass elements
			searchInClassLevel(
				tree,
				pacItem,
				cls,
				meth,
				trcObj);
			return;
		}
		else if (item != null)
		{
			tree.setSelection(new TreeItem[] { item });
			return;
		}		
		
		tree.deselectAll();
	}

	/**
	 * search the element to select in the class level tree
	 */
	protected void searchInClassLevel(
		Tree tree,
		TreeItem item,
		TRCClass cls,
		TRCMethod meth,
		TRCObject trcObj) {
		TreeItem[] classItems = null;
		TreeItem classItemBis = null;

		if (item != null)
			classItems = item.getItems();
		else
			classItems = tree.getItems();		
		
		classItemBis = searchItem(classItems, cls);

		if (classItemBis != null) {
			if (meth == null && trcObj == null) { // TRCClass selected
				tree.setSelection(new TreeItem[] { classItemBis });
				return;
			}
			
			((MultiLevelStatisticTreeViewer)getTreeViewer()).expandItem(classItemBis);			
			
			//at least one of trcObj and meth must be null
			if (trcObj != null)
			{
				searchInObjectLevel(tree, classItemBis, trcObj);
				return;
			}	
			else if (meth != null)
			{
				searchInMethodLevel(tree, classItemBis, meth);
				return;
			}
		}
		else if (item != null)
		{
			tree.setSelection(new TreeItem[] { item });	
			return;			
		}
		
		tree.deselectAll();
	}
	
	/**
	 * search the element to select in the method level tree
	 *
	 */
	protected void searchInMethodLevel(
		Tree tree,
		TreeItem item,
		TRCMethod meth) {
		TreeItem[] methodItems = null;
		
		if (item != null)
			methodItems = item.getItems();
		else
			methodItems = tree.getItems();			

		TreeItem methodItem = searchItem(methodItems, meth);

		if (methodItem != null) {
			tree.setSelection(new TreeItem[] { methodItem });
			return;
		}
		else if (item != null)
		{
			tree.setSelection(new TreeItem[] { item });		
			return;			
		}

		tree.deselectAll();
	}
	
	/**
	 * search the element to select in the object level tree
	 *
	 */
	protected void searchInObjectLevel(
		Tree tree,
		TreeItem item,
		TRCObject obj) {
		TreeItem[] objItems = null;
		
		if (item != null)
			objItems = item.getItems();
		else
			objItems = tree.getItems();			

		TreeItem objItem = searchItem(objItems, obj);

		if (objItem != null) {
			tree.setSelection(new TreeItem[] { objItem });
			return;
		}
		else if (item != null)
		{
			tree.setSelection(new TreeItem[] { item });		
			return;			
		}

		tree.deselectAll();
	}
	
	
	protected void searchInLevel(Tree tree,
			TreeItem item,
			TRCPackage pack,
			TRCClass cls,
			TRCMethod meth,
			TRCObject trcObj)
	{
		switch (getLevel()) {
		case PACKAGE_LEVEL :
			searchInPackageLevel(
				tree,
				item,
				pack,
				cls,
				meth,
				null);
			break;
		case CLASS_LEVEL :
			if (cls != null)
				searchInClassLevel(
					tree,
					item,
					cls,
					meth,
					null);
			
			break;
		}
		
	}
	
	protected Object getItemModelData(TreeItem item)
	{
		return item.getData();
	}
	
	protected TreeItem searchItem(TreeItem[] items, Object obj) {
		TreeItem item = null;
		for (int j = 0; j < items.length; j++) {
			item = items[j];

			if (item!=null && !item.isDisposed()) {
				if (item.getData() != null &&
						obj == getItemModelData(item))
					return item;
			}
		}
		return null;
	}		

	/**
	 * Update the "model selection" when an element is selected in the tree
	 */
	public void updateModelSelection() {

		ISelection selection = getTreeViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			if(sel instanceof MethodCallDetails)
				notifyViewSelectionChanged(this, ((MethodCallDetails)sel).getMethod());
			else	
				notifyViewSelectionChanged(this,sel);
		}
	}

	/**
	 * if the selection event is sent by the active tree: discard the event
	 * else handle it
	 */
	protected void doHandleViewSelectionChangedEvent(ViewSelectionChangedEvent event, boolean isPostponedOperation, int processedOperations) {
		Object  source = event.getSource();
		if(source!=this){	
			handleSelectionEvent();
		}
		else
		{
			ITraceSelection model =
				UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
			if (model.size() > 0) {
				Object sel = model.getFirstElement();
				if (sel != null && sel instanceof EObject)
						updateStatusContext(ContextManager.getContextLanguage(ContextUpdaterHelper.getContext((EObject)sel)));
			}
		}
	}

	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {

		menu.add(getUpdateAction());
		menu.add(fSeparator);
		resetChooseColumnsAction();
		menu.add(
			getChooseColumnsAction(
				getColumnDataList(),
				getColumnsPreferencesKey()));
		menu.add(
			getSortByColumnAction());
					
	}

	/**
	 * Redraw the tree to display the three levels
	 * (package level, class level and method level)
	 * 
	 */
	public void redrawTable() {
		getTree().setRedraw(false);


		// bug 259195, retrieve column info presisted.
		ArrayList<ColumnData> list = ColumnData.createColumnData(getColumnsPreferencesKey(), getDefaultColumnsTemplate());
		
		TreeColumn[] treeColumns=getTree().getColumns();
		List<Integer> widthInfor=new LinkedList<Integer>();
		List<String> columnNames=new LinkedList<String>();
		for(int i=0;i<treeColumns.length;i++)
		{
			widthInfor.add(new Integer(treeColumns[i].getWidth()));
			columnNames.add(treeColumns[i].getText());
		}
		resetColumns(list);
		
		for(int i=0;i<getTree().getColumns().length;i++)
		{
			for(int j=0;j<columnNames.size();j++)
			{
				if(columnNames.get(j).equals(getTree().getColumn(i).getText()))
				{
					getTree().getColumn(i).setWidth(widthInfor.get(j).intValue());
					StatisticTableColumnInfo treeColumnInfo = getColumnInfo(getTree().getColumn(i));
					int index=this.getColumnIndexByColumnKey(list, treeColumnInfo.getColumnData().key());
					if(index!=-1)
					{
						list.get(index).width(widthInfor.get(j).intValue());
					}
					break;
				}
			}
		}
		_currentColumns = list;
		getTree().setRedraw(true);
		
		refresh();
	}
	
			// bug 259195, retrieve column info presisted.
	private int getColumnIndexByColumnKey(ArrayList<ColumnData> columnData,String key)
	{
		if(columnData==null)
			return -1;
		else
		{
			for(int i=0;i<columnData.size();i++)
			{
				if(columnData.get(i).key().equals(key))
					return i;
			}
			return -1;
		}
	}

	/**
	 * dispose the active columns and create others
	 */
	protected void resetColumns(ArrayList list) {
		super.resetColumns(list);
		expandFirstElement();
//		handleSelectionEvent();
	}

	/**
	 * reset the choose column Action
	 */
	protected void resetChooseColumnsAction() {
		_chooseColumnAction = null;
	}

	public void dispose() {
		super.dispose();
	}

	protected void doUpdateButtons(boolean isPostponedOperation, int processedOperations) {
		((MultiLevelStatisticPage) getTraceViewerPage()).updateButtons(getLevel());
		((MultiLevelStatisticPage) getTraceViewerPage()).openSource().setEnabled(canOpenSourceOnThisObject());
	}	
	
	protected void doUpdate(boolean refresh, boolean isPostponedOperation, int processedOperations)
	{
		if(_page instanceof MultiLevelStatisticPage)
		{
			if (_contextInfo != null)
				_contextInfo.setMOFObject(_page.getMOFObject());
		}
		
		super.doUpdate(refresh, isPostponedOperation, processedOperations);
		
		if(_page instanceof MultiLevelStatisticPage)
		{
			((MultiLevelStatisticPage)_page).updateFilterAppliedDescription();
			_isEmpty = isEmptyUpdate();
		}
		
		//bug 126261
		Tree tree = getTree();
		setLinesVisible(tree);
		_viewerSorter = getViewerSorterInstance();
		if(_previousColumnText != null){
			for(int i = 0; i < tree.getColumnCount(); i++) {
				if(tree.getColumn(i).getText().equals(_previousColumnText)) {
					_viewerSorter.setSortedColumn(_previousSequence,tree.getColumn(i));
					return;
					}
				}
			}
		}
		//end bug 126261	
	
	protected SimpleSearchQuery getCurrentFilter()
	{
		if(_page instanceof MultiLevelStatisticPage)
			return ((MultiLevelStatisticPage)_page).getCurrentFilter();
		
		return null;
	}	

	protected Composite createControl(Composite parent, ArrayList cols) {
		final Composite vc = super.createControl(parent, cols);
		
		if (_page.getTraceViewer() instanceof MultiLevelStatisticViewer)
		{
			_contextInfo = new ContextInfoContainer();
			_contextInfo.setViewer((IContextViewer)_page.getTraceViewer());
			_contextInfo.createControl(vc);
			_contextInfo.addContextInfoContainerListener(
					new IContextInfoContainerListener()
					{
						public void visibilityChanged(boolean isVisible)
						{
							vc.layout(true, true);
						}
					});
		}
		
		return vc;
	}
	
	protected void updateStatusContext(IContextLanguage language)
	{
		if (_contextInfo != null)
			_contextInfo.updateStatusContext(language);
	}

	protected void handleHidden() {
		super.handleHidden();
	}

	protected void handleVisible() {
		super.handleVisible();
	}

	public void updateButtons() {
		super.updateButtons();
	}

	class MultiLevelStatisticViewSelectionListener extends StatViewSelectionListener {
		
		public MultiLevelStatisticViewSelectionListener() {
			super();
		}
		
		public void widgetSelected(org.eclipse.swt.events.SelectionEvent arg0) {
			super.widgetSelected(arg0);
						
			if (arg0.widget instanceof TableColumn) {
				expandFirstElement();
				handleSelectionEvent();
			}
		}
		
	}
	
	public SelectionListener getSelectionListener() {
		return new MultiLevelStatisticViewSelectionListener();
	}

}
