/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: StdDrawStrategy.java,v 1.1 2005/07/22 14:49:34 hleung Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import org.eclipse.jface.action.*;
import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;

public abstract class StdDrawStrategy implements DrawStrategy, KeyListener,
		MouseListener, MouseMoveListener, DisposeListener {
	protected float fWidth = 0;

	protected float fHeight = 0;

	protected boolean _mousePressed = false; // AWT problem

	protected float _origx, _origy, _lastx, _lasty;

	protected boolean _controlDown = false;

	protected boolean _shiftDown = false;

	protected boolean _dragScrolling = false;

	protected JCanvas _canvas;

	private boolean _dirty = true;

	protected boolean _nonpropzoom = false;

	protected void constrainAspect() {
		Rectangle visible = _canvas.getSize();

		float dx = _origx - _lastx;
		float dy = _origy - _lasty;
		float scalingRatio = _canvas.xscale() / _canvas.yscale();
		float aspect = (float) visible.width / (float) visible.height
				/ scalingRatio;

		if (Math.abs(dx / dy) < aspect) {
			dx = (dx < 0 ? -Math.abs(dy) * aspect : Math.abs(dy) * aspect);
		} else {
			dy = (dy < 0 ? -Math.abs(dx) / aspect : Math.abs(dx) / aspect);
		}

		_lastx = _origx - dx;
		_lasty = _origy - dy;
	}

	protected boolean dragZoomed() {
		return Math.abs(_lastx - _origx) * _canvas.xscale() > 6
				|| Math.abs(_lasty - _origy) * _canvas.yscale() > 6;
	}

	protected void drawZoomRect() {
		_canvas.drawRectXor(Math.min(_origx, _lastx), Math.min(_origy, _lasty),
				Math.abs(_origx - _lastx), Math.abs(_origy - _lasty));
	}

	/**
	 * This method should fill (or update) the given menu. It is called just
	 * before the menu opens.
	 */
	public abstract void fillContextMenu(IMenuManager menu);

	public float getHeight() {
		return fHeight;
	}

	public float getWidth() {
		return fWidth;
	}

	public synchronized boolean isDirty() {
		return _dirty;
	}

	public synchronized boolean isDirtyAndReset() {
		boolean currentstate = _dirty;
		_dirty = false;
		return currentstate;
	}

	public boolean isNonPropZoom() {
		return _nonpropzoom;
	}

	public JCanvas jcanvas() {
		return _canvas;
	}

	public void jcanvas(JCanvas c) {
		if (c != _canvas) {
			if (_canvas != null) {
				_canvas.canvas().removeDisposeListener(this);
			}

			_canvas = c;
			if (_canvas != null) {
				_canvas.canvas().addDisposeListener(this);
				resetArea();
			}
		}
	}

	public void keyPressed(KeyEvent e) {
		if (e.keyCode == SWT.CONTROL) {
			_controlDown = true;

		} else if (e.keyCode == SWT.SHIFT) {
			_shiftDown = true;
		} else if (e.keyCode == SWT.HOME) {
			JCanvas canvas = jcanvas();
			if (canvas == null)
				return;

			canvas.home();
		}
	}

	public void keyReleased(KeyEvent e) {
		if (e.keyCode == SWT.CONTROL) {
			_controlDown = false;
		} else if (e.keyCode == SWT.SHIFT)
			_shiftDown = false;

	}

	public void keyTyped(KeyEvent e) {
	}

	public float left() {
		return 0;
	}

	/**
	 * @param event
	 *            org.eclipse.swt.events.MouseEvent
	 */
	public void mouseDoubleClick(MouseEvent event) {
	}

	public void mouseDown(MouseEvent e) {
		if (_canvas.isZoom()) {
			_canvas.zoom(e.x, e.y);
			return;
		}

		if (_dragScrolling || _mousePressed)
			return;
		// in case of bogus call from AWT

		if (_controlDown) {
			_dragScrolling = true;

			_lastx = e.x;
			_lasty = e.y;

		} else {
			_mousePressed = true;

			_origx = _lastx = _canvas.normX(e.x);
			_origy = _lasty = _canvas.normY(e.y);
		}
	}

	public void mouseMove(MouseEvent e) {
		if (_canvas == null)
			return;

		float newx = _canvas.normX(e.x);
		float newy = _canvas.normY(e.y);

		if (_mousePressed) {
			if (_dragScrolling) {
				int dx = (int) _lastx - e.x;
				int dy = (int) _lasty - e.y;

				_lastx = e.x;
				_lasty = e.y;

				_canvas.scrollBy(dx, dy);
				_canvas.redraw();

			} else {
				drawZoomRect();
				_lastx = newx;
				_lasty = newy;
				if (!_nonpropzoom)
					constrainAspect();
				drawZoomRect();
			}
		} else {
			float nx = _canvas.normX(e.x);
			float ny = _canvas.normY(e.y);
			moved(nx, ny);

		}
	}

	public void mouseUp(MouseEvent e) {
		if (_canvas.isZoom())
			return;

		if (_dragScrolling) {
			_dragScrolling = false;
		} else if (_mousePressed) {
			_mousePressed = false;
			drawZoomRect();

			_lastx = _canvas.normX(e.x);
			_lasty = _canvas.normY(e.y);

			if (dragZoomed()) {
				if (!_nonpropzoom)
					constrainAspect();
				float x = Math.min(_origx, _lastx);
				float y = Math.min(_origy, _lasty);
				float w = Math.abs(_origx - _lastx);
				float h = Math.abs(_origy - _lasty);
				_canvas.zoomRect(x, y, w, h);
				_canvas.redraw();

			} else {
				selected(_lastx, _lasty, _shiftDown, _controlDown, false);
				_canvas.redraw();
			}
		}
	}

	/**
	 * A convenience function for resetting the total scrollable area of the
	 * canvas.
	 */
	public void resetArea() {
		JCanvas canvas = jcanvas();
		if (canvas == null)
			return;

		canvas.left(left());
		canvas.top(top());
		canvas.setArea(width(), height());
	}

	public synchronized void setDirty() {
		_dirty = true;
	}

	/**
	 * Insert the method's description here. Creation date: (6/8/2001 11:30:19
	 * AM)
	 * 
	 * @param height
	 *            float
	 */
	public void setHeight(float height) {
		fHeight = height;
	}

	public void setNonPropZoom(boolean nonpropzoom) {
		_nonpropzoom = nonpropzoom;
	}

	/**
	 * Insert the method's description here. Creation date: (6/8/2001 11:30:19
	 * AM)
	 * 
	 * @param width
	 *            float
	 */
	public void setWidth(float width) {
		fWidth = width;
	}

	public float top() {
		return 0;
	}

	public void widgetDisposed(DisposeEvent e) {
		if (_canvas != null && e.widget == _canvas.canvas()) {

			_canvas.dispose();
		}
		_canvas = null;
	}
}
