/**********************************************************************
 * Copyright (c) 2006 Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SummaryGraph.java,v 1.2 2006/11/01 18:13:33 ewchan Exp $
 * 
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.ui.widgets.grapher.GraphCanvas;
import org.eclipse.hyades.ui.widgets.grapher.graphs.LineGraph;
import org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSliderConfigurationException;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSliderEvent;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSliderListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

public class SummaryGraph implements ZoomSliderListener {
	private Composite namesContainer;
	private Composite graphComposite;
	private final List sourceList = new ArrayList();
	private GraphCanvas canvas;
	private TimeZoomSlider xSlider;
	private ZoomSlider ySlider;

	public SummaryGraph(final Composite parent, final String title) throws ZoomSliderConfigurationException {
        Composite container = new Composite(parent, SWT.NONE);
        GridLayout gridLayout = new GridLayout(1, true);
		container.setLayout(gridLayout);
		gridLayout.verticalSpacing = 0;
        Button titleLabel = new Button(container, SWT.PUSH | SWT.LEFT);
        titleLabel.setText(title);
        titleLabel.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
		
		createGraphComposite(container);                

		namesContainer = new Composite(container, SWT.NONE);
		namesContainer.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
		GridLayout namesLayout = new GridLayout(0, true);
		namesContainer.setLayout(namesLayout);
		namesLayout.verticalSpacing = 0;
		namesLayout.marginHeight = 0;
	}

	private void createGraphComposite(final Composite parent) throws ZoomSliderConfigurationException {
		graphComposite = new Composite(parent, SWT.NONE);
		graphComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		GridLayout gridLayout = new GridLayout(2, false);
		gridLayout.horizontalSpacing = 0;
		gridLayout.verticalSpacing = 0;            
		graphComposite.setLayout(gridLayout);

		//Create the Y Slider            
		ySlider = new ZoomSlider(graphComposite);
		ySlider.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, true));
		ySlider.addZoomSliderListener(this);
		// Set the initial configuration for the y slider
		ySlider.configure(0, 200, 50, 150, 2);
		ySlider.setEnabled(false);

		//Create the graph canvas
		canvas = new GraphCanvas(graphComposite, SWT.NONE);
		canvas.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));                              

		Composite emptyFill = new Composite(graphComposite, SWT.NONE);
		emptyFill.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));

		//Create the X Slider
		xSlider = new TimeZoomSlider(graphComposite);
		xSlider.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));                
		xSlider.setDirection(ZoomSlider.DECREASING);                
		xSlider.setOrientation(ZoomSlider.HORIZONTAL);
		xSlider.addZoomSliderListener(this);
		// Configure the initial settings for this slider
		xSlider.configure(-10000, 10000, -200, 200, 2);
		xSlider.setEnabled(false);
	}

	public void addGraphSource(final SummaryGraphSource graphSource) {
        // Construct a line graph to plot the points of sin data source.
        // Set the PLOTTING_TYPE_AVERAGE to plot average results. These results are obtained
        // from the graph source via the getAv
        LineGraph lineGraph = new LineGraph(LineGraph.PLOTTING_TYPE_AVERAGE, graphSource, xSlider, ySlider, graphSource.getColor());
        canvas.addGraph(lineGraph);
        addName(graphSource);
        sourceList.add(graphSource);
	}
	
	private void addName(final SummaryGraphSource graphSource) {
		((GridLayout)namesContainer.getLayout()).numColumns++;
		Composite container = new Composite(namesContainer, SWT.NONE);
		GridLayout gridLayout = new GridLayout(2, false);
		gridLayout.verticalSpacing = 0;
		gridLayout.marginHeight = 0;
		container.setLayout(gridLayout);
		container.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
		final Canvas tick = new Canvas(container, SWT.NONE);
		tick.setLayoutData(new GridData(20, 20));
		tick.addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent e) {
				e.gc.setBackground(graphSource.getColor());
				e.gc.fillRectangle(tick.getBounds());
				e.gc.setForeground(e.display.getSystemColor(SWT.COLOR_BLACK));
				e.gc.drawRectangle(tick.getBounds());
			}
		});
		Label title = new Label(container, SWT.NONE);
		title.setText(graphSource.getName());
		title.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
	}

	public void zoomSliderChanged(final ZoomSliderEvent event) {
        canvas.redraw();
	}

	public void update() {
        canvas.redraw();
        if (sourceList.isEmpty()) {
        	return;
        }
        double maxX = 0;
		double minX = Double.MAX_VALUE;
		double maxY = 0;
		double minY = Double.MAX_VALUE;
		for (int i = 0; i < sourceList.size(); i++) {
			SummaryGraphSource source = (SummaryGraphSource) sourceList.get(i);
			if (source.getMax() > maxX) {
				maxX = source.getMax(); 
			}
			if (source.getMin() < minX) {
				minX = source.getMin(); 
			}
			if (source.getValueMax() > maxY) {
				maxY = source.getValueMax(); 
			}
			if (source.getValueMin() < minY) {
				minY = source.getValueMin();
			}
		}
		try {
			xSlider.configure(0.8*minX, 1.2*maxX, minX, maxX, 2);
			ySlider.configure(0.8*minY, 1.2*maxY, minY, maxY, 2);
		} catch (ZoomSliderConfigurationException e) {
//			e.printStackTrace();
		}
	}

	public void dispose() {
		ArrayList graphs = canvas.getGraphs();
		for (int i = 0; i < graphs.size(); i++) {
			((LineGraph)graphs.get(i)).dispose();
		}
	}
}
