/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceUIPlugin.java,v 1.12 2010/07/27 02:06:57 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.trace.TracePackage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.views.adapter.internal.ActiveFilter;
import org.eclipse.hyades.trace.views.adapter.internal.FilterEventsUI;
import org.eclipse.hyades.trace.views.adapter.internal.GetSetFilter;
import org.eclipse.hyades.trace.views.adapter.internal.ThresholdFilter;
import org.eclipse.hyades.trace.views.adapter.internal.TraceConstants;
import org.eclipse.hyades.trace.views.util.internal.IColorChangedListener;
import org.eclipse.hyades.trace.views.util.internal.IRefChangedListener;
import org.eclipse.hyades.trace.views.util.internal.ITimeChangedListener;
import org.eclipse.hyades.trace.views.util.internal.SpectrumColorMap;
import org.eclipse.hyades.trace.views.util.internal.UpdateModel;
import org.eclipse.hyades.ui.filters.internal.util.FilterQueries;
import org.eclipse.hyades.ui.filters.internal.util.FilterResourceHandler;
import org.eclipse.hyades.uml2sd.trace.TraceSDUtil;
import org.eclipse.hyades.uml2sd.trace.preferences.ITraceInteractionPreferenceListener;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferenceConverter;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * 
 * Plugin class that provides way to access plugin instance, resource bundle and
 * debug enablement.
 * <p>
 * <b>Note:</b>This class is not intended to be extended and use externally,
 * and it is subject to be changed without notice.
 * 
 */
public class TraceUIPlugin extends AbstractUIPlugin {
	private static TraceUIPlugin _plugin;

	public static final ActiveFilter THRESHOLD_FILTER = new ThresholdFilter();
	public static final ActiveFilter GETSET_FILTER = new GetSetFilter();
	
	public static final String PLUGIN_ID = "org.eclipse.hyades.trace.views"; //$NON-NLS-1$

	private static final String FILTER_TYPE = "org.eclipse.hyades.trace.views.ProfilingFilterType";
	
	private static final boolean ADD_GET_SET_FILTER = true;
	private static final boolean DO_NOT_ADD_GET_SET_FILTER = false;
	
	private ResourceBundle fResourceBundle;

	/**
	 * The collection of color changed event listeners.
	 */
	private ListenerList fColorChangedEventListeners = new ListenerList();

	/**
	 * The collection of reference changed event listeners.
	 */
	private ListenerList fRefChangedEventListeners = new ListenerList();

	/**
	 * The collection of time changed event listeners.
	 */
	private ListenerList fTimeChangedEventListeners = new ListenerList();

	/**
	 * This method creates and stores the single instance on the first
	 * invocation.
	 */
	public TraceUIPlugin() {

		if (_plugin == null)
			_plugin = this;
	}

	/**
	 * This method returns the single instance of this class.
	 */
	public static TraceUIPlugin getDefault() {
		return _plugin;
	}

	/**
	 * @return org.eclipse.jface.resource.ImageDescriptor
	 * @param name
	 *            java.lang.String
	 */
	public static ImageDescriptor getImageDescriptor(String name) {
		return TracePluginImages.getImageDescriptor(name);
	}

	public static void setImageDescriptors(IAction action, String type,
			String relPath) {
		TracePluginImages.setImageDescriptors(action, type, relPath);
	}

	/**
	 * @return java.util.ArrayList
	 */

	public static String getPluginId() {
		return PLUGIN_ID;
	}

	public ResourceBundle getResourceBundle() {

		if (fResourceBundle == null) {
			try {
				fResourceBundle = Platform.getResourceBundle(Platform
						.getBundle(PLUGIN_ID));
			} catch (MissingResourceException e) {
				e.printStackTrace();
				fResourceBundle = null;
			}
		}

		return fResourceBundle;
	}

	public static String getString(String key) {
		ResourceBundle bundle = TraceUIPlugin.getDefault().getResourceBundle();
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the string from the plugin's resource bundle, or 'key' if not
	 * found.
	 */
	public static String getResourceString(String key) {
		return getString(key);
	}

	/**
	 * Insert the method's description here. Creation date: (8/8/2001 5:11:57
	 * PM)
	 * 
	 * @param store
	 *            org.eclipse.jface.preference.IPreferenceStore
	 */
	protected void initializeDefaultPreferences(IPreferenceStore store) {

		PreferenceConverter.setDefault(store, TraceConstants.BACKGROUND_RGB,
				new RGB(255, 255, 255));
		PreferenceConverter.setDefault(store, TraceConstants.SELECTION_RGB,
				new RGB(255, 255, 149));
		PreferenceConverter.setDefault(store, TraceConstants.UNKNOWN_RGB,
				new RGB(128, 128, 128));
		PreferenceConverter.setDefault(store, TraceConstants.CLASSES_RGB,
				new RGB(71, 141, 141));
		store.setDefault(TraceConstants.BACKGROUND_OPTION,
				TraceConstants._89);
		store.setDefault(TraceConstants.CLASSES_OPTION,
				TraceConstants.CLASS_DEFAULT_COLOR);
		store.setDefault(TraceConstants.OBJ_REF_OPTION,
				TraceConstants.OBJ_REF_REFERANDS);
		store.setDefault(TraceConstants.TIME_OPTION,
				TraceConstants.COMPENSATED_TIME);
		store.setDefault(TraceConstants.PERCENTAGE, 0);

		store.setDefault(ITraceInteractionPreferenceListener.PAGE_SIZE,
				128 * 1024);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);

		savePluginPreferences();
		// dispose the colors
		SpectrumColorMap.shutdown();

		TracePluginImages.shutdown();
		TraceSDUtil.shutdown();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);

		addDefaultFilters();

		UIPlugin.getDefault().addProfileEventListener(new UpdateModel());

		SpectrumColorMap.startup();
	}

	/**
	 * Adds the given listener to the collection of registered event listeners.
	 * Has no effect if an identical listener is already registered.
	 * 
	 * @param listener
	 *            the listener to add
	 */
	public void addColorChangedEventListener(IColorChangedListener listener) {
		fColorChangedEventListeners.add(listener);
	}

	/**
	 * Adds the given listener to the collection of registered event listeners.
	 * Has no effect if an identical listener is already registered.
	 * 
	 * @param listener
	 *            the listener to add
	 */
	public void addRefChangedEventListener(IRefChangedListener listener) {
		fRefChangedEventListeners.add(listener);
	}

	/**
	 * Adds the given listener to the collection of registered event listeners.
	 * Has no effect if an identical listener is already registered.
	 * 
	 * @param listener
	 *            the listener to add
	 */
	public void addTimeChangedEventListener(ITimeChangedListener listener) {
		fTimeChangedEventListeners.add(listener);
	}

	/**
	 * Removes the given listener from the collection of registered event
	 * listeners. Has no effect if an identical listener is not already
	 * registered.
	 * 
	 * @param listener
	 *            the listener to remove
	 */
	public void removeColorChangedEventListener(IColorChangedListener listener) {
		fColorChangedEventListeners.remove(listener);
	}

	/**
	 * Removes the given listener from the collection of registered event
	 * listeners. Has no effect if an identical listener is not already
	 * registered.
	 * 
	 * @param listener
	 *            the listener to remove
	 */
	public void removeRefChangedEventListener(IRefChangedListener listener) {
		fRefChangedEventListeners.remove(listener);
	}

	/**
	 * Removes the given listener from the collection of registered event
	 * listeners. Has no effect if an identical listener is not already
	 * registered.
	 * 
	 * @param listener
	 *            the listener to remove
	 */
	public void removeTimeChangedEventListener(ITimeChangedListener listener) {
		fTimeChangedEventListeners.remove(listener);
	}

	public void notifyColorChangedEventListener() {
		Object[] listeners = fColorChangedEventListeners.getListeners();
		for (int idx = 0; idx < listeners.length; idx++) {
			((IColorChangedListener) listeners[idx]).handleColorChangedEvent();
		}
	}

	public void notifyRefChangedEventListener() {
		Object[] listeners = fRefChangedEventListeners.getListeners();
		for (int idx = 0; idx < listeners.length; idx++) {
			((IRefChangedListener) listeners[idx]).handleRefChangedEvent();
		}
	}

	public void notifyTimeChangedEventListener() {
		Object[] listeners = fTimeChangedEventListeners.getListeners();
		for (int idx = 0; idx < listeners.length; idx++) {
			((ITimeChangedListener) listeners[idx]).handleTimeChangedEvent();
		}
	}

	public void log(IStatus status) {
		getLog().log(status);
	}

	public void log(Throwable e) {
		log(new Status(IStatus.ERROR, PLUGIN_ID, IStatus.ERROR, "Error", e)); //$NON-NLS-1$
	}

	/**
	 * Returns the workspace instance.
	 */
	public static IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}

	/**
	 * @return the plugin id
	 */
	public static String getId() {
		return getDefault().getBundle().getSymbolicName();
	}

	private void addDefaultFilters() {
		String traceViewsPluginDefaultFiltersKey = "org.eclipse.hyades.trace.views";

		if (!FilterResourceHandler
				.isDefaultFiltersAddedForKey(traceViewsPluginDefaultFiltersKey)) {
			List filtersToAdd = new ArrayList();

			FilterQueries fq = FilterResourceHandler.createFilterQueries(
					FILTER_TYPE, CommonUIMessages._111, null, null);
			filtersToAdd.add(fq);

			fq = createFilterQuery(TraceUIMessages._216,
					TracePackage.eINSTANCE.getTRCPackage_BaseTime(),
					OrderByOperators.DESC_LITERAL, DO_NOT_ADD_GET_SET_FILTER);
			filtersToAdd.add(fq);

			fq = createFilterQuery(TraceUIMessages._217,
					TracePackage.eINSTANCE.getTRCPackage_CumulativeTime(),
					OrderByOperators.DESC_LITERAL, DO_NOT_ADD_GET_SET_FILTER);
			filtersToAdd.add(fq);

			fq = createFilterQuery(TraceUIMessages._219,
					TracePackage.eINSTANCE.getTRCPackage_Calls(),
					OrderByOperators.DESC_LITERAL, DO_NOT_ADD_GET_SET_FILTER);
			filtersToAdd.add(fq);

			fq = createFilterQuery(TraceUIMessages._218,
					TracePackage.eINSTANCE.getTRCPackage_TotalSize(),
					OrderByOperators.DESC_LITERAL, DO_NOT_ADD_GET_SET_FILTER);
			filtersToAdd.add(fq);

			fq = createFilterQuery(TraceUIMessages._220,
					TracePackage.eINSTANCE.getTRCPackage_TotalInstances(),
					OrderByOperators.DESC_LITERAL, DO_NOT_ADD_GET_SET_FILTER);
			filtersToAdd.add(fq);

			fq = createFilterQuery(TraceUIMessages._236, null, null, DO_NOT_ADD_GET_SET_FILTER);
			filtersToAdd.add(fq);

			FilterResourceHandler.addFiltersOnInstanceLoad(filtersToAdd);
			FilterResourceHandler.defaultFiltersAddedForKey(
					traceViewsPluginDefaultFiltersKey, true);
		}
	}

	private static FilterQueries createFilterQuery(final String name,
			final EStructuralFeature feature, final OrderByOperators operator,
			final boolean addGetSetFilter) {
		FilterQueries filterQuery = FilterResourceHandler.createFilterQueries(
				FILTER_TYPE, name, null, null);
		SimpleSearchQuery query = filterQuery.standard();
		if (feature != null && operator != null) {
			FilterEventsUI.addOrderByExpressionAndSetMaxElements(query, feature,
					operator, 10);
		}
		addGetterThresholdFilters(query, addGetSetFilter);
		return filterQuery;
	}

	private static void addGetterThresholdFilters(
			final SimpleSearchQuery filter, final boolean addGetSetFilter) {
		WhereExpression expr = filter.getWhereExpression();
		if (expr == null) {
			expr = ExtensionsFactory.eINSTANCE.createLogicalExpression();
			((LogicalExpression)expr).setOperator(LogicalOperators.AND_LITERAL);
		}
		if (expr instanceof LogicalExpression) {
			EList arguments = ((LogicalExpression)expr).getArguments();
			if (addGetSetFilter) {
				arguments.add(THRESHOLD_FILTER.createActiveExpression());
				arguments.add(GETSET_FILTER.createActiveExpression());
			} else {
				arguments.add(THRESHOLD_FILTER.createDisabledExpression());
				arguments.add(GETSET_FILTER.createDisabledExpression());
			}
		}
		filter.setWhereExpression(expr);
	}
}