/*******************************************************************************
 * Copyright (c) 2008 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Intel Corporation - Initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.views.internal.fragment;

import org.eclipse.jface.viewers.IElementComparer;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.graphics.Image;

public class ContentProviderNewLazyAdaptor implements ILazyFragmentTreeContentWithIndexProvider {
	private ITreeContentProvider provider;
	private ElementInfo inputElementInfo;
	private TreeViewer viewer;
	private ECWrapper comparerWrapper;
	private ViewerComparator comparator;
	
	private static class TableLPWrapper extends LabelProvider implements ITableLabelProvider {
		private ITableLabelProvider lp;

		TableLPWrapper(ITableLabelProvider provider){
			this.lp = provider;
		}
		
		public void addListener(ILabelProviderListener listener) {
			lp.addListener(listener);
		}

		public void dispose() {
			lp.dispose();
		}

		public Image getColumnImage(Object element, int columnIndex) {
			Object o = getObject(element);
			return lp.getColumnImage(o, columnIndex);
		}

		public String getColumnText(Object element, int columnIndex) {
			Object o = getObject(element);
			return lp.getColumnText(o, columnIndex);
		}

		public boolean isLabelProperty(Object element, String property) {
			Object o = getObject(element);
			return lp.isLabelProperty(o, property);
		}

		public void removeListener(ILabelProviderListener listener) {
			lp.removeListener(listener);
		}
	}
	
	private static class ECWrapper implements IElementComparer{
		private IElementComparer comparer;
		
		public ECWrapper(IElementComparer comparer){
			this.comparer = comparer;
		}
		
		public boolean equals(Object a, Object b) {
			if(comparer == null)
				return a.equals(b);
			return comparer.equals(getObject(a), getObject(b));
		}

		public int hashCode(Object element) {
			if(comparer == null)
				return element.hashCode();
			return comparer.hashCode(getObject(element));
		}
	}
	
	private class ElementInfo {
		ElementInfo parent;
		Object element;
		Object[] children;
		boolean isSorted;
		boolean isChildCountInited;
		Boolean hasChildren;
		
		ElementInfo(ElementInfo parent, Object element){
			if(element instanceof ElementInfo){
				int i = 0;
				i++;
			}
				
			this.parent = parent;
			this.element = element;
		}
		
		public void elementChanged(Object element){
			this.element = element;
			reset();
		}

		public void reset(){
			children = null;
			isSorted = false;
			hasChildren = null;
		}

		public int hashCode() {
			return element.hashCode();
		}

		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			ElementInfo other = (ElementInfo) obj;
			return element.equals(other.element);
		}
		
		public ContentProviderNewLazyAdaptor getContainer(){
			return ContentProviderNewLazyAdaptor.this;
		}
		
		public Object getParent(){
			if(parent == null){
				if(equals(inputElementInfo))
					return null;
				Object parentEl = provider.getParent(element); 
				if(parentEl == null){
					if(inputElementInfo.indexOf(this.element) >= 0){
						parent = inputElementInfo;
						return inputElementInfo.element;
					}
					return null;
				}
				parentEl = toViewerElement(parentEl);
				if(parentEl instanceof ElementInfo){
					parent = (ElementInfo)parentEl;
				} else {
					parent = inputElementInfo;
				}
			}
			if(parent.equals(inputElementInfo))
				return parent.element;
			return parent;
		}
		
		public Object[] getSortedChildren(){
			getRawChildren();
			if(!isSorted){
				if(children.length != 0 && comparator != null){
					comparator.sort(viewer, children);
				}
				if(children.getClass() != Object[].class){
					Object[] tmp = new Object[children.length];
					System.arraycopy(children, 0, tmp, 0, tmp.length);
					children = tmp;
				}
				isSorted = true;
			}
			return children;
		}

		public int indexOf(Object element){
			Object[] children = getSortedChildren();
			if(element instanceof ElementInfo)
				element = ((ElementInfo)element).element;
			for(int i = 0; i < children.length; i++){
				Object c = children[i];
				if(c instanceof ElementInfo)
					c = ((ElementInfo)c).element;
				if(element.equals(c))
					return i;
			}
			return -1;
		}

		private Object[] getRawChildren(){
			if(children == null){
				if(this.equals(inputElementInfo))
					children = provider.getElements(element);
				else
					children = provider.hasChildren(element) ? provider.getChildren(element) : new Object[0];
				if(children == null)
					children = new Object[0];
			}
			return children;
		}
		
		public boolean hasChildren(){
			if(hasChildren == null){
				if(this.equals(inputElementInfo))
					hasChildren = Boolean.TRUE;
				else
					hasChildren = Boolean.valueOf(provider.hasChildren(element));
			}
			return hasChildren.booleanValue();
		}
		
		public int getChildrenLength(){
			int length = getRawChildren().length; 
			return length;
		}
	}
	
	public ContentProviderNewLazyAdaptor(ITreeContentProvider cp, 
			IElementComparer comparer,
			ViewerComparator c){
		this.provider = cp;
		this.comparerWrapper = new ECWrapper(comparer);
		this.comparator = c;
	}
	
	public static Object getObject(Object element){
		return element instanceof ElementInfo ? ((ElementInfo)element).element : element;
	}
	
	public Object toViewerElement(Object element){
		if(element instanceof ElementInfo){
			ElementInfo info = (ElementInfo)element;
			if(info.getContainer() == this)
				return info; 
			element = info.element;
		} else if(element.equals(inputElementInfo.element)){
			return element;
		}
		return new ElementInfo(null, element);
	}

	public Object getParent(Object element) {
		ElementInfo info = getElementInfo(element);
		return info != null ? info.getParent() : null;
	}

	public void updateChildCount(Object element, int currentChildCount, IProviderContext context) {
		ElementInfo info = getElementInfo(element);
		if(info == null)
			return; 
		int count = info.getChildrenLength();
		if(!info.isChildCountInited)
			info.isChildCountInited = true;
		context.setChildCount(element, count);
	}

	public void updateElement(Object parent, int index, IProviderContext context) {
		ElementInfo info = getElementInfo(parent);
		if(info == null)
			return;
//		if(!info.isChildCountInited){
//			info.isChildCountInited = true;
//			if(info.hasChildren()){
//				viewer.setChildCount(parent, info.getChildrenLength());
//			}
//		}
		
		Object[] children = info.getSortedChildren();
		
		Object c = children[index];
		if(c instanceof ElementInfo)
			c = ((ElementInfo)c).element;
		ElementInfo child = info.equals(inputElementInfo) ? new ElementInfo(null, c) : new ElementInfo(info, c);
		children[index] = child;
		
		context.replace(parent, index, child);
		
		context.setHasChildren(child, child.hasChildren());
//		updateChildCount(child, 0);
	}
	

//	private int getChildrenLength(Object element){
//		ElementInfo info = getElementInfo(element);
//		return info.getChildrenLength();
//	}

	private ElementInfo getElementInfo(Object element){
		if(element == null || element.equals(inputElementInfo.element)){
			return inputElementInfo;
		}
		return element instanceof ElementInfo ? (ElementInfo)element : null;
	}

	public void dispose() {
		inputElementInfo = null;
		provider.dispose();
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		if(inputElementInfo == null) {
			inputElementInfo = new ElementInfo(null, newInput);
			inputElementInfo.isChildCountInited = true;
		} else {
			inputElementInfo.elementChanged(newInput);
		}
		
		this.viewer = (TreeViewer)viewer;
		provider.inputChanged(viewer, oldInput, newInput);
	}

	public ITableLabelProvider wrapLableProvider(ITableLabelProvider provider){
		return new TableLPWrapper(provider);
	}
	
	public IElementComparer getComparerWrapper(){
		return comparerWrapper.comparer != null ? comparerWrapper : null;
	}
	
	public void setViewerComparator(ViewerComparator c){
		if(inputElementInfo != null){
			inputElementInfo.reset();
		}
		this.comparator = c;
	}

	public int indexOf(Object parent, Object element) {
		ElementInfo elInfo = getElementInfo(parent);
		return elInfo != null ? elInfo.indexOf(element) : -1;
	}
	
	public ITreeContentProvider getWrappedContentProvider(){
		return provider;
	}
}
