/*******************************************************************************
 * Copyright (c) 2008 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Intel Corporation - Initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.views.internal.fragment;

import java.util.Iterator;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.viewers.TreePath;

public class Expansion implements IElementInfo, Comparable{
	private int directChildSize;
	private int allChildSize;
	private Object element;
	private Expansion parent;
	private TreeMap indexToChildMap;
	private boolean isCollapsed;
	private Integer index;
	private Integer rootIndex;
	private RootExpansion rootExpansion;
	private boolean isVirtual;
	private boolean isOutOfRangeMarker;
	
	Expansion(Expansion parent, Expansion base){
		copyFrom(parent, base);
	}
	
	private void copyFrom(Expansion parent, Expansion base){
		this.directChildSize = base.directChildSize;
		this.allChildSize = base.allChildSize;
		this.element = base.element;
		this.parent = parent;
		if(base.indexToChildMap != null && base.indexToChildMap.size() != 0){
			this.indexToChildMap = (TreeMap)base.indexToChildMap.clone();
			for(Iterator iter = this.indexToChildMap.entrySet().iterator(); iter.hasNext();){
				Map.Entry entry = (Map.Entry)iter.next();
				entry.setValue(new Expansion(this, (Expansion)entry.getValue()));
			}
		}
		this.isCollapsed = base.isCollapsed;
		this.index = base.index;
		this.rootIndex = base.rootIndex;
		if(parent != null) {
			this.rootExpansion = parent.getRoot();
		} else {
			this.rootExpansion = (RootExpansion)parent;
		}
		this.isVirtual = base.isVirtual;
		this.isOutOfRangeMarker = base.isOutOfRangeMarker;
	}
	
	Expansion(Object element, int size){
		setInfo(element, size);
	}
	
	void IAmRoot(){
		if(getClass() == RootExpansion.class)
			rootExpansion = (RootExpansion)this;
	}

	public Expansion expandChild(int elementIndex, Object childElement, int childSize){
		if(indexToChildMap != null && indexToChildMap.size() != 0){
			Integer i = new Integer(elementIndex);
			Expansion exp = (Expansion)indexToChildMap.get(i);
			if(exp != null){
				if(exp.directChildSize == childSize 
						&& exp.element.equals(childElement) 
						&& exp.isCollapsed){
					exp.expand();
					return exp;
				}
				Assert.isTrue(false);	
			}
		}
		return createChild(elementIndex, childElement, childSize, false, false, true);
	}
	
	public Object getData(){
		return element;
	}

	private Expansion createChild(int elementIndex, Object childElement, int childSize, boolean isOutOfRangeMarker, boolean virtual, boolean expanded){
		validateChildIndex(elementIndex, isOutOfRangeMarker);
		if(isOutOfRangeMarker)
			virtual = true;
		if(!virtual){
			validateExpansionInfo(childElement, childSize);
		} else {
			expanded = false;
//			childElement = null;
//			childSize = 0;
		}
		Integer i = new Integer(elementIndex);
		Expansion child = new Expansion(childElement, childSize);
		child.isOutOfRangeMarker = isOutOfRangeMarker;
		child.isVirtual = virtual;
		child.isCollapsed = !expanded;
		linkChild(i, child);
		Assert.isTrue(child.getParent() == this);
		return child;
	}
	
	public Expansion getCommonRelative(Expansion other){
		return getCommonRelative(this, other);
	}
	
	public static Expansion getCommonRelative(Expansion exp1, Expansion exp2){
		if(exp1.equals(exp2))
			return exp1;
		if(!exp1.getRoot().equals(exp2.getRoot()))
			return null;
		
		int d1 = exp1.getDepth();
		int d2 = exp2.getDepth();
		int d;
		if(d1 > d2){
			int delta = d1 - d2;
			for(int i = 0; i < delta; i++){
				exp1 = exp1.getParent();
			}
			d = d2;
		} else if (d2 > d1){
			int delta = d2 - d1;
			for(int i = 0; i < delta; i++, exp2 = exp2.getParent());
			d = d1;
		} else {
			d = d1;
		}
		if(d == 0){
			return exp1;
		}
		for(int i = 0; i < d; i++, exp1 = exp1.getParent(), exp2 = exp2.getParent()){
			if(exp1.equals(exp2))
				return exp1;
		}
		
		//should never be here
		Assert.isTrue(false);
		return null;
	}

	private void validateChildIndex(int index, boolean isOutOfRangeMarker){
		Assert.isTrue(!(index >= (isOutOfRangeMarker ? directChildSize + 1 : directChildSize) 
				|| index < 0));
	}
	
	public boolean isVirtual(){
		return isVirtual;
	}
	
	public int getDepth(){
		int i = 0;
		for(Expansion exp = this; !exp.equals(exp.getRoot()); exp = exp.getParent(), i++);
		return i;
	}
	
	public void materialize(Object element, int childSize, boolean expand){
		Assert.isTrue(!(this.isOutOfRangeMarker
				|| !this.isVirtual
				|| this.directChildSize != 0
				|| this.allChildSize != 0
				|| (this.element != null && !this.element.equals(element))));
		
		validateExpansionInfo(element, childSize);
		parent.validateChildIndex(index.intValue(), false);
		setInfo(element, childSize);
		isCollapsed = !expand;
		isVirtual = false;
		parent.linkChild(index, this);
	}
	
	private void setInfo(Object element, int childSize){
		this.directChildSize = childSize;
		this.allChildSize = childSize;
		this.element = element;
	}
	
//	private void internalSetVirtual(){
//		if(isRoot())
//			throw new UnsupportedOperationException();
//		
//		isVirtual = true;
//	}
	
	public void collapse(boolean preserveChildInfo){
		Assert.isTrue(!isVirtual);

		if(isCollapsed)
			return;
		
		if(parent != null){
			if(preserveChildInfo && hasChildExpansions()){
				if(allChildSize != 0){
					parent.childSizeChanged(this, -allChildSize);
				}
				isCollapsed = true;
			} else {
				virtualize();
			}
		}
	}
	
	public void virtualize(){
		if(isVirtual)
			return;
		
		if(isCollapsed){
			allChildSize = directChildSize = 0;
		}
		Expansion curParent = parent;
		Integer curIndex = index; 
		parent.unlinkChild(this);
		
		indexToChildMap = null;
		allChildSize = directChildSize = 0;
		isVirtual = true;
		isCollapsed = true;

		parent = curParent;
		parent.linkChild(curIndex, this);
	}


//	public boolean hasChildExpansions(int startOffset, int endOffset, boolean checkCollapsed){
//		if(startOffset > endOffset)
//			throw new IllegalArgumentException();
//		
//		if(allChildSize == 0)
//			return false;
//		
//		if(endOffset > allChildSize) 
//			endOffset = allChildSize;
//		
//		if(endOffset < startOffset)
//			return false;
//		
//		Expansion startExp = getChildExpansionByOffset(startOffset, true);
//		Expansion endExp = getChildExpansionByOffset(endOffset, true);
//		if(!startExp.isVirtual() || !endExp.isVirtual())
//			return true;
//		int index1 = startExp.getIndex();
//		int index2 = endExp.getIndex();
//		if(index1 == index2)
//			return false;
//		
//		SortedMap subMap = getIndexToChildMap(false).subMap(index1 + 1, index2);
//		if(subMap.isEmpty())
//			return false;
//		
//		for(Iterator iter = subMap.values().iterator(); iter.hasNext();){
//			Expansion exp = (Expansion)iter.next();
//			if(checkCollapsed || !exp.isCollapsed)
//				return true;
//		}
//		
//		return false;
//		
//	}
	
	public void expand(){
		Assert.isTrue(!isVirtual);
		
		if(!isCollapsed)
			return;

		isCollapsed = false;
		
		if(parent != null){
			if(allChildSize != 0){
				parent.childSizeChanged(this, allChildSize);
			}
		}
	}

	private void unlinkChild(Expansion child){
		Integer index = child.index;
		child.reverseUnlinkParent();
		rootExpansion.unlinkExpansion(child);
		getIndexToChildMap(false).remove(index);
	}
	
	public int getDirectChildSize(boolean checkExpanded){
		return checkExpanded && isCollapsed ? 0 : directChildSize;
	}

	public int getDirectChildCount(){
		return getDirectChildSize(false);
	}
	
	public int getAllChildSize(boolean checkExpanded){
		return checkExpanded && isCollapsed ? 0 : allChildSize;
	}
	
	public boolean hasChildExpansions(){
		return allChildSize != directChildSize;
	}

	public boolean hasDirectChildren(boolean expanded){
		if(!hasChildExpansions())
			return false;
		
		return hasDirectChildren(indexToChildMap, expanded);
	}

	public boolean hasDirectChildren(int startIndex, int stopIndex, boolean expanded){
		if(!hasChildExpansions())
			return false;
		
		SortedMap map = indexToChildMap.subMap(new Integer(startIndex), new Integer(stopIndex));
		return hasDirectChildren(map, expanded);
	}

	public boolean hasDirectChildren(SortedMap map, boolean expanded){
		if(map == null || map.size() == 0)
			return false;
		
		for(Iterator iter = map.values().iterator(); iter.hasNext();){
			Expansion child = (Expansion)iter.next();
			if(child.isExpanded() == expanded)
				return true;
		}
		return false;
	}

	private void reverseUnlinkParent(){
		if(parent != null){
			if(!isCollapsed && allChildSize != 0){
				parent.childSizeChanged(this, -allChildSize);
			}
			this.parent = null;
			index = null;
		}
	}
	
	public Expansion getDirectExistingChildByIndex(int index){
		if(indexToChildMap == null || indexToChildMap.size() == 0)
			return null;
		
		return (Expansion)indexToChildMap.get(new Integer(index));
	}

	private void linkChild(Integer index, Expansion child){
		if(!child.isCollapsed && !child.isVirtual){
			TreeMap iMap = getIndexToChildMap(true);
			Assert.isTrue(iMap.get(index) == null);
			iMap.put(index, child);
		}
		if(!child.isVirtual())
			rootExpansion.linkExpansion(child);
		child.reverseLinkParent(this, index, child.isVirtual ? null : new Integer(rootIndexOf(index)));
	}
	
//	private int calcChildRootIndex(Integer childIndex){
//		int root = rootIndex != null ? rootIndex.intValue() + childIndex.intValue() + 1 : childIndex.intValue();
//		if()
//		SortedMap map = getIndexToChildMap(false).headMap(childIndex);
//		for(Iterator iter = map.values().iterator(); iter.hasNext();){
//			Expansion pre = (Expansion)iter.next();
//			if(!pre.isCollapsed)
//				root += pre.allChildSize;
//		}
//		return root;
//	}
	
	private void reverseLinkParent(Expansion parent, Integer index, Integer rootIndex){
		this.parent = parent;
		this.index = index;
		this.rootIndex = rootIndex;
		this.rootExpansion = parent.rootExpansion;
		if(!isCollapsed)
			parent.childSizeChanged(this, allChildSize);
	}
	
	public boolean isRoot(){
		return rootExpansion == this;
	}
	
	public RootExpansion getRoot(){
		return rootExpansion;
	}
	
	private void childSizeChanged(Expansion directChild, int change){
		if(change == 0)
			return;
//		if(change < 0)
//			throw new IllegalArgumentException();
		int newAllSize = allChildSize + change;
		Assert.isTrue(newAllSize >= 0);
		allChildSize = newAllSize;
		SortedMap subMap = getIndexToChildMap(false).tailMap(new Integer(directChild.index.intValue() + 1));
		changeGlobalIndexForChildren(subMap, change);
		if(parent != null && !isCollapsed)
			parent.childSizeChanged(this, change);
	}
	
	public int getRootIndex(){
		if(isVirtual)
			return parent.rootIndexOf(this);
		return rootIndex != null ? rootIndex.intValue() : -1;
	}
	
	private void changeGlobalIndex(int change){
		int newRootIndex = rootIndex.intValue() + change;
		Assert.isTrue(newRootIndex >= 0);

		if(change != 0){
			rootIndex = new Integer(newRootIndex);
			TreeMap map = getIndexToChildMap(false);
			if(map != null){
				changeGlobalIndexForChildren(map, change);
			}
		}
	}

	private int rootIndexOf(Expansion directChild){
		return rootIndexOf(directChild.index);
	}

	private int rootIndexOf(Integer index){
		int indexIntValue = index.intValue();
		if(indexToChildMap == null || indexToChildMap.size() == 0)
			return getRootIndex() + indexIntValue + 1;
		SortedMap subMap = indexToChildMap.headMap(index);
		if(subMap.isEmpty())
			return getRootIndex() + indexIntValue + 1;
		
		Expansion exp = (Expansion)subMap.get(subMap.lastKey());
		int result = exp.rootIndex.intValue() + /*exp.getAllChildSize(true) +*/ indexIntValue - exp.index.intValue();
//		if(!exp.isCollapsed)
		result += exp.getAllChildSize(true);
		return result;
	}
	
	private void changeGlobalIndexForChildren(SortedMap map, int change){
		Iterator iter = map.values().iterator();
		while(iter.hasNext()){
			Expansion exp = (Expansion)iter.next();
			exp.changeGlobalIndex(change);
		}
	}
	
	private TreeMap getIndexToChildMap(boolean create){
		if(indexToChildMap == null && create)
			indexToChildMap = new TreeMap();
		return indexToChildMap;
	}

	private static void validateExpansionInfo(Object element, int size) {
		Assert.isTrue(!(element == null
				|| size < 0));
	}

	public Expansion getDirectChildByIndex(int index, boolean generateOutOfRangeMarker){
		validateChildIndex(index, isOutOfRangeMarker);
		if(generateOutOfRangeMarker && index == directChildSize)
			return createChild(index, null, 0, true, true, false);
		Expansion result = null;
		if(indexToChildMap != null && indexToChildMap.size() != 0)
			result = (Expansion)indexToChildMap.get(new Integer(index));
		if(result != null)
			return result;
		return createChild(index, null, 0, false, true, false);
	}

	public Expansion getChildExpansionByOffset(int offset, boolean generateOutOfRangeMarker){
		if(isCollapsed || isVirtual)
			return null;
		if(allChildSize <= offset){
			if(generateOutOfRangeMarker && allChildSize == offset)
				return createChild(directChildSize, null, 0, true, true, false);
			return null;
		}
		
		TreeMap map = getIndexToChildMap(false);
		if(map == null || map.size() == 0){
			return createChild(offset, null, 0, false, true, false);
		}

		int rootIndex = getRootIndex();
		
		int tailNonexpendedChildOffset = 0;
		int tailNonExpendedChildIndex = 0;
		for(Iterator iter = map.values().iterator(); iter.hasNext(); ){
			Expansion child = (Expansion)iter.next();
			int childOffset = child.getRootIndex() - rootIndex - 1;
			if(childOffset == offset){
				return child;
			} else if(childOffset < offset) {
				int childSize = child.getAllChildSize(true);
				if(childOffset + childSize >= offset){
					return child.getChildExpansionByOffset(offset - childOffset - 1, false);
				}
				tailNonexpendedChildOffset = childOffset + childSize + 1;
				tailNonExpendedChildIndex = child.getIndex() + 1;
			} else /*if (delta > offset)*/{
				break;
//				if(child.isCollapsed){
//					offsetOffset += 1;
//				} else {
//					if(child.getAllChildSize(false) >= delta - offset){
//						return child.getChildExpansionByOffset(delta - offset - 1, false);
//					} else {
//						offsetOffset += child.getAllChildSize(false) + 1;
//					}
//				}
			}
		}
		
		Assert.isTrue(offset >= tailNonexpendedChildOffset);
		return createChild(tailNonExpendedChildIndex + offset - tailNonexpendedChildOffset, null, 0, isOutOfRangeMarker, true, false);
	}

//	public Expansion getExpansionReversed(int offset){
//		return getExpansion(allChildSize - offset - 1);
//	}

	public boolean isChildOf(Expansion exp){
		for(Expansion p = parent; p != null; p = p.parent){
			if(p == exp)
				return true;
		}
		return false;
	}
	
	public boolean isParentOf(Expansion exp){
		return exp.isChildOf(this);
	}
	
	public Expansion getParent(){
		return parent;
	}
	
	public int getIndex(){
		return index.intValue();
	}
	
	public boolean isOutOfRangeMarker(){
		return isOutOfRangeMarker;
	}
	
	public Expansion getRelativeExpansion(int offset){
		if(offset == 0)
			return this;
		if(offset > 0) {
			if(!isCollapsed && allChildSize >= offset){
				return getChildExpansionByOffset(offset - 1, false);
			} else if (parent != null){
				return parent.getRelativeExpansion(offset + getRootIndex() - parent.getRootIndex());
			} 
			return null;
		}
		if(getRootIndex() < -offset)
			return null;
		if(parent != null)
			return parent.getRelativeExpansion(offset + getRootIndex() - parent.getRootIndex());
		return null;
	}
	
	public Expansion[] getParents(boolean includeThis, boolean bottomUp){
		int depth = getDepth();
		Expansion[] parents = new Expansion[includeThis ? depth + 1 : depth];
		
		if(bottomUp){
			int cursor = parents.length - 1;
			Expansion start = includeThis ? this : parent;
			for(;start != null; start = start.parent){
				parents[cursor--] = start;
			}
		} else {
			int cursor = 0;
			Expansion start = includeThis ? this : parent;
			for(;start != null; start = start.parent){
				parents[cursor++] = start;
			}
		}
		return parents;
	}
	
	public boolean isExpanded(){
		return !isCollapsed;
	}

	public boolean isVisible(){
		return parent != null ? !parent.isCollapsed && parent.isVisible() : true;
	}

	public Expansion[] getDirectChildren(){
		if(indexToChildMap == null || indexToChildMap.size() == 0)
			return new Expansion[0];
		
		Expansion[] result = new Expansion[indexToChildMap.size()];
		indexToChildMap.values().toArray(result);
		return result;
	}

	public Expansion[] getDirectChildren(int startIndex, int endIndex, boolean expanded){
		assertStartEndIndex(startIndex, endIndex);
		if(indexToChildMap == null || indexToChildMap.size() == 0)
			return new Expansion[0];
		
		SortedMap map = indexToChildMap.subMap(new Integer(startIndex), new Integer(endIndex));
		return getDirectChildren(map, expanded);
	}
	
	public boolean isDirectChildExpanded(int index){
		return indexToChildMap != null && indexToChildMap.size() != 0 && indexToChildMap.containsKey(new Integer(index));
	}
	
	private void assertStartEndIndex(int startIndex, int endIndex){
		Assert.isTrue(startIndex >= 0
				&& startIndex <= endIndex
				&& endIndex <= directChildSize);
	}

	public Expansion[] getDirectChildren(boolean expanded){
		return getDirectChildren(indexToChildMap, expanded);
	}

	private Expansion[] getDirectChildren(SortedMap map, boolean expanded){
		if(map == null || map.size() == 0)
			return new Expansion[0];
		
		Expansion[] result = new Expansion[map.size()];
		int cursor = 0;
		for(Iterator iter = map.values().iterator(); iter.hasNext();){
			Expansion exp = (Expansion)iter.next();
			if(exp.isExpanded() == expanded){
				result[cursor++] = exp;
			}
		}
		if(cursor < result.length){
			if(cursor == 0){
				result = new Expansion[0];
			} else {
				Expansion tmp[] = new Expansion[cursor];
				System.arraycopy(result, 0, tmp, 0, cursor);
				result = tmp;
			}
		}
		return result;
	}

	/**
	 * hashCode and equals behavior:
	 * same virtual expansion may be presented with different objects
	 * this is NOT true for "real" objects  
	 */
	
	public int hashCode() {
		if(!isVirtual)
			return super.hashCode();
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ rootExpansion.hashCode();
		result = prime * result + rootIndex.hashCode();
		return result;
	}

	public boolean equals(Object obj) {
		if(!isVirtual)
			return super.equals(obj);
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Expansion other = (Expansion) obj;
		if(!other.isVirtual)
			return false;

		if(!rootExpansion.equals(other.rootExpansion))
			return false;
		
		if(getRootIndex() != other.getRootIndex())
			return false;
		
		return true;
	}
	
	public TreePath toTreePath(){
		Assert.isTrue(element != null);
		Object[] path = new Object[getDepth()];
		int cursor = path.length-1;
		for(Expansion start = this; !start.equals(rootExpansion); start = start.parent){
			path[cursor--] = start.element;
		}

		return new TreePath(path);
	}

	public int compareTo(Object o) {
		return getRootIndex() - ((Expansion)o).getRootIndex();
	}
	
	public void setData(Object data){
		Assert.isTrue(isVirtual || element.equals(data));
		element = data;
	}
}
