/*******************************************************************************
 * Copyright (c) 2008 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Intel Corporation - Initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.views.internal.fragment;

import org.eclipse.core.runtime.Assert;

public class ExpansionRange {
	private Expansion base;
	private int startOffset;
	private int size;
	
	private Expansion startExpansion;
	private Expansion endExpansion;
	private Expansion lastIncludedExpansion;

	private ExpansionRange(Expansion base, int offset, int size){
		this.base = base;
		this.startOffset = offset;
		this.size = size;
	}
	
	private static void assertRange(Expansion base, int offset, int size){
		Assert.isTrue(size >= 0); 
		Assert.isTrue(offset >= 0);
		Assert.isTrue(base.isExpanded());
		Assert.isTrue(base.getAllChildSize(false) >= offset);
		Assert.isTrue(base.getAllChildSize(false) - offset >= size);
	}

	public static ExpansionRange createExpansionRange(Expansion base, int offset, int size){
		assertRange(base, offset, size);
		return new ExpansionRange(base, offset, size);
	}
	
	public int getSize(boolean includeReferencedParents){
		return includeReferencedParents ? size + getNumParentRefs() : size;
		
	}
	
	public int getDepth(Expansion exp){
		for(int i = 0; exp != null; exp = exp.getParent(), i++){
			if(exp.equals(base))
				return i;
		}
		return -1;
	}

	public void clearCache(){
		startExpansion = null;
		endExpansion = null;
		lastIncludedExpansion = null;
	}
	
	public Expansion getStartExpansion(){
		if(startExpansion == null){
			if(size == 0)
				return null;
			startExpansion = base.getChildExpansionByOffset(startOffset, false);
		}
		return startExpansion;
	}

	private Expansion getEndExpansion(){
		if(endExpansion == null){
			if(startOffset + size == base.getAllChildSize(false))
				return null;
			endExpansion = base.getChildExpansionByOffset(startOffset + size, false);
		}
		return endExpansion;
	}
	
	public Expansion getLastIncludedExpansion(){
		if(lastIncludedExpansion == null && size != 0){
			lastIncludedExpansion = base.getChildExpansionByOffset(startOffset + size - 1, false);
		}
		return lastIncludedExpansion;
	}

	public int getMaxMove(boolean forward){
		return forward ? getMaxForwardMove() : getMaxBackwardMove();
	}
	
	public int getMaxForwardMove(){
		int move = base.getAllChildSize(false) - startOffset - size;
		Assert.isTrue(move >= 0);
		return move;
	}

	public int getMaxBackwardMove(){
		return startOffset;
	}

	public int getStartOffset(){
		return startOffset;
	}

	public int limitByMaxMove(int num){
		return num > 0 ? Math.min(num, getMaxForwardMove()) : Math.min(num, getStartOffset());
	}

	public ExpansionRange moveToStart(int newStart, boolean preservingSize){
		assertValid();
		Assert.isTrue(newStart >= 0);
		int newSize = size;
		int childSize = base.getAllChildSize(false);
		if(newStart + newSize > childSize){
			if(preservingSize){
				newStart = childSize - newSize;
			} else {
				newSize = childSize - newStart; 
			}
		}
		return createExpansionRange(base, newStart, newSize);
	}
	
	public ExpansionRange getInterection(ExpansionRange other){
		ExpansionRange result;
		if(startOffset == other.startOffset){
			result = other.size > size ? this : other;
		} else {
			int endOffset = startOffset + size;
			int otherEndOffset = other.startOffset + other.size;
			if(startOffset > other.startOffset){
				if(endOffset <= otherEndOffset){
					result = this;
				} else if (startOffset <= otherEndOffset){
					result = createExpansionRange(base, startOffset, otherEndOffset - startOffset);
				} else {
					result = null;
				}
			} else {
				if(otherEndOffset <= endOffset){
					result = other;
				} else if (other.startOffset <= endOffset){
					result = createExpansionRange(base, other.startOffset, endOffset - other.startOffset);
				} else {
					result = null;
				}
			}
		}
		
		return result;
	}
	
	public static final int NONE = 0;
	public static final int CONTAINMENT = 1;
	public static final int REFERENCE = 1 << 1;
	
	public int getRelation(Expansion exp){
		if(exp.equals(base))
			return REFERENCE;
		if(size == 0)
			return NONE;
		if(!exp.isChildOf(base))
			return NONE;
		int ri = internalGetRawIndex(exp);
		if(ri >= startOffset && ri < startOffset + size)
			return CONTAINMENT;
		if(exp.isParentOf(getLastIncludedExpansion())
				|| exp.isParentOf(getStartExpansion())){
			return REFERENCE;
		}
		return NONE;
	}
	
	private int internalGetRawIndex(Expansion exp){
		return exp.getRootIndex() - base.getRootIndex() - 1;
	}
	
	public Expansion getBase(){
		return base;
	}
	
	public int getNumParentRefs(){
		Expansion exp = getStartExpansion();
		if(exp == null)
			return size;
		int depth = getDepth(exp);
		Assert.isTrue(depth > 0);
		return depth - 1; //we are not including the base Expansion 
	}
	
	public int getStartIndex(final Expansion exp){
		switch(getRelation(exp)){
		case NONE:
			return -1;
		case CONTAINMENT:
			return 0;
		case REFERENCE:
			Expansion expDirectChild = getStartExpansion();
			if(expDirectChild == null){
				return -1;
			}
			for(;!expDirectChild.getParent().equals(exp); expDirectChild = expDirectChild.getParent());
			return expDirectChild.getIndex();
		default:
			Assert.isTrue(false);
			return 0;
		}
	}

	public int getEndIndex(Expansion exp){
		switch(getRelation(exp)){
		case NONE:
			return -1;
		case CONTAINMENT:
			Expansion exp2 = getLastIncludedExpansion();
			if(exp2 == null)
				return -1;//exp.getDirectChildSize(false);
			if(exp2.equals(exp))
				return 0;
			for(Expansion p = exp2.getParent(); p != null; exp2 = p, p = p.getParent()){
				if(p.equals(exp))
					return exp2.getIndex() + 1;
			}
			
			return exp.getDirectChildSize(false);
		case REFERENCE:
			Expansion expDirectChild = getLastIncludedExpansion();
			if(expDirectChild == null)
				return -1;
			if(!expDirectChild.isChildOf(exp)){
				return exp.getDirectChildSize(false);
			}
//				expDirectChild = getStartExpansion();
//			Assert.isTrue(expDirectChild != null || exp.equals(base));
			
			for(; !expDirectChild.getParent().equals(exp); expDirectChild = expDirectChild.getParent());
			return expDirectChild.getIndex() + 1;
		default:
			Assert.isTrue(false);
			return 0;
		}
	}

	public int getIndex(Expansion exp){
		int offset = getStartIndex(exp.getParent());
		if(offset == -1)
			return -1;
		return exp.getIndex() - offset;
	}

	public int getDirectChildCount(Expansion exp){
		int offset = getStartIndex(exp);
		if(offset == -1)
			return -1;
		int endIndex = getEndIndex(exp);
		Assert.isTrue(endIndex != -1
				&& offset <= endIndex);
		int result = endIndex - offset;
		Assert.isTrue(result <= size);
		return result;
	}
	
	public Expansion[] getDirectChildren(Expansion exp, boolean expanded){
		int offset = getStartIndex(exp);
		if(offset == -1)
			return new Expansion[0];
		int endIndex = getEndIndex(exp);
		Assert.isTrue(endIndex != -1
				&& offset <= endIndex);

		return exp.getDirectChildren(offset, endIndex, true);
	}
	
	public boolean hasDirectChildren(Expansion exp, boolean expanded){
		int offset = getStartIndex(exp);
		if(offset == -1)
			return false;
		int endIndex = getEndIndex(exp);
		Assert.isTrue(endIndex != -1
				&& offset <= endIndex);

		return exp.hasDirectChildren(offset, endIndex, true);
	}
	
	public ExpansionRange adjustOnExpand(Expansion exp){
		Assert.isTrue(exp.isChildOf(base));
		int newStart, newSize;
		int expIndex = internalGetRawIndex(exp);
		if(expIndex < startOffset){
			//change occured before the range
			newStart = startOffset + exp.getAllChildSize(false);
			newSize = size;
		} else if (expIndex < startOffset + size - 1){
			//change occured within the range
			newStart = startOffset;
			newSize = size + exp.getAllChildSize(false);
		} else {
			//change occured after the range
			newStart = startOffset;
			newSize = size;
		}
		
		return createExpansionRange(base, newStart, newSize);
	}
	
	public ExpansionRange adjustOnCollapse(Expansion exp){
		Assert.isTrue(exp.isChildOf(base));
		int newStart, newSize;
		int expIndex = internalGetRawIndex(exp);
		int childSize = exp.getAllChildSize(false);
		int nextExpIndex = expIndex + childSize + 1;
		if(expIndex < startOffset){
			//change occured before the range
			if(nextExpIndex <= startOffset){
				newStart = startOffset - exp.getAllChildSize(false);
				newSize = size;
			} else {
				newStart = expIndex + 1;
				if(nextExpIndex <= startOffset + size){
					newSize = startOffset + size - nextExpIndex;
				} else {
					newSize = 0;
				}
			}
		} else if (expIndex < startOffset + size - 1){
			//change occured within the range
			newStart = startOffset;
			if(nextExpIndex <= startOffset + size){
				newSize = size - exp.getAllChildSize(false);
			} else {
				newSize = expIndex - startOffset + 1;
			}
			
		} else {
			//change occured after the range
			newStart = startOffset;
			newSize = size;
		}
		
		return createExpansionRange(base, newStart, newSize);
	}

	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((base == null) ? 0 : base.hashCode());
		result = prime * result + size;
		result = prime * result + startOffset;
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ExpansionRange other = (ExpansionRange) obj;
		if (base == null) {
			if (other.base != null)
				return false;
		} else if (!base.equals(other.base))
			return false;
		if (size != other.size)
			return false;
		if (startOffset != other.startOffset)
			return false;
		return true;
	}
	
	public void assertValid(){
		assertRange(base, startOffset, size);
	}
	
	public ExpansionRange expand(Expansion exp, Object data, int childCount){
		if(exp.isExpanded())
			return this;
		
		if(exp.isVirtual()){
			Assert.isTrue(childCount >= 0);
			exp.materialize(data, childCount, true);
		} else {
			Assert.isTrue(childCount == -1 || exp.getDirectChildSize(false) == childCount);
			Assert.isTrue(data == null || exp.getData().equals(data));
			exp.expand();
		}

//		if(getRelation(exp) != CONTAINMENT)
//			return this;
		
		return adjustOnExpand(exp);
	}
	
	public ExpansionRange collapse(Expansion exp, boolean preserveChildInfo){
		if(!exp.isExpanded())
			return this;
		
		ExpansionRange newRange = adjustOnCollapse(exp);

		exp.collapse(preserveChildInfo);
		
		return newRange;
	}
}
