/*******************************************************************************
 * Copyright (c) 2008 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Intel Corporation - Initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.views.internal.fragment;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeExpansionEvent;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseWheelListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.TreeEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;

public class FragmentedTreeViewer extends TreeViewer {

	private boolean contentProviderIsLazyFragmented;
	private boolean contentProviderSupportsIndexInfo;

	private int numItemsInFragmentMode = 500;
	
	private int fragmentModeThreashold = 10000;
	
	private boolean preservingSelection;
	
	private ExpansionRange range;
	
	private MoveInfo movingInfo;

	private RootExpansion rootExpansion;
	
	private int itemHeight = -1;
	
	private abstract class BaseMaterializationContext implements IProviderContext, IElementInfo {
		private int childCount;
		protected Object element;
		private Boolean hasChildren;
		
		BaseMaterializationContext(){
			childCount = -1;
		}

		BaseMaterializationContext(Object element){
			this.element = element;
			this.childCount = -1;
		}

		public void replace(Object parent, int index, Object element) {
			if(getIndex() == index && getParent().equals(parent)){
				if(this.element == null){
					this.element = element;
				} else {
					Assert.isTrue(this.element.equals(element));
				}
			}
		}
		
		protected abstract Object getParent();
		public abstract int getIndex();
		
		public void setChildCount(Object element, int count) {
			if(element.equals(this.element)){
				this.childCount = count;
			}
		}

		public void setHasChildren(Object element, boolean hasChildren) {
			if(element.equals(this.element)){
				this.hasChildren  = Boolean.valueOf(hasChildren);
			}
		}
		
		public Object getElement(){
			if(this.element == null){
				ILazyFragmentTreeContentProvider provider = (ILazyFragmentTreeContentProvider)getContentProvider();
				provider.updateElement(getParent(), getIndex(), this);
				Assert.isTrue(this.element != null);
			}
			return this.element;
		}
		
		public Object getData(){
			return getElement();
		}
		
		public int getDirectChildCount(){
			return getChildCount();
		}
		
		public int getChildCount(){
			if(childCount < 0){
				ILazyFragmentTreeContentProvider provider = (ILazyFragmentTreeContentProvider)getContentProvider();
				Object element = getElement();
				if(childCount < 0){
					if(element != null){
						provider.updateChildCount(element, 0, this);
						if(childCount < 0){
							childCount = 0;
						}
					} else {
						childCount = 0;
					}
				}
			}
			return childCount;
		}
		
		public boolean getHashChildren(){
			if(hasChildren == null){
				if(childCount < 0){
					getChildCount();
				} 
				hasChildren = Boolean.valueOf(childCount > 0);
			}
			return hasChildren.booleanValue();
		}
	}
	
	private class SimpleMaterializationContext extends BaseMaterializationContext {
		private int elementIndex;
		private Object parent;
		
		public SimpleMaterializationContext(Object parent, int elementIndex) {
			this(parent, elementIndex, null);
		}

		public SimpleMaterializationContext(Object parent, int elementIndex, Object element) {
			super(element);
			Assert.isTrue(parent != null && (elementIndex >= 0 || element != null));
			this.parent = parent;
			this.elementIndex = elementIndex;
		}

		public int getIndex() {
			if(elementIndex == -1){
				//avoid recursion
				elementIndex = -2;
				ILazyFragmentTreeContentWithIndexProvider provider = (ILazyFragmentTreeContentWithIndexProvider)getContentProvider();
				elementIndex = provider.indexOf(getParent(), getElement());
				if(elementIndex == -1)
					elementIndex = -2;
			}
			return elementIndex;
		}

		protected Object getParent() {
			return parent;
		}
	}
	
	private abstract class BaseExpansionMaterializationContext extends BaseMaterializationContext {

		public BaseExpansionMaterializationContext() {
			super();
		}

		public BaseExpansionMaterializationContext(Object element) {
			super(element);
		}

		public abstract Expansion getMaterializedExpansion();
	}

	private class ExpansionMaterializationContext extends BaseExpansionMaterializationContext {
		private MoveInfo moveInfo;
		private Expansion childExpansion;
		
		public ExpansionMaterializationContext(MoveInfo info, Expansion exp) {
			this(info, exp, null);
		}

		public ExpansionMaterializationContext(MoveInfo info, Expansion exp, Object element) {
			super(element);
			this.moveInfo = info;
			childExpansion = exp;
		}


		public Expansion getMaterializedExpansion(){
			if(!childExpansion.isVirtual())
				return childExpansion;
			int childCount = getChildCount();
			Object element = getElement();
			if(childCount > 0){
				childExpansion = moveInfo.expandExpansion(childExpansion, element, childCount);
			} else {
				childExpansion.setData(element);
			}
			return childExpansion;
		}

		public int getIndex() {
			return childExpansion.getIndex();
		}

		protected Object getParent() {
			return childExpansion.getParent().getData();
		}
	}
	
	private class RootExpansionMaterializationContext extends BaseExpansionMaterializationContext {
		RootExpansionMaterializationContext(Object element){
			super(element);
		}

		public void replace(Object parent, int index, Object element) {
			throw new UnsupportedOperationException();
		}

		public int getIndex() {
			throw new UnsupportedOperationException();
		}

		protected Object getParent() {
			throw new UnsupportedOperationException();
		}

		public Expansion getMaterializedExpansion(){
			return getMaterializedRootExpansion();
		}

		public RootExpansion getMaterializedRootExpansion(){
			Object element = getElement();
			int childCount = getChildCount();
			return new RootExpansion(element, childCount);
		}
		
		public Object getElement(){
			return element;
		}
	}

	private class MoveInfo {
		private ExpansionRange newRange;
		private ExpansionRange oldRange;
		private Expansion newTopExpansion;
		private Expansion curTopExpansion;
		private Map forceExpansionMap;
		private boolean skipTopInfoAdjustment;
		private RootExpansion rootExpansion;
		
		private Set itemsToClear;
		
		private Expansion[] selection;
		
		MoveInfo(RootExpansion rootExpansion, ExpansionRange range){
			this.rootExpansion = rootExpansion;
			this.oldRange = range;
		}

		public void addForceExpansion(Expansion exp){
			addForceExpansion(exp, null);
		}

		public void addForceExpansion(Expansion exp, TreeItem item){
			Assert.isTrue(!exp.isVirtual());
			
			forceExpansionMap = addExpansionToMap(forceExpansionMap, exp, item);
		}

		void processItemRequestedWhileMove(TreeItem item){
			if(itemsToClear == null)
				itemsToClear = new HashSet();
			item.setText(" "); //$NON-NLS-1$
			itemsToClear.add(item);
		}
		
		public Map addExpansionToMap(Map map, Expansion exp, Object value){
			if(map == null){
				map = new HashMap();
				map.put(exp, value);
				return map;
			}

			if(!isExpansionInMap(map, exp)){
				for(Iterator iter = map.keySet().iterator(); iter.hasNext();){
					if(exp.isParentOf((Expansion)iter.next()))
						iter.remove();
				}
				map.put(exp, value);
			}
			return map;
		}
		
		public boolean isForceExpansion(Expansion exp){
			return isExpansionInMap(forceExpansionMap, exp);
		}
		
		public boolean isExpansionInMap(Map map, Expansion exp){
			if(map == null)
				return false;
			for(; exp != null; exp = exp.getParent()){
				if(map.containsKey(exp))
					return true;
			}
			return false;
		}
		
		public void validateMoveInfo(){
			if(newTopExpansion != null)
				Assert.isTrue(newRange.getRelation(newTopExpansion) == ExpansionRange.CONTAINMENT);
		}
		
		public void makeVisible(Expansion exp){
			if(exp.isVisible())
				return;
			Expansion[] parents = exp.getParents(false, true);
			for(int i = 0; i < parents.length; i++){
				if(!parents[i].isExpanded()){
					expandExpansion(parents[i], null, -1);
				}
			}
			Assert.isTrue(exp.isVisible());
		}

		public void apply(){
			Assert.isTrue(movingInfo == null);
			validateMoveInfo();
			boolean oldPreserving = preservingSelection;
			try{
				preservingSelection = true;
				movingInfo = this;
				Assert.isTrue(newRange != null);
				
				boolean processForceExpansionsOnly = false;
				boolean processSynch = true;
				if((newRange.equals(oldRange)
						|| (oldRange.getStartOffset() == 0 && oldRange.getSize(false) == rootExpansion.getAllChildSize(false)
								&& newRange.getStartOffset() == 0 && newRange.getSize(false) == rootExpansion.getAllChildSize(false)))){
	//				Assert.isTrue(topExpansion != null);
					if(forceExpansionMap == null || forceExpansionMap.size() == 0){
						processSynch = false;
					} else {
						processForceExpansionsOnly = true;
					}
				}

				if(oldPreserving == false){//we are not in the preserving context
					if(selection == null && processSynch)
						selection = getSelectedExpansions();
				} else {
					selection = null;
				}
				
				range = newRange;

				if(processSynch){
					if(processForceExpansionsOnly){
						for(Iterator iter = forceExpansionMap.entrySet().iterator(); iter.hasNext();){
							Map.Entry entry = (Map.Entry)iter.next();
							Expansion exp = (Expansion)entry.getKey();
							TreeItem item = (TreeItem)entry.getValue();
							if(item == null && !exp.isRoot()){
								item = getTreeItem(exp);
								Assert.isTrue(item != null);
							}
							
							synchWithExpansion(item == null ? getTree() : null, item, exp, newRange, exp, oldRange);
						}
					} else {
						synchWithExpansion(getTree(), null, rootExpansion, newRange, rootExpansion, oldRange);
					}
				}
				
				movingInfo = null;

				if(selection != null)
					setSelection(selection);
				
				if(newTopExpansion != null){
					setTopVisibleExpansion(newTopExpansion);
				}
			} finally {
				if(newRange != null)
					range = newRange;
				movingInfo = null;
				preservingSelection = oldPreserving; 
				postProcessItemsToClear();
			}
		}
			
		private void postProcessItemsToClear(){
			if(itemsToClear == null || itemsToClear.size() == 0)
				return;
			
			for(Iterator iter = itemsToClear.iterator(); iter.hasNext();){
				TreeItem item = (TreeItem)iter.next();
				if(item.isDisposed())
					continue;
				
				TreeItem parentItem = item.getParentItem();
				if(parentItem != null){
					int index = parentItem.indexOf(item);
					parentItem.clear(index, true);
				} else {
					Tree tree = item.getParent();
					int index = tree.indexOf(item);
					tree.clear(index, true);
				}
			}
		}
		
		private boolean removeItemToClear(TreeItem item){
			if(itemsToClear == null)
				return false;
			return itemsToClear.remove(item);
		}
		
		public Expansion itemExpanded(TreeItem item, Object data, int childCount){
			Expansion exp = getExpansion(item/*, item.getData()*/);
			return expandExpansion(exp, data, childCount);
		}
		
		public Expansion expandExpansion(Expansion exp, Object data, int childCount){
			Assert.isTrue(!exp.isExpanded()); 

			//the method is not reenterable
//			Assert.isTrue(expandCollapseMoveInfo.oldRange == null);

			if(!exp.isExpanded()){
				oldRange = oldRange.expand(exp, data, childCount);

				if(!skipTopInfoAdjustment){
					skipTopInfoAdjustment = true;
					if(newTopExpansion.isChildOf(exp)){
						newTopExpansion = exp;
						curTopExpansion = null;
					}
					
					if(newTopExpansion.equals(exp)){
						//make sure we're using one and the same instance
						newTopExpansion = exp;
						if(curTopExpansion != null)
							curTopExpansion = exp;
					}

					int numVisibleItems = getNumVisibleItems();
					int topVisibleRoot = newTopExpansion.getRootIndex();
					int expRoot = exp.getRootIndex();
					if(topVisibleRoot > expRoot //expandee is higher than top
							|| topVisibleRoot + numVisibleItems < expRoot + exp.getAllChildSize(false)) //expandee lowes child is invisible
					{
						newTopExpansion = exp;
					}
				}
					
				if(newTopExpansion != null){
					calcNewRangeInfoPreservingTopVisible(newTopExpansion);
				} else {
					Assert.isTrue(newRange == null);
				}
			}
			return exp;
		}
		
		public void setNewTopVisibleExpansion(Expansion exp){
			newTopExpansion = exp;
			calcNewRangeInfoPreservingTopVisible(newTopExpansion);
		}
		
		public void itemCollapsed(TreeItem item){
			Expansion exp = getExpansion(item);
			collapseExpansion(exp);
		}
		
		public void collapseExpansion(Expansion exp){

			Assert.isTrue(exp.isExpanded());

			int rootIndex = exp.getRootIndex();
			Expansion invalidExp = exp; 
			oldRange = oldRange.collapse(exp, true);

			exp = rootExpansion.getChildExpansionByOffset(rootIndex, false);
			Assert.isTrue(exp != null);

			if(newTopExpansion == invalidExp){
				newTopExpansion = exp;
			}
			if(curTopExpansion == invalidExp){
				curTopExpansion = exp;
			}

			if(!skipTopInfoAdjustment){
				skipTopInfoAdjustment = true;
				if(newTopExpansion.isChildOf(exp)){
					newTopExpansion = exp;
					curTopExpansion = null;
				}
			}
			
			if(newTopExpansion != null){
				calcNewRangeInfoPreservingTopVisible(newTopExpansion);
				Assert.isTrue(newRange.getRelation(newTopExpansion) != ExpansionRange.NONE);
			} else {
				Assert.isTrue(newRange == null);
			}
		}
		
		public void calcNewRangeInfoOnScroll(boolean scaleMode){
			if(scaleMode){
				calcNewRangeInfoPreservingScale(oldRange, curTopExpansion);
			} else {
				newTopExpansion = curTopExpansion;
				calcNewRangeInfoPreservingTopVisible(newTopExpansion);
			}
		}
		
		/*
		 * core logic for move calculation:
		 * 
		 *  topvis - start         topvis
		 * ---------------- = -----------------
		 *    Ni - Nivis         Nel - Nivis
		 *   
		 * where:
		 * topvis - top visible expansion root index
		 * start - range start root index
		 * Ni - num items in range
		 * Nivis - num visible items
		 */

		private void calcNewRangeInfoPreservingTopVisible(Expansion topVisExp){
			long Ni = getMaxRangeSize(rootExpansion);
			long Nivis = getNumVisibleItems();
			long Nel = rootExpansion.getAllChildSize(false);
			if(Nel == Ni){
				newRange = ExpansionRange.createExpansionRange(rootExpansion, 0, (int)Nel);
				newTopExpansion = topVisExp;
			} else {
				Assert.isTrue(Nel > Ni);
				long topvis = topVisExp.getRootIndex();
				if(topvis >= Nel - Nivis){
					newRange = ExpansionRange.createExpansionRange(rootExpansion, (int)(Nel - Ni), (int)Ni);
					newTopExpansion = topVisExp;
				} else if (topvis == 0){
					newRange = ExpansionRange.createExpansionRange(rootExpansion, 0, (int)Ni);
					newTopExpansion = topVisExp;
				} else {
					long start = (long)(((double)(topvis*(Nel - Ni)))/(Nel - Nivis + 1));
					if(start != 0){
						Assert.isTrue(topvis >= start);
						int numItemsVisible = getNumVisibleItems();
						if(topvis - start < numItemsVisible){
							start = topvis - numItemsVisible;
							if(start<0)
								start = 0;
						} else if(topvis - start >= Ni - Nivis){
							start = topvis - Ni + Nivis + 1;
						}
						if(start > Nel - Ni){
							start = Nel - Ni;
						}
					}
					newTopExpansion = topVisExp;
					newRange = ExpansionRange.createExpansionRange(rootExpansion, (int)start, (int)Ni);
//					int topIndex = newTopExpansion.getIndex();
//					if(topIndex != 0){
//						Expansion parentExp = newTopExpansion.getParent();
//						if(!parentExp.isRoot()){
//							int startIndex = newRange.getStartIndex(parentExp);
//							Assert.isTrue(startIndex >= 0 && startIndex <= topIndex);
//							if(startIndex != 0){
//								if(topIndex == startIndex){
//									newRange.moveToStart(newRange.getStartOffset() - 1, true);
//								}
//							}
//						}
//					}
				}
			}
		}
		
		private void calcNewRangeInfoPreservingScale(ExpansionRange range, Expansion topVisible){
			long Nivis = getNumVisibleItems();
			long Nel = rootExpansion.getAllChildSize(false);
			long Ni = range.getSize(false);
			if(Nel == Ni){
				Assert.isTrue(range.getStartOffset() == 0);
				newRange = range;
				newTopExpansion = topVisible;
			} else {
				Assert.isTrue(Nel > Ni);
				int start = range.getStartOffset();
				int topvis = topVisible.getRootIndex();
				int topVisibleOffset = topvis - start;
				if(topVisibleOffset < 0)
					topVisibleOffset = 0;
//				Assert.isTrue(topVisibleOffset >= 0);
//				Assert.isTrue(topVisibleOffset < range.getSize(false));
				if(topVisibleOffset >= Ni - Nivis - 4){
					if(oldRange.getStartOffset() == Nel - Nivis){
						newRange = oldRange;
						newTopExpansion = topVisible;
					} else {
						int newIndex = (int)(Nel - Nivis);
						newTopExpansion = rootExpansion.getChildExpansionByOffset(newIndex, false);
						newRange = ExpansionRange.createExpansionRange(rootExpansion, (int)(Nel - Ni), (int)Ni);
					}
				} else if(topvis <= start){
					newTopExpansion = rootExpansion.getChildExpansionByOffset(0, false);
					newRange = ExpansionRange.createExpansionRange(rootExpansion, 0, (int)Ni);
				} else {
					long newTopVis = (long)(((double)(topVisibleOffset*(Nel - Nivis)))/(Ni-Nivis));
					if(newTopVis >= Nel - Nivis){
						newTopVis = Nel - Nivis;
						newTopExpansion = rootExpansion.getChildExpansionByOffset((int)newTopVis, false);
						newRange = ExpansionRange.createExpansionRange(rootExpansion, (int)(Nel - Ni), (int)Ni);
					} else if (newTopVis == 0){
						newTopExpansion = rootExpansion.getChildExpansionByOffset(0, false);
						newRange = ExpansionRange.createExpansionRange(rootExpansion, 0, (int)Ni);
					} else {
						newRange = range.moveToStart((int)(newTopVis - topVisibleOffset), true);
						newTopExpansion = rootExpansion.getChildExpansionByOffset(newRange.getStartOffset() + topVisibleOffset, false);
					}
				}
			}
		}
		
		private void synchWithExpansion(Tree tr, TreeItem item, Expansion newExp, ExpansionRange range, Expansion oldExp, ExpansionRange oldRange){
			int newStart = range.getStartIndex(newExp); 
			int oldStart = oldExp != null ? oldRange.getStartIndex(oldExp) : -1;
			int newSize = range.getDirectChildCount(newExp);
			int oldSize;
			TreeItem[] items;
			boolean isForce = !newExp.equals(oldExp) || isForceExpansion(newExp);
			boolean newAsOld = newStart == oldStart && !isForce;
			Expansion[] newExpansions = range.getDirectChildren(newExp, true);
			Expansion[] oldExpansions = oldExp != null ? oldRange.getDirectChildren(oldExp, true) : new Expansion[0];
			Widget w;
			if(item != null){
				w = item;
				oldSize = item.getItemCount();
				if(newSize != oldSize){
					item.setItemCount(newSize);
				}
//				if(!item.getExpanded())
					item.setExpanded(true);
				if(newAsOld && newExpansions.length == 0 && oldExpansions.length == 0)
					return;
				items = item.getItems();
			} else {
				w = tr;
				oldSize = tr.getItemCount();
				if(newSize != oldSize){
					tr.setItemCount(newSize);
				}
				if(newAsOld && newExpansions.length == 0 && oldExpansions.length == 0)
					return;
				items = tr.getItems();
			}
			
			
			int newCursor = 0;
			int newIndex = newExpansions.length != 0 ? newExpansions[0].getIndex() : -1;
			
			int oldCursor = 0;
			int oldIndex = oldExpansions.length != 0 ? oldExpansions[0].getIndex() : -1;

			
			if(newAsOld){
				for(int i = 0; i < items.length; i++){
					Expansion newChildExp = null;
					Expansion oldChildExp = null;
					if(newStart + i == newIndex){
						newChildExp = newExpansions[newCursor++];
						newIndex = newExpansions.length > newCursor ? newExpansions[newCursor].getIndex() : -1;
					}
					if(oldStart >= 0 && oldStart + i == oldIndex){
						oldChildExp = oldExpansions[oldCursor++];
						oldIndex = oldExpansions.length > oldCursor ? oldExpansions[oldCursor].getIndex() : -1;
					}

					if(newChildExp != null){
						TreeItem child = items[i];
						
						if(child.getData() != null){
							//materialized
							synchWithExpansion(null, child, newChildExp, range, oldChildExp, oldRange);
						} else {
							replace(w, i, newChildExp.getData());
							synchWithExpansion(null, child, newChildExp, range, oldChildExp, oldRange);
						}
						removeItemToClear(child);
						if(newIndex == -1 && oldIndex == -1)
							break;
					} else if(oldChildExp != null){
						TreeItem child = items[i];
						child.setText(" "); //$NON-NLS-1$
						child.setExpanded(false);
						child.setItemCount(0);
						if(newIndex == -1 && oldIndex == -1)
							break;
						
						if(item != null){
							item.clear(i, true);
						} else {
							tr.clear(i, true);
						}
					}
				} 
			} else {
				for(int i = 0; i < items.length; i++){
					TreeItem child = items[i];
					Expansion newChildExp = null;
					Expansion oldChildExp = null;
					if(newStart + i == newIndex){
						newChildExp = newExpansions[newCursor++];
						newIndex = newExpansions.length > newCursor ? newExpansions[newCursor].getIndex() : -1;
					}
					if(oldStart >= 0 && oldStart + i == oldIndex){
						oldChildExp = oldExpansions[oldCursor++];
						oldIndex = oldExpansions.length > oldCursor ? oldExpansions[oldCursor].getIndex() : -1;
					}


					if(child.getData() != null){
						//child is initialized
						disassociate(child);
							
						if(newChildExp != null){
							if(item != null){
								item.clear(i, true);
							} else {
								tr.clear(i, true);
							}
							
							replace(w, i, newChildExp.getData());
							synchWithExpansion(null, child, newChildExp, range, oldChildExp, oldRange);
						} else {
							//should not have a callback here
							child.setExpanded(false);
							child.setItemCount(0);
							if(item != null){
								item.clear(i, true);
							} else {
								tr.clear(i, true);
							}
						}
					} else {
						//the child is not initialized
						if(newChildExp != null){

							replace(w, i, newChildExp.getData());
							synchWithExpansion(null, child, newChildExp, range, oldChildExp, oldRange);					
						} else if (oldChildExp != null || isForce){
							//avoid callback
							child.setText(" "); //$NON-NLS-1$

							child.setExpanded(false);
							child.setItemCount(0);
							
							if(item != null){
								item.clear(i, true);
							} else {
								tr.clear(i, true);
							}
						} 
					} 
					removeItemToClear(child);
				}
			}
		} 
	}

	private class TreeInfo {
		private Expansion topExpansion;
		private Expansion bottomExpansion;
//		private ExpansionRange range;
		private int numVisibleExpansions = -1;
		private int maxItemForwardMove = -1;
		private int maxItemBackwardMove = -1;
		
		private TreeItem topItem;
		
		public TreeInfo(){
			
		}
		
		public TreeItem getTopItem(){
			if(topItem == null){
				topItem = getTree().getTopItem();
			}
			return topItem;
		}
		
		public Expansion getTopVisibleExpansion(){
			if(topExpansion == null){
				TreeItem topItem = getTopItem();
				if(topItem != null)
					topExpansion = FragmentedTreeViewer.this.getExpansion(topItem);
			}
			return topExpansion;
		}
		
		public int getNumVisibleItems(){
			if(numVisibleExpansions == -1){
				numVisibleExpansions = FragmentedTreeViewer.this.getNumVisibleItems();
			}
			return numVisibleExpansions;
		}

		
		public Expansion getVisibleExpansion(boolean bottom){
			return bottom ? getBottomVisibleExpansion() : getTopVisibleExpansion();
		}

		public Expansion getBottomVisibleExpansion(){
			if(bottomExpansion == null){
				Expansion topExpansion = getTopVisibleExpansion();
	//			Expansion topExpansion = getTopVisibleExpansion();
				if(topExpansion == null){//tree is empty
					Tree tree = getTree();
					Assert.isTrue(tree.getItemCount() == 0);
					return null;
				}

				bottomExpansion = FragmentedTreeViewer.this.getBottomVisibleExpansion(topExpansion, getNumVisibleItems());
			}
			return bottomExpansion;
		}
		
		public boolean isVisible(Expansion exp){
			int expIndex = exp.getRootIndex();
			return getTopVisibleExpansion().getRootIndex() <= expIndex
				&& getBottomVisibleExpansion().getRootIndex() >= expIndex;
		}

	}
	
	public FragmentedTreeViewer(Composite parent) {
		this(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
	}

	public FragmentedTreeViewer(Composite parent, int style) {
		this(new Tree(parent, style));
	}

	public FragmentedTreeViewer(Tree tree) {
		super(tree);
		setUseHashlookup(true);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.jface.viewers.AbstractTreeViewer#hookControl(org.eclipse.swt.widgets.Control)
	 */
	protected void hookControl(Control control) {
		super.hookControl(control);
		Tree treeControl = (Tree) control;

		if ((treeControl.getStyle() & SWT.VIRTUAL) != 0) {
// we have this inherited from TreeViewer
//			treeControl.addDisposeListener(new DisposeListener() {
//				public void widgetDisposed(DisposeEvent e) {
//					treeIsDisposed = true;
//					unmapAllElements();
//				}
//			});
			treeControl.addListener(SWT.SetData, new Listener() {

				public void handleEvent(Event event) {
					if (contentProviderIsLazyFragmented) {
						if(movingInfo != null){
							if(DbgUtil.DEBUG)
								DbgUtil.println("data event while move"); //$NON-NLS-1$
							movingInfo.processItemRequestedWhileMove((TreeItem)event.item);
							return;
						}
						TreeItem item = (TreeItem) event.item;
						TreeItem parentItem = item.getParentItem();
						int index = event.index;
						virtualLazyUpdateWidget(
								parentItem == null ? (Widget) getTree()
										: parentItem, item, index, range, true);
					}
				}

			});
			
			treeControl.getVerticalBar().addSelectionListener(new SelectionListener(){

				public void widgetDefaultSelected(SelectionEvent e) {
				}

				public void widgetSelected(SelectionEvent e) {
					if(contentProviderIsLazyFragmented){
						handleScroll(e);
					}
				}
				
			});
			
			treeControl.addKeyListener(new KeyListener(){

				public void keyPressed(KeyEvent e) {
					if(contentProviderIsLazyFragmented){
						handleKey(e);
					}
					
				}

				public void keyReleased(KeyEvent e) {
					
				}
				
			});
			
			treeControl.addMouseWheelListener(new MouseWheelListener(){

				public void mouseScrolled(MouseEvent e) {
					if(contentProviderIsLazyFragmented){
						handleMouseWeel(e);
					}
				}
				
			});
		}
	}
	
	/*
	 * (non-Javadoc) Method declared in AbstractTreeViewer.
	 */
	protected void setExpanded(Item node, boolean expand) {
		if (contentProviderIsLazyFragmented) {
			TreeItem item = (TreeItem)node;
			if(expand){
				item.setExpanded(true);
				handleTreeExpand(item, false);
			} else {
				item.setExpanded(false);
				handleTreeCollapse(item);
			}

			// force repaints to happen
			getControl().update();
		}
	}

	protected void setSelectionToWidget(List v, boolean reveal) {
		if(contentProviderIsLazyFragmented){
			MoveInfo info;
			Expansion[] exps;
			if(v == null || v.size() == 0){
				info = null;
				exps = ExpansionSelection.EMPTY_EXPANSION_ARRAY;
			} else {
				Expansion[] currentSel = getSelectedExpansions();
				if(currentSel.length == v.size()){
					Set curSel = new HashSet();
					for(int i = 0; i < currentSel.length; i++){
						curSel.add(currentSel[i].getData());
					}
					curSel.removeAll(v);
					if(curSel.size() == 0)
						return;
				}
				if(reveal){
					info = new MoveInfo(rootExpansion, range);
					info.skipTopInfoAdjustment = true;
					exps = getExpansionsFromList(v, true, info);
					if(info.newRange == null)
						info = null;
				} else {
					info = null;
					exps = getExpansionsFromList(v, false, null);
				}
			}
			
			setSelectionToWidget(info, exps);

			return;
		}
		super.setSelectionToWidget(v, reveal);
	}
	
	
	private void setSelectionToWidget(MoveInfo mInfo, Expansion[] exps){
		if(mInfo != null){
			mInfo.selection = exps;
			mInfo.apply();
		} else {
			setSelection(exps);
		}
	}
	
	private Expansion[] getExpansionsFromList(List list, boolean reveal, MoveInfo info){
		int size = list.size();
		Expansion[] result = size != 0 ? new Expansion[size] : ExpansionSelection.EMPTY_EXPANSION_ARRAY;
		int cursor = 0;
		for (int i = 0; i < size; ++i) {
			Object elementOrTreePath = list.get(i);
			Expansion exp = findExpansion(info, elementOrTreePath, reveal);
			if(exp == null)
				continue;
			if(reveal || (exp.isVisible() && range.getRelation(exp) != ExpansionRange.NONE)){
				result[cursor++] = exp;
			}
		}
		
		if(cursor < result.length){
			if(cursor == 0){
				result = ExpansionSelection.EMPTY_EXPANSION_ARRAY; 
			} else {
				Expansion[] tmp = new Expansion[cursor];
				System.arraycopy(result, 0, tmp, 0, cursor);
				result = tmp;
			}
		}
		return result;
	}
	
	protected void setSelection(List items) {
		if(contentProviderIsLazyFragmented){
			Assert.isTrue(false);
			return;
		}
		super.setSelection(items);
	}
	
	protected boolean isSameSelection(List items, Item[] current) {
		if(contentProviderIsLazyFragmented){
			Assert.isTrue(false);
			return true;
		}
		return super.isSameSelection(items, current);
	}
	
	private Expansion materializeExpansionIfHasChildren(MoveInfo info, Expansion exp){
		Assert.isTrue(exp.isVirtual());
		return new ExpansionMaterializationContext(info, exp).getMaterializedExpansion();
	}

	private Expansion findExpansion(MoveInfo mInfo, Object elementOrTreePath, boolean revealAndExpandParents){
		Object element = getElementFromElementOrTreePath(elementOrTreePath);

		Expansion exp = rootExpansion.getExpansion(element);
		Expansion expandedRoot = null;
		if(exp == null && contentProviderSupportsIndexInfo){
			ArrayList elList = revealAndExpandParents ? new ArrayList(5) : new ArrayList(1);
			Object childEl = element;
			Object parentElOrPath = getParentElement(elementOrTreePath);
			Object parentEl;
			Expansion parentExp = null;
			for(; parentElOrPath != null; 
			        childEl = parentEl, 
			           parentElOrPath = getParentElement(parentElOrPath)){
				parentEl = getElementFromElementOrTreePath(parentElOrPath);
				Expansion e = rootExpansion.getExpansion(parentEl);
				if(e != null){
					parentExp = e;
					elList.add(new SimpleMaterializationContext(e.getData(), -1, childEl));
					break;
				} else if(revealAndExpandParents){
					elList.add(new SimpleMaterializationContext(parentEl, -1, childEl));
				} else {
					break;
				}
			}
			
			if(parentExp != null){
				if(elList != null && elList.size() != 0){
					for(int i = elList.size() - 1; i >= 0; i--){
						SimpleMaterializationContext c = (SimpleMaterializationContext)elList.get(i);
						if(c.getIndex() < 0/* || c.getIndex() >= c.getChildCount()*/)
							return null;
					}
				}

				if(elList != null && elList.size() != 0){
					for(int i = elList.size() - 1; i >= 0; i--){
						SimpleMaterializationContext c = (SimpleMaterializationContext)elList.get(i);
						parentExp = parentExp.getDirectChildByIndex(c.getIndex(), false);
						parentExp.setData(c.getData());
						if(i != 0){
							parentExp = mInfo.expandExpansion(parentExp, c.getData(), c.getDirectChildCount());
							if(expandedRoot == null){
								expandedRoot = parentExp;
							}
						}
					}
					
					exp = parentExp;
				}
			}
		}
		
		if(exp != null){
			if(expandedRoot != null){
				Expansion forceExp = null;
				if(expandedRoot.isVisible()){
					forceExp = expandedRoot;
				} else {
					Expansion[] parents = expandedRoot.getParents(false, true);
					for(int k = 0; k < parents.length; k++){
						if(!parents[k].isExpanded()){
							forceExp = parents[k];
							break;
						}
					}
					mInfo.makeVisible(forceExp);
				}
				Assert.isTrue(forceExp != null);
				mInfo.addForceExpansion(forceExp);
				mInfo.oldRange.clearCache();
				if(mInfo.newRange == null){
					mInfo.newRange = mInfo.oldRange;
				}
			}
			if(mInfo.newTopExpansion == null){
				Expansion curTop = getTopVisibleExpansion();
				if(curTop == null){
					mInfo.newTopExpansion = exp;
				} else {
					int topRoot = curTop.getRootIndex();
					int newTopRoot = exp.getRootIndex();
					if(topRoot >= newTopRoot){
						mInfo.setNewTopVisibleExpansion(exp);
					} else {
						int numVis = getNumVisibleItems();
						if(topRoot + numVis <= newTopRoot ){
							mInfo.setNewTopVisibleExpansion(exp);
						} else {
							mInfo.newTopExpansion = curTop;
						}
					}
				}
			}
		}
			
		return exp;
	}

	/*
	 * (non-Javadoc) Method declared in AbstractTreeViewer.
	 */
	protected void showItem(Item item) {
		getTree().showItem((TreeItem) item);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.jface.viewers.AbstractTreeViewer#getChild(org.eclipse.swt.widgets.Widget,
	 *      int)
	 */
	protected Item getChild(Widget widget, int index) {
		if (widget instanceof TreeItem) {
			return ((TreeItem) widget).getItem(index);
		}
		if (widget instanceof Tree) {
			return ((Tree) widget).getItem(index);
		}
		return null;
	}

	protected void assertContentProviderType(IContentProvider provider) {
		if (provider instanceof ILazyFragmentTreeContentProvider) {
			return;
		}
		super.assertContentProviderType(provider);
	}

	protected Object[] getRawChildren(Object parent) {
		if (contentProviderIsLazyFragmented) {
			return new Object[0];
		}
		return super.getRawChildren(parent);
	}

	public void setChildCount(Object elOrTreePath, int c) {
		if(contentProviderIsLazyFragmented){
			//TODO: implement
			throw new UnsupportedOperationException();
		}
		super.setChildCount(elOrTreePath, c);
	}

	private int calcChildCount(ExpansionRange range, Expansion exp){
		return Math.max(1, range.getDirectChildCount(exp));
	}
	
	private Object getElementFromElementOrTreePath(Object elOrTreePath){
		if(elOrTreePath instanceof TreePath){
			TreePath path = (TreePath)elOrTreePath;
			if(path.getSegmentCount() == 0)
				return getInput();
			return path.getLastSegment();
		}
		return elOrTreePath;
	}
	
	private int getMaxRangeSize(RootExpansion rootExpansion){
		return rootExpansion.getAllChildSize(false) > fragmentModeThreashold ? numItemsInFragmentMode : rootExpansion.getAllChildSize(false);
	}
	
	private Object getData(Widget w, boolean access){
		return w instanceof TreeItem ? getData((TreeItem)w, access) : getInput();
	}

	private Object getData(TreeItem item, boolean access){
		if(access)
			item.getText();
		return item.getData();
	}
	
	private TreeItem getTreeItem(Expansion exp){
		if(DbgUtil.DEBUG) {
			Assert.isTrue(movingInfo == null || range.equals(movingInfo.oldRange));
		}
		if(exp.isRoot()
				|| range.getRelation(exp) == ExpansionRange.NONE)
			return null;
		Expansion[] exps = exp.getParents(true, true);
		
		int index = range.getIndex(exps[1]);
		Assert.isTrue(index >= 0);
		TreeItem item = getTree().getItem(index);
		for(int i = 2; i < exps.length; i++){
			index = range.getIndex(exps[i]);
			Assert.isTrue(index >= 0);
			item = item.getItem(index);
		}
		return item;
	}

	private Expansion getExpansion(Widget w){
		return w instanceof TreeItem ? getExpansion((TreeItem)w) : rootExpansion;
	}

	private Expansion getExpansion(TreeItem item){
		if(DbgUtil.DEBUG) {
			Assert.isTrue(movingInfo == null || range.equals(movingInfo.oldRange));
		}

		TreeItem parentItem = item.getParentItem();
		Tree tree = parentItem == null ? item.getParent() : null;
		
		int itemIndex;
		Expansion parentExpansion;
		if(parentItem != null){
			Object parentElement = getData(parentItem, false);
			if(parentElement == null){
				Assert.isTrue(parentItem.getExpanded());
				Expansion parentExp = getExpansion(parentItem);
				if(parentExp == null || parentExp.isVirtual()){
					return null;
				}
//				Assert.isTrue(!parentExp.isVirtual());
				Assert.isTrue(parentExp.isExpanded());
				TreeItem parentParentItem = parentItem.getParentItem();
				Widget parentParent;
				int parentIndex;
				if(parentParentItem != null){
					parentParent = parentParentItem;
					parentIndex = parentParentItem.indexOf(parentItem);
				} else {
					Tree ptree = parentItem.getParent();
					parentParent = ptree;
					parentIndex = ptree.indexOf(parentItem);
				}
				virtualLazyUpdateWidget(parentParent, parentItem, parentIndex, range, true);
				
				parentElement = getData(parentItem, false);
			}
			Assert.isTrue(parentElement != null);
			parentExpansion = rootExpansion.getExpansion(parentElement);
			Assert.isTrue(parentExpansion != null);
			itemIndex = parentItem.indexOf(item);
		} else {
			parentExpansion = rootExpansion;
			itemIndex = tree.indexOf(item);
		}
		
		int startOffset = range.getStartIndex(parentExpansion);
		Assert.isTrue(startOffset >= 0);
		Expansion result = parentExpansion.getDirectChildByIndex(itemIndex + startOffset, false);
		if(result.getData() == null){
			result.setData(item.getData());
		}
		Assert.isTrue(range.getRelation(result) != ExpansionRange.NONE);
		return result;
	}

	private void setChildCountToTree(final Tree tree, final int c) {
		preservingSelection(new Runnable() {
			public void run() {
				tree.setItemCount(c);
				return;
			}
		});
	}

	private void setChildCountToItem(final TreeItem item, final int c) {
		preservingSelection(new Runnable() {
			public void run() {
				item.setItemCount(c);
			}
		});
	}

	public void replace(Object parentElementOrTreePath, int index,
			Object element) {
		if(contentProviderIsLazyFragmented){
			//TODO: implement
			throw new UnsupportedOperationException();
		}
		super.replace(parentElementOrTreePath, index, element);
	}
	
	private void replace(Widget parent, final int index,
			final Object element) {
		if (checkBusy())
			return;
		Item[] selItems;
		ExpansionSelection sel;
		if(preservingSelection){
			selItems = null;
			sel = null;
		} else {
			selItems = getSelection(getControl());
			sel = (ExpansionSelection) getSelection();
		}
		Widget[] itemsToDisassociate;
//		if (parentElementOrTreePath instanceof TreePath) {
//			TreePath elementPath = ((TreePath) parentElementOrTreePath)
//					.createChildPath(element);
//			itemsToDisassociate = internalFindItems(elementPath);
//		} else {
			itemsToDisassociate = internalFindItems(element);
//		}
		if (parent instanceof Tree) {
			Tree tree = (Tree)parent;
			if (index < tree.getItemCount()) {
				TreeItem item = tree.getItem(index);
				if(!preservingSelection)
					sel = adjustSelectionForReplace(selItems, sel, item, element, getRoot());
				// disassociate any different item that represents the
				// same element under the same parent (the tree)
				for (int i = 0; i < itemsToDisassociate.length; i++) {
					if (itemsToDisassociate[i] instanceof TreeItem) {
						TreeItem itemToDisassociate = (TreeItem) itemsToDisassociate[i];
						if (itemToDisassociate != item
								&& itemToDisassociate.getParentItem() == null) {
							int indexToDisassociate = getTree().indexOf(
									itemToDisassociate);
							disassociate(itemToDisassociate);
							getTree().clear(indexToDisassociate, true);
						}
					}
				}
				Object oldData = item.getData();
				updateItem(item, element);
				if (!FragmentedTreeViewer.this.equals(oldData, element)) {
					item.clearAll(true);
				}
			}
		} else {
//			Widget[] parentItems = internalFindItems(parentElementOrTreePath);
//			for (int i = 0; i < parentItems.length; i++) {
				TreeItem parentItem = (TreeItem) parent;//parentItems[i];
				if (index < parentItem.getItemCount()) {
					TreeItem item = parentItem.getItem(index);
					if(!preservingSelection)
						sel = adjustSelectionForReplace(selItems, sel, item, element, parentItem.getData());
					// disassociate any different item that represents the
					// same element under the same parent (the tree)
					for (int j = 0; j < itemsToDisassociate.length; j++) {
						if (itemsToDisassociate[j] instanceof TreeItem) {
							TreeItem itemToDisassociate = (TreeItem) itemsToDisassociate[j];
							if (itemToDisassociate != item
									&& itemToDisassociate.getParentItem() == parentItem) {
								int indexToDisaccociate = parentItem
										.indexOf(itemToDisassociate);
								disassociate(itemToDisassociate);
								parentItem.clear(indexToDisaccociate, true);
							}
						}
					}
					Object oldData = item.getData();
					updateItem(item, element);
					if (!FragmentedTreeViewer.this.equals(oldData, element)) {
						item.clearAll(true);
					}
				}
//			}
		}
		// Restore the selection if we are not already in a nested preservingSelection:
		if (!preservingSelection) {
			setSelectionToWidget(sel, false);
			// send out notification if old and new differ
			ISelection newSelection = getSelection();
			if (!newSelection.equals(sel)) {
				handleInvalidSelection(sel, newSelection);
			}
		}
	}

	/**
	 * Fix for bug 185673: If the currently replaced item was selected, add it
	 * to the selection that is being restored. Only do this if its getData() is
	 * currently null
	 *
	 * @param selectedItems
	 * @param selection
	 * @param item
	 * @param element
	 * @return
	 */
	private ExpansionSelection adjustSelectionForReplace(Item[] selectedItems,
			ExpansionSelection selection, TreeItem item, Object element, Object parentElement) {
		if (item.getData() != null || selectedItems.length == selection.size()
				|| parentElement == null) {
			// Don't do anything - we are not seeing an instance of bug 185673
			return selection;
		}
		for (int i = 0; i < selectedItems.length; i++) {
			if (item == selectedItems[i]) {
				// The current item was selected, but its data is null.
				// The data will be replaced by the given element, so to keep
				// it selected, we have to add it to the selection.
				Expansion exp = getExpansion(item);
				if(exp == null)
					continue;

				if(exp.getData() == null){
					exp.setData(element);
				}
				Expansion[] originalExpansions = selection.getExpansions();
				int length = originalExpansions.length;
				Expansion[] exps = new Expansion[length + 1];
				System.arraycopy(originalExpansions, 0, exps, 0, length);
				exps[length] = exp;

				return new ExpansionSelection(exps);
			}
		}
		// The item was not selected, return the given selection
		return selection;
	}

	public boolean isExpandable(Object element) {
		if (contentProviderIsLazyFragmented) {
			TreeItem treeItem = (TreeItem) internalExpand(element, false);
			if (treeItem == null) {
				return false;
			}
			virtualMaterializeItem(treeItem, range);
			return treeItem.getItemCount() > 0;
		}
		return super.isExpandable(element);
	}

	protected Object getParentElement(Object element) {
		boolean oldBusy = isBusy();
		setBusy(true);
		try {
			if (contentProviderIsLazyFragmented && !(element instanceof TreePath)) {
				ILazyFragmentTreeContentProvider lazyTreeContentProvider = (ILazyFragmentTreeContentProvider) getContentProvider();
				return lazyTreeContentProvider.getParent(element);
			}
			return super.getParentElement(element);
		} finally {
			setBusy(oldBusy);
		}
	}

	protected void createChildren(Widget widget) {
		if (contentProviderIsLazyFragmented) {
			if(widget instanceof TreeItem){
				TreeItem item = (TreeItem)widget;
				if(item.getExpanded()){
					Expansion exp = getExpansion(item);
					if(!exp.isExpanded()){
						exp = handleTreeExpand(item, true);
						if(exp.getDirectChildSize(false) == 0)
							return;
						Assert.isTrue(exp.isExpanded());
						widget = getTreeItem(exp);
					}
				} else {
//					throw new UnsupportedOperationException();
					//we are not supporting this scenario
					return;
				}
			} else {
				if(widget.getData() == null)
					return;
			}

			// touch all children to make sure they are materialized

			TreeItem[] children = (TreeItem[])getChildren(widget);

			for (int i = 0; i < children.length; i++) {
				if (children[i].getData() == null) {
					virtualLazyUpdateWidget(widget, children[i], i, range, true);
				}
			}
			return;
		}
		super.createChildren(widget);
	}

	protected void internalAdd(Widget widget, Object parentElement,
			Object[] childElements) {
		if (contentProviderIsLazyFragmented) {
			if (widget instanceof TreeItem) {
				TreeItem ti = (TreeItem) widget;
				int count = ti.getItemCount() + childElements.length;
				ti.setItemCount(count);
				ti.clearAll(false);
			} else {
				Tree t = (Tree) widget;
				t.setItemCount(t.getItemCount() + childElements.length);
				t.clearAll(false);
			}
			return;
		}
		super.internalAdd(widget, parentElement, childElements);
	}

	private void virtualMaterializeItem(TreeItem treeItem, ExpansionRange range) {
		if (treeItem.getData() != null) {
			// already materialized
			return;
		}
		if (!contentProviderIsLazyFragmented) {
			return;
		}
		virtualLazyUpdateWidget(treeItem, range, true);
	}

	private void virtualLazyUpdateWidget(TreeItem treeItem, ExpansionRange range, boolean fullUpdate) {
		int index;
		Widget parent = treeItem.getParentItem();
		if (parent == null) {
			parent = treeItem.getParent();
		}
		Object parentElement = parent.getData();
		if (parentElement != null) {
			if (parent instanceof Tree) {
				index = ((Tree) parent).indexOf(treeItem);
			} else {
				index = ((TreeItem) parent).indexOf(treeItem);
			}
			virtualLazyUpdateWidget(parent, treeItem, index, range, true);
		}
	}	
	
	protected void internalRefreshStruct(Widget widget, Object element,
			boolean updateLabels) {
		if (contentProviderIsLazyFragmented) {
				Expansion expansion;
//				int index = 0;
//				Widget parent = null;
				if (widget instanceof TreeItem) {
					//TODO:
					throw new UnsupportedOperationException();
//					TreeItem treeItem = (TreeItem) widget;
//					expansion = getExpansion(treeItem);
//					Expansion parentExp = expansion.getParent();
//					int expIndex = expansion.getIndex();
//					boolean expanded = false;
//					int count = 0;
//					if(expansion.isExpanded()){
//						expanded = true;
//						count = expansion.getDirectChildSize(false);
//						range = range.collapse(expansion, false);
//						expansion = parentExp.getDirectChildByIndex(expIndex, false);
//					} else if (!expansion.isVirtual()){
//						count = expansion.getDirectChildSize(false);
//						expansion.virtualize();
//						expansion = parentExp.getDirectChildByIndex(expIndex, false);
//					}
//					
//					if(expanded){
//						try {
//							setExpansionInfoToItem(parent, item, itemIndex, range, exp)
//							virtualLazyUpdateChildCount(widget, count);
//							expansion = parentExp.getDirectChildByIndex(expIndex, false);
//							Assert.isTrue(expansion.isExpanded());
//						} finally {
//						}
//					} else {
//						virtualLazyUpdateHasChildren(treeItem, count);
//						expansion = parentExp.getDirectChildByIndex(expIndex, false);
//						Assert.isTrue(!expansion.isExpanded());
//					}

				} else {
					if(element == null){
						((Tree)widget).setItemCount(0);
						rootExpansion = null;
						return;
					}
					initializeTreeInfo((Tree)widget);
					expansion = rootExpansion;
				}

				MoveInfo info = new MoveInfo(rootExpansion, range);
				info.skipTopInfoAdjustment = true;
	
				if(rootExpansion.getDirectChildSize(false) != 0){
					range = adjustRangeOnRefresh(range);
				} 
				
				info.newRange = range;
				if(expansion.isExpanded())
					info.addForceExpansion(expansion);
				
				info.apply();
			return;
		}
		super.internalRefreshStruct(widget, element, updateLabels);
	}

	private ExpansionRange adjustRangeOnRefresh(ExpansionRange range){
		range.assertValid();
		int numEls = rootExpansion.getAllChildSize(false);
		int numItemsAllowed = getMaxRangeSize(rootExpansion);
		if(numEls <= numItemsAllowed)
			return ExpansionRange.createExpansionRange(rootExpansion, 0, numEls);
		int oldSize = range.getSize(false); 
		int oldStart = range.getStartOffset();
		int newSize = numItemsAllowed;
		int newStart;
		if(oldSize < numItemsAllowed){
			if(oldStart + newSize <= numEls){
				newStart = oldStart;
			} else {
				newStart = numEls - newSize;
			}
		} else {
			newStart = oldStart; 
		}
		return ExpansionRange.createExpansionRange(rootExpansion, newStart, newSize);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.jface.viewers.AbstractTreeViewer#internalInitializeTree(org.eclipse.swt.widgets.Control)
	 */
	protected void internalInitializeTree(Control widget) {
		if (contentProviderIsLazyFragmented) {
			if (widget instanceof Tree && widget.getData() != null) {
				initializeTreeInfo((Tree)widget);
				return;
			}
		}
		super.internalInitializeTree(widget);
	}
	
	private void initializeTreeInfo(Tree tree){
		RootExpansionMaterializationContext context = new RootExpansionMaterializationContext(getInput());
		rootExpansion = context.getMaterializedRootExpansion();
		range = ExpansionRange.createExpansionRange(rootExpansion, 0, getMaxRangeSize(rootExpansion));
		setChildCountToTree(tree, range.getDirectChildCount(rootExpansion));
	}
	
	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.jface.viewers.AbstractTreeViewer#updatePlus(org.eclipse.swt.widgets.Item,
	 *      java.lang.Object)
	 */
	protected void updatePlus(Item item, Object element) {
		if (contentProviderIsLazyFragmented) {
			virtualLazyUpdateWidget((TreeItem)item, range, false);
		} else {
			super.updatePlus(item, element);
		}
	}

	/**
	 * Removes the element at the specified index of the parent.  The selection is updated if required.
	 *
	 * @param parentOrTreePath the parent element, the input element, or a tree path to the parent element
	 * @param index child index
	 * @since 3.3
	 */
	public void remove(final Object parentOrTreePath, final int index) {
		if (checkBusy())
			return;
		
		if(true)
			throw new UnsupportedOperationException();
		final List oldSelection = new LinkedList(Arrays
				.asList(((TreeSelection) getSelection()).getPaths()));
		preservingSelection(new Runnable() {
			public void run() {
				TreePath removedPath = null;
				if (internalIsInputOrEmptyPath(parentOrTreePath)) {
					Tree tree = (Tree) getControl();
					if (index < tree.getItemCount()) {
						TreeItem item = tree.getItem(index);
						if (item.getData() != null) {
							removedPath = getTreePathFromItem(item);
							disassociate(item);
						}
						item.dispose();
					}
				} else {
					Widget[] parentItems = internalFindItems(parentOrTreePath);
					for (int i = 0; i < parentItems.length; i++) {
						TreeItem parentItem = (TreeItem) parentItems[i];
						if (index < parentItem.getItemCount()) {
							TreeItem item = parentItem.getItem(index);
							if (item.getData() != null) {
								removedPath = getTreePathFromItem(item);
								disassociate(item);
							}
							item.dispose();
						}
					}
				}
				if (removedPath != null) {
					boolean removed = false;
					for (Iterator it = oldSelection.iterator(); it
							.hasNext();) {
						TreePath path = (TreePath) it.next();
						if (path.startsWith(removedPath, getComparer())) {
							it.remove();
							removed = true;
						}
					}
					if (removed) {
						setSelection(new TreeSelection(
								(TreePath[]) oldSelection
										.toArray(new TreePath[oldSelection
												.size()]), getComparer()),
								false);
					}

				}
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.AbstractTreeViewer#handleTreeExpand(org.eclipse.swt.events.TreeEvent)
	 */
	protected void handleTreeExpand(TreeEvent event) {
		if (contentProviderIsLazyFragmented) {
			if(isExpandCollapseHandlingDisabled()){
				//TODO: do we need to fire the event?
//				fireTreeExpanded(new TreeExpansionEvent(this, event.item
//						.getData()));
			} else {
				Expansion exp = handleTreeExpand((TreeItem)event.item, false);
				if(exp != null)
					fireTreeExpanded(new TreeExpansionEvent(this, exp.getData()));
			}
			
			return;
		}
		super.handleTreeExpand(event);
	}
	
	private Expansion handleTreeExpand(TreeItem item, boolean forceParentCreate){
		Object data = item.getData(); 
		if (data != null || forceParentCreate) {
			MoveInfo expandCollapseMoveInfo = new MoveInfo(rootExpansion, range);
			expandCollapseMoveInfo.selection = getSelectedExpansions();
			expandCollapseMoveInfo.newTopExpansion = getTopVisibleExpansion();
			expandCollapseMoveInfo.curTopExpansion = expandCollapseMoveInfo.newTopExpansion;
			Expansion exp = data != null ? rootExpansion.getExpansion(data) : null;
			if(exp == null){
				exp = getExpansion(item);
				ExpansionMaterializationContext context = new ExpansionMaterializationContext(expandCollapseMoveInfo, exp, data);
				exp = context.getMaterializedExpansion();
			} else {
				expandCollapseMoveInfo.expandExpansion(exp, null, -1);
			}
			
			if(exp.getDirectChildSize(false) != 0){
				expandCollapseMoveInfo.addForceExpansion(exp, item);
				Assert.isTrue(exp.isExpanded());
				expandCollapseMoveInfo.apply();
			} else {
				item.setItemCount(0);
			}
			return exp;
		} 
		return null;
	}
	
	protected void handleTreeCollapse(TreeEvent event) {
		if (contentProviderIsLazyFragmented) {
			if(isExpandCollapseHandlingDisabled()){
				//TODO: do we need to fire the event?
//				fireTreeCollapsed(new TreeExpansionEvent(this, event.item.getData()));
			} else {
				if(event.item.getData() != null){
					Object data  = event.item.getData();
					handleTreeCollapse((TreeItem)event.item);
					fireTreeCollapsed(new TreeExpansionEvent(this, data));
				}
			}
			return;
		}
		super.handleTreeCollapse(event);
	}
	
	protected void handleTreeCollapse(TreeItem item) {
		if(item.getData() == null)
			return;
		MoveInfo expandCollapseMoveInfo = new MoveInfo(rootExpansion, range);
		expandCollapseMoveInfo.selection = getSelectedExpansions();
		expandCollapseMoveInfo.newTopExpansion = getTopVisibleExpansion();
		expandCollapseMoveInfo.curTopExpansion = expandCollapseMoveInfo.newTopExpansion;
		expandCollapseMoveInfo.itemCollapsed(item);
		expandCollapseMoveInfo.apply();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.AbstractTreeViewer#setContentProvider(org.eclipse.jface.viewers.IContentProvider)
	 */
	public void setContentProvider(IContentProvider provider) {
		contentProviderIsLazyFragmented = (provider instanceof ILazyFragmentTreeContentProvider);
		contentProviderSupportsIndexInfo = (provider instanceof ILazyFragmentTreeContentWithIndexProvider);
		super.setContentProvider(provider);
	}

	public void setHasChildren(Object elementOrTreePath, boolean hasChildren) {
		if(contentProviderIsLazyFragmented){
			throw new UnsupportedOperationException();
		}
		super.setHasChildren(elementOrTreePath, hasChildren);
	}

	private void setHasChildrenToItem(final TreeItem item, final boolean hasChildren) {

		preservingSelection(new Runnable() {
			public void run() {
				if (!hasChildren) {
					item.setItemCount(0);
				} else {
					item.setItemCount(1);
					TreeItem child = item.getItem(0);
					if (child.getData() != null) {
						disassociate(child);
					}
					item.clear(0, true);
				}
			}
		});
	}
	
	private void setChildInfoToItem(Widget parent, TreeItem item, int itemIndex, ExpansionRange range, Expansion exp){
		if(exp.isExpanded()){
			setChildCountToItem(item, calcChildCount(range, exp));
		} else {
			setHasChildrenToItem(item, exp.getDirectChildSize(false) != 0);
		}
	}

	private void virtualLazyUpdateWidget(Widget parent, TreeItem child, int itemIndex, ExpansionRange range, boolean fullUpdate) {
		boolean oldBusy = isBusy();
		setBusy(false);
		try {
			Object parentData = getData(parent, false);
			if(parentData == null)
				return;
			Expansion parentExp = getExpansion(parent/*, parentData*/);
			int offset = range.getStartIndex(parentExp);
			Assert.isTrue( offset >= 0);
			int expIndex = itemIndex + offset;
			Expansion existingChild = parentExp.getDirectExistingChildByIndex(expIndex);
			if(existingChild != null){
				if(child.getData() == null || fullUpdate)
					replace(parent, itemIndex, existingChild.getData());
				setChildInfoToItem(parent, child, itemIndex, range, existingChild);
			} else {
				Object element;
				if(!fullUpdate){
					element = child.getData();
					if(element == null)
						fullUpdate = true;
				} else {
					element = null;
				}
				SimpleMaterializationContext context = new SimpleMaterializationContext(parentData, expIndex, element);
				if(fullUpdate){
					element = context.getElement();
					replace(parent, itemIndex, element);
				}
				boolean hasChildren = context.getHashChildren();
				setHasChildrenToItem(child, hasChildren);
			}
		} finally {
			setBusy(oldBusy);
		}
	}
	
	protected void disassociate(Item item) {
		if (contentProviderIsLazyFragmented) {
			// avoid causing a callback:
			item.setText(" "); //$NON-NLS-1$
		}
		super.disassociate(item);
	}

	private boolean isScrollDisabled(){
		return movingInfo != null;
	}

	private boolean isExpandCollapseHandlingDisabled(){
		return movingInfo != null;
	}

	private void handleScroll(SelectionEvent vScrolEvent){
		if(!isScrollDisabled()){
			boolean scrollMode;
			if(DbgUtil.DEBUG)
				DbgUtil.println("scrolling type=" + vScrolEvent.detail); //$NON-NLS-1$
			switch(vScrolEvent.detail){
			case SWT.NONE:
				if(DbgUtil.DEBUG)
					DbgUtil.println("Scroll NONE, skipping .."); //$NON-NLS-1$
				return;
			case SWT.DRAG:
			case SWT.HOME:
			case SWT.END:
				scrollMode = true;
				break;
			default:
//					SWT.ARROW_DOWN. SWT.ARROW_UP. SWT.PAGE_DOWN. SWT.PAGE_UP
				scrollMode = false;
				break;
			}

			MoveInfo info = calcMoveInfoOnScroll(scrollMode);
			if(info != null){	
			    info.apply();
			}
		}
	}
	
	private void handleKey(KeyEvent keyEvent){
		if(!isScrollDisabled()){
			MoveInfo info;
			if(DbgUtil.DEBUG)
				DbgUtil.println("key code=" + keyEvent.keyCode); //$NON-NLS-1$
			switch(keyEvent.keyCode){
			case SWT.PAGE_UP:
			case SWT.PAGE_DOWN:
			case SWT.ARROW_UP:
			case SWT.ARROW_DOWN:
				info = calcMoveInfoOnScroll(false);
				break;
			case SWT.HOME:
				info = calcMoveInfoOnHomeEnd(true);
				break;
			case SWT.END:
				info = calcMoveInfoOnHomeEnd(false);
				break;
			default:
				return;
			}

			if(info != null){	
			    info.apply();
			}
		}
	}
	
	private void handleMouseWeel(MouseEvent e){
		if(!isScrollDisabled()){
			if(DbgUtil.DEBUG)
				DbgUtil.println("mouse weel event"); //$NON-NLS-1$
			
			MoveInfo info = calcMoveInfoOnScroll(false);
			if(info != null){	
			    info.apply();
			}
		}
	}
	
	private MoveInfo calcMoveInfoOnScroll(boolean scrollMode){
		if(range.getSize(false) == rootExpansion.getAllChildSize(false))
			return null;

		TreeInfo info = new TreeInfo();
		Expansion topExp = info.getTopVisibleExpansion();
		if(topExp == null)
			return null;
		if(range.getRelation(topExp) == ExpansionRange.REFERENCE){
			if(DbgUtil.DEBUG)
				DbgUtil.println("reference became visible, choose first expansion in range"); //$NON-NLS-1$
			topExp = range.getStartExpansion();
		}
		if(DbgUtil.DEBUG)
			DbgUtil.println("-----------------------"); //$NON-NLS-1$
		
		MoveInfo mInfo = new MoveInfo(rootExpansion, range);
		mInfo.curTopExpansion = topExp;
		mInfo.calcNewRangeInfoOnScroll(scrollMode);
		
		return mInfo;
	}

	private MoveInfo calcMoveInfoOnHomeEnd(boolean home){
		if(range.getSize(false) == rootExpansion.getAllChildSize(false))
			return null;
		Expansion topExp;
		if(home){
			topExp = rootExpansion.getChildExpansionByOffset(0, false);
		} else {
			int numVis = getNumVisibleItems();
			int offset = rootExpansion.getAllChildSize(false) - numVis;
			if(offset < 0)
				offset = 0;
			topExp = rootExpansion.getChildExpansionByOffset(offset, false);
		}
		if(topExp == null)
			return null;
//		if(range.getRelation(topExp) == ExpansionRange.REFERENCE){
//			if(DbgUtil.DEBUG)
//				DbgUtil.println("reference became visible, choose first expansion in range"); //$NON-NLS-1$
//			topExp = range.getStartExpansion();
//		}
		if(DbgUtil.DEBUG)
			DbgUtil.println("-----------------------"); //$NON-NLS-1$
		
		MoveInfo mInfo = new MoveInfo(rootExpansion, range);
		mInfo.curTopExpansion = topExp;
		mInfo.calcNewRangeInfoOnScroll(true);
		
		return mInfo;
	}

	private Expansion getTopVisibleExpansion(){
		TreeItem topItem = getTree().getTopItem();
		if(topItem == null){//tree is empty
//			Assert.isTrue(getTree().getItemCount() == 0);
			return null;
		}
		Expansion exp = getExpansion(topItem);
//		Assert.isTrue(range.getRelation(exp) != ExpansionRange.NONE);
		return exp;
	}
//	
	private int getNumVisibleItems(){
		Tree tree = getTree();
		int height = tree.getClientArea().height;
		int headerHeight = tree.getHeaderHeight();
		height = height - headerHeight;
		if(height < 0){
			if(DbgUtil.DEBUG)
				DbgUtil.println("height < 0"); //$NON-NLS-1$
			height = 0;
		}
		int itemHeight = getItemHeight();
		int numVisibleItems = height / itemHeight;
		if(height % itemHeight != 0)
			numVisibleItems++;
		else if(numVisibleItems == 0)
			numVisibleItems = 1;
		
		return numVisibleItems;
	}
	
	private int getItemHeight(){
		if(itemHeight == -1){
			itemHeight = getTree().getItemHeight();
		}
		return itemHeight;
	}
	
	private Expansion getBottomVisibleExpansion(Expansion topExpansion, int numVisibleItems){
		Expansion exp = topExpansion.getRelativeExpansion(numVisibleItems - 1);
		if(exp == null || range.getRelation(exp) == ExpansionRange.NONE){
			exp = range.getLastIncludedExpansion();
		}
		Assert.isTrue(range.getRelation(exp) != ExpansionRange.NONE);
		return exp;
	}

	protected void preservingSelection(Runnable updateCode) {
		if(preservingSelection){
			updateCode.run();
			return;
		}
		try {
			preservingSelection = true;
			super.preservingSelection(updateCode);
		} finally {
			preservingSelection = false;
		}
	}
	
	public void setTopVisibleExpansion(Expansion exp){
		Assert.isTrue(range.getRelation(exp) == ExpansionRange.CONTAINMENT);

		TreeItem item = getTreeItem(exp);
		Assert.isTrue(item != null);
		getTree().setTopItem(item);
	}
	
	public ISelection getSelection() {
		if(contentProviderIsLazyFragmented){
			Expansion[] exps = getSelectedExpansions();
			return exps.length != 0 ? new ExpansionSelection(exps) : ExpansionSelection.EMPTY;
		}
		return super.getSelection();
	}
	
	private Expansion[] getSelectedExpansions(){
		Tree tree = getTree();
		if (tree == null || tree.isDisposed()) {
			return ExpansionSelection.EMPTY_EXPANSION_ARRAY;
		}
		Item[] items = getSelection(tree);
		if(items.length == 0)
			return ExpansionSelection.EMPTY_EXPANSION_ARRAY;
		
		Expansion[] exps = new Expansion[items.length];
		int cursor = 0;
		for(int i = 0; i < items.length; i++){
			TreeItem item = (TreeItem)items[i];
			if(item.getData() != null){
				Expansion exp = getExpansion((TreeItem)items[i]);
				if(exp != null){
					exps[cursor++] = exp;
				}
			}
		}
		if(cursor < exps.length){
			if(cursor == 0){
				exps = ExpansionSelection.EMPTY_EXPANSION_ARRAY; 
			} else {
				Expansion[] tmp = new Expansion[cursor];
				System.arraycopy(exps, 0, tmp, 0, cursor);
				exps = tmp;
			}
		}
		
		Arrays.sort(exps);
		return exps;
	}
	
	protected void setSelectionToWidget(ISelection selection, boolean reveal) {
		if (selection instanceof IExpansionSelection) {
			setSelectionToWidget(null, ((IExpansionSelection) selection).getExpansions());
		} else {
			super.setSelectionToWidget(selection, reveal);
		}
	}
	
	private void setSelection(Expansion[] exps){
		Arrays.sort(exps);
		Expansion[] curSelection = getSelectedExpansions();
//already sorted		
//		Arrays.sort(curSelection);
		if(Arrays.equals(exps, curSelection))
			return;
		
		TreeItem[] items = getItems(exps);
		getTree().setSelection(items);
	}
	
	private TreeItem[] getItems(Expansion[] exps){
		TreeItem[] items = new TreeItem[exps.length];
		int cursor = 0;
		for(int i = 0; i < exps.length; i++){
			TreeItem item = getTreeItem(exps[i]);
			if(item != null){
				items[cursor++] = item;
			}
		}
		if(cursor < items.length){
			if(cursor == 0){
				items = new TreeItem[0];
			} else {
				TreeItem[] tmp = new TreeItem[cursor];
				System.arraycopy(items, 0, tmp, 0, cursor);
				items = tmp;
			}
		}
		return items;
	}

}
