/**********************************************************************
 * Copyright (c) 2006 Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AggregatedInvocation.java,v 1.1 2006/10/11 22:11:11 ewchan Exp $
 * 
 **********************************************************************/
package org.eclipse.hyades.trace.views.util.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.models.trace.TRCThread;

public class AggregatedInvocation {
	
	public static class Item {
		protected TRCFullMethodInvocation _invocation;
		protected AggregatedInvocation _parent;
		
		public Item(TRCFullMethodInvocation invocation, AggregatedInvocation parent) {
			_invocation = invocation;
			_parent = parent;
		}
		
		public Item getInvokedBy() {
			TRCMethodInvocation invokedBy = _invocation.getInvokedBy();
			return (invokedBy instanceof TRCFullMethodInvocation) ?
					new Item((TRCFullMethodInvocation)invokedBy, _parent) : null;
		}

		public double getTimePercentPerThread() {
			double time = getSumTime();
			double threadTime = _parent.getThreadTime();
			return threadTime > 0 ? time / threadTime : 0;
		}

		public double getSumTime() {
			double start = _invocation.getEntryTime();
			double end = _invocation.getExitTime();
			if (end <= 0 || end > _parent.getSnapshotTime()) {
				end = _parent.getSnapshotTime();
			}
			return Math.max(end - start - _invocation.getOverhead(), 0);
		}

		public TRCMethod getMethod() {
			return _invocation.getMethod();
		}
		
		public TRCFullMethodInvocation getInvocation() {
			return _invocation;
		}
		
		public AggregatedInvocation getParent() {
			return _parent;
		}
	}

	protected TRCThread _thread;
	protected TRCMethod _method;
	protected AggregatedInvocation _parent;
	protected ArrayList _allInvocations = new ArrayList();
	protected HashMap _nestedCalls;
	protected boolean _running;
	protected double _snapshotTime;

	protected double _minTime = Double.MAX_VALUE;
	protected double _maxTime;
	protected double _sumTime;

	public AggregatedInvocation(TRCFullMethodInvocation call) {
		_thread = call.getThread();
		_method = call.getMethod();
		_snapshotTime = call.getProcess().getLastEventTime();
		add(call);
	}

	public AggregatedInvocation(AggregatedInvocation inv, TRCMethod method) {
		_thread = inv._thread;
		_method = method;
		_snapshotTime = inv._snapshotTime;
		_parent = inv;
	}

	public TRCMethod getMethod() {
		return _method;
	}

	public TRCThread getThread() {
		return _thread;
	}

	public List getAllInvocations() {
		return _allInvocations;
	}

	public AggregatedInvocation getParent() {
		return _parent;
	}

	public Collection getNestedCalls() {
		if (_nestedCalls == null) {
			_nestedCalls = new HashMap();
			updateNestedCalls();
		}
		return _nestedCalls.values();
	}

	public double getSnapshotTime() {
		return _snapshotTime;
	}

	public double getMaxTime() {
		return _maxTime;
	}

	public double getMinTime() {
		return !_allInvocations.isEmpty() ? _minTime : 0;
	}

	public double getAvgTime() {
		return !_allInvocations.isEmpty() ? _sumTime / (double)_allInvocations.size(): 0;
	}

	public double getSumTime() {
		return _sumTime;
	}

	public static Collection getThreadCalls(TRCThread thread) {
		HashMap calls = new HashMap();

		EList invocations = thread.getInitialInvocations();
		for (Iterator it = invocations.iterator(); it.hasNext();) {
			Object obj = it.next();
			if (obj instanceof TRCFullMethodInvocation) {
				TRCFullMethodInvocation call = (TRCFullMethodInvocation)obj;
				TRCMethod method = call.getMethod();
				AggregatedInvocation inv = (AggregatedInvocation) calls.get(method);
				if (inv == null) {
					inv = new AggregatedInvocation(call);
					calls.put(method, inv);
				} else {
					inv.add(call);
				}
			}
		}
		return calls.values();
	}

	public double getTimePercentPerThread() {
		double time = getThreadTime();
		return time > 0 ? getSumTime() / time : 0;
	}
	
	public double getThreadTime() {
		double start = _thread.getStartTime();
		double end = _thread.getStopTime();
		if (end > _snapshotTime || end <= 0) {
			end = _snapshotTime;
		}
		return Math.max(end - start, 0);
	}

	protected void updateNestedCalls() {
		for (int i=0; i < _allInvocations.size(); i++) {
			TRCFullMethodInvocation inv = ((Item)_allInvocations.get(i))._invocation;
			EList invokes = inv.getInvokes();
			for (Iterator it = invokes.iterator(); it.hasNext();) {
				Object obj = it.next();
				if (!(obj instanceof TRCFullMethodInvocation)) {
					continue;
				}
				TRCFullMethodInvocation call = (TRCFullMethodInvocation)obj;
				if (call.getEntryTime() > _snapshotTime) {
					break;
				}
				TRCMethod method = call.getMethod();
				AggregatedInvocation nestedCall = (AggregatedInvocation)_nestedCalls.get(method);
				if (nestedCall == null) {
					nestedCall = new AggregatedInvocation(this, method);
					_nestedCalls.put(method, nestedCall);
				}
				nestedCall.add(call);
			}
		}
	}	

	protected void add(TRCFullMethodInvocation call) {
		_allInvocations.add(new Item(call, this));
		double start = call.getEntryTime();
		double end = call.getExitTime();
		if (end <= 0 || end > _snapshotTime) {
			end = _snapshotTime;
		}
		double time = Math.max(end - start - call.getOverhead(), 0);
		_minTime = Math.min(time, _minTime);
		_maxTime = Math.max(time, _maxTime);
		_sumTime += time;
	}
}
