/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BufferedCanvas.java,v 1.1 2005/07/22 14:49:36 hleung Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.util.internal;

import org.eclipse.hyades.trace.views.internal.*;
import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;

/**
 * A Canvas which reduces flicker by drawing in an off screen buffer.
 */
public abstract class BufferedCanvas extends Canvas
										implements IColorChangedListener
{

	/** The drawable for double buffering */
	private Image fBuffer;

	public BufferedCanvas(Composite parent, int flags) {
		super(parent, flags + SWT.NO_BACKGROUND);

		addPaintListener(
			new PaintListener() {
				public void paintControl(PaintEvent event) {
					doubleBufferPaint(event.gc);
				}
			}
		);

		addDisposeListener(
			new DisposeListener() {
				public void widgetDisposed(DisposeEvent e) {
					if (fBuffer != null) {
						
						TraceUIPlugin.getDefault().removeColorChangedEventListener(BufferedCanvas.this);
						fBuffer.dispose();
						fBuffer= null;
					}
				}
			}
		);
		
		TraceUIPlugin.getDefault().addColorChangedEventListener(this);
		
	}
	abstract public void doPaint(GC gc);
	/**
	 * Double buffer drawing.
	 * @private
	 */
	private void doubleBufferPaint(GC dest) {

		Point size= getSize();

		if (size.x <= 0 || size.y <= 0)
			return;

		if (fBuffer != null) {
			Rectangle r= fBuffer.getBounds();
			if (r.width != size.x || r.height != size.y) {
				fBuffer.dispose();
				fBuffer= null;
			}
		}
		if (fBuffer == null)
			fBuffer= new Image(getDisplay(), size.x, size.y);

		GC gc= new GC(fBuffer);
		try {
			gc.setBackground(getBackground());
			gc.fillRectangle(0, 0, size.x, size.y);
			doPaint(gc);
		} finally {
			gc.dispose();
		}

		dest.drawImage(fBuffer, 0, 0);
	}
	public void repaint() {
		if (!isDisposed()) {
			GC gc= new GC(this);
			
			FontData data= SpectrumColorMap.getFontData();
			Font font = new Font(getDisplay(), data);
			gc.setFont(font);
			
			doubleBufferPaint(gc);

			font.dispose();
			gc.dispose();
		}
	}
	
	/**
	 * see IColorChangedListener
	 *
	 */
	public void handleColorChangedEvent()
	{	
		if(!isDisposed())	
		{
			setBackground(SpectrumColorMap.getBackgroundColor());	  			
			repaint();
		}
		else
		{
			TraceUIPlugin.getDefault().removeColorChangedEventListener(this);						
		}
		
	}
	
}
