/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CoverageUtil.java,v 1.4 2007/08/01 00:29:38 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.util.internal;

import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCFilter;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCSourceInfo;

/**
 * Here are some utilities related to Coverage feature
 */
public class CoverageUtil {

	/**
	 * @param e
	 * @return source location signature if any
	 */
	private static String getSourceSig(TRCSourceInfo e) {
		return (e != null ? e.getLocation()+":" : "");
	}
	
	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCMethod e) {
		return  getSourceSig(e.getSourceInfo())+
				e.getLineNo()+":"+
				getSig(e.getDefiningClass())+"."+
				e.getName()+
				e.getSignature();
	}

	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCClass e) {
		return  getSourceSig(e.getSourceInfo())+
				e.getLineNo()+":"+
				getSig(e);
	}

	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCPackage e) {
		return e.getName();
	}
	
	/**
	 * @param e
	 * @return qualified name of the class
	 */
	private static String getSig(TRCClass e) {
		return (e.getPackage().getName() == null ? "" : (e.getPackage().getName()+"."))+
				e.getName();
	}

	/**
	 * Private boolean class for convenience.
	 */
	private static class LocalBoolean
	{
		public boolean bool;
		public LocalBoolean(boolean b)
		{
			bool=b;
		}
	}

	/**
	 * This is for post-filtering needed because of the boundary classes collected even if excluded
	 * by the filter set.<br>
	 * The empty packages are filtered out and
	 * those listed in the active filters for this agent configuration
	 * @param pac
	 * @return
	 */
	public static boolean isFiltered(TRCPackage pac) {
		if (pac.getClasses() == null ||
			pac.getClasses().size() == 0) {
			//if empty package
			return true;
		}

		List cllist = pac.getClasses();
		int remaining = cllist.size();
		for(int idx=0;idx<remaining;idx++){
			Object obj = cllist.get(idx);
			if(obj instanceof TRCClass){
				if(CoverageUtil.isFiltered((TRCClass)obj)){
					remaining--;
				}
			}
		}
		if (remaining == 0) {
			//if packages doesn't contain class
			return true;
		}
		
		//No need to filter packages if they contains included classes
//		Object[] conflist = pac.getProcess().getAgent().getAgentProxy().getConfigurations().toArray();
//		int idxmax = conflist.length;
//		for(int idx = 0; idx < idxmax; idx++){
//			TRCConfiguration c = (TRCConfiguration) conflist[idx];
//			if(!c.isActive())continue;
//			
//			Object[] filtlist = c.getFilters().toArray();
//			for (int k = 0; k < filtlist.length; ++k) {
//				TRCFilter f = (TRCFilter)filtlist[k];
//				if(!f.getActive().booleanValue())
//					continue;
//				LocalBoolean exclude = new LocalBoolean(true);
//				if(isFilteredPackage(f,pac,exclude)){
//					if(exclude.bool)
//						return true;
//					else
//						return false;
//				}
//			}
//		}
		return false;
	}

	/**
	 * The empty classes are filtered out too
	 * @param c
	 * @return
	 */
	public static boolean isFiltered(TRCClass cl) {
		if (cl.getMethods() == null ||
			cl.getMethods().size() == 0) {
			//if empty class
			return true;
		}
		Object[] conflist = cl.getPackage().getProcess().getAgent().getAgentProxy().getConfigurations().toArray();
		int idxmax = conflist.length;
		for(int idx = 0; idx < idxmax; idx++){
			TRCConfiguration c = (TRCConfiguration) conflist[idx];
			if(!c.isActive())continue;
			
			Object[] filtlist = c.getFilters().toArray();
			for (int k = 0; k < filtlist.length; ++k) {
				TRCFilter f = (TRCFilter)filtlist[k];
				if(!f.getActive().booleanValue())
					continue;
				LocalBoolean exclude = new LocalBoolean(true);
				if(isFilteredClass(f,cl,exclude)){
					if(exclude.bool)
						return true;
					else
						return false;
				}
			}
		}
		return false;
	}

	/**
	 * It is no more used.
	 * @param filter
	 * @param pack
	 * @return 
	 */
//	private static boolean isFilteredPackage(TRCFilter filter, TRCPackage pack, LocalBoolean exclude)
//	{
//		String pattern = filter.getPattern();
//		String mode = filter.getMode();
//		String type = filter.getType();
//		if(type.equals(mode)){
//			type = parseFilterPattern(pattern);
//			pattern = externFilterPattern;
//		}
//		String methodPattern = filter.getOperation();
//		if(methodPattern!=null && (methodPattern.length()>0 && !methodPattern.equals("*"))){
//			return false;
//		}
//		String name = pack.getName();
//		if(pattern.endsWith(".")){
//			name = name+".";
//		}
//		if(mode.equals("INCLUDE")){//$NON-NLS-1$
//			exclude.bool = false;
//		}else{
//			exclude.bool = true;
//		}
//		if(type.equals("NONE")){
//			if (name.equals(pattern)) {
//				return true;
//			}
//		}else if(type.equals("PREFIX")){
//			if (name.endsWith(pattern)) {
//				return true;
//			}
//		}else if(type.equals("SUFFIX")){
//			if (name.startsWith(pattern)) {
//				return true;
//			}
//		}
//		return false;
//	}

	/**
	 * @param filter
	 * @param cl
	 * @return 
	 */
	private static final String Path_Separator = ".";
	private static boolean isFilteredClass(TRCFilter filter, TRCClass cl, LocalBoolean exclude)
	{
		String pattern = filter.getPattern();
		String mode = filter.getMode();
		String type = filter.getType();
		if(type.equals(mode)){
			type = parseFilterPattern(pattern);
			pattern = externFilterPattern;
		}
		String methodPattern = filter.getOperation();
		if(methodPattern!=null && methodPattern.length()>0 && !methodPattern.equals("*")){
			return false;
		}
		String clname = cl.getName();
		String fullname = cl.getPackage().getName() + Path_Separator + clname;
		
		if(mode.equals("INCLUDE")){//$NON-NLS-1$
			exclude.bool = false;
		}else{
			exclude.bool = true;
		}
		if(type.equals("NONE")){
			if (fullname.equals(pattern)) {
				return true;
			}else if (clname.equals(pattern)) {
				return true;
			}
		}else if(type.equals("PREFIX")){
			if (fullname.endsWith(pattern)) {
				return true;
			}else if (clname.endsWith(pattern)) {
				return true;
			}
		}else if(type.equals("SUFFIX")){
			if (fullname.startsWith(pattern)) {
				return true;
			}else if (clname.startsWith(pattern)) {
				return true;
			}
		}
		return false;
	}

	private static String externFilterPattern=null;
	private static String parseFilterPattern(String filterPattern)
	{
	 String genericPattern = "NONE";
	 
	 if(filterPattern.length()==0){
	 	externFilterPattern = filterPattern;
	 	return genericPattern;
	 }

	 int p = filterPattern.indexOf('*');
	 if (p == 0)
	 {
	  genericPattern = "PREFIX";
	  externFilterPattern = filterPattern.substring(1);
	 }
	 else if (p == (filterPattern.length()-1))
	 {
	  genericPattern = "SUFFIX";
	  externFilterPattern = filterPattern.substring(0,(filterPattern.length()-1));
	 }else{
	 	externFilterPattern = filterPattern;
	 }
	 return genericPattern;
	}
	
	/**
	 * check if trace object contains method coverage information
	 * @param mofObject
	 * @return true if method coverage info is collected, false otherwise
	 */
	public static boolean isCoverageSupport(EObject mofObject) {
	        boolean isSupport=false;

	        if (mofObject != null) {
	            if (mofObject instanceof TRCAgentProxy) {
	            	List configs = ((TRCAgentProxy) mofObject).getConfigurations();
	            	
	        		for (int i = 0; i < configs.size(); i++)
	        		{
	        			//need to loop through the configurations to find the correct one
	        			TRCConfiguration config = (TRCConfiguration)configs.get(i);
	        			List options = config.getOptions();
	        			for (int opt = 0; opt < options.size(); opt++)
	        			{
	        				//need to loop through all the options to find ones 
	        				//relating to the coverage method count option
	        				TRCOption option = (TRCOption)options.get(opt);
	        				if (option.getKey().equals("METHOD_COUNTS")) {
	        					isSupport = Boolean.valueOf(option.getValue()).booleanValue();
	        					return isSupport;
	        				}
	        			}
	        		}
	            	
	            } else if (mofObject instanceof TRCProcessProxy) {
	                Object[] agents = ((TRCProcessProxy) mofObject).getAgentProxies().toArray();
	                for (int idx = 0; idx < agents.length; idx++) {
	                    	isSupport = isCoverageSupport((TRCAgentProxy) agents[idx]);
	                    	if (isSupport) return true;
	                }
	            } else if (mofObject instanceof TRCNode) {
	                Object[] processes = ((TRCNode) mofObject).getProcessProxies().toArray();
	                for (int idx = 0; idx < processes.length; idx++) {
	                    	isSupport = isCoverageSupport((TRCProcessProxy) processes[idx]);
	                    	if (isSupport) return true;
	                }
	            } else if (mofObject instanceof TRCMonitor) {
	                Object[] nodes = ((TRCMonitor) mofObject).getNodes().toArray();
	                for (int k = 0; k < nodes.length; k++) {
	                    	isSupport = isCoverageSupport((TRCNode) nodes[k]);
	                    	if (isSupport) return true;
	                }
	            }
	        }

		return isSupport;
	}

}
