/**********************************************************************
 * Copyright (c) 2006, 2007 Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ThreadDetails.java,v 1.2 2007/04/09 04:49:30 ewchan Exp $
 * 
 **********************************************************************/

package org.eclipse.hyades.trace.views.util.internal;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadDeadEvent;
import org.eclipse.hyades.models.trace.TRCThreadDeadLockEvent;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.models.trace.TRCThreadRunningEvent;
import org.eclipse.hyades.models.trace.TRCThreadSleepingEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForLockEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForObjectEvent;
import org.eclipse.swt.SWT;

public class ThreadDetails {
	
	public static final int UNKNOWN = 0;
	public static final int RUNNING = 1;
	public static final int SLEEPING = 2;
	public static final int WAITING = 3;
	public static final int BLOCKED = 4;
	public static final int DEADLOCK = 5;
	public static final int STOPPED = 6;
	
	protected static final int[] _colorCodes = 
			{
				SWT.COLOR_WHITE,    //unknown
				SWT.COLOR_GREEN,    //running
				SWT.COLOR_BLUE,     //sleeping
				SWT.COLOR_CYAN,     //waiting
				SWT.COLOR_YELLOW,   //blocked
				SWT.COLOR_RED,      //deadlock
				SWT.COLOR_DARK_GRAY //stopped
			};

	protected TRCThread _thread;
	protected int _threadState;
	
	protected double _runningTime;
	protected double _sleepingTime;
	protected double _waitingTime;
	protected double _blockedTime;
	protected double _deadlockedTime;
	protected int _blockCount;
	protected int _deadlockCount;
	
	public ThreadDetails(TRCThread thread) {
		_thread = thread;
		updateNumbers();
	}
	
	public TRCThread getThread() {
		return _thread;
	}
	
	public int getState() {
		return _threadState;
	}
	
	public double getRunningTime() {
		return _runningTime;
	}
	
	public int getBlockCount() {
		return _blockCount;
	}
	
	public int getDeadlockCount() {
		return _deadlockCount;
	}
	
	public double getBlockedTime() {
		return _blockedTime;
	}
	
	public double getDeadlockedTime() {
		return _deadlockedTime;
	}
	
	public double getSleepingTime() {
		return _sleepingTime;
	}
	
	public double getWaitingTime() {
		return _waitingTime;
	}
	
	public static int getStateColor(int state) {
		return _colorCodes[state];
	}
	
	protected int getEventType(TRCThreadEvent event) {
		if (event instanceof TRCThreadRunningEvent) {
			return RUNNING;
		} else if (event instanceof TRCThreadSleepingEvent) {
			return SLEEPING;
		} else if (event instanceof TRCThreadDeadLockEvent) {
			return DEADLOCK;
		} else if (event instanceof TRCThreadWaitingForLockEvent) {
			return BLOCKED;
		} else if (event instanceof TRCThreadWaitingForObjectEvent) {
			return WAITING;
		} else if (event instanceof TRCThreadDeadEvent) {
			return STOPPED;
		}
		return UNKNOWN;
	}
	
	protected void updateNumbers() {
		EList events = _thread.getThreadEvents();
		/*
		 * https://bugs.eclipse.org/bugs/show_bug.cgi?id=181195
		 * ConcurrentModificationException is thrown when using iterator
		 * when the list is modified at the same time (by loader)
		 */
//		Iterator it = events.iterator();
//		if (!it.hasNext()) {
//			return;
//		}
//		TRCThreadEvent event = (TRCThreadEvent)it.next();
//		while (it.hasNext()) {
//			TRCThreadEvent next = (TRCThreadEvent)it.next();
//			double timeSpan = Math.max(next.getTime() - event.getTime(), 0);
//			addEvent(getEventType(event), timeSpan);
//			event = next;
//		}
		int count = events.size();
		if (count < 1)
			return;
		TRCThreadEvent event = (TRCThreadEvent)events.get(0);
		for (int i = 1; i < count; i++){
			TRCThreadEvent next = (TRCThreadEvent)events.get(i);
			double timeSpan = Math.max(next.getTime() - event.getTime(), 0);
			addEvent(getEventType(event), timeSpan);
			event = next;
		}
		_threadState = getEventType(event);
		double timeSpan = Math.max(_thread.getProcess().getLastEventTime() - event.getTime(), 0);
		addEvent(_threadState, timeSpan);
	}
	
	protected void addEvent(int eventType, double timeSpan) {
		switch (eventType) {
		case BLOCKED:
			_blockedTime += timeSpan;
			_blockCount++;
			break;
		case DEADLOCK:
			_deadlockedTime += timeSpan;
			_deadlockCount ++;
			break;
		case SLEEPING:
			_sleepingTime += timeSpan;
			break;
		case WAITING:
			_waitingTime += timeSpan;
			break;
		case RUNNING:
			_runningTime += timeSpan;
			break;
		}
	}
}
