/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JavaOpenSource.java,v 1.3 2007/05/31 04:48:17 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.views.utils;

import java.util.Iterator;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.models.trace.TRCMethodProperties;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCObjectReference;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.views.adapter.internal.TraceConstants;
import org.eclipse.hyades.trace.views.internal.TraceUIMessages;
import org.eclipse.hyades.trace.views.internal.TraceUIPlugin;
import org.eclipse.hyades.ui.internal.util.OpenJavaSource;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;

public class JavaOpenSource {

	/**
	 * Open the source of Java type EObject in editor if it is available in workbench.  
	 * 
	 * @param eObj EObject to try open in editor for its source
	 * @param showError preference on showing error message upon error locating source. 
	 * @return false when object provided is not valid. true otherwise. 
	 */
	public static boolean openSource(EObject eObj, boolean showError) {
		
		if (eObj != null) {
			final String pattern = getPattern(eObj);
			final int javaType = getType(eObj);
			
			final boolean showErrorMessage = showError;
			if (pattern != null && javaType != IJavaSearchConstants.UNKNOWN) {
				// Now call the core utility function to actually locate and open the source for the object.
				if(!OpenJavaSource.openSource(pattern, javaType, SearchEngine.createWorkspaceScope(), true))
				{
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							if (showErrorMessage) {
								MessageDialog.openInformation(
								UIPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(),
								TraceMessages.TRC_MSGT,
								NLS.bind(TraceUIMessages._68, pattern));
							}
						}
					});
				}
				return true;
			}
		}
		
		return false;
	}
	
	private static String getPattern(Object selection) {
		// Added: Handle the case where the user selects a package, then clicks on the
		// Open Source button on the toolbar - see bugzilla 51854
		if(selection == null || selection instanceof TRCPackage) {
			return null;
		}
				
		if(selection instanceof TRCMethodInvocation)
		{
			selection = ((TRCMethodInvocation)selection).getMethod();
		}
		
	   	if (selection instanceof TRCMethod) {
			TRCMethod trcMethod = (TRCMethod) selection;
			
			if((trcMethod.getModifier() & TRCMethodProperties.JAVA_CONSTRUCTOR) !=0 )
			{
				// For constructors we only require the parameter signature appended to the class name
				TRCClass clas = trcMethod.getDefiningClass();
				
				String packName = clas.getPackage().getName();
				if(packName.equals(""))
					return (new StringBuffer())
					                     .append(clas.getName()).append(trcMethod.getSignature()).toString();			
				else			
					return (new StringBuffer()).append(clas.getPackage().getName()).append(".")
					                     .append(clas.getName()).append(trcMethod.getSignature()).toString();
			}
			
			else
			 {
				TRCClass clas = trcMethod.getDefiningClass();
				
				String packName = clas.getPackage().getName();
				if(packName.equals(""))
					return (new StringBuffer())
					                     .append(clas.getName()).append(".")
					                     .append(trcMethod.getName()).append(trcMethod.getSignature()).toString();			
				else			
					return (new StringBuffer()).append(clas.getPackage().getName()).append(".")
					                     .append(clas.getName()).append(".")
					                     .append(trcMethod.getName()).append(trcMethod.getSignature()).toString();
				
			}
		} 
		else {
			if (selection instanceof TRCObjectReference)
			{
				TRCObjectReference obj = (TRCObjectReference)selection;
				if (TraceUIPlugin.getDefault().getPreferenceStore().getInt(TraceConstants.OBJ_REF_OPTION)==TraceConstants.OBJ_REF_REFERANDS)
				{
					selection = obj.getOwner();
				}
				else
				{
					selection = obj.getTarget();
				}
			}
			if (selection instanceof TRCObject) {
				
				TRCObject obj = (TRCObject)selection;
				if(obj.getIsA()==obj.getProcess().getClassClass())
					selection = findClassType(obj);
				else
					selection = obj.getIsA();
			} 
			
			if (selection instanceof TRCClass) {
				
				TRCClass clas = (TRCClass) selection;
				
				String packName = clas.getPackage().getName();
				if(packName.equals(""))
					return clas.getName();
				else			
					return (new StringBuffer()).append(clas.getPackage().getName()).append(".")
					                     .append(clas.getName()).toString();
			} 
		}
		return null;
	}

	private static int getType(Object selection) {
		int javaType;
		
		// Added: Handle the case where the user selects a package, then clicks on the
		// Open Source button on the toolbar - see bugzilla 51854
		if(selection == null || selection instanceof TRCPackage) {
			return IJavaSearchConstants.UNKNOWN;
		}
				
		if(selection instanceof TRCMethodInvocation) {
			selection = ((TRCMethodInvocation)selection).getMethod();
		}
		
	   	if (selection instanceof TRCMethod) {
			TRCMethod trcMethod = (TRCMethod) selection;
			
			if((trcMethod.getModifier() & TRCMethodProperties.JAVA_CONSTRUCTOR) !=0 ) {
				javaType = IJavaSearchConstants.CONSTRUCTOR;
			}
			else {
				javaType = IJavaSearchConstants.METHOD;
			}
		} 
		else {
			javaType = IJavaSearchConstants.TYPE;
		}
		return javaType;
	}

	private static Object findClassType(TRCObject obj) {
		for (Iterator iter1 = obj.getProcess().getPackages().iterator(); iter1.hasNext();) {
			TRCPackage aPackage = (TRCPackage) iter1.next();
			for (Iterator iter2 = aPackage.getClasses().iterator(); iter2.hasNext();) {
				TRCClass aClass = (TRCClass) iter2.next();
				for (Iterator iter3 = aClass.getClassObjects().iterator();iter3.hasNext();) {
					TRCObject aObject = (TRCObject) iter3.next();
					if(aObject == obj)
						return aClass;
				}
			}
		}
		return obj.getIsA();
	}
}
