/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BaseTraceInteractions.java,v 1.7 2005/09/26 12:42:36 sduguet Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.trace.loaders;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.ui.IProfileEventListener;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.views.actions.internal.OpenPatternViewAction;
import org.eclipse.hyades.trace.views.internal.TraceUIMessages;
import org.eclipse.hyades.uml2sd.trace.TraceSDUtil;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.ContextIds;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.DateComparator;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCallStack;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUpdate;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceLifelineDraft;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceProcesses;
import org.eclipse.hyades.uml2sd.trace.selection.IDateSelection;
import org.eclipse.hyades.uml2sd.trace.util.TIUtils;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDExtendedActionBarProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFindProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDGraphNodeSupporter;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.Criteria;
import org.eclipse.hyades.uml2sd.ui.core.BasicExecutionOccurrence;
import org.eclipse.hyades.uml2sd.ui.core.Frame;
import org.eclipse.hyades.uml2sd.ui.core.GraphNode;
import org.eclipse.hyades.uml2sd.ui.load.BackgroundLoader;
import org.eclipse.hyades.uml2sd.ui.view.SDView;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;

/**
 * The abstract trace loader.
 * It is based on the parsing of MethodInvocations from the model.
 * Lifelines attribution is made by the concrete classes.
 */
public abstract class BaseTraceInteractions
	implements ISelectionListener, ISDFindProvider, IProfileEventListener, ISDExtendedActionBarProvider {
	
	public static final String ACTIONS_SHOWSTATTABLE = "org.eclipse.hyades.uml2sd.trace.actions.ShowStatTable"; //$NON-NLS-1$
	/**
	 * With this we know how to implement IUML2SDLoader.setViewer()
	 */
	protected SDView view;
	
	/**
	 * The frame containing the current page
	 */
	protected Frame frame;
	
	protected TraceProcesses currentProcesses;

	/**
	 * Implementation of IUml2SDLoader
	 */
	public void setViewer(SDView view_) {
		if (TraceSDUtil.debugEvents) {
			TraceSDUtil.debugUml2SDTraceEvents("Class "+getClass().getName()+" setViewer()"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		view = view_;
	    //listen to selection from other part.
	    view.getSite().getWorkbenchWindow().getSelectionService().addSelectionListener(this);
	    view.setEnableAction(ACTIONS_SHOWSTATTABLE, true);
		UIPlugin.getDefault().addProfileEventListener(this);		
		view.setSDFindProvider(this);
		view.setSDExtendedActionBarProvider(this);
	    onSetViewer();
	    PlatformUI.getWorkbench().getHelpSystem().setHelp(this.view.getSDWidget(), ContextIds.INTERACTION_VIEW);
		parseModelFromNavigator(false);
	}
	
	protected void parseModelFromNavigator(boolean optional) {
		List list = TIUtils.collectMultiSelectionInNavigator();
		if (optional &&
			currentMofObjectsFromPDProjectExplorer != null &&
			currentMofObjectsFromPDProjectExplorer.size() == list.size() &&
			currentMofObjectsFromPDProjectExplorer.containsAll(list)) {
			return;
		}
		currentMofObjectsFromPDProjectExplorer = list;
		parseModelBackground();
	}

	abstract public void onSetViewer();
	
	public void aboutToBeReplaced() {
		if (view != null) {
			view.setEnableAction(ACTIONS_SHOWSTATTABLE,false);
		}
		UIPlugin.getDefault().removeSelectionListener(this);
		UIPlugin.getDefault().removeProfileEventListener(this);		
		clearInternals();
		view = null;
		if (TraceSDUtil.debugEvents) {
			TraceSDUtil.debugUml2SDTraceEvents("Class "+getClass().getName()+" aboutToBeReplaced()"); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}
	
	/**
	 * Let's share the view...<br>
	 * @return the view or null if the loader is not associated to a view currently
	 */
	public SDView getView() {
		return view;
	}

	protected List currentMofObjectsFromPDProjectExplorer;
	private Cursor waitCursor;
	
	/**
	 * 
	 */
	protected void clearInternals() {
		currentMofObjectsFromPDProjectExplorer = null;
		currentProcesses = null;
		frame = null;
		if (waitCursor != null && !waitCursor.isDisposed()) {
			waitCursor.dispose();
		}
		waitCursor = null;
		clearFindInternals();
	}
	
	/**
	 * 
	 * @todo Generated comment
	 */
	protected void clearFindInternals() {
		findResults = null;
		findCriteria = null;
		currentFindIndex = 0;
	}
	
	abstract protected void parseModel(List list);

	abstract protected void computeModel();
	
	abstract protected void updateSD(TraceInteractionUpdate update);
	
	/**
	 * monitor is the IProgressMonitor to be checked not cancelled during long tasks
	 */
	private IProgressMonitor monitor_;
	
	protected boolean isCurrentLoadCanceled() {
		return monitor_ != null && monitor_.isCanceled();
	}
	
	/**
	 * @param monitor_
	 * @param taskName
	 */
	private void beforeLongTask(IProgressMonitor monitor, String taskName) {
		if (view == null || view.getSDWidget().isDisposed()) {
			monitor.setCanceled(true);
			return;
		}
		monitor_ = monitor;
		view.toggleWaitCursorSync(true);
		monitor.beginTask(taskName, 100/*???*/);
	}
	
	/**
	 * 
	 */
	private void afterLongTask(IProgressMonitor monitor) {
		monitor.done();
		if (view != null && !view.getSDWidget().isDisposed()) {
			view.toggleWaitCursorSync(false);
		}
	}
	
	/**
	 * parseModel() in a background task
	 */
	protected void parseModelBackground() {
		BackgroundLoader.getInstance().newTask(new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InterruptedException {
				beforeLongTask(monitor, TraceUIMessages._148); //$NON-NLS-1$
				if (!isCurrentLoadCanceled()) {
					parseModel(currentMofObjectsFromPDProjectExplorer);
				}
				afterLongTask(monitor);
			}
		});
	}
	
	/**
	 * updateSD() in a background task
	 */
	protected void updateSDBackground(final TraceInteractionUpdate update) {
		BackgroundLoader.getInstance().newTask(new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InterruptedException {
				beforeLongTask(monitor, TraceUIMessages._149); //$NON-NLS-1$
				if (!isCurrentLoadCanceled()) {
					updateSD(update);
				}
				afterLongTask(monitor);
			}
		});
	}
	
	/**
	 * @param eObject
	 * @return
	 */
	protected void createFrame() {
		if (currentProcesses == null && currentMofObjectsFromPDProjectExplorer != null) {
			currentProcesses = new TraceProcesses();
			for (Iterator i = currentMofObjectsFromPDProjectExplorer.iterator();
				 i.hasNext() && ! isCurrentLoadCanceled(); ) {
				TraceProcesses someProcesses = TraceProcesses.getTraceProcesses((EObject)i.next(), monitor_);
				if (someProcesses != null) {
					currentProcesses.addAll(someProcesses);
				}
			}
			if (currentProcesses.size() == 0) {
				currentProcesses = null;
			}
		}
		frame = new Frame();
		if (currentProcesses == null) {
			frame.setName(TraceUIMessages._147); //$NON-NLS-1$
		} else {
			frame.setTimeUnitName(TraceUIMessages._158); //$NON-NLS-1$
		}
	}

	/**
	 * @param process
	 */
	abstract public void setFrameName();

	/**
	 *
	 */
	abstract public void fillPage(TraceInteractionUpdate update);
	
	protected HashMap traceThreadMap;
	protected ArrayList traceThreadList;

	/**
	 * @param eThread
	 * @return
	 */
	protected TraceCallStack getTraceCallStack(TRCThread eThread) {
		TraceCallStack ret = (TraceCallStack)traceThreadMap.get(eThread);
		if (ret == null) {
			ret = new TraceCallStack(this, eThread);
			traceThreadMap.put(eThread, ret);
			traceThreadList.add(ret);
		}
		return ret;
	}
	
	/**
	 * @param tcs
	 */
	public void sortTraceThreadList(TraceCallStack tcs) {
		traceThreadList.remove(tcs);
		for (int i = 0; i < traceThreadList.size(); i++) {
			TraceCallStack tcs2 = (TraceCallStack)traceThreadList.get(i);
			if (tcs.getCurrentExitTime() < tcs2.getCurrentExitTime()) {
				traceThreadList.add(i, tcs);
				return;
			}
		}
		traceThreadList.add(tcs);
	}
	
	/**
	 * @return
	 */
	protected double getTraceThreadListFirstTime() {
		double ret = (traceThreadList != null && traceThreadList.size() > 0) ?
				((TraceCallStack)traceThreadList.get(0)).getCurrentExitTime() :
				Double.MAX_VALUE;
		return ret;
	}
	
	/**
	 * @return
	 */
	protected long getTraceThreadListFirstTicket() {
		long ret = (traceThreadList != null && traceThreadList.size() > 0) ?
				((TraceCallStack)traceThreadList.get(0)).getCurrentTicket() :
				Long.MAX_VALUE;
		return ret;
	}
	
	/**
	 * @return
	 */
	protected TRCThread getTraceThreadListFirstThread() {
		TRCThread ret = (traceThreadList != null && traceThreadList.size() > 0) ?
				((TraceCallStack)traceThreadList.get(0)).getEThread() :
				null;
		return ret;
	}
	
	/**
	 * @return
	 */
	protected short getTraceThreadListFirstStackDepth() {
		short ret = (traceThreadList != null && traceThreadList.size() > 0) ?
				((TraceCallStack)traceThreadList.get(0)).getCurrentStackDepth() :
				Short.MAX_VALUE;
		return ret;
	}
	
	/**
	 * Place to create the execution occurence for the lifeline
	 * @param current
	 */
	protected void setExecutionOccurence(TraceLifelineDraft draft) {
		if (draft != null) {
			if (draft.getStart() == Integer.MAX_VALUE) {
				draft.setStart(0);
			}
			BasicExecutionOccurrence occ = new BasicExecutionOccurrence();
			occ.setStartOccurrence(draft.getStart());
			occ.setEndOccurrence(draft.getEnd());
			draft.getLifeline().addExecution(occ);
		}
	}

	/**
	 * Get the lifeline to be used for this message
	 * @param mi
	 * @return
	 */
	protected abstract EObject getLifelineEObjectFromMethodInvocation(TRCFullMethodInvocation mi);

	/**
	 * Get the name of the class
	 * @param to
	 * @param long_
	 * @return
	 */
	protected abstract String getLifeLineTitle(EObject to, boolean long_);

	/**
	 * Get the category of the lifeline corresponding to this message
	 * @param mi
	 * @return
	 */
	protected abstract int getLifeLineCategory(EObject to);

	
	/**
	 * Implementation of ISelectionListener
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if (TraceSDUtil.debugEvents) {
			TraceSDUtil.debugUml2SDTraceEvents("selectionChanged("+part+")="+selection); //$NON-NLS-1$ //$NON-NLS-2$
		}
	    if( part instanceof PDProjectExplorer ) {
			if (!((PDProjectExplorer)part).isLinkingEnabled()) {
				return;
			}
			if (view != null &&
				view.getSDWidget() != null &&
				!view.getSDWidget().isDisposed()) {
				parseModelFromNavigator(false);
			}
	    } else if (part instanceof SDView) {
    		if (TraceSDUtil.debugEvents) {
    			showInternalSelection(selection);
    		}
	    } else {
			if (frame == null) {
				return;
			}
			TraceInteractionUpdate update;
			update = new TraceInteractionUpdate();
    		externalSelectionChanged(part, selection, update);
			finishExternalExtendedSelection(update);
    		if (update.needsUpdate()) {
        		updateSDBackground(update);
    		}
	    }
	}
	
	/**
	 * Implementation of IProfileEventListener
	 */
	public void handleProfileEvent(ProfileEvent event) {
		if (TraceSDUtil.debugEvents) {
			TraceSDUtil.debugUml2SDTraceEvents("handleProfileEvent("+event+")="/*+event.getSource()*/+" "+event.getType()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		if ((event.getSource() == null || event.getSource() instanceof EObject) &&
			(event.getType() & (ProfileEvent.REFRESH_VIEWS |
								ProfileEvent.STOP_COLLECTING |
								ProfileEvent.UPDATE_MODEL |
								ProfileEvent.CLEAN_UP)) != 0) {
			if (view != null &&
				view.getSDWidget() != null &&
				!view.getSDWidget().isDisposed()) {
				parseModelFromNavigator(event.getType() == ProfileEvent.REFRESH_VIEWS);
			}
		}
	}
	
	/**
	 * @param selection
	 */
	private void showInternalSelection(Object selection) {
		if (TraceSDUtil.debugEvents) {
			TraceSDUtil.debugUml2SDTraceEvents("showInternalSelection("+selection+")"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		// Do nothing
		if (selection instanceof IDateSelection) {
			IDateSelection ds = (IDateSelection)selection;
			if (TraceSDUtil.debugEvents) {
				TraceSDUtil.debugUml2SDTraceEvents(Double.toString(ds.getStartDate()));
			}
		} else if (selection instanceof IStructuredSelection) {
			IStructuredSelection iss = (IStructuredSelection)selection;
			for (Iterator i = iss.iterator(); i.hasNext(); ) {
				Object o = i.next();
				if (TraceSDUtil.debugEvents) {
					TraceSDUtil.debugUml2SDTraceEvents("IStructuredSelection:showInternalSelection("+o+")"); //$NON-NLS-1$ //$NON-NLS-2$
				}
				showInternalSelection(o);
			}
		}
	}

	/**
	 * @param part
	 * @param selection
	 * @param update
	 */
	protected void externalSelectionChanged(IWorkbenchPart part,
										    Object selection,
										    TraceInteractionUpdate update) {
		if (TraceSDUtil.debugEvents) {
			TraceSDUtil.debugUml2SDTraceEvents("External selection from "+part+" is "+selection); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (selection instanceof IStructuredSelection) {
			if (TraceSDUtil.debugEvents) {
				TraceSDUtil.debugUml2SDTraceEvents("IStructuredSelection"); //$NON-NLS-1$
			}
			IStructuredSelection iss = (IStructuredSelection)selection;
			for (Iterator i = iss.iterator(); i.hasNext(); ) {
				Object o = i.next();
				externalSelectionChanged(part, o, update);
			}
    	} else if (externalExtendedSelectionChanged(part, selection, update)) {
    		// done by concrete class
    	}
	}

	abstract protected boolean externalExtendedSelectionChanged(IWorkbenchPart part,
															    Object selection,
															    TraceInteractionUpdate update);

	abstract protected void finishExternalExtendedSelection(TraceInteractionUpdate update);

	/**
	 * Implementation of ISDGraphNodeSupporter
	 */
	public boolean isNodeSupported(int nodeType) {
		switch (nodeType) {
			case ISDGraphNodeSupporter.LIFELINE:
			case ISDGraphNodeSupporter.SYNCMESSAGE:
				return true;
			default:
				return false;
		}
	}

	protected ArrayList findResults;
	protected Criteria findCriteria;
	protected int currentFindIndex;
	
	/**
	 * implementation of ISDFindProvider
	 */
	public boolean find(Criteria toSearch_) {
		if (frame == null) {
			return false;
		}
		if (findResults == null ||
			findCriteria == null ||
			!findCriteria.equals(toSearch_)) {
			findResults = new ArrayList();
			findCriteria = toSearch_;
			// search in the current page:
			if (findCriteria.isLifeLineSelected()) {
				for (int i = 0; i < frame.lifeLinesCount(); i++) {
					if (TIUtils.matchCriteria(frame.getLifeline(i).getName(), findCriteria)) {
						findResults.add(frame.getLifeline(i));
					}
				}
			}
			ArrayList msgs = new ArrayList();
			if (findCriteria.isSyncMessageSelected()) {
				for (int i = 0; i < frame.syncMessageCount(); i++) {
					if (TIUtils.matchCriteria(frame.getSyncMessage(i).getName(), findCriteria)) {
						msgs.add(frame.getSyncMessage(i));
					}
				}
				for (int i = 0; i < frame.syncMessageReturnCount(); i++) {
					if (TIUtils.matchCriteria(frame.getSyncMessageReturn(i).getName(), findCriteria)) {
						msgs.add(frame.getSyncMessageReturn(i));
					}
				}
			}
			if (msgs.size() > 0) {
				Object[] temp = msgs.toArray();
				Arrays.sort(temp, new DateComparator());
				findResults.addAll(Arrays.asList(temp));
			}
			List selection = view.getSDWidget().getSelection();
			if (selection != null && selection.size() == 1) {
				currentFindIndex = findResults.indexOf(selection.get(0))+1;
			} else {
				currentFindIndex = 0;
			}
		} else {
			currentFindIndex++;
		}
		if (findResults.size() > currentFindIndex) {
			GraphNode current = (GraphNode)findResults.get(currentFindIndex);
			view.getSDWidget().moveTo(current);
			return true;			
		}
		
		return notFoundYet(findCriteria);
	}
	
	/**
	 * @see org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFindProvider#cancel()
	 */
	public void cancel() {
		clearFindInternals();
	}
	
	/**
	 * Implementation of ISDExtendedActionBarProvider
	 */
	public void supplementCoolbarContent(IActionBars bar) {
		OpenPatternViewAction openExec = new OpenPatternViewAction();
		openExec.setText(TraceUIMessages._102); //$NON-NLS-1$
		openExec.setToolTipText(TraceUIMessages._102); //$NON-NLS-1$
		openExec.setId(ACTIONS_SHOWSTATTABLE);
		openExec.setImageDescriptor(TraceSDUtil.getResourceImageDescriptor("full/cview16/exec_flow_view.gif")); //$NON-NLS-1$
		bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS", openExec); //$NON-NLS-1$
		bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS", openExec); //$NON-NLS-1$
	}

	/**
	 * Called when find in the current shown graph nodes returned an empty strings.<br>
	 * It's the moment to search in other pages if any.
	 * @return
	 */
	abstract public boolean notFoundYet(Criteria toSearch);

}
