/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ObjectInteractions.java,v 1.7 2005/10/19 11:35:14 sduguet Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.trace.loaders;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCTraceObject;
import org.eclipse.hyades.trace.views.internal.TraceUIMessages;
import org.eclipse.hyades.trace.views.internal.TraceUIPlugin;

/**
 * This is the concrete class for object interaction loader
 */
public class ObjectInteractions extends BaseLifelineLoader {
	
	/**
	 * Constructor
	 */
	public ObjectInteractions() {
		parent = new ClassInteractions();
	}
	
	/**
	 * The lifeline to be used for this message is the owning object
	 */
	public EObject getLifelineEObjectFromMethodInvocation(TRCFullMethodInvocation mi) {
		if (mi.getOwningObject() == null ||
			mi.getOwningObject().getIsA() == mi.getProcess().getClassClass())
			return mi.getMethod().getDefiningClass(); // TRCClass
		return mi.getOwningObject(); // TRCTraceObject
	}

	/**
	 * The name of the lifeline is the name of the class,
	 * prefixed with the id of the owning object if any
	 */
	public String getLifeLineTitle(EObject eo, boolean long_) {
		TRCClass cl;
		String owningObject = ""; //$NON-NLS-1$
		if (eo instanceof TRCTraceObject) {
			TRCTraceObject to = (TRCTraceObject)eo;
			cl = to.getIsA();
			owningObject = ":" + to.getId(); //$NON-NLS-1$
		} else {
			cl = (TRCClass)eo;
		}
		return cl.getName() + owningObject +
			   (!long_ && cl.getPackage() != null ? (" ("+cl.getPackage().getName()+")") : ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}

	/**
	 * Returns the distinguishedtitle of a lifeline
	 * @param eo the EObject denoting this lifeline
	 * @param rank its rank
	 * @return the distinguished title of the lifeline
	 */
	public String getDistinguishedLifeLineTitle(EObject eo, int rank) {
		TRCProcess process;
		if (eo instanceof TRCTraceObject) {
			process = ((TRCTraceObject)eo).getProcess();
		} else {
			process = ((TRCClass)eo).getPackage().getProcess();
		}
		return getLifeLineTitle(eo, true) + " ["+process.getPid()+"]"; //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * The runtimeId of agentProxy and the lifeline title
	 */
	public String getLifeLineId(EObject eo) {
		TRCAgentProxy ap;
		if (eo instanceof TRCTraceObject) {
			ap = ((TRCTraceObject)eo).getProcess().getAgent().getAgentProxy();
		} else {
			ap = ((TRCClass)eo).getPackage().getProcess().getAgent().getAgentProxy();
		}
		return ap.getRuntimeId()+"#"+getLifeLineTitle(eo, true); //$NON-NLS-1$
	}
	
	/**
	 * The category of such a lifeline is:
	 * - the Classes category if and only if the name of the class of the owning object is "class";
	 *   that means that the method is static usually
	 * - the Object category in any other situation
	 */
	public int getLifeLineCategory(EObject eo) {
		if (eo instanceof TRCClass) {
			return TraceInteractions.CATEGORY_CLASSES;
		}
		return TraceInteractions.CATEGORY_OBJECTS;
	}
	
	/**
	 * This is the object interactions, nothing else
	 */
	public String getFinalTitleString() {
		return TraceUIPlugin.getResourceString("41"); //$NON-NLS-1$
	}

	/**
	 * Does not allow drill down
	 */
	public boolean drillDownEnabled(String value) {
		return "false".equalsIgnoreCase(value);  //$NON-NLS-1$
	}

	/**
	 * @see org.eclipse.hyades.uml2sd.trace.loaders.ILifelineLoader#getLifelineType(org.eclipse.emf.ecore.EObject)
	 */
	public String getLifelineType(EObject eObject) {
		if (eObject instanceof TRCClass) {
			return TraceUIMessages._143; //$NON-NLS-1$;
		}
		return TraceUIMessages._144; //$NON-NLS-1$;
	}

	/**
	 * @return a string carrying the lifeline type
	 */
	public String getLifelineTypeName() {
		return TraceUIMessages._145; //$NON-NLS-1$;
	}

}
