/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileOnServerWizardPage.java,v 1.7 2010/08/22 22:10:09 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.profile.server.core.internal;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.ILaunchConfigurationTab;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.ProfileLaunchUtil;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentsTab;
import org.eclipse.hyades.trace.ui.internal.launcher.DestinationTab;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.action.LegacyActionTools;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;

/**
 * This class handles the UI content that appears in the new Attach UI when profiling
 * a WAS runtime.
 *  
 * @author sshiah
 * @author ewchan
 */
public class ProfileOnServerWizardPage extends WizardPage {

	private ILaunchConfigurationTab[] tabList;
	private ILaunchConfigurationWorkingCopy workingCopy;
	private boolean widgetCreated = false; 
	private String message;

	/**
	 * Constructor of the wizard page. Expects the name of this wizard page as well as
	 * a working copy of the ILaunchConfiguration of the server that was just started.
	 * 
	 * @param name name of this wizard page.
	 * @param wc a working copy of the ILaunchConfiguration of the server that was just started.
	 */
	public ProfileOnServerWizardPage(String name, ILaunchConfigurationWorkingCopy workingCopy, String _message) {
		super(name);
        this.workingCopy = workingCopy;
		setTitle(PluginMessages.STR_PROFILE_SERVER_TITLE);
		widgetCreated = false;
		message = _message;
	}

	/**
	 * Creates the control for this wizard page.
	 * 
	 * @param parent parent's composite
	 */
	public void createControl(Composite parent) {

		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		data.minimumWidth = 600;
		data.minimumHeight = 430;
		parent.setLayoutData(data);
		
		// Create a new composite.
		Composite content = new Composite(parent, SWT.NONE);
		content.setLayout(new GridLayout());
		
		List list = new ArrayList();

		list.add(new AttachAgentsTab() {
			protected void updateLaunchConfigurationDialog() {
				updateDialog();
			}
		});
		
		list.add(new DestinationTab() {
			protected void updateLaunchConfigurationDialog() {
				updateDialog();
			}
		});
		
		tabList = new ILaunchConfigurationTab[list.size()];
		list.toArray(tabList);		

		// Create the tab folder which contains the tabs.
		TabFolder folder = new TabFolder(content, SWT.TOP);
		data = new GridData(SWT.FILL, SWT.FILL, true, true);
		data.grabExcessHorizontalSpace = true;
		data.grabExcessVerticalSpace = true;
		data.minimumWidth = 450;
		folder.setLayoutData(data);

		// Iterate through the list of tabs that we instantiated. For each one...
		// - Create a tab item for it.
		// - Create the control for that tab.
		// - Set the name of the tab item.
		// - Set the little icon that will appear next to the tab name/label
		// - Set the control of the tab.
		for(int i = 0; i < tabList.length; i ++) {
			TabItem tab = new TabItem(folder, SWT.TOP);
			tabList[i].createControl(folder);
			tab.setText(tabList[i].getName());
			tab.setImage(tabList[i].getImage());
			tab.setControl(tabList[i].getControl());
		}
		
		initialize();
		setMessage(message);
		
		widgetCreated = true;

		setImageDescriptor(UIPlugin.getImageDescriptor(PDPluginImages.IMG_UI_WZ_ATTACH));

		setControl(content);
		parent.pack();
	}

	public void initialize() {
		for(int i = 0; i < tabList.length; i ++) {
				tabList[i].initializeFrom(workingCopy);
		}
	}
	
	public boolean finish() {
		for(int i = 0; i < tabList.length; i ++) {
			tabList[i].performApply(workingCopy);
		}
		return true;
	}
	
	/** Locate the attach tab in the tab list */
	private AttachAgentsTab getAttachTab() {
		
		if(tabList != null) {
			for(int x = 0; x < tabList.length; x++) {
				if(tabList[x] instanceof AttachAgentsTab) {
					return (AttachAgentsTab)tabList[x]; 
				}
			}
			
		}
		
		return null;
	}

	public boolean isPageComplete() {
		IStatus status = ProfileLaunchUtil.checkValidity(workingCopy.getOriginal(), ((AttachAgentsTab)getAttachTab()).getAgentsCheckBoxTree(),new ProfilingSetsManagerCopy());
		if (status.getSeverity()==IStatus.ERROR || tabList[1].isValid(workingCopy)==false) {
			return false;
		}
		return true;
	}
	
	/**
	 * Updates all of the UI in the dialog:
	 * - updates the Finish/Next/Back/Cancel buttons to reflect the current status
	 *   of the UI
	 * - updates/clears error messages (if any)
	 * - sets the message that should be displayed in the current tab.
	 */
	public void updateDialog() {

		if (!widgetCreated) return;
		
		setPageComplete(isPageComplete());

		String errorMsg = null;

		IStatus status = ProfileLaunchUtil.checkValidity(workingCopy.getOriginal(), ((AttachAgentsTab)getAttachTab()).getAgentsCheckBoxTree(),new ProfilingSetsManagerCopy());
		if (status.getSeverity()==IStatus.ERROR) {
			errorMsg = "[" + LegacyActionTools.removeMnemonics(tabList[0].getName()) + "] " + status.getMessage(); //$NON-NLS-1$ //$NON-NLS-2$
		}
		else if (tabList[1].getErrorMessage()!=null) {
			errorMsg = "[" + LegacyActionTools.removeMnemonics(tabList[1].getName()) + "] " + tabList[1].getErrorMessage(); //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		setErrorMessage(errorMsg);
	}
}
