/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EclipseClass.java,v 1.1 2005/12/19 20:30:10 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.profile.server.core.internal.reflect;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;

import org.eclipse.core.runtime.Platform;

/**
 * This utility class is capable of finding arbitrary classes in arbitrary Eclipse plug-ins.  It can also instantiate
 * objects and invoke arbitrary methods on these objects and classes.
 * 
 * @author Qiyan Li
 *
 */
public class EclipseClass {

    /**
     * Locates the given class in the given bundle.  If the class is found and can be loaded, the class is returned.
     * Otherwise, a <code>LinkageException</code> is thrown to indicate an unsuccessful class lookup.
     * 
     * @param bundleId The symbolic name of the bundle (e.g., plug-in ID)
     * @param classId The fully-qualified class ID
     * @return the class with the given class ID from the given bundle.
     * @throws LinkageException if the bundle or class is not found or if the class is not loadable.
     */
    public static Class findClass(String bundleId, String classId) throws LinkageException {
        try {
            return Platform.getBundle(bundleId).loadClass(classId);
        } catch (Throwable e) {
            throw new LinkageException();
        }
    }
    
    /**
     * Instantiates an instance of the given class from the given bundle with the constructor having the given
     * signature.  Any failure in this process will cause a <code>LinkageException</code> to be thrown.
     * 
     * @param bundleId The symbolic name of the bundle (e.g., plug-in ID)
     * @param classId The fully-qualified class ID
     * @param signature The signature of the constructor with which an instance will be created
     * @param arguments The parameters for the constructor
     * @return an instance of the given class from the given bundle, created by the given constructor with the given
     *      parameters
     * @throws LinkageException if an error has occurred in the process.
     */
    public static Object newInstance(String bundleId, String classId, Class[] signature, Object[] arguments) throws LinkageException {
        Class foreignClass = findClass(bundleId, classId);
        try {
            Constructor constructor = foreignClass.getConstructor(signature);
            constructor.setAccessible(true);
            return constructor.newInstance(arguments);
        } catch (Throwable e) {
            throw new LinkageException();
        }
    }

    /**
     * Invokes the method identified by the given method ID and signature on the given object using the given
     * parameters.  Any failure in this process will cause a <code>LinkageException</code> to be thrown.
     * 
     * @param object The object on which the method invocation should be made
     * @param methodId The name of the method to be called
     * @param signature The signature of the method to be called
     * @param arguments The parameters to be passed into the method
     * @return any return value from the method invocation.
     * @throws LinkageException if an error has occurred in the process.
     */
    public static Object invoke(Object object, String methodId, Class[] signature, Object[] arguments) throws LinkageException {
        try {
            Method method = object.getClass().getMethod(methodId, signature);
            method.setAccessible(true);
            return method.invoke(object, arguments);
        } catch (Throwable e) {
            throw new LinkageException();
        }
    }

    /**
     * Invokes the static method identified by the given method ID and signature on the given class using the given
     * parameters.  Any failure in this process will cause a <code>LinkageException</code> to be thrown.
     * 
     * @param object The class on which the method invocation should be made
     * @param methodId The name of the method to be called
     * @param signature The signature of the method to be called
     * @param arguments The parameters to be passed into the method
     * @return any return value from the method invocation.
     * @throws LinkageException if an error has occurred in the process.
     */
    public static Object invokeStatic(Class clazz, String methodId, Class[] signature, Object[] arguments) throws LinkageException {
        try {
            Method method = clazz.getMethod(methodId, signature);
            method.setAccessible(true);
            return method.invoke(clazz, arguments);
        } catch (Throwable e) {
            throw new LinkageException();
        }
    }
}
