/**********************************************************************
 * Copyright (c) 2007, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.integration.pde.internal.util;

import java.util.ArrayList;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.TreeSet;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.trace.ui.internal.util.WorkspaceFiltersExtractor;
import org.eclipse.jdt.launching.IVMRunner;
import org.eclipse.jdt.launching.VMRunnerConfiguration;
import org.eclipse.osgi.service.resolver.BundleDescription;
import org.eclipse.pde.core.plugin.IPluginModelBase;
import org.eclipse.pde.core.plugin.PluginRegistry;
import org.eclipse.pde.ui.launcher.IPDELauncherConstants;
import org.eclipse.pde.ui.launcher.JUnitLaunchConfigurationDelegate;
import org.eclipse.ui.IWorkingSet;

/**
 * A utility class that is primarily used for launching
 * 
 * @author Ali Mehregani
 */
public class PDEIntegrationLauncherUtility
{
	/*
	 * All funcs below are copied from:
	 *    package org.eclipse.pde.internal.ui.launcher
	 *    class LaunchPluginValidator
	 */

	private static void addToMap(Map<String, IPluginModelBase> map, IPluginModelBase[] models) {
		for (int i = 0; i < models.length; i++) {
			addToMap(map, models[i]);
		}
	}

	private static void addToMap(Map<String, IPluginModelBase> map, IPluginModelBase model) {
		BundleDescription desc = model.getBundleDescription();
		if (desc != null) {
			String id = desc.getSymbolicName();
			if (!map.containsKey(id)) {
				map.put(id, model);						
			} else {
				map.put(id + "_" + desc.getBundleId(), model); //$NON-NLS-1$
			}
		}
	}

	private static IPluginModelBase[] getSelectedWorkspacePlugins(ILaunchConfiguration configuration)
			throws CoreException {
		boolean usedefault = configuration.getAttribute(IPDELauncherConstants.USE_DEFAULT, true);
		boolean useFeatures = configuration.getAttribute(IPDELauncherConstants.USEFEATURES, false);
		IPluginModelBase[] models = PluginRegistry.getWorkspaceModels();
		if (usedefault || useFeatures || models.length == 0)
			return models;
		ArrayList<IPluginModelBase> list = new ArrayList<IPluginModelBase>();
		if (configuration.getAttribute(IPDELauncherConstants.AUTOMATIC_ADD, true)) {
			TreeSet<String> deselected = parsePlugins(configuration, IPDELauncherConstants.DESELECTED_WORKSPACE_PLUGINS);
			if (deselected.size() == 0)
				return models;
			for (int i = 0; i < models.length; i++) {
				String id = models[i].getPluginBase().getId();
				if (id != null && !deselected.contains(id)) 
					list.add(models[i]);
			}		
		} else {
			TreeSet<String> selected = parsePlugins(configuration, IPDELauncherConstants.SELECTED_WORKSPACE_PLUGINS);
			for (int i = 0; i < models.length; i++) {
				String id = models[i].getPluginBase().getId();
				if (id != null && selected.contains(id)) 
					list.add(models[i]);
			}
		}
		return list.toArray(new IPluginModelBase[list.size()]);
	}

	public static TreeSet<String> parsePlugins(ILaunchConfiguration configuration, String attribute)
			throws CoreException {
		TreeSet<String> set = new TreeSet<String>();
		String ids = configuration.getAttribute(attribute, (String) null);
		if (ids != null) {
			StringTokenizer tok = new StringTokenizer(ids, ","); //$NON-NLS-1$
			while (tok.hasMoreTokens())
				set.add(tok.nextToken());
		}
		return set;
	}

	public static IPluginModelBase[] getPluginList(ILaunchConfiguration config) throws CoreException {
		Map<String, IPluginModelBase> map = getPluginsToRun(config);
		return map.values().toArray(new IPluginModelBase[map.size()]);
	}

	public static Map<String, IPluginModelBase> getPluginsToRun(ILaunchConfiguration config)
			throws CoreException {
		//checkBackwardCompatibility(config, true); <-- is it needed here?
		TreeMap<String, IPluginModelBase> map = new TreeMap<String, IPluginModelBase>();
		if (config.getAttribute(IPDELauncherConstants.USE_DEFAULT, true)) {
			addToMap(map, PluginRegistry.getActiveModels());
			return map;
		}
		if (config.getAttribute(IPDELauncherConstants.USEFEATURES, false)) {
			addToMap(map, PluginRegistry.getWorkspaceModels());
			return map;
		}
		addToMap(map, getSelectedWorkspacePlugins(config));
		Set<String> exModels = parsePlugins(config, IPDELauncherConstants.SELECTED_TARGET_PLUGINS);
		IPluginModelBase[] exmodels = PluginRegistry.getExternalModels();
		for (int i = 0; i < exmodels.length; i++) {
			String id = exmodels[i].getPluginBase().getId();
			if (id != null && exModels.contains(id)) {
				IPluginModelBase existing = map.get(id);
				// only allow dups if plug-in existing in map is not a workspace plug-in
				if (existing == null || existing.getUnderlyingResource() == null)
					addToMap(map, exmodels[i]);
			}
		}
		return map;
	}

	public static ArrayList getWorkspacePluginFilters(final ILaunchConfiguration config, final IWorkingSet[] workingSets) throws CoreException {
		ArrayList result = new ArrayList(); 
		IPluginModelBase[] plugins = getPluginList(config);
		for (int i = 0; i < plugins.length; i++) {
			IResource res = plugins[i].getUnderlyingResource();
			if (res != null) {
				ArrayList filters = WorkspaceFiltersExtractor.extractFilters(res.getProject(), workingSets);
				result.addAll(filters);
			}
		}
		return result;
	}

	/**
	 * Used to extract attributes from configuration associated with
	 * a JUnit plug-in launch item
	 * 
	 * @author Ali Mehregani
	 */
	public static class JUnitPluginConfigurationExtended extends JUnitLaunchConfigurationDelegate
	{
		private IVMRunner runner;
		private VMRunnerConfiguration configuration;
		
		public IVMRunner getVMRunner(ILaunchConfiguration configuration, String mode) throws CoreException
		{
			if (runner == null)
			{
				runner = new IVMRunner()
				{
					public void run(VMRunnerConfiguration configuration, ILaunch launch, IProgressMonitor monitor) throws CoreException
					{
						JUnitPluginConfigurationExtended.this.configuration = configuration;
					}
				};
			}
			
			return runner;
		}		
		
		
		public VMRunnerConfiguration getVMRunnerConfiguration()
		{
			return configuration;
		}
	}
}
