/********************************************************************** 
 * Copyright (c) 2005, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ProbeBuilder.java,v 1.2 2006/02/06 20:18:16 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.tptp.platform.probekit.builder;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.internal.probekit.DocumentRoot;
import org.eclipse.hyades.models.internal.probekit.Probe;
import org.eclipse.hyades.models.internal.probekit.Probekit;
import org.eclipse.hyades.probekit.IProbeCompiler;
import org.eclipse.hyades.probekit.ProbekitCompileProblemException;
import org.eclipse.hyades.probekit.ProbekitException;
import org.eclipse.hyades.probekit.CompilerFactory;
import org.eclipse.hyades.probekit.ProbekitPlugin;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistry;
import org.eclipse.tptp.platform.probekit.util.ProbeResourceBundle;
import org.eclipse.tptp.platform.probekit.util.ProbekitConstants;


/**
 * Builder class for probes. 
 * Runs before the Java compiler to turn probe source into Java.
 */
public class ProbeBuilder extends IncrementalProjectBuilder {
	/**
	 * This variable controls behavior I can't turn on yet.
	 * Maybe the bundler is better-enough now that we can enable it.
	 */
	static final boolean put_probes_in_package_not_yet = false;
	
	static final ProbeRegistry registry = ProbeRegistry.getRegistry();
	
	/* (non-Javadoc)
	 * @see org.eclipse.core.internal.events.InternalBuilder#build(int, java.util.Map, org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor)
			throws CoreException {
		if (kind == IncrementalProjectBuilder.FULL_BUILD) {
			BuilderUtils.trace("--- builder FULL BUILD ---");	//$NON-NLS-1$
			getProject().accept(new ProbeBuilderVisitor(monitor));
		}
		else {
			// Incremental build
			IResourceDelta delta = getDelta(getProject());
			if (delta == null) {
				// Run the visitor as a visitor, not a delta visitor.
				BuilderUtils.trace("--- builder INCREMENTAL BUILD, no delta ---");	//$NON-NLS-1$
				getProject().accept(new ProbeBuilderVisitor(monitor));
			}
			else {
				// Run the visitor as a delta visitor
				BuilderUtils.trace("--- builder INCREMENTAL BUILD, has delta ---");	//$NON-NLS-1$
				delta.accept(new ProbeBuilderVisitor(monitor));
			}
		}
		BuilderUtils.trace("---builder end---");
		return null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.core.internal.events.InternalBuilder#clean(org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected void clean(IProgressMonitor monitor) throws CoreException 
	{
		BuilderUtils.trace("---builder CLEAN---");
		super.clean(monitor);
		getProject().accept(new CleanVisitor(monitor));
		BuilderUtils.trace("---builder CLEAN end---");
	}


	boolean recursiveMkdir(IContainer baseDir, IPath p) {
		IFolder f = baseDir.getFolder(p);
		if (!f.exists()) {
			if (!recursiveMkdir(baseDir, p.removeLastSegments(1))) {
				return false;
			}
			try {
				f.create(true, true, null);
			} catch (CoreException e) {
				return false;
			}
		}
		return true;
	}

	IFolder getPackageSubdir(IContainer baseDir, String pkgName) {
		// Build a Path from the package name we used
		StringBuffer dirpath = new StringBuffer(pkgName); //$NON-NLS-1$
		for (int i = 0; i < dirpath.length(); i++) {
			if (dirpath.charAt(i) == '.') dirpath.setCharAt(i, '/');
		}
		Path p = new Path(dirpath.toString());
		recursiveMkdir(baseDir, p);
		IFolder folder = baseDir.getFolder(p);
		return folder;
	}

	private static void reportError(String msg, IResource ires) {
		try {
			IMarker m = ires.createMarker(IProbeCompiler.PROBEKIT_PROBLEM_MARKER);
			m.setAttribute(IMarker.MESSAGE, msg);
			m.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
		} catch (CoreException e) {
			// Report this failure to the error log
			IStatus status = new Status(IStatus.ERROR, 
					"org.eclipse.tptp.platform.probekit",  //$NON-NLS-1$ 
					0,
					ProbekitPlugin.getResourceString("Builder.markerCreateError"),  //$NON-NLS-1$				
					null);
			ProbekitPlugin.getDefault().getLog().log(status);
		}
	}

	/**
	 * Remove all markers of the Probekit problem type from the indicated resource.
	 * Called when we add something to the compiler.
	 * @param ires the resource for which to remove all Probekit markers.
	 */
	private static void removeProbekitMarkers(IResource ires) {
		try {
			ires.deleteMarkers(IProbeCompiler.PROBEKIT_PROBLEM_MARKER, true, IResource.DEPTH_ZERO);
		} catch (CoreException e) {
			// Well, we tried...
		}
	}
	
	private static String generateProbeBasename(IProbeCompiler compiler, IResource r)
	{
		String basename = getBaseName(r.getName());
		return compiler.makeValidJavaIdentifier(basename);
	}
	
	private static String generateJavaFileName(IProbeCompiler compiler, IResource r)
	{
		return generateJavaFileName(compiler, generateProbeBasename(compiler, r));
	}
	
	private static String generateJavaFileName(IProbeCompiler compiler, String basename)
	{
		return basename + compiler.getClassSuffix() + ".java";	//$NON-NLS-1$
	}

	/**
	 * Function that actually runs the probe compiler on a resource 
	 * @param r the resource to tell the probe compiler to generate Java source from
	 */
	private void compileProbeResource(IResource r, IProgressMonitor monitor) {
		String src;
		String eng;
		String suffix;
		String mangled_basename;
		IProbeCompiler pc = CompilerFactory.INSTANCE.createCompiler();
		
		try {
			registry.remove(r);
			removeProbekitMarkers(r);
			mangled_basename = generateProbeBasename(pc, r);
			pc.setClassPrefix(mangled_basename);
			// TODO: get the ID from the probe and use it
			// as the basis for a package name for the probe class.
			// But go on using the source file name as the basis for
			// the probe class name.
			pc.addIFile((IFile)r);
			
			if (put_probes_in_package_not_yet) {
				// This code is not used yet - when we put generated classes in
				// a package, the bundler doesn't work. This is in RATLC as of 4/28/04.
				pc.setPackageName(ProbekitConstants.PROBE_PACKAGE_NAME);
			}

			src = pc.getGeneratedSource();
			eng = pc.getEngineScript();
			suffix = pc.getClassSuffix();
		}
		catch (ProbekitCompileProblemException e) {
			// The resource failed to compile, but any problems have been
			// reported as markers.
			// TODO: remove the probescript file, java files
			return;
		}
		catch (ProbekitException e) {
			// This is worse: something really bad happened and we can't get up
			// TODO: report the problem somehow. 
			reportError(ProbekitPlugin.getResourceString("Builder.compilerError"), r);  //$NON-NLS-1$ 
			IStatus status = new Status(IStatus.ERROR, 
					"org.eclipse.tptp.platform.probekit", 0,  //$NON-NLS-1$
					ProbekitPlugin.getResourceString("Builder.compilerError"), //$NON-NLS-1$ 					
					e);
			ProbekitPlugin.getDefault().getLog().log(status);
			return;
		}

		// Put the generated files in the directory that this resource 
		// is in. When we write the *.java file, it will 
		// automagically be compiled by the JDT build nature.
		// TODO: it is probably rude to put generated resources into the same directory as the source.
		try {
			IFile javaSourceIFile;
			if (put_probes_in_package_not_yet) {
				// This code is not used yet - when we put generated classes in
				// a package, the bundler doesn't work. This is in RATLC as of 4/28/04.
				IFolder f = getPackageSubdir(r.getParent(), ProbekitConstants.PROBE_PACKAGE_NAME);
				javaSourceIFile = f.getFile(generateJavaFileName(pc, mangled_basename));
			}
			else {
				// Only this code is actually used.
				IPath p = new Path(generateJavaFileName(pc, mangled_basename));
				javaSourceIFile = r.getParent().getFile(p); 
			}
			ByteArrayInputStream newJavaContents = new ByteArrayInputStream(src.getBytes());
			// TODO: why do I have to test exists() and use setContents vs. create? 
			// Seems like "force" should handle that, but it doesn't.
			if (javaSourceIFile.exists()) {
				javaSourceIFile.setContents(newJavaContents, true, false, monitor);
			}
			else {
				javaSourceIFile.create(newJavaContents, true, monitor);
			}
			// Tag the resource as "derived."
			javaSourceIFile.setDerived(true);
			
			IFile engineScriptIFile = r.getParent().getFile(new Path(mangled_basename + ProbekitConstants.PROBE_SCRIPT_EXT));
			ByteArrayInputStream newEngineScriptContents = new ByteArrayInputStream(eng.getBytes("UTF-8"));
			if (engineScriptIFile.exists()) {
				engineScriptIFile.setContents(newEngineScriptContents, true, false, monitor);
			}
			else {
				engineScriptIFile.create(newEngineScriptContents, true, monitor);
			}
			// Tag the engine script file as "derived."
			engineScriptIFile.setDerived(true);
			generateProbeInfo(r, mangled_basename);
		}
		catch (Exception e) {
			reportError(ProbekitPlugin.getResourceString("Builder.savingError"), r);  //$NON-NLS-1$
			IStatus status = new Status(IStatus.ERROR, 
					"org.eclipse.tptp.platform.probekit", 0,  //$NON-NLS-1$
					ProbekitPlugin.getResourceString("Builder.savingError"),  //$NON-NLS-1$
					e);
			ProbekitPlugin.getDefault().getLog().log(status);
			return;
		}
	}
	
	/**
	 * Generate a stripped down model file, suitable for export. The probeinfo
	 * file is the original probe model with the probe code fragments stripped
	 * out, persisted as XML.
	 * 
	 * <p>To achieve this, we instantiate the original model, remove the
	 * list of fragments on each Probe element, null out any class scoped
	 * fragment, and save the result to a probeinfo file.</p>
	 * 
	 * @param probeSrc Original .probe file, authored in the workspace
	 * @param basename Basename to use for the generated probeinfo file
	 * @throws IOException
	 * @throws CoreException
	 */
	protected static void generateProbeInfo(IResource probeSrc, String basename) 
		throws IOException, CoreException
	{
		URI fileURI = URI.createFileURI(probeSrc.getLocation().toOSString());
		ResourceSet resourceSet = new ResourceSetImpl();
		Iterator iter = null;
		Probekit probekit = null;
		Resource emfRes;
		IFile probeInfoFile = probeSrc.getParent().getFile(new Path(basename + 
				ProbekitConstants.PROBE_INFO_EXT));
		
		// instantiate the model, associate the new instance with the output 
		// instead of the original (probeSrc) input file. Changing this
		// association causes EMF to automatically persist the modified model
		// to the output file when save() is called, below.
		try {
			// instantiate the model
			emfRes = resourceSet.getResource(fileURI, true);
			// change the file association
			String fname = probeInfoFile.getFullPath().toString();
			emfRes.setURI(URI.createPlatformResourceURI(fname));
		} catch(RuntimeException ex) {
			BuilderUtils.trace("Unable to instantiate model from " + probeSrc);	//$NON-NLS-1$
			throw ex;
		}
		
		iter = emfRes.getContents().iterator();
		while ( iter.hasNext() )
		{
			Object o = iter.next();
			// find the top level Probekit element
			if ( o instanceof DocumentRoot )
			{
				DocumentRoot root = (DocumentRoot)o;
				probekit = root.getProbekit();
				org.eclipse.emf.ecore.resource.Resource rs;
			} else if ( o instanceof Probekit ) {
				probekit = (Probekit)o;
			} 
			// walk the list of probes, stripping out the Fragment lists.
			// A Probekit contains a list of Probe's; each Probe contains
			// a list of Fragment's and possibly a FragmentAtClassScope.
			EList probeList = probekit.getProbe();
			ListIterator liter = probeList.listIterator();
			while ( liter.hasNext() ) {
				Object lo = liter.next();
				if ( lo instanceof Probe ) {
					Probe probe = (Probe)lo;
					EList fragments = probe.getFragment();
					try {
						fragments.clear();
					} catch (UnsupportedOperationException ex) {
						// TODO this operation should fail
						BuilderUtils.trace("Unable to remove code fragments");	//$NON-NLS-1$
					}
					String csFrag = probe.getFragmentAtClassScope();
					if ( csFrag != null && csFrag.length() > 0 ) {
						probe.setFragmentAtClassScope("");
					}
				}
			}
		}
		// save the modified model to the probeinfo file
		emfRes.save(null);
		probeInfoFile.setDerived(true);
	}
	
	/**
	 * Helper function that returns the basename (minus the extension)
	 * for a file name.
	 * @param name The input file name to chop down to its basename. Must not have any path info.
	 * @return the part of the input name before the last period character in it.
	 */
	private static String getBaseName(String name) {
		if (name == null) return null;
		int lastdot = name.lastIndexOf(".");  //$NON-NLS-1$
		if (lastdot == -1) return name;
		return name.substring(0, lastdot);
     }

	/**
	 * Inner class implementing both IResourceDeltaVisitor and IResourceVisitor. 
	 * When the delta visitor hits a file resource, it calls the non-delta visit function.
	 * When the non-delta visit function sees a FILE whose name ends with the probe source suffix,
	 * it calls compileProbeResource (which is in the outer class). 
	 */
	private class ProbeBuilderVisitor implements IResourceDeltaVisitor, IResourceVisitor 
	{
		private IProbeCompiler probeCompiler;
		private IProgressMonitor monitor;
		public ProbeBuilderVisitor(IProgressMonitor m) {
			monitor = m;
			probeCompiler = CompilerFactory.INSTANCE.createCompiler();
		}
		
		public boolean visit(IResource r) {
			if (r.getType() == IResource.FILE && BuilderUtils.isProbeSource(r)) {
				BuilderUtils.trace("Builder visit hit a probe resource. Compiling: " + r.getFullPath());	//$NON-NLS-1$
				compileProbeResource(r, monitor);
			}
			return true;
		}
		
		public boolean visit(IResourceDelta delta) {
			IResource r = delta.getResource();
			int kind = delta.getKind();
			
			BuilderUtils.trace("builder visit delta kind=" + kind + " on resource " + r);	//$NON-NLS-1$
			switch (kind) {
				case IResourceDelta.ADDED :
				case IResourceDelta.CHANGED :
					visit(r);
					break;
				case IResourceDelta.REMOVED :
				{
					// If this is a probe source file, remove the registry entry
					// and the generated java file. Once we remove the java file,
					// the java builder will remove the class files. Very tidy.
					//
					// For foo.probe, the java file is (generally) foo_probe.java.
					// Build this up as a project relative path so we can get the
					// resource, then remove it, if it exists.
					if (r.getType() == IResource.FILE && BuilderUtils.isProbeSource(r)) {
						registry.remove(r);
						// Get the proj relative path, sans the probe src file name
						IPath loc = r.getProjectRelativePath();
						loc = loc.removeLastSegments(1).addTrailingSeparator();
						
						String filename = generateJavaFileName(probeCompiler, r);
						IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
						IResource javaFile = r.getProject().findMember(loc.append(filename));
						
						if ( javaFile != null ) {
							// remove the java file
							try {
								BuilderUtils.trace("Removing " + javaFile);
								javaFile.delete(true, monitor);
							} catch (CoreException e) {
								// well, we tried
							}
						}
					}
					break;
				}
			}
			return true;
		}
	} // class ProbeBuilderVisitor
	
	/**
	 * Resource visitor class for the clean() operation. During a clean, we
	 * remove the generated .java and .probescript files only. We need not
	 * remove the class files because the Java builder will reap them for us.
	 * 
	 * <p>For the same reason, we do not bother to reap the .probe and 
	 * .probescript files in the output directory. They're copied there by the
	 * Java builder and it happily manages cleaning them up, too.</p>
	 * 
	 * <p>Cleaning a .probe source file means removing it from the ProbeRegistry
	 * and removing any (error) markers.</p>
	 * 
	 * <p>Note that this is not also a resource delta visitor. Deltas are not
	 * interesting in this context. We just want to watch for our .probe and
	 * .probescript files to go by.</p>
	 * 
	 * @author kcoleman
	 */
	protected static class CleanVisitor implements IResourceVisitor
	{
		protected IProbeCompiler probeCompiler;
		protected IProgressMonitor monitor;
		
		public CleanVisitor(IProgressMonitor m)
		{
			monitor = m;
		 	probeCompiler = CompilerFactory.INSTANCE.createCompiler();
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.core.resources.IResourceVisitor#visit(org.eclipse.core.resources.IResource)
		 */
		public boolean visit(IResource r) throws CoreException 
		{
			String leafname = r.getName();
			
			if (leafname.endsWith(ProbekitConstants.PROBE_SRC_EXT)) {
				if (BuilderUtils.isInSrcDir(r)) {
					registry.remove(r);
					removeProbekitMarkers(r);
					// Remove properties which connect the source to the
					// generated files.
					ProbeResourceBundle.removeResourceAssociations(r);
				}
				// else it is a copy put in the output dir by the JavaBuilder.
			} else if (r.isDerived()) {
				// We only care about derived files because only such files
				// could have been generated by our build process. We don't
				// want to get carried away and remove user-created copies, etc.
				if (leafname.endsWith(ProbekitConstants.PROBE_SCRIPT_EXT) ||
						leafname.endsWith(ProbekitConstants.PROBE_INFO_EXT)) {
					if ( BuilderUtils.isInSrcDir(r)) {
						r.delete(true, monitor);
					}
					// else, this is a copy of the original src
				} else {
					// Is this the generated .java file? Ignore the class files:
					// they'll be reaped by the java builder.
					String classNameSuffix = probeCompiler.getClassSuffix();
					String javaFileSuffix = classNameSuffix + ".java"; //$NON-NLS-1$
					if (leafname.endsWith(javaFileSuffix)) {
						// TODO do we put markers on the java file?
						r.delete(true, monitor);
					}
				}
			}
			return true;
		}
	}

}
